/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "CoreTools.h"
#include "Equiv.h"
#include "math/Functions.h"
#include "graph/Digraph.h"
#include "graph/DigraphTemplate.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/MergeCopyNumberTwo.h"
#include "paths/MergeTwoGreedy.h"
#include "paths/PairedPair.h"

// PrintFold: output a string, folding at column "maxline".

void PrintFold( ostream& out, const String& s, int& count, const int maxline )
{    if ( count >= maxline )
     {    out << "\n";
          count = 0;    }
     out << s;
     count += s.size( );    }

// An "altgraph", or "alternative graph" is a directed graph with the property that 
// there can only be edges from vertex v to vertex w if w = v + 1, and such that if 
// v and v + 1 are vertices, there is at least one edge from v to v + 1.  Thus an 
// altgraph can be laid out in a line, e.g. as in the following cartoon:
// 0 ------> 1 ===========> 2 ---------> 3 ==========> 4.
// An altgraph could be represented by class digraph<T>, but we have chosen to give
// a separate implementation.
//
// There is an altgraph ostream operator for class pp_read.  Something very
// similar could be implemented for other classes.

template<class T> class altgraph {

     public:

     altgraph( ) { }

     altgraph( const T& t ) // construct single edge graph
     {    edges_.resize(1);
          edges_[0].resize(1);
          edges_[0][0] = t;    }

     altgraph( const vec< vec<T> >& edges ) : edges_(edges) { }

     int Count( ) const { return edges_.size( ); }
     int N( ) const { return edges_.size( ) + 1; }
     int Depth( int i ) const { return edges_[i].size( ); }

     const vec<T>& EdgeSet( int i ) const { return edges_[i]; }

     // Replace: replace the ith edge set by another altgraph.

     void Replace( int i, const altgraph<pp_read>& G )
     {    int oldc = Count( );
          edges_.resize( Count( ) + G.Count( ) - 1 );
          for ( int j = oldc - 1; j > i; j-- )
               edges_[ j + G.Count( ) - 1 ] = edges_[j];
          for ( int j = 0; j < G.Count( ); j++ )
               edges_[ j + i ] = G.edges_[j];    }

     friend ostream& operator<<( ostream& out, const altgraph<pp_read>& G )
     {    int count = 0, maxline = 75;
          for ( int i = 0; i < G.edges_.isize( ); i++ )
          {    if ( i > 0 ) PrintFold( out, ".", count, maxline );
               if ( G.edges_[i].size( ) > 1 ) PrintFold( out, "{", count, maxline );
               for ( int j = 0; j < G.edges_[i].isize( ); j++ )
               {    if ( j > 0 ) PrintFold( out, ",", count, maxline );
                    for ( int k = 0; k < G.edges_[i][j].isize( ); k++ )
                    {    if ( k > 0 ) PrintFold( out, ".", count, maxline );
                         PrintFold( out, BaseAlpha( G.edges_[i][j][k] ), 
                              count, maxline );    }    }
               if ( G.edges_[i].size( ) > 1 ) 
                    PrintFold( out, "}", count, maxline );    }
          return out;    }

     private:

     vec< vec<T> > edges_;

};

// A pp_altgraph_pos gives a position on an altgraph<pp_read>.

class pp_altgraph_pos {

     public:

     pp_altgraph_pos( ) { }
     pp_altgraph_pos( const int stage, const int level, const int pos )
          : stage_(stage), level_(level), pos_(pos) { }

     int Stage( ) const { return stage_; }
     int Level( ) const { return level_; }
     int Pos( ) const { return pos_; }

     private:

     int stage_;
     int level_;
     int pos_;

};

// A placed_pp_altgraph is an altgraph<pp_read>, together with a start position
// start +/- dev, satisfying:
// - no edge has zero length;
// - no more than two parallel edges;
// - two parallel edges must disagree with each other on both ends;
// - no adjacent cases of depth one;
// - parallel edges not at an end must have same length in kmers
// - if there are two parallel edges at an end, the longer one comes first
// - start represents the position at the beginning, measured from the beginning of
//   the longer edge if there are two parallel edges at the beginning.

class placed_pp_altgraph : public altgraph<pp_read> {

     public:

     placed_pp_altgraph( ) { }
     placed_pp_altgraph( const pp_read& r, const double start, const double dev,
          const vec<int>& L )
          : altgraph<pp_read>(r), start_(start), dev_(dev), L_(&L) { }
     placed_pp_altgraph( const vec< vec<pp_read> >& altedges, 
          const double start, const double dev, const vec<int>& L )
          : altgraph<pp_read>(altedges), start_(start), dev_(dev), L_(&L) { }

     double Start( ) const { return start_; }
     double Dev( ) const { return dev_; }
     double Stop( ) const
     {    double stop = Start( );
          for ( int i = 0; i < Count( ); i++ )
               stop += EdgeSet(i)[0].Length(*L_);
          return stop;    }
     const vec<int>& L( ) const { return *L_; }
     int L( int i ) const { return (*L_)[i]; }

     friend ostream& operator<<( ostream& out, const placed_pp_altgraph& G )
     {    out << "["; 
          if ( G.Dev( ) > 0 || G.Start( ) != double(int(floor( G.Start( ) )))  )
          {    RightPrecisionOut( out, G.Start( ), 2 );
               out << " +/- ";
               RightPrecisionOut( out, G.Dev( ), 2 );    }
          else out << int(floor( G.Start( ) ));
          out << "] " << altgraph<pp_read>(G) << " [";
          if ( G.Stop( ) != double(int(floor( G.Stop( ) )))  )
               RightPrecisionOut( out, G.Stop( ), 2 );
          else out << int(floor( G.Stop( ) ));
          return out << "]";    }

     private:

     double start_, dev_;
     const vec<int>* L_;

};

// A cluster represents the result of merging a bunch of pp_reads, whose starting
// positions are known (relative to some fixed point).
//
// ids: indices of the reads (twice the pair id, plus one if it's the right read)
// body: the merger 
// start: the start points of the pp_reads on body
// pos: the predicted distance in kmers between the fixed point and the body
// dev: the deviation value for pos.
//
// Note that a cluster is really a special case of a painted_placed_pp_altgraph,
// defined below.

class cluster {

     public:

     cluster( ) { }
     cluster( const vec<int>& ids, const pp_read& body, const vec<int>& start,
          const double pos, const double dev ) 
          : ids(ids), body(body), start(start), pos(pos), dev(dev) { }

     vec<int> ids;
     pp_read body;
     vec<int> start;
     double pos;
     double dev;

};

// A painted_placed_pp_altgraph is a placed_pp_altgraph that has reads painted
// onto it.  These reads are represented in the data structure by their identifiers
// and start points.  The identifiers are 2*id for the left read of pair id and
// 2*id+1 for its right read.

class painted_placed_pp_altgraph : public placed_pp_altgraph {

     public:

     painted_placed_pp_altgraph( ) { }
     painted_placed_pp_altgraph( const pp_read& r, const double start, 
          const double dev, const vec<int>& L, const vec<int>& read_ids,
          const vec<int>& read_starts )
          : placed_pp_altgraph( r, start, dev, L ), read_ids_(read_ids)
     {    for ( int i = 0; i < read_starts.isize( ); i++ )
          {    read_starts_.push_back( 
                    pp_altgraph_pos( 0, 0, read_starts[i] ) );    }    }
     painted_placed_pp_altgraph( const cluster& c, const vec<int>& L )
          : placed_pp_altgraph( c.body, c.pos, c.dev, L ), 
          read_ids_(c.ids) 
     {    for ( int i = 0; i < c.start.isize( ); i++ )
               read_starts_.push_back( pp_altgraph_pos( 0, 0, c.start[i] ) );    }
     painted_placed_pp_altgraph( const vec< vec<pp_read> >& altedges, 
          const double start, const double dev, const vec<int>& L,
          const vec<int>& read_ids, const vec<pp_altgraph_pos>& read_starts )
          : placed_pp_altgraph( altedges, start, dev, L ), read_ids_(read_ids),
          read_starts_(read_starts) { }

     const vec<int>& ReadIds( ) const { return read_ids_; }
     int ReadIdCount( ) const { return read_ids_.size( ); }
     int ReadId( int i ) const { return read_ids_[i]; }
     const vec<pp_altgraph_pos>& ReadStarts( ) const { return read_starts_; }
     const pp_altgraph_pos& ReadStarts( int i ) const { return read_starts_[i]; }

     friend ostream& operator<<( ostream& out, const painted_placed_pp_altgraph& G )
     {    static vec<int> ids;
          ids = G.read_ids_;
          Sort(ids);
          out << "{";
          int count = 1, maxline = 75;
          for ( int i = 0; i < ids.isize( ); i++ )
          {    if ( i > 0 ) 
               {    out << ",";
                    ++count;    }
               if ( ids[i] % 2 == 0 ) 
               {    static String s;
                    s = ToString( ids[i]/2 ) + "L";
                    if ( i < ids.isize( ) - 1 && ids[i+1] == ids[i] + 1 )
                    {    s += "R";
                         ++i;    }
                    PrintFold( out, s, count, maxline );    }
               else PrintFold( out, ToString( ids[i]/2 ) + "R", count, maxline );   }
          return out << "} -->\n" << placed_pp_altgraph(G);    }

     void PrintStarts( ) const
     {    for ( int i = 0; i < read_ids_.isize( ); i++ )
          {    int id = read_ids_[i];
               cout << id/2 << ( id % 2 == 0 ? "L" : "R" ) << " ";
               const pp_altgraph_pos& p = read_starts_[i];
               PRINT3( p.Stage( ), p.Level( ), p.Pos( ) );    }    }

     private:

     vec<int> read_ids_;
     vec<pp_altgraph_pos> read_starts_;

};

// Merge( G, C ): attempt to merge C into G.  Return False if merger fails.

Bool Merge( painted_placed_pp_altgraph& G, const cluster& C,
     const vec<pp_pair>& ppp, const Bool verbose )
{
     const vec<int>& L = G.L( );
     if (verbose) 
     {    cout << "Entering Merge:\nG = " << G << "\nC = " 
               << painted_placed_pp_altgraph( C, L ) << "\n\n";    }

     // First determine if G and C could be disjoint.

     double dmult = 3.0;
     double G_start = G.Start( ), G_stop = G.Stop( );
     double C_start = C.pos, C_stop = C.pos;
     for ( int i = 0; i < C.body.isize( ); i++ )
          C_stop += L[ C.body[i] ];
     double G_dev = G.Dev( ), C_dev = C.dev;
     double dev = G_dev + C_dev;
     if ( G_stop + dmult * dev <= C_start ) return False;
     if ( C_stop + dmult * dev <= G_start ) return False;

     // Find possible offsets in kmers for an alignment between G and C.

     double start = G_start;
     static vec<int> offsets;
     offsets.clear( );
     int gpos = 0;
     for ( int i = 0; i < G.Count( ); i++ )
     {    const vec<pp_read>& v = G.EdgeSet(i);
          for ( int j = 0; j < v.isize( ); j++ )
          {    const pp_read& r = v[j];
               double gstart = start;
               int Gpos = gpos;
               if ( i == 0 && j == 1 ) Gpos += v[0].Length(L) - v[1].Length(L);
               for ( int k = 0; k < r.isize( ); k++ )
               {    int m = r[k];
                    double cstart = C_start;
                    int Cpos = 0;
                    for ( int u = 0; u < C.body.isize( ); u++ )
                    {    if ( C.body[u] == m )
                         {    if ( Abs( gstart - cstart ) <= dmult * dev )
                                   offsets.push_back( Gpos - Cpos );    }
                         cstart += L[ C.body[u] ];
                         Cpos += L[ C.body[u] ];    }
                    gstart += L[ r[k] ];
                    Gpos += L[ r[k] ];    }    }
          start += v[0].Length(L);
          gpos += v[0].Length(L);    }
     UniqueSort(offsets);

     // Determine if there is a winning offset.

     vec<int> kmers_aligning;
     for ( int i = 0; i < offsets.isize( ); i++ )
     {    int offset = offsets[i], kmer_count = 0, gpos = 0;
          for ( int i = 0; i < G.Count( ); i++ )
          {    const vec<pp_read>& v = G.EdgeSet(i);
               int max_kc = 0;
               for ( int j = 0; j < v.isize( ); j++ )
               {    int kc = 0, Gpos = gpos;
                    if ( i == 0 && j == 1 ) Gpos += v[0].Length(L) - v[1].Length(L);
                    const pp_read& r = v[j];
                    for ( int k = 0; k < r.isize( ); k++ )
                    {    int m = r[k], Cpos = 0;
                         for ( int u = 0; u < C.body.isize( ); u++ )
                         {    if ( C.body[u] == m && offset == Gpos - Cpos ) 
                                   kc += L[m];
                              Cpos += L[ C.body[u] ];    }
                         Gpos += L[ r[k] ];    }
                    max_kc = Max( kc, max_kc );    }
               kmer_count += max_kc;
               gpos += v[0].Length(L);    }
          kmers_aligning.push_back(kmer_count);    }
     ReverseSortSync( kmers_aligning, offsets );
     if (verbose)
     {    cout << "offset   kmers_aligning\n";
          for ( int i = 0; i < offsets.isize( ); i++ )
               cout << offsets[i] << "    " << kmers_aligning[i] << "\n";    }
     if ( offsets.empty( ) ) return False;
     if ( !offsets.solo( ) && kmers_aligning[0] < 10 * kmers_aligning[1] )
          return False;
     int offset = offsets[0];

     // Compute start point g +/- d for merger.

     double g, d;
     if ( !CombineMeasurements( 
          G.Start( ), C.pos - offset, G.Dev( ), C.dev, dmult, g, d ) )
     {    return False;    }
     if ( offset < 0 ) g += offset;

     // Create a merged data structure that consists of vertices V = {(pi,ui)}, 
     // where pi is a coordinate and ui is a unipath id, and edges E = {(i,j)} that 
     // represent adjacencies (pi,ui) --> (pj,uj).

     vec< pair<int,int> > Cv, Ce, Gv, Ge, V, E;
     int cpos = offset;
     for ( int i = 0; i < C.body.isize( ); i++ )
     {    Cv.push_back( make_pair( cpos, C.body[i] ) );
          if ( i < C.body.isize( ) - 1 ) Ce.push_back( make_pair( i, i+1 ) );
          cpos += L[ C.body[i] ];    }
     gpos = 0;
     vec<int> Glastback;
     for ( int i = 0; i < G.Count( ); i++ )
     {    const vec<pp_read>& v = G.EdgeSet(i);
          vec<int> Gfront, Gback;
          for ( int j = 0; j < v.isize( ); j++ )
          {    const pp_read& r = v[j];
               int Gpos = gpos;
               if ( i == 0 && j == 1 ) Gpos += v[0].Length(L) - v[1].Length(L);
               for ( int k = 0; k < r.isize( ); k++ )
               {    if ( k == 0 ) Gfront.push_back( Gv.size( ) );
                    if ( k == r.isize( ) - 1 ) Gback.push_back( Gv.size( ) );
                    if ( k < r.isize( ) - 1 )
                         Ge.push_back( make_pair( Gv.size( ), Gv.size( ) + 1 ) );
                    Gv.push_back( make_pair( Gpos, r[k] ) );
                    Gpos += L[ r[k] ];    }    }
          for ( int j1 = 0; j1 < Glastback.isize( ); j1++ )
          {    for ( int j2 = 0; j2 < Gfront.isize( ); j2++ )
                    Ge.push_back( make_pair( Glastback[j1], Gfront[j2] ) );    }
          Glastback = Gback;
          gpos += v[0].Length(L);    }
     V.SetCat( Cv, Gv );
     E = Ce;
     for ( int i = 0; i < Ge.isize( ); i++ )
     {    E.push_back( make_pair( 
               Ge[i].first + Cv.isize( ), Ge[i].second + Cv.isize( ) ) );    }
     vec<int> Vi( V.size( ), vec<int>::IDENTITY ), Vlow;
     SortSync( V, Vi );
     int count = 0;
     for ( int i = 0; i < V.isize( ); i++ )
     {    if ( i > 0 && V[i] == V[i-1] ) 
          {    Vlow.push_back(count-1);
               continue;    }
          else Vlow.push_back(count);
          if ( i != count ) V[count] = V[i];
          ++count;    }
     V.resize(count);
     vec<int> Vlookup( Vi.size( ) );
     for ( int i = 0; i < Vi.isize( ); i++ )
          Vlookup[ Vi[i] ] = Vlow[i];
     for ( int i = 0; i < E.isize( ); i++ )
          E[i] = make_pair( Vlookup[ E[i].first ], Vlookup[ E[i].second ] );

     // Turn the data structure into a graph and simplify it.

     equiv_rel e( 2 * V.size( ) );
     for ( int i = 0; i < E.isize( ); i++ )
          e.Join( 2 * E[i].first + 1, 2 * E[i].second );
     vec<int> reps;
     e.OrbitRepsAlt(reps);
     int N = reps.size( );
     vec< vec<vrtx_t> > from(N), to(N);
     vec< vec<int> > from_edge_obj(N), to_edge_obj(N);
     vec<int> efrom( V.size( ) ), eto( V.size( ) );
     for ( int i = 0; i < V.isize( ); i++ )
     {    efrom[i] = BinPosition( reps, e.ClassId( 2*i ) );
          eto[i] = BinPosition( reps, e.ClassId( 2*i + 1 ) );    }
     for ( int i = 0; i < V.isize( ); i++ )
     {    from[ efrom[i] ].push_back( eto[i] );
          from_edge_obj[ efrom[i] ].push_back(i);
          to[ eto[i] ].push_back( efrom[i] );
          to_edge_obj[ eto[i] ].push_back(i);    }
     vec<pp_read> edges( V.size( ) );
     for ( int i = 0; i < edges.isize( ); i++ )
     {    pp_read r;
          r.push_back( V[i].second );
          edges[i] = r;    }
     for ( int v = 0; v < N; v++ )
     {    SortSync( from[v], from_edge_obj[v] );
          SortSync( to[v], to_edge_obj[v] );    }
     digraphE<pp_read> X( from, to, edges, to_edge_obj, from_edge_obj );
     X.RemoveUnneededVertices( );
     X.RemoveDeadEdgeObjects( );

     // Test to see if graph is an altgraph, and that parallel edges have the
     // same length.

     vec<vrtx_t> sources, sinks;
     X.Sources(sources), X.Sinks(sinks);
     if ( sources.size( ) != 1 && sources.size( ) != 2 ) return False;
     if ( sinks.size( ) != 1 && sinks.size( ) != 2 ) return False;
     int v, w;
     if ( sources.solo( ) ) v = sources[0];
     else
     {    if ( X.FromSize( sources[0] ) != 1 ) return False;
          if ( X.FromSize( sources[1] ) != 1 ) return False;
          int v1 = X.From( sources[0] )[0], v2 = X.From( sources[1] )[0];
          if ( v1 != v2 ) return False;
          if ( X.ToSize(v1) != 2 ) return False;
          v = v1;    }
     if ( sinks.solo( ) ) w = sinks[0];
     else
     {    if ( X.ToSize( sinks[0] ) != 1 ) return False;
          if ( X.ToSize( sinks[1] ) != 1 ) return False;
          int w1 = X.To( sinks[0] )[0], w2 = X.To( sinks[1] )[0];
          if ( w1 != w2 ) return False;
          if ( X.FromSize(w1) != 2 ) return False;
          w = w1;    }
     while(1)
     {    if ( X.FromSize(v) != 1 && X.FromSize(v) != 2 ) return False;
          int u = X.From(v)[0];
          if ( u == v ) return False;
          if ( X.FromSize(v) == 2 && X.From(v)[1] != u ) return False;
          if ( X.FromSize(v) != X.ToSize(u) ) return False;
          if ( X.FromSize(v) == 2 
               && X.EdgeObjectByIndexFrom( v, 0 ).Length(L)
                    != X.EdgeObjectByIndexFrom( v, 1 ).Length(L) )
          {    return False;    }
          if ( u == w ) break;
          v = u;    }

     // Convert the graph X into an altgraph structure.

     vec< vec<pp_read> > altedges;
     if ( sources.solo( ) ) v = sources[0];
     else
     {    vec<pp_read> front_edges(2);
          front_edges[0] = X.EdgeObjectByIndexFrom( sources[0], 0 );
          front_edges[1] = X.EdgeObjectByIndexFrom( sources[1], 0 );
          if ( front_edges[0].Length(L) < front_edges[1].Length(L) )
               swap( front_edges[0], front_edges[1] );
          altedges.push_back(front_edges);
          v = X.From( sources[0] )[0];    }
     while(1)
     {    int u = X.From(v)[0];
          vec<pp_read> edges;
          for ( int j = 0; j < X.FromSize(v); j++ )
               edges.push_back( X.EdgeObjectByIndexFrom( v, j ) );
          altedges.push_back(edges);
          if ( u == w ) break;
          v = u;    }
     if ( !sinks.solo( ) )
     {    vec<pp_read> back_edges(2);
          back_edges[0] = X.EdgeObjectByIndexTo( sinks[0], 0 );
          back_edges[1] = X.EdgeObjectByIndexTo( sinks[1], 0 );
          if ( back_edges[0].Length(L) < back_edges[1].Length(L) )
               swap( back_edges[0], back_edges[1] );
          altedges.push_back(back_edges);    }

     // Get the read placements.  To do this we first compute the read starts in
     // kmer space, relative to the start of G (read_starts_k).

     vec<int> read_ids, read_starts_k;
     read_ids.SetCat( G.ReadIds( ), C.ids );
     vec<int> G_start_edges( G.Count( ) );
     G_start_edges[0] = 0;
     for ( int i = 1; i < G.Count( ); i++ )
          G_start_edges[i] = G_start_edges[i-1] + G.EdgeSet(i-1)[0].Length(L);
     for ( int i = 0; i < G.ReadStarts( ).isize( ); i++ )
     {    const pp_altgraph_pos& p = G.ReadStarts(i);
          int stage = p.Stage( ), level = p.Level( ), pos = p.Pos( );
          int startk = G_start_edges[stage];
          if ( stage == 0 && level == 1 )
          {    startk += G.EdgeSet(stage)[0].Length(L) 
                    - G.EdgeSet(stage)[1].Length(L);    }
          for ( int j = 0; j < pos; j++ )
               startk += L[ G.EdgeSet(stage)[level][j] ];
          read_starts_k.push_back(startk);    }
     for ( int i = 0; i < C.start.isize( ); i++ )
     {    int startk = offset;
          for ( int j = 0; j < C.start[i]; j++ )
               startk += L[ C.body[j] ];
          read_starts_k.push_back(startk);    }
     vec<int> start_edges( altedges.size( ) + 1 );
     start_edges[0] = 0;
     for ( int i = 1; i <= altedges.isize( ); i++ )
          start_edges[i] = start_edges[i-1] + altedges[i-1][0].Length(L);
     vec<pp_altgraph_pos> read_starts;
     for ( int i = 0; i < read_starts_k.isize( ); i++ )
     {    int x = read_starts_k[i], id = read_ids[i];
          if ( offset < 0 ) x -= offset;
          const pp_read& r 
               = ( id % 2 == 0 ?  ppp[id/2].Left( ) : ppp[id/2].Right( ) );
          int stage, level, pos = -1;
          for ( stage = 0; stage < altedges.isize( ); stage++ )
               if ( x < start_edges[stage+1] ) break;
          ForceAssertLt( stage, altedges.isize( ) );
          for ( level = 0; level < altedges[stage].isize( ); level++ )
          {    int x0 = start_edges[stage];
               if ( stage == 0 && level == 1 )
               {    x0 += altedges[stage][0].Length(L) 
                         - altedges[stage][1].Length(L);    }
               for ( pos = 0; pos < altedges[stage][level].isize( ); pos++ )
               {    if ( x0 == x && r[0] == altedges[stage][level][pos] )
                         goto found_stage_level_pos;
                    x0 += L[ altedges[stage][level][pos] ];    }    }
          ForceAssert( 0 == 1 );
          found_stage_level_pos:
          read_starts.push_back( pp_altgraph_pos( stage, level, pos ) );    }
     painted_placed_pp_altgraph A( altedges, g, d, L, read_ids, read_starts );
     if (verbose) cout << "Merged graph:\n" << A << "\n";
     G = A;
     return True;    }

void MergeClusterGreedy( const vec< pair< vec<int>, int > >& pairs_edge, 
     const HyperKmerPath& h, const KmerBaseBroker& kbb, int u, 
     vec<pp_pair>& ppp, vec<int>& p0, const vec<int>& L, const digraphE<int>& G, 
     const vec<vrtx_t>& to_left_vertex, const vec<vrtx_t>& to_right_vertex, 
     const vec<int>& edge_copyno, const Bool stop_at_branch, 
     vec<Bool>& remove, const Bool verbose, 
     // return values:
     vec<painted_placed_pp_altgraph>& mclusters,
     vec< vec<int> >& morbits, 
     vec< pair< int, vec<int> > >& signature,
     vec<int>& ids_used )
{
     // Compute gaps between u (on left edge) and right edge.
     // Then reorder using the gaps.

     vec<double> left_pos, gaps;
     for ( int i = 0; i < p0.isize( ); i++ )
     {    const pp_pair& p = ppp[ p0[i] ];
          double left = 0.0, gap = p.Gap( );
          int j;
          for ( j = 0; j < p.LeftSize( ); j++ )
          {    left -= L[ p.Left(j) ];
               if ( p.Left(j) == u ) break;    }
          left_pos.push_back(left);
          for ( int k = j + 1; k < p.LeftSize( ); k++ )
               gap += L[ p.Left(k) ];
          gaps.push_back(gap);    }
     SortSync( gaps, p0, left_pos );

     // Define initial clusters.

     vec<cluster> left_clusters, right_clusters;
     for ( int i = 0; i < p0.isize( ); i++ )
     {    static vec<int> ids(1), start(1);
          int id = p0[i];
          ids[0] = 2*id;
          start[0] = 0;
          left_clusters.push_back( cluster( ids, ppp[id].Left( ), start, 
               left_pos[i], 0.0 ) );
          ++ids[0];
          right_clusters.push_back( cluster( ids, ppp[id].Right( ), start, 
               gaps[i], ppp[id].Dev( ) ) );    }

     // If the left read of a pair overlaps a left read of one of the given pairs
     // along a copy-number-two point, add it in.

     vec<int> us;
     for ( int i = 0; i < p0.isize( ); i++ )
     {    const pp_read& l = ppp[ p0[i] ].Left( );
          for ( int j = 0; j < l.isize( ); j++ )
          {    if ( l[j] != u && edge_copyno[ l[j] ] == 2 ) 
                    us.push_back( l[j] );    }    }
     UniqueSort(us);
     vec<Bool> used( ppp.size( ), False );
     for ( int i = 0; i < p0.isize( ); i++ )
          used[ p0[i] ] = True;
     for ( int m = 0; m < pairs_edge.isize( ); m++ )
     {    int u2 = pairs_edge[m].second;
          if ( !BinMember( us, u2 ) ) continue;
          const vec<int>& v = pairs_edge[m].first;
          for ( int j = 0; j < v.isize( ); j++ )
          {    int id = v[j];
               if ( used[id] ) continue;
               used[id] = True;
               int w = Position( ppp[id].Left( ), u2 );
               for ( int i = 0; i < p0.isize( ); i++ )
               {    const pp_read& l = ppp[ p0[i] ].Left( );
                    int u_pos = Position( l, u ), u2_pos = Position( l, u2 );
                    if ( u2_pos < 0 ) continue;
                    static vec<int> ids(1), start(1);
                    ids[0] = 2*id;
                    start[0] = 0;
                    double left_pos = left_clusters[i].pos;
                    for ( int q = 0; q < u2_pos; q++ )
                         left_pos += L[ l[q] ];
                    for ( int q = 0; q < w; q++ )
                         left_pos -= L[ ppp[id].Left(q) ];
                    left_clusters.push_back( cluster( ids, ppp[id].Left( ), 
                         start, left_pos, 0.0 ) );
                    ++ids[0];
                    double right_pos 
                         = left_pos + ppp[id].Left( ).Length(L) + ppp[id].Gap( );
                    right_clusters.push_back( cluster( ids, ppp[id].Right( ),
                         start, right_pos, ppp[id].Dev( ) ) );
                    break;    }    }    }

     // Try to merge left clusters.

     vec<Bool> dead( left_clusters.size( ), False );
     while(1)
     {    Bool progress = False;
          for ( int i1 = 0; i1 < left_clusters.isize( ); i1++ )
          {    for ( int i2 = i1 + 1; i2 < left_clusters.isize( ); i2++ )
               {    if ( dead[i1] || dead[i2] ) continue;
                    const cluster &c1 = left_clusters[i1], &c2 = left_clusters[i2];
                    int offset = 0;
                    double delta;
                    if ( c1.pos <= c2.pos )
                    {    delta = c2.pos - c1.pos;
                         for ( int j = 0; j < c1.body.isize( ); j++ )
                         {    if ( delta <= 0.0 ) break;
                              delta -= L[ c1.body[j] ];
                              offset++;    }    }
                    else
                    {    delta = c1.pos - c2.pos;
                         for ( int j = 0; j < c2.body.isize( ); j++ )
                         {    if ( delta <= 0.0 ) break;
                              delta -= L[ c2.body[j] ];
                              offset--;    }    }
                    if ( offset <= - c2.body.isize( ) ) continue;
                    if ( offset >= c1.body.isize( ) ) continue;
                    if ( delta != 0.0 ) continue;
                    const pp_read &r1 = c1.body, &r2 = c2.body;
                    int start1 = Max( 0, offset );
                    int stop1 = Min( r1.isize( ), r2.isize( ) + offset );
                    int j;
                    for ( j = start1; j < stop1; j++ )
                         if ( r1[j] != r2[j-offset] ) break;
                    if ( j < stop1 ) continue;
                    static vec<int> ids, start;
                    ids.SetCat( c1.ids, c2.ids );
                    static pp_read c;
                    JoinReads( c1.body, c2.body, offset, c );
                    start = c1.start;
                    start.append( c2.start );
                    double g = ( offset >= 0 ? c1.pos : c2.pos );
                    if ( offset >= 0 )
                    {    for ( int j = c1.start.isize( ); j < start.isize( ); j++ )
                              start[j] += offset;    }
                    else
                    {    for ( int j = 0; j < c1.start.isize( ); j++ )
                              start[j] -= offset;    }
                    left_clusters.push_back( cluster( ids, c, start, g, 0.0 ) );
                    dead[i1] = dead[i2] = True;
                    dead.push_back(False);
                    progress = True;
                    break;    }    }
          if ( !progress ) break;    }
     EraseIf( left_clusters, dead );

     // Try to merge right clusters.

     double dmult = 3.0;
     dead.resize_and_set( right_clusters.size( ), False );
     while(1)
     {    Bool progress = False;
          for ( int i1 = 0; i1 < right_clusters.isize( ); i1++ )
          {    for ( int i2 = i1 + 1; i2 < right_clusters.isize( ); i2++ )
               {    if ( dead[i1] || dead[i2] ) continue;
                    const cluster &c1 = right_clusters[i1], &c2 = right_clusters[i2];
                    if ( MaybeOneBeforeTwo( c1.body, c1.pos, c1.dev, c2.body,
                         c2.pos, c2.dev, dmult, L, G, to_left_vertex,
                         to_right_vertex ) )
                    {    continue;    }
                    if ( MaybeOneBeforeTwo( c2.body, c2.pos, c2.dev, c1.body,
                         c1.pos, c1.dev, dmult, L, G, to_left_vertex,
                         to_right_vertex ) )
                    {    continue;    }
                    static vec<pp_read> c;
                    static vec<double> g, d;
                    static vec<int> offsets;
                    FindJoins( c1.body, c1.pos, c1.dev, c2.body, c2.pos, c2.dev,
                         dmult, L, c, g, d, offsets );
                    if ( !c.solo( ) ) continue;
                    static vec<int> ids, start;
                    ids.SetCat( c1.ids, c2.ids );
                    start = c1.start;
                    start.append( c2.start );
                    if ( offsets[0] >= 0 )
                    {    for ( int j = c1.start.isize( ); j < start.isize( ); j++ )
                              start[j] += offsets[0];    }
                    else
                    {    for ( int j = 0; j < c1.start.isize( ); j++ )
                              start[j] -= offsets[0];    }
                    right_clusters.push_back( 
                         cluster( ids, c[0], start, g[0], d[0] ) );
                    dead[i1] = dead[i2] = True;
                    dead.push_back(False);
                    progress = True;
                    break;    }    }
          if ( !progress ) break;    }
     EraseIf( right_clusters, dead );

     // Announce results.

     if (verbose) 
          cout << "\nResults of merging cluster at " << BaseAlpha(u) << ":\n\n";
     int count = 0;
     Bool duplicate = False;
     for ( int pass = 1; pass <= 2; pass++ )
     {    const vec<cluster>& clusters 
               = ( pass == 1 ? left_clusters : right_clusters );
          for ( int i = 0; i < clusters.isize( ); i++ )
          {    const cluster& c = clusters[i];
               if ( c.body.CountValue(u) > 1 ) duplicate = True;
               // cout << count++ << ". " // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
               //      << painted_placed_pp_altgraph( c, L ) << "\n"; // XXXXXXXXXXX
               count++;    }    }

     vec<cluster> clusters;
     clusters.SetCat( left_clusters, right_clusters );
     vec<Bool> cused( clusters.size( ), False );
     int mcount = 1;
     mclusters.clear( );
     for ( int i1 = 0; i1 < clusters.isize( ); i1++ )
     {    if ( cused[i1] ) continue;
          painted_placed_pp_altgraph A( clusters[i1], L );
          cused[i1] = True;
          Bool merge_verbose = False;
          // merge_verbose = True; // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
          while(1)
          {    Bool progress = False;
               for ( int i2 = 0; i2 < clusters.isize( ); i2++ )
               {    if ( !cused[i2] && Merge( A, clusters[i2], ppp, merge_verbose ) )
                    {    progress = True;
                         cused[i2] = True;    }    }
               if ( !progress ) break;    }
          if (verbose) cout << mcount++ << ". " << A << "\n";
          mclusters.push_back(A);    }
     if ( verbose && duplicate ) 
          cout << "Warning: " << BaseAlpha(u) << " is duplicated.\n";

     // Determine signature of each read.

     ids_used.clear( );
     for ( int i = 0; i < used.isize( ); i++ )
          if ( used[i] ) ids_used.push_back( 2*i, 2*i + 1 );
     signature.resize( ids_used.size( ) );
     for ( int i = 0; i < mclusters.isize( ); i++ )
     {    const painted_placed_pp_altgraph& A = mclusters[i];
          for ( int j = 0; j < A.ReadIds( ).isize( ); j++ )
          {    int id = A.ReadIds( )[j];
               const pp_altgraph_pos& p = A.ReadStarts(j);
               const pp_read& r 
                    = ( id % 2 == 0 ? ppp[id/2].Left( ) : ppp[id/2].Right( ) );
               vec<int> sig( A.Count( ), -1 );
               int stage = p.Stage( ), level = p.Level( ), pos = p.Pos( );
               sig[stage] = level;
               int rpos = A.EdgeSet(stage)[level].isize( ) - pos;
               for ( stage++; stage < A.Count( ); stage++ )
               {    for ( level = 0; level < A.Depth(stage); level++ )
                    {    if ( rpos >= r.isize( ) ) goto end_of_read;
                         if ( r[rpos] == A.EdgeSet(stage)[level][0] ) 
                         {    rpos += A.EdgeSet(stage)[level].isize( );
                              if ( rpos >= r.isize( ) ) 
                              {    sig[stage] = level;
                                   goto end_of_read;    }
                              break;    }    }
                    ForceAssertLt( level, A.Depth(stage) );
                    sig[stage] = level;    }
               end_of_read:
               signature[ BinPosition( ids_used, id ) ]
                    = make_pair( i, sig );    }    }
     if (verbose)
     {    cout << "\nRead signatures:\n";
          for ( int i = 0; i < signature.isize( ); i++ )
          {    int id = ids_used[i];
               cout << id/2 << ( id % 2 == 0 ? "L" : "R" ) << " ";
               int m = signature[i].first;
               cout << m+1 << ".";
               for ( int j = 0; j < signature[i].second.isize( ); j++ )
               {    int s = signature[i].second[j];
                    if ( s == -1 ) cout << "-";
                    else if ( mclusters[m].Depth(j) == 1 ) cout << "*";
                    else cout << s + 1;    }
               cout << "\n";    }    }

     // Classify pairs by signature.

     equiv_rel esig( ids_used.size( )/2 );
     for ( int i1 = 0; i1 < ids_used.isize( ); i1 += 2 )
     {    for ( int i2 = i1+2; i2 < ids_used.isize( ); i2 += 2 )
          {    if ( signature[i1] == signature[i2]
                    && signature[i1+1] == signature[i2+1] )
               {    esig.Join( i1/2, i2/2 );    }    }    }
     vec<int> reps, o, morbit;
     esig.OrbitRepsAlt(reps);
     if (verbose) cout << "\nPair orbits:\n";
     morbits.clear( );
     for ( int i = 0; i < reps.isize( ); i++ )
     {    esig.Orbit( reps[i], o );
          Sort(o);
          if (verbose) cout << "[" << i+1 << "] ";
          morbit.clear( );
          for ( int j = 0; j < o.isize( ); j++ )
               morbit.push_back( ids_used[ 2 * o[j] ] / 2 );
          morbits.push_back(morbit);
          if (verbose)
          {    CompactPrint( cout, morbit );
               cout << "\n";    }    }    }

inline bool vec_size_greater( const vec<int>& left, const vec<int>& right )
{    return left.size( ) > right.size( );    }

void MergeCopyNumberTwoCoreGreedy( const HyperKmerPath& h, const KmerBaseBroker& kbb,
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove, const Bool verbose )
{    vec<int> used;
     vec< vec<painted_placed_pp_altgraph> > MCLUSTERS;
     vec< vec< vec<int> > > MORBITS;
     vec< vec< pair< int, vec<int> > > > SIGNATURE;
     vec< vec<int> > IDS_USED;
     while(1)
     {    static vec< pair< vec<int>, int > > pairs_edge;
          GeneratePairsEdge( ppp, edge_copyno, pairs_edge, remove );
          if ( pairs_edge.empty( ) ) return;
          int pei;
          for ( pei = 0; pei < pairs_edge.isize( ); pei++ )
          {    int u = pairs_edge[pei].second;
               if ( !Member( used, u ) ) break;    }
          if ( pei == pairs_edge.isize( ) ) break;
          int u = pairs_edge[pei].second;
          used.push_back(u);
          vec<int>& p0 = pairs_edge[pei].first;
          vec<painted_placed_pp_altgraph> mclusters;
          vec< vec<int> > morbits;
          vec< pair< int, vec<int> > > signature;
          vec<int> ids_used;
          MergeClusterGreedy( pairs_edge, h, kbb, u, ppp, p0, L, G, to_left_vertex, 
               to_right_vertex, edge_copyno, True, remove, verbose, mclusters,
               morbits, signature, ids_used );    

          // Do not allow copy-number-two unipath to appear twice.

          vec<int> unipaths;
          for ( int i = 0; i < mclusters.isize( ); i++ )
          {    for ( int j = 0; j < mclusters[i].Count( ); j++ )
               {    const vec<pp_read>& E = mclusters[i].EdgeSet(j);
                    for ( int k = 0; k < E.isize( ); k++ )
                    {    for ( int l = 0; l < E[k].isize( ); l++ )
                              unipaths.push_back( E[k][l] );    }    }    }
          Sort(unipaths);
          Bool fail = False;
          for ( int i = 1; i < unipaths.isize( ); i++ )
          {    if ( edge_copyno[ unipaths[i] ] != 2 ) continue;
               if ( unipaths[i] == unipaths[i-1] )
               {    cout << "REJECT: " << BaseAlpha( unipaths[i] )
                         << " has copy number two and appears twice.\n";
                    fail = True;
                    break;    }    }
          if (fail) continue;

          // Do not allow copy-number-one unipath to occur at depth one.

          for ( int i = 0; i < mclusters.isize( ); i++ )
          {    for ( int j = 0; j < mclusters[i].Count( ); j++ )
               {    const vec<pp_read>& E = mclusters[i].EdgeSet(j);
                    if ( E.size( ) == 2 ) continue;
                    for ( int l = 0; l < E[0].isize( ); l++ )
                    {    if ( edge_copyno[ E[0][l] ] == 1 )
                         {    cout << "REJECT: " << BaseAlpha( E[0][l] )
                                   << " has copy number one and appears in a "
                                   << "singleton edge.\n";
                              fail = True;
                              break;    }    }
                    if (fail) break;    }
               if (fail) break;    }
          if (fail) continue;

          // Save the clusters.

          MCLUSTERS.push_back(mclusters); 
          MORBITS.push_back(morbits);
          SIGNATURE.push_back(signature);
          IDS_USED.push_back(ids_used);    }
     vec<int> morbits_index;
     vec< vec<int> > morbits_flat;
     for ( int i = 0; i < MORBITS.isize( ); i++ )
     {    for ( int j = 0; j < MORBITS[i].isize( ); j++ )
          {    morbits_flat.push_back( MORBITS[i][j] );
               morbits_index.push_back(i);    }    }
     UniqueSortSync( morbits_flat, morbits_index );
     SortSync( morbits_flat, morbits_index, vec_size_greater );    
     vec<Bool> pused( ppp.size( ), False );
     vec< vec<int> > morbits_to_use;
     vec<int> morbits_to_use_index;
     for ( int i = 0; i < morbits_flat.isize( ); i++ )
     {    const vec<int>& o = morbits_flat[i];
          if ( o.solo( ) ) continue;
          int j;
          for ( j = 0; j < o.isize( ); j++ )
               if ( pused[ o[j] ] ) break;
          if ( j < o.isize( ) ) continue;
          for ( j = 0; j < o.isize( ); j++ )
               pused[ o[j] ] = True;
          morbits_to_use.push_back(o);
          morbits_to_use_index.push_back( morbits_index[i] );    }
     for ( int i = 0; i < morbits_to_use.isize( ); i++ )
     {    cout << "[" << i+1 << " ] using orbit ";
          CompactPrint( cout, morbits_to_use[i] );
          cout << " from cluster " << morbits_to_use_index[i] << "\n";    }    
     for ( int i = 0; i < morbits_to_use.isize( ); i++ )
     {    const vec<int>& o = morbits_to_use[i];
          int mid = morbits_to_use_index[i];
          const vec<painted_placed_pp_altgraph>& mc = MCLUSTERS[mid];
          const vec< pair< int, vec<int> > >& signature = SIGNATURE[mid];
          const vec<int>& ids_used = IDS_USED[mid];
          int lid = -1;
          for ( int j = 0; j < ids_used.isize( ); j += 2 )
               if ( ids_used[j] == 2 * o[0] ) lid = j;
          const pair< int, vec<int> >& sigL = signature[lid];
          const pair< int, vec<int> >& sigR = signature[lid+1];
          const painted_placed_pp_altgraph& mcL = mc[ sigL.first ];
          const painted_placed_pp_altgraph& mcR = mc[ sigR.first ];
          pp_read left, right;
          for ( int stage = 0; stage < sigL.second.isize( ); stage++ )
          {    int level = sigL.second[stage];
               if ( level < 0 ) continue;
               left.append( mcL.EdgeSet(stage)[ sigL.second[stage] ] );    }
          for ( int stage = 0; stage < sigR.second.isize( ); stage++ )
          {    int level = sigR.second[stage];
               if ( level < 0 ) continue;
               right.append( mcR.EdgeSet(stage)[ sigR.second[stage] ] );    }
          vec<int> pids, left_starts, left_stops, right_starts, right_stops;
          int npairs = o.size( );
          for ( int j = 0; j < npairs; j++ )
          {    int pid = o[j];
               pids.push_back(pid);
               const pp_pair& p = ppp[pid];
               int left_start = -1, right_start = -1;
               for ( int k = 0; k < mcL.ReadIdCount( ); k++ )
               {    if ( 2 * pid == mcL.ReadId(k) )
                    {    left_start = mcL.ReadStarts(k).Pos( );
                         break;    }    }
               for ( int k = 0; k < mcR.ReadIdCount( ); k++ )
               {    if ( 2 * pid + 1 == mcR.ReadId(k) )
                    {    right_start = mcR.ReadStarts(k).Pos( );
                         break;    }    }
               left_starts.push_back(left_start);
               left_stops.push_back( left_start + p.LeftSize( ) );
               right_starts.push_back(right_start);
               right_stops.push_back( right_start + p.RightSize( ) );    }
          vec<double> gaps, devs;
          for ( int j = 0; j < npairs; j++ )
          {    int pid = pids[j];
               const pp_pair& p = ppp[pid];
               double gap = p.Gap( );
               for ( int k = left_stops[j]; k < left.isize( ); k++ )
                    gap -= L[ left[k] ];
               for ( int k = 0; k < right_starts[j]; k++ )
                    gap -= L[ right[k] ];
               gaps.push_back(gap), devs.push_back( p.Dev( ) );    }
          cout << "\nPROPOSED MERGER OF:\n";
          for ( int j = 0; j < npairs; j++ )
               cout << "[" << j+1 << "] " << ppp[ pids[j] ] << "\n";
          cout << "NEW LEFT = " << left << "\n" << "NEW RIGHT = " << right << "\n";
          cout << "NEW GAPS:\n";
          for ( int j = 0; j < npairs; j++ )
          {    cout << "[" << j+1 << "] ";
               RightPrecisionOut( cout, gaps[j], 2 );
               cout << " +/- ";
               RightPrecisionOut( cout, devs[j], 2 ); 
               cout << "\n";    }
          double dmult = 4.0;
          double g = gaps[0], d = devs[0];
          Bool fail_to_combine = False;
          for ( int j = 1; j < npairs; j++ )
          {    double gnew, dnew;
               if ( !CombineMeasurements( 
                    g, gaps[j], d, devs[j], dmult, gnew, dnew ) )
               {    cout << "failed to combine\n";
                    fail_to_combine = True;
                    break;    }
               g = gnew, d = dnew;    }
          if (fail_to_combine) continue;

          // Don't allow mergers across dinucleotide repeats.

          Bool illegal_merger = False;
          for ( int pass = 1; pass <= 2; pass++ )
          {    const pp_read& r = ( pass == 1 ? left : right );
               for ( int j = 1; j < r.isize( ); j++ )
               {    if ( h.EdgeLength( r[j] ) == 2 && r[j] != r[j-1] )
                    {    int k;
                         for ( k = j+1; k < r.isize( ); k++ )
                              if ( r[k] != r[j] ) break;
                         if ( k < r.isize( ) && k - j >= 2 )
                         {    static pp_read s;
                              s.clear( );
                              for ( int l = j-1; l <= k; l++ )
                                   s.push_back( r[l] );
                              Bool contained = False;
                              for ( int x = 0; x < npairs; x++ )
                              {    if ( ppp[ pids[x] ].Left( ).Contains(s) )
                                        contained = True;
                                   else if ( ppp[ pids[x] ].Right( ).Contains(s) )
                                        contained = True;
                                   if (contained) break;    }
                              if ( !contained )
                              {    cout << "illegal merger over dinucleotide "
                                        << "repeat\n";
                                   PRINT(s);
                                   illegal_merger = True;
                                   goto test_illegal;    }    }    }    }    }
          test_illegal: if (illegal_merger) continue;

          pp_pair p( left, right, g, d );
          cout << "MERGER: " << p << "\n";
          ppp[ pids[0] ] = p;
          for ( int j = 1; j < pids.isize( ); j++ )
               remove[ pids[j] ] = True;    }    }

void MergeCopyNumberTwoGreedy( const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove, const Bool verbose )
{    for ( int pass = 1; pass <= 2; pass++ )
     {    if (verbose) cout << "\nEntering MergeCopyNumberTwoGreedy\n";
          cout << "FIRST ROUND\n";
          MergeCopyNumberTwoCoreGreedy( h, kbb, ppp, L, to_left_vertex, 
               to_right_vertex, G, edge_copyno, remove, verbose );
          cout << "SECOND ROUND\n";
          MergeCopyNumberTwoCoreGreedy( h, kbb, ppp, L, to_left_vertex, 
               to_right_vertex, G, edge_copyno, remove, verbose );
          /*
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );
          digraphE<int> Grc(G);
          Grc.Reverse( );
          MergeCopyNumberTwoCoreGreedy( h, kbb, ppp, L, to_right_vertex, 
               to_left_vertex, Grc, edge_copyno, remove, verbose );
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );    
          */
               }    }

template digraphE<istring>::digraphE(vec<vec<vrtx_t> > const&, vec<vec<vrtx_t> > const&, 
     vec<istring> const&, vec<vec<int> > const&, vec<vec<int> > const&);
template void digraphE<istring>::RemoveUnneededVertices();
template void digraphE<istring>::RemoveDeadEdgeObjects();
