/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// MergeCopyNumberTwo: experimental code to merge copy number two stuff.   This
// operates on a vec<pp_pair> ppp, replacing it by a smaller and merged version.
// Other inputs to various functions in this file:
// vec<int> L: the lengths in kmers of the edges;
// vec<int> edge_copyno: the predicted copy number of the edges;
// vec<vrtx_t> to_left_vertex: map from graph edges to their left vertices;
// vec<vrtx_t> to_right_vertex: map from graph edges to their right vertices;
// digraphE<int> G: the underlying graph of the HyperKmerPath, with edges labelled
//                  by their lengths in kmers.

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "CoreTools.h"
#include "math/Functions.h"
#include "paths/BridgeGaps.h"
#include "graph/Digraph.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/PairedPair.h"

Bool cmp_pairs_edge( const pair< vec<int>, int >& pe1, 
     const pair< vec<int>, int>& pe2 )
{    return pe1.first.size( ) < pe2.first.size( );    }

// MaybeOneBeforeTwo.  Determine if read r1 at position p1 +/- d1 could be before
// read r2 at position p2 +/- d2.

Bool MaybeOneBeforeTwo( const pp_read& r1, const double p1, const double d1,
     const pp_read& r2, const double p2, const double d2, const double dmult,
     const vec<int>& L, const digraphE<int>& G, const vec<vrtx_t>& to_left_vertex,
     const vec<vrtx_t>& to_right_vertex )
{    double p1x = p1;
     for ( int j = 0; j < r1.isize( ); j++ )
          p1x += L[ r1[j] ];
     double space = p2 - p1x + dmult * ( d1 + d2 );
     return G.ThisClose( to_right_vertex[ r1.back( ) ], 
          to_left_vertex[ r2.front( ) ], int(floor(space)) );    }

// GeneratePairsEdge.  Generate a vec< pair< vec<int>, int > > such that the
// second member of a pair is a copy number two edge id u, and the first member
// is {i: u is in ppp[i].Left}.  However, the pair for u is not returned in the
// event that ppp[i].Left contains more than one copy of u, for some i.

void GeneratePairsEdge( const vec<pp_pair>& ppp, const vec<int>& edge_copyno,
     vec< pair< vec<int>, int > >& pairs_edge, const vec<Bool>& remove )
{    static vec< pair<int,int> > edge_pair;
     edge_pair.clear( ), pairs_edge.clear( );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          for ( int j = 0; j < ppp[i].LeftSize( ); j++ )
          {    int u = ppp[i].Left(j);
               if ( edge_copyno[u] != 2 ) continue;
               edge_pair.push_back( make_pair( u, i ) );    }    }
     Sort(edge_pair);
     for ( int i = 0; i < edge_pair.isize( ); i++ )
     {    int u = edge_pair[i].first;
          int j;
          for ( j = i + 1; j < edge_pair.isize( ); j++ )
               if ( edge_pair[j].first != u ) break;
          Bool dupped = False;
          for ( int k = i + 1; k < j; k++ )
          {    if ( edge_pair[k].second == edge_pair[k-1].second ) 
                    dupped = True;    }
          if (dupped) continue;
          static vec<int> p;
          p.clear( );
          for ( int k = i; k < j; k++ )
               p.push_back( edge_pair[k].second );
          pairs_edge.push_back( make_pair( p, u ) );
          i = j - 1;    }
     sort( pairs_edge.rbegin( ), pairs_edge.rend( ), cmp_pairs_edge );    }

// BranchPoint.  Determine if a read M has a branch point, in the sense that there
// is some read ( ppp[i].Left or ppp[i].Right ) that shares a copy number two edge
// with M, but does not align to M.  Return k such that {M[i]: 0 <= i < k} is
// branch point free if there is a branch, else -1.
//
// Also test for copy number one.

int BranchPoint( const HyperKmerPath& h, const vec<vrtx_t>& to_right_vertex,
     const pp_read& M, const vec<pp_pair>& ppp, const vec<int>& edge_copyno, 
     const vec<Bool>& remove, const Bool verbose )
{    if (verbose) PRINT(M);
     int answer = -1;
     for ( int i = 0; i < M.isize( ); i++ )
     {    if ( edge_copyno[ M[i] ] == 1 )
          {    if (verbose)
               {    cout << "Stop after ";
                    for ( int u = 0; u < i; u++ )
                    {    if ( u > 0 ) cout << ".";
                         cout << BaseAlpha( M[u] );    }
                    cout << "\n";    }
               if ( answer == -1 ) answer = i;
               else answer = Min( answer, i );
               break;    }    }    
     for ( int i = 0; i < M.isize( ); i++ )
     {    int v = to_right_vertex[ M[i] ];
          for ( int j = 0; j < h.From(v).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( v, j );
               if ( edge_copyno[e] == 1 )
               {    if (verbose)
                    {    cout << "Stop after ";
                         for ( int u = 0; u < i+1; u++ )
                         {    if ( u > 0 ) cout << ".";
                              cout << BaseAlpha( M[u] );    }
                         cout << "\n";    }
                    if ( answer == -1 ) answer = i+1;
                    else answer = Min( answer, i+1 );
                    break;    }    }    }
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          for ( int pass = 1; pass <= 2; pass++ )
          {    const pp_read& r = ( pass == 1 ? ppp[i].Left( ) : ppp[i].Right( ) );
               for ( int j = 0; j < r.isize( ); j++ )
               {    int v = r[j];
                    if ( edge_copyno[v] != 2 ) continue;
                    for ( int k = 0; k < M.isize( ); k++ )
                    {    if ( M[k] != v ) continue;
                         for ( int l = k + 1; l < M.isize( ); l++ )
                         {    if ( l + j - k == r.isize( ) ) break;
                              if ( M[l] != r[l+j-k] )
                              {    if (verbose)
                                   {    cout << "Stop after ";
                                        for ( int u = j; u < l+j-k; u++ )
                                        {    if ( u > j ) cout << ".";
                                             cout << BaseAlpha( r[u] );    }
                                        cout << "\n";    }
                                   if ( answer == -1 ) answer = l;
                                   else answer = Min( answer, l );
                                   break;    }    }    
                         for ( int l = k - 1; l >= 0; l-- )
                         {    if ( l + j - k < 0 ) break;
                              if ( M[l] != r[l+j-k] )
                              {    if (verbose)
                                   {    cout << "Stop after ";
                                        for ( int u = 0; u <= l+j-k; u++ )
                                        {    if ( u > 0 ) cout << ".";
                                             cout << BaseAlpha( r[u] );    }
                                        cout << "\n";    }
                                   if ( answer == -1 ) answer = l+1;
                                   else answer = Min( answer, l+1 );
                                   break;    }    }    }    }    }    }
     return answer;    }

// MergeCluster.  Given a collection p0 of read pairs, whose left reads all contain
// the edge u, merge subsets of the collection where possible.

void MergeCluster( const HyperKmerPath& h, const KmerBaseBroker& kbb, int u, 
     vec<pp_pair>& ppp, vec<int>& p0, const vec<int>& L, const digraphE<int>& G, 
     const vec<vrtx_t>& to_left_vertex, const vec<vrtx_t>& to_right_vertex, 
     int& total_eliminated, const vec<int>& edge_copyno, const Bool stop_at_branch, 
     vec<Bool>& remove, const Bool verbose )
{
     // Compute gaps between u (on left edge) and right edge.
     // Then reorder using the gaps.

     vec<double> gaps;
     for ( int i = 0; i < p0.isize( ); i++ )
     {    const pp_pair& p = ppp[ p0[i] ];
          double gap = p.Gap( );
          int j;
          for ( j = 0; j < p.LeftSize( ); j++ )
          if ( p.Left(j) == u ) break;
          for ( int k = j + 1; k < p.LeftSize( ); k++ )
               gap += L[ p.Left(k) ];
          gaps.push_back(gap);    }
     SortSync( gaps, p0 );

     // Starting at the beginning, try to merge all the right edges.

     pp_read M = ppp[ p0[0] ].Right( );
     double g = gaps[0], d = ppp[ p0[0] ].Dev( );
     double dmult = 3.0;
     vec< vec<int> > pools;
     vec<int> pool;
     vec<pp_read> Ms;
     vec<double> gs, ds;
     pool.push_back(0);
     for ( int i = 1; i < p0.isize( ); i++ )
     {    const pp_pair& p = ppp[ p0[i] ];
          const pp_read& ri = p.Right( );
          const double gi = gaps[i], di = p.Dev( );
          static vec<pp_read> Mnew;
          static vec<double> gnew, dnew;
          static vec<int> offsets;
          if (verbose) cout << "trying to merge in read " << i << ": " << ri << "\n";

          // First see if M and ri could be disjoint from each other.

          if ( MaybeOneBeforeTwo( ri, gi, di, M, g, d, dmult, L, G, 
               to_left_vertex, to_right_vertex ) )
          {    if (verbose) cout << "Could be disjoint.  Stopping merger process.\n";
               return;    }
          if ( MaybeOneBeforeTwo( M, g, d, ri, gi, di, dmult, L, G, 
               to_left_vertex, to_right_vertex ) )
          {    
               // Could it be the case that everything else must be to the right
               // of M?  If so we can close out the old pool and start a new one.
               // Otherwise, give up.

               double p1x = g;
               for ( int j = 0; j < M.isize( ); j++ )
                    p1x += L[ M[j] ];
               int j;
               for ( j = i; j < p0.isize( ); j++ )
               {    const double dj = ppp[ p0[j] ].Dev( );
                    if ( gaps[j] - p1x < dmult * ( d + dj ) ) break;    }
               if ( j == p0.isize( ) )
               {    pools.push_back(pool);
                    pool.clear( );
                    pool.push_back(i);
                    Ms.push_back(M);
                    gs.push_back(g);
                    ds.push_back(d);
                    M = ri;
                    g = gaps[i];
                    d = ppp[ p0[i] ].Dev( );
                    if (verbose) cout << "Break point reached.\n";
                    continue;    }
               else
               {    if (verbose) 
                         cout << "Could be disjoint.  Stopping merger process.\n";
                    return;    }    }

          // Now look for joins.

          FindJoins( M, g, d, ri, gi, di, dmult, L, Mnew, gnew, dnew, offsets );
          if ( Mnew.empty( ) )
          {    FindJoins( M, g, d, ri, gi, di, dmult + 0.5, L,
                    Mnew, gnew, dnew, offsets );    }
          if ( Mnew.solo( ) )
          {    M = Mnew[0], g = gnew[0], d = dnew[0];
               pool.push_back(i);
               if (verbose) cout << "Done.\n";
               continue;    }
          if ( Mnew.size( ) > 1 )
          {    if (verbose)
               {    cout << "Found " << Mnew.size( ) << " joins.  Stopping merger "
                         << "process.\n";    }
               return;    }
          if ( Mnew.empty( ) )
          {    if (verbose) cout << "No join found.  Trimming M.\n";
               while( M.nonempty( ) )
               {    if (verbose) cout << "Downsizing M by 1:\n";
                    M.pop_back( );
                    if (verbose) PRINT(M);
                    if ( M.empty( ) ) break;
                    if ( MaybeOneBeforeTwo( M, g, d, ri, gi, di, dmult, L, G, 
                         to_left_vertex, to_right_vertex ) )
                    {    if (verbose) cout << "Looks good enough (1).\n";
                         break;    }
                    FindJoins( M, g, d, ri, gi, di, dmult + 0.5, L,
                         Mnew, gnew, dnew, offsets );
                    if ( Mnew.nonempty( ) )
                    {    if (verbose) cout << "Looks good enough (2).\n";
                         break;    }    }
               if (verbose) cout << "Checking the rest:\n";
               for ( int j = i + 1; j < p0.isize( ); j++ )
               {    const pp_pair& p = ppp[ p0[j] ];
                    const pp_read& rj = p.Right( );
                    const double gj = gaps[j], dj = p.Dev( );
                    if (verbose) 
                         cout << "Checking " << j << " of " << p0.size( ) << "\n";
                    while( M.nonempty( ) )
                    {    if ( MaybeOneBeforeTwo( M, g, d, rj, gj, dj, dmult, 
                              L, G, to_left_vertex, to_right_vertex ) )
                         {    if (verbose) cout << "Looks good enough (1).\n";
                              break;    }
                         FindJoins( M, g, d, rj, gj, dj, dmult + 0.5, L,
                              Mnew, gnew, dnew, offsets );
                         if ( Mnew.nonempty( ) )
                         {    if (verbose) cout << "Looks good enough (2).\n";
                              break;    }
                         if (verbose) cout << "Downsizing M by 1:\n";
                         M.pop_back( );
                         if (verbose) PRINT(M);    }    }
               break;    }    }
     pools.push_back(pool);
     Ms.push_back(M);
     gs.push_back(g);
     ds.push_back(d);

     // Go through the pools.

     int old_actives = ppp.isize( ) - Sum(remove);
     int old_ppp_size = ppp.size( );
     if (verbose) 
     {    cout << "Pools:\n";
          for ( int pi = 0; pi < pools.isize( ); pi++ )
          {    const vec<int>& pool = pools[pi];
               cout << "[" << pi+1 << "]: ";
               CompactPrint( cout, pool );
               cout << "\n";    }
          cout << "\n";    }
     for ( int pi = 0; pi < pools.isize( ); pi++ )
     {    const vec<int>& pool = pools[pi];
          pp_read& M = Ms[pi];
          const double g = gs[pi], d = ds[pi];

          // Test M to see if it contains a branch point.

          if (stop_at_branch)
          {    if (verbose) PRINT(M);
               int l = BranchPoint( h, to_right_vertex, M, ppp, edge_copyno, remove,
                    verbose );
               if ( l >= 0 ) M.resize(l);    }

          // Define the mergeable set.

          if ( M.empty( ) ) continue;
          vec<Bool> mergeable( p0.size( ), False );
          vec<int> shift( p0.size( ), 1000000000 );
          int maxshift = 0;
          for ( int ip = 0; ip < pool.isize( ); ip++ )
          {    int i = pool[ip];
               const pp_pair& p = ppp[ p0[i] ];
               const pp_read& ri = p.Right( );
               const double gi = gaps[i], di = p.Dev( );
               static vec<pp_read> Mnew;
               static vec<double> gnew, dnew;
               static vec<int> offsets;
               if ( MaybeOneBeforeTwo( ri, gi, di, M, g, d, dmult, L, G, 
                    to_left_vertex, to_right_vertex ) )
               {    continue;    }
               if ( MaybeOneBeforeTwo( M, g, d, ri, gi, di, dmult, L, G, 
                    to_left_vertex, to_right_vertex ) )
               {    continue;    }
               FindJoins( M, g, d, ri, gi, di, dmult, L, Mnew, gnew, dnew, offsets );
               if ( Mnew.empty( ) )
               {    FindJoins( M, g, d, ri, gi, di, dmult + 0.5, L,
                         Mnew, gnew, dnew, offsets );    }
               if ( !Mnew.solo( ) ) continue;
               if ( offsets[0] < 0 ) continue;
               if ( offsets[0] + ri.isize( ) > M.isize( ) ) continue;
               mergeable[i] = True;
               shift[i] = offsets[0];
               maxshift = Max( maxshift, shift[i] );    }
          for ( int i = 0; i < p0.isize( ); i++ )
               if ( !mergeable[i] ) shift[i] = maxshift + 1;

          // Reorder cluster by shift.

          vec<double> gapsx = gaps;
          vec<int> p0x = p0;
          SortSync( shift, p0x, mergeable, gapsx );

          // Print cluster.
     
          if (verbose)
          {    cout << "\nLargest cluster (" << BaseAlpha(u) << "):\n";
               vec< vec<String> > rows;
               for ( int i = 0; i < p0x.isize( ); i++ )
               {    const pp_pair& p = ppp[ p0x[i] ];
                    static vec<String> row;
                    row.clear( );
                    row.push_back( mergeable[i] ? "*" : "" );
                    ostrstream out1, out2;
                    RightPrecisionOut( out1, gapsx[i], 2 );
                    out1 << " +/- ";
                    RightPrecisionOut( out1, p.Dev( ), 2 );
                    out1 << ends;
                    static String sh;
                    sh.resize(0);
                    for ( int j = 0; j < shift[i]; j++ )
                    {    sh += ".";
                         sh += BaseAlpha( M[j] );    }
                    for ( unsigned int j = 0; j < sh.size( ); j++ )
                         out2 << " ";
                    out2 << p.Right( ) << ends;
                    row.push_back( out1.str( ), out2.str( ) );
                    rows.push_back(row);    }
               PrintTabular( cout, rows, 2 );
               cout << "\n";    }

          // Combine mergeable pairs which have the same left hand side.
     
          vec<pp_read> lefts;
          vec<int> ids;
          for ( int i = 0; i < p0x.isize( ); i++ )
          {    if ( mergeable[i] ) 
               {    lefts.push_back( ppp[ p0x[i] ].Left( ) );
                    ids.push_back(i);    
                    if (verbose) cout << "old pair: " << ppp[ p0x[i] ] << "\n";
                    remove[ p0x[i] ] = True;    }    }
          SortSync( lefts, ids );
          for ( int i = 0; i < lefts.isize( ); i++ )
          {    int j;
               for ( j = i + 1; j < lefts.isize( ); j++ )
                    if ( lefts[j] != lefts[i] ) break;
               double gnew = 0, dnew = 0;
               Bool merge_failed = False;
               for ( int k = i; k < j; k++ )
               {    const pp_pair& p = ppp[ p0x[ ids[k] ] ];
                    double gk = p.Gap( ), dk = p.Dev( );
                    for ( int u = 0; u < shift[ ids[k] ]; u++ )
                         gk -= L[ M[u] ];
                    if ( k == i )
                    {    gnew = gk, dnew = dk;    }
                    else
                    {    double gn, dn;
                         Bool combined = CombineMeasurements( gnew, gk, dnew, dk, 
                              dmult, gn, dn );
                         if ( !combined )
                         {    combined = CombineMeasurements( gnew, gk, dnew, dk, 
                                   dmult + 0.5, gn, dn );    }
                         if ( !combined )
                         {    merge_failed = True;
                              break;    }
                         gnew = gn, dnew = dn;    }    }
               if (merge_failed) // Don't know if this can happen.
               {    if (verbose)
                    {    cout << "Warning: merge failed, using default "
                              << "values for g and d.\n";    }
                    gnew = g, dnew = d;
                    int k;
                    for ( k = 0; k < lefts[i].isize( ); k++ )
                         if ( lefts[i][k] == u ) break;
                    for ( int x = k + 1; x < lefts[i].isize( ); x++ )
                         gnew -= L[ lefts[i][x] ];    }
               ppp.push_back( pp_pair( lefts[i], M, gnew, dnew ) );
               remove.push_back(False);
               if (verbose) cout << "new pair: " << ppp.back( ) << "\n";
               i = j - 1;    }    }

     // Clean up.

     remove.resize( ppp.size( ), False );
     int new_actives = ppp.isize( ) - Sum(remove);
     int elim = old_actives - new_actives;
     total_eliminated += elim;
     if (verbose)
     {    cout << "Eliminated " << elim << " pairs.\n\n";
          cout << "\nBEFORE SimpleExtend:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) cout << "[" << i << "] " << ppp[i] << "\n";    }
     SimpleExtendPairedPairs( h, kbb, ppp, L, dmult, remove, True );
     if (verbose)
     {    cout << "\nAFTER SimpleExtend:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) cout << "[" << i << "] " << ppp[i] << "\n";   }   }

// Align: Suppose that r1 and r2 agree at positions pos1 and pos2, respectively.
// Determine if this partial alignment extends to a full proper alignment.

Bool Align( const pp_read& r1, const pp_read& r2, int pos1, int pos2 )
{    for ( int j = 1; ; j++ )
     {    if ( pos1 + j >= r1.isize( ) ) break;
          if ( pos2 + j >= r2.isize( ) ) break;
          if ( r1[ pos1 + j ] != r2[ pos2 + j ] ) return False;    }
     for ( int j = 1; ; j++ )
     {    if ( pos1 - j < 0 ) break;
          if ( pos2 - j < 0 ) break;
          if ( r1[ pos1 - j ] != r2[ pos2 - j ] ) return False;    }
     return True;    }

// SplitReads.  Given a collection of reads containing copy-number-two unipath u,
// classify the reads as pile1, pile2, or unpiled.  All reads in pile1 should
// overlap on the genome, and likewise for pile2.

void SplitReads( /* inputs: */  const vec<pp_read>& R, int u,
                 /* outputs: */ vec<int>& pile, pp_read& type1, pp_read& type2,
                                int& tpos1, int& tpos2 )
{    vec<int> pos;
     for ( int i = 0; i < R.isize( ); i++ )
     {    int j;
          for ( j = 0; j < R[i].isize( ); j++ )
               if ( R[i][j] == u ) break;
          pos.push_back(j);    }
     Bool conflict = False;
     pile.resize_and_set( R.size( ), 0 );
     for ( int i1 = 0; i1 < R.isize( ); i1++ )
     {    for ( int i2 = i1 + 1; i2 < R.isize( ); i2++ )
          {    if ( !Align( R[i1], R[i2], pos[i1], pos[i2] ) )
               {    conflict = True;
                    type1 = R[i1], type2 = R[i2];
                    tpos1 = pos[i1], tpos2 = pos[i2];
                    pile[i1] = 1, pile[i2] = 2;
                    goto found;    }    }    }
     found: if ( !conflict ) return;
     Bool problem = False;
     while(1)
     {    Bool progress = False;
          for ( int i = 0; i < R.isize( ); i++ )
          {    if ( pile[i] != 0 ) continue;
               Bool a1 = Align( R[i], type1, pos[i], tpos1 );
               Bool a2 = Align( R[i], type2, pos[i], tpos2 );
               if ( a1 && a2 ) continue;
               if ( !a1 && !a2 ) 
               {    problem = True;
                    break;    }
               if (a1) 
               {    pile[i] = 1;
                    for ( int j = pos[i] - tpos1 - 1; j >= 0; j-- )
                    {    type1.push_front( R[i][j] );
                         ++tpos1;    }
                    int right =
                         ( R[i].isize() - pos[i] ) - ( type1.isize() - tpos1 );
                    for ( int j = R[i].isize() - right; j < R[i].isize(); j++ )
                         type1.push_back( R[i][j] );    }
               else 
               {    pile[i] = 2;
                    for ( int j = pos[i] - tpos2 - 1; j >= 0; j-- )
                    {    type2.push_front( R[i][j] );
                         ++tpos2;    }
                    int right =
                         ( R[i].isize() - pos[i] ) - ( type2.isize() - tpos2 );
                    for ( int j = R[i].isize() - right; j < R[i].isize(); j++ )
                         type2.push_back( R[i][j] );    }
               progress = True;    }
          if ( problem || !progress ) break;    }
     if (problem)
     {    for ( int i = 0; i < pile.isize( ); i++ )
               pile[i] = 0;    }    }

// MergeCopyNumberTwoCore.  This does mergers based on taking all the 
// ppp[i].Left that contain a given copy-number-two edge u.

int MergeCopyNumberTwoCore( const HyperKmerPath& h, const KmerBaseBroker& kbb,
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove, const Bool verbose )
{    if (verbose) cout << "\nThinking about copy number two mergers.\n";
     vec<int> used;
     int total_eliminated = 0;
     while(1)
     {    if (verbose) cout << "\nLooking for copy number two to merge on.\n";
          static vec< pair< vec<int>, int > > pairs_edge;
          GeneratePairsEdge( ppp, edge_copyno, pairs_edge, remove );
          if ( pairs_edge.empty( ) )
          {    if (verbose) cout << "No clusters seen.\n";
               return total_eliminated;    }
          int pei;
          for ( pei = 0; pei < pairs_edge.isize( ); pei++ )
          {    int u = pairs_edge[pei].second;
               if ( !Member( used, u ) ) break;    }
          if ( pei == pairs_edge.isize( ) ) return total_eliminated;
          int u = pairs_edge[pei].second;
          used.push_back(u);
          vec<int>& p0 = pairs_edge[pei].first;

          static vec<int> p0new;
          p0new.clear( );
          for ( int j = 0; j < p0.isize( ); j++ )
          {    const pp_pair& p = ppp[ p0[j] ];
               if ( BranchPoint( h, to_right_vertex, p.Right( ), ppp, 
                    edge_copyno, remove, verbose ) == -1 )
               {    p0new.push_back( p0[j] );    }    }
          if ( p0new.empty( ) ) continue;

          MergeCluster( h, kbb, u, ppp, p0new, L, G, to_left_vertex, 
               to_right_vertex, total_eliminated, edge_copyno, True, remove,
               verbose );    }
     return total_eliminated;    }

// MergeCopyNumberTwoCoreSplit.  This does mergers based on taking all the 
// ppp[i].Left that contain a given copy-number-two edge u, under the hypothesis
// that these can be split into two conflicting piles -- the mergers are then 
// separately performed on each of the two piles.

int MergeCopyNumberTwoCoreSplit( const HyperKmerPath& h, const KmerBaseBroker& kbb,
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove, const Bool verbose )
{    if (verbose) cout << "\nThinking about copy number two split mergers.\n";
     vec<int> used;
     int total_eliminated = 0;
     while(1)
     {    if (verbose) cout << "Looking for copy number two to split merge on.\n";
          static vec< pair< vec<int>, int > > pairs_edge;
          GeneratePairsEdge( ppp, edge_copyno, pairs_edge, remove );
          if ( pairs_edge.empty( ) )
          {    if (verbose) cout << "No clusters seen.\n";
               return total_eliminated;    }
          int pei;
          for ( pei = 0; pei < pairs_edge.isize( ); pei++ )
          {    int u = pairs_edge[pei].second;
               if ( !Member( used, u ) ) break;    }
          if ( pei == pairs_edge.isize( ) ) return total_eliminated;
          int u = pairs_edge[pei].second;
          used.push_back(u);
          vec<int>& p0 = pairs_edge[pei].first;
          vec<pp_read> R;
          for ( int i = 0; i < p0.isize( ); i++ )
               R.push_back( ppp[ p0[i] ].Left( ) );
          vec<int> pile;
          pp_read type1, type2;
          int tpos1, tpos2;
          SplitReads( R, u, pile, type1, type2, tpos1, tpos2 );
          if ( Sum(pile) == 0 ) continue;
          vec<int> p1, p2;
          for ( int i = 0; i < p0.isize( ); i++ )
          {    if ( pile[i] == 1 ) p1.push_back( p0[i] );
               if ( pile[i] == 2 ) p2.push_back( p0[i] );    }
          if ( p1.solo( ) ) p1 = p2;
          MergeCluster( h, kbb, u, ppp, p1, L, G, to_left_vertex, to_right_vertex, 
               total_eliminated, edge_copyno, False, remove, verbose );    }
     return total_eliminated;    }

void MergeCopyNumberTwo( const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove, const Bool verbose )
{    int total_eliminated = 0;
     if (verbose)
     {    cout << "\nUPON ENTRY:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove[i] ) continue;
               cout << "[" << i << "] " << ppp[i] << "\n";    }    }
     for ( int pass = 1; pass <= 2; pass++ )
     {    if (verbose) cout << "MergeCopyNumberTwo, starting pass " << pass << "\n";
          total_eliminated += MergeCopyNumberTwoCore( h, kbb, ppp, L, 
               to_left_vertex, to_right_vertex, G, edge_copyno, remove, verbose );
          if (verbose)
          {    cout << "\nAFTER FW MERGE, PASS = " << pass << ":\n";
               for ( int i = 0; i < ppp.isize( ); i++ )
                    if ( !remove[i] ) cout << "[" << i << "] " << ppp[i] << "\n";   }
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );
          digraphE<int> Grc(G);
          Grc.Reverse( );
          total_eliminated += MergeCopyNumberTwoCore( h, kbb, ppp, L, 
               to_right_vertex, to_left_vertex, Grc, edge_copyno, remove, verbose );
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );
          if (verbose)
          {    cout << "\nAFTER RC MERGE, PASS = " << pass << ":\n";
               for ( int i = 0; i < ppp.isize( ); i++ )
                    if ( !remove[i] ) cout << "[" << i << "] " << ppp[i] << "\n";
               cout << "Total pairs eliminated = " << total_eliminated 
                    << ".\n\n";    }    }
     if (verbose)
     {    cout << "\nBEFORE splits:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) cout << "[" << i << "] " << ppp[i] << "\n";    }

     // Look for splits.

     /*
     vec<int> used;
     while(1)
     {    
          // Find large cluster of reads containing the same copy-number-two edge.

          static vec< pair< vec<int>, int > > reads_edge;
          static vec< pair<int,int> > edge_read;
          edge_read.clear( ), reads_edge.clear( );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove[i] ) continue;
               for ( int j = 0; j < ppp[i].LeftSize( ); j++ )
               {    int u = ppp[i].Left(j);
                    if ( edge_copyno[u] != 2 ) continue;
                    edge_read.push_back( make_pair( u, i ) );    }
               for ( int j = 0; j < ppp[i].RightSize( ); j++ )
               {    int u = ppp[i].Right(j);
                    if ( edge_copyno[u] != 2 ) continue;
                    edge_read.push_back( make_pair( u, -i-1 ) );    }    }
          Sort(edge_read);
          for ( int i = 0; i < edge_read.isize( ); i++ )
          {    int u = edge_read[i].first;
               int j;
               for ( j = i + 1; j < edge_read.isize( ); j++ )
                    if ( edge_read[j].first != u ) break;
               Bool dupped = False;
               for ( int k = i + 1; k < j; k++ )
               {    if ( edge_read[k].second == edge_read[k-1].second ) 
                         dupped = True;    }
               if (dupped) continue;
               static vec<int> p;
               p.clear( );
               for ( int k = i; k < j; k++ )
                    p.push_back( edge_read[k].second );
               reads_edge.push_back( make_pair( p, u ) );
               i = j - 1;    }
          sort( reads_edge.rbegin( ), reads_edge.rend( ), cmp_pairs_edge );
          if ( reads_edge.empty( ) )
          {    if (verbose) cout << "No read clusters seen.\n";
               break;    }
          int rei;
          for ( rei = 0; rei < reads_edge.isize( ); rei++ )
          {    int u = reads_edge[rei].second;
               if ( !Member( used, u ) ) break;    }
          if ( rei == reads_edge.isize( ) ) break;
          int u = reads_edge[rei].second;
          used.push_back(u);
          vec<int>& r0 = reads_edge[rei].first;
     
          // Get the reads.

          vec<pp_read> R;
          for ( int i = 0; i < r0.isize( ); i++ )
          {    if ( r0[i] >= 0 ) R.push_back( ppp[ r0[i] ].Left( ) );
               else R.push_back( ppp[ -r0[i]-1 ].Right( ) );    }
          UniqueSort(R);

          // Split into piles and then extend the reads.

          vec<int> pile;
          pp_read type1, type2;
          int tpos1, tpos2;
          SplitReads( R, u, pile, type1, type2, tpos1, tpos2 );
          if ( Sum(pile) == 0 ) continue;
          int npile1 = 0, npile2 = 0;
          vec<pp_read> Pile1, Pile2;
          for ( int i = 0; i < pile.isize( ); i++ )
          {    if ( pile[i] == 1 ) ++npile1;
               if ( pile[i] == 2 ) ++npile2;
               if ( pile[i] == 1 ) Pile1.push_back( R[i] );
               if ( pile[i] == 2 ) Pile2.push_back( R[i] );    }
          if ( npile1 > 1 || npile2 > 1 )
          {    for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( remove[i] ) continue;
                    pp_pair& p = ppp[i];
                    int n1 = type1.size( ), n2 = type2.size( );
                    int np = p.LeftSize( );
                    if ( Member( p.Left( ), u ) )
                    {    int x = Position( p.Left( ), u );
                         if ( Member( Pile1, p.Left( ) ) )
                         {    pp_read rnew;
                              JoinReads( p.Left( ), type1, x - tpos1, rnew );
                              if ( rnew.isize( ) > p.LeftSize( ) )
                              {    if (verbose)
                                   {    cout << "OLD: " << p << "\n";
                                        cout << "replacing " << p.Left( ) << " by "
                                             << rnew << "\n";    }
                                   p.SetLeft(rnew);
                                   double gap = p.Gap( );
                                   int right = ( n1 - tpos1 ) - ( np - x );
                                   if (verbose) 
                                        PRINT4( n1, tpos1, p.LeftSize( ), x );
                                   for ( int j = n1 - right; j < n1; j++ )
                                        gap -= L[ type1[j] ];
                                   p.SetGap(gap);    
                                   if (verbose) 
                                        cout << "NEW: " << p << "\n";    }    }
                         else if ( Member( Pile2, p.Left( ) ) )
                         {    pp_read rnew;
                              JoinReads( p.Left( ), type2, x - tpos2, rnew );
                              if ( rnew.isize( ) > p.LeftSize( ) )
                              {    if (verbose)
                                   {    cout << "replacing " << p.Left( ) << " by "
                                             << rnew << "\n";    }
                                   p.SetLeft(rnew);
                                   double gap = p.Gap( );
                                   int right = ( n2 - tpos2 ) - ( np - x );
                                   for ( int j = n2 - right; j < n2; j++ )
                                        gap -= L[ type2[j] ];
                                   p.SetGap(gap);    }    }    }
                    if ( Member( p.Right( ), u ) )
                    {    int x = Position( p.Right( ), u );
                         if ( Member( Pile1, p.Right( ) ) )
                         {    pp_read rnew;
                              JoinReads( p.Right( ), type1, x - tpos1, rnew );
                              if ( rnew.isize( ) > p.RightSize( ) )
                              {    if (verbose)
                                   {    cout << "replacing " << p.Right( ) << " by "
                                             << rnew << "\n";    }
                                   p.SetRight(rnew);
                                   double gap = p.Gap( );
                                   int left = tpos1 - x;
                                   for ( int j = 0; j < left; j++ )
                                        gap -= L[ type1[j] ];
                                   p.SetGap(gap);    }    }
                         else if ( Member( Pile2, p.Right( ) ) )
                         {    pp_read rnew;
                              JoinReads( p.Right( ), type2, x - tpos2, rnew );
                              if ( rnew.isize( ) > p.RightSize( ) )
                              {    if (verbose)
                                   {    cout << "replacing " << p.Right( ) << " by "
                                             << rnew << "\n";    }
                                   p.SetRight(rnew);
                                   double gap = p.Gap( );
                                   int left = tpos2 - x;
                                   for ( int j = 0; j < left; j++ )
                                        gap -= L[ type2[j] ];
                                   p.SetGap(gap);    }    }    }    }    }    }
     */

     // Try to merge split.

     total_eliminated = 0;
     for ( int pass = 1; pass <= 2; pass++ )
     {    if (verbose)
               cout << "MergeCopyNumberTwoSplit, starting pass " << pass << "\n";
          total_eliminated += MergeCopyNumberTwoCoreSplit( h, kbb, ppp, L, 
               to_left_vertex, to_right_vertex, G, edge_copyno, remove, verbose );
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );
          digraphE<int> Grc(G);
          Grc.Reverse( );
          total_eliminated += MergeCopyNumberTwoCoreSplit( h, kbb, ppp, L, 
               to_right_vertex, to_left_vertex, Grc, edge_copyno, remove, verbose );
          for ( int i = 0; i < ppp.isize( ); i++ )
               if ( !remove[i] ) ppp[i].ReverseMe( );
          if (verbose)
          {    cout << "Total pairs eliminated = " << total_eliminated 
                    << ".\n\n";    }    }

     /*

     // Find pairs cn2op = {(x,i)} consisting of a pp_read x and a position i on x 
     // such that x[i] has copy number two, such that there exists another such pair 
     // (y,j) for which x[i] = y[j] but x, y do not align as an extension of the 
     // match at i, j.

     class cn2event
     {    public:
          cn2event( ) { }
          cn2event( int id, int pos, int u ) : id(id), pos(pos), u(u) { }
          int id, pos, u;
          friend Bool operator<( const cn2event& e1, const cn2event& e2 )
          {    return e1.u < e2.u;    }    };
     vec<cn2event> cn2, cn2op;
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          pp_pair& p1 = ppp[i];
          for ( int j = 0; j < p1.LeftSize( ); j++ )
          {    if ( edge_copyno[ p1.Left(j) ] == 2 )
                    cn2.push_back( cn2event( 2 * i, j, p1.Left(j) ) );    }
          for ( int j = 0; j < p1.RightSize( ); j++ )
          {    if ( edge_copyno[ p1.Right(j) ] == 2 )
                    cn2.push_back( cn2event( 2 * i + 1, j, p1.Right(j) ) );    }    }
     Sort(cn2);
     for ( int i1 = 0; i1 < cn2.isize( ); i1++ )
     {    int i2;
          for ( i2 = i1 + 1; i2 < cn2.isize( ); i2++ )
               if ( cn2[i2].u != cn2[i1].u ) break;
          for ( int j1 = i1; j1 < i2; j1++ )
          {    const pp_read& r1 = ( cn2[i1].id % 2 == 0 
                    ? ppp[ cn2[i1].id/2 ].Left( )
                    : ppp[ ( cn2[i1].id - 1 ) / 2 ].Right( ) );
               for ( int j2 = i1; j2 < i2; j2++ )
               {    const pp_read& r2 = ( cn2[i2].id % 2 == 0 
                         ? ppp[ cn2[i2].id/2 ].Left( )
                         : ppp[ ( cn2[i2].id - 1 ) / 2 ].Right( ) );
                    if ( !Align( r1, r2, cn2[i1].pos, cn2[i2].pos ) )
                    {    cn2op.push_back( cn2[i1] );
                         break;    }    }    }
          i1 = i2 - 1;    }

     // Look for cases where there are pairs p1, p2, p3, all sharing a 
     // copy-number-two point x, such that p1 and p2 do not align (extending x),
     // and such that p3 aligns as a subset of p1.  Then eliminate p3.

     if (verbose) cout << "SPECIAL REMOVE 1\n";
     for ( int i1 = 0; i1 < cn2op.isize( ); i1++ )
     {    const cn2event& e1 = cn2op[i1];
          int id1 = ( e1.id % 2 == 0 ? e1.id/2 : (e1.id - 1)/2 );
          if ( remove[id1] ) continue;
          const pp_pair& p1 = ppp[id1];
          for ( int i2 = 0; i2 < cn2.isize( ); i2++ )
          {    const cn2event& e2 = cn2[i2];
               if ( e1.u != e2.u ) continue;
               if ( e1.id == e2.id ) continue;
               if ( ( e1.id - e2.id ) % 2 != 0 ) continue;
               int id2 = ( e2.id % 2 == 0 ? e2.id/2 : (e2.id - 1)/2 );
               if ( remove[id2] ) continue;
               const pp_pair& p2 = ppp[id2];
               if ( e1.id % 2 == 0 )
               {    if ( !Align( p1.Left( ), p2.Left( ), e1.pos, e2.pos ) ) continue;
                     
                    ...

     ...............................................................................

     */

     // Look for cases where there is a closed pair C, another pair of the form
     // C --> D or D --> C, and a read E that shares a copy number two point with 
     // C, but does not align to it.  Eliminate the closed pair C.
     // NOTE: not yet robust to bad pairs.

     if (verbose) cout << "SPECIAL REMOVE 2\n";
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          pp_pair& C = ppp[i];
          if ( !IsClosed( C, L ) ) continue;
          static vec<int> twos;
          twos.clear( );
          for ( int j = 0; j < C.LeftSize( ); j++ )
               if ( edge_copyno[ C.Left(j) ] == 2 ) twos.push_back(j);
          if ( twos.empty( ) ) continue;
          Bool have_CD = False;
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( remove[j] || j == i ) continue;
               if ( ppp[j].Left( ) == C.Left( ) || ppp[j].Right( ) == C.Left( ) )
               {    have_CD = True;
                    break;    }    }
          if ( !have_CD ) continue;
          Bool have_E = False;
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( remove[j] || j == i ) continue;
               for ( int pass = 1; pass <= 2; pass++ )
               {    const pp_read& r 
                         = ( pass == 1 ? ppp[j].Left( ) : ppp[j].Right( ) );
                    for ( int u = 0; u < r.isize( ); u++ )
                    {    if ( edge_copyno[ r[u] ] != 2 ) continue;
                         for ( int v = 0; v < twos.isize( ); v++ )
                         {    if ( r[u] != C.Left( twos[v] ) ) continue;
                              if ( Align( r, C.Left( ), u, twos[v] ) ) continue;
                              if (verbose)
                              {    cout << "using [" << j << "] "
                                        << ppp[j] << "\n";    }
                              have_E = True;
                              goto have_or_have_not;    }    }    }    }
          have_or_have_not:
          if ( have_E && verbose ) 
               cout << "removing [" << i << "] " << ppp[i] << "\n";
          if (have_E) remove[i] = True;    }    }



/*

....................................................................................

// A forked_read is a pp_read that can have two alternate branches appended to it
// on the left and likewise on the right.  In the "unforked" state, these branches
// are empty.

class forked_read {

     public:

     forked_read( ) { }
     forked_read( const pp_read& M ) : M_(M) { }

     const pp_read& L1( ) const { return L1_; }
     const pp_read& L2( ) const { return L2_; }
     const pp_read& M( ) const { return M_; }
     const pp_read& R1( ) const { return R1_; }
     const pp_read& R2( ) const { return R2_; }

     int L1( int j ) const { return L1_[j]; }
     int L2( int j ) const { return L2_[j]; }
     int M( int j ) const { return M_[j]; }
     int R1( int j ) const { return R1_[j]; }
     int R2( int j ) const { return R2_[j]; }

     int L1Size( ) const { return L1_.size( ); }
     int L2Size( ) const { return L2_.size( ); }
     int MSize( ) const { return M_.size( ); }
     int R1Size( ) const { return R1_.size( ); }
     int R2Size( ) const { return R2_.size( ); }

     Bool LeftUnforked( ) const { return L1( ).empty( ); }

     // MergeReadMeetingM fails if it can't add a given read X.

     This can change boundaries of M - how to report?

     Bool MergeReadMeetingM( const pp_read& X, int Xpos, int Mpos, 
          const vec<int>& L )
     {    ForceAssertEq( X[Xpos], M(Mpos) );
          for ( int i = Xpos - 1; i >= 0; i-- )
          {    int j = i - Xpos + Mpos;
               if ( j < 0 )
               {    if ( LeftUnforked( ) )
                    {    for ( int k = i; k >= 0; k-- )
                              M_.push_front( X[k] );    }
                    break;    }

               .........................

               if ( X[i] != M(j) ) break;    }

     private:

     pp_read L1_, L2_;
     pp_read M_;
     pp_read R1_, R2_;

};

// A forked_pair is a pair of forked_reads, with a gap value for the separation
// between their middles.

class forked_pair {

     public:

     forked_pair( ) { }
     forked_pair( const pp_pair& p ) : left_( p.Left( ) ), right_( p.Right( ) )
     {    SetGap( p.Gap( ) );
          SetDev( p.Dev( ) );    }

     const forked_read& Left( ) const { return left_; }
     const forked_read& Right( ) const { return right_; }
     forked_read& LeftMutable( ) { return left_; }
     forked_read& RightMutable( ) { return right_; }

     double Gap( ) const { return gap_; }
     void SetGap( double g ) { gap_ = g; }
     double Dev( ) const { return dev_; }
     void SetDev( double d ) { dev_ = d; }

     private:

     forked_read left_, right_;
     double gap_, dev_;

};

....................................................................................

// A bounded_pp_pair is a pp_pair with left/right boundary marks on each read.
// It is intended that what is between the marks is unbranched.

class bounded_pp_pair : public pp_pair {

     public:

     bounded_pp_pair( ) { }
     bounded_pp_pair( const pp_pair& p ) : pp_pair(p)
     {    startL_ = 0;
          stopL_ = p.LeftSize( );
          startR_ = 0;
          stopR_ = p.RightSize( );    }

     int StartL( ) const { return startL_; }
     int StopL( ) const { return stopL_; }
     int StartR( ) const { return startR_; }
     int StopR( ) const { return stopR_; }

     void SetStartL( int x ) { startL_ = x; }
     void SetStopL( int x ) { stopL_ = x; }
     void SetStartR( int x ) { startR_ = x; }
     void SetStopR( int x ) { stopR_ = x; }

     private:

     int startL_, stopL_;
     int startR_, stopR_;

};

Bool EnlargePair( bounded_pp_pair& p, const vec<pp_pair>& ppp, 
     const vec<Bool>& remove, const vec<int>& edge_copyno )
{    for ( int pass = 1; pass <= 2; pass++ )
     {    pp_read& ps = ( pass == 1 ? p.Left( ) : p.Right( ) );
          static vec<Bool> twopos;
          twopos.clear( );
          int start = ( pass == 1 ? p.StartL( ) : p.StartR( ) );
          int stop = ( pass == 1 ? p.StopL( ) : p.StopR( ) );
          for ( int i = start; i < stop; i++ )
               if ( edge_copyno[i] == 2 ) twopos.push_back(i);
          if ( twopos.empty( ) ) continue;
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove[i] ) continue;
               const pp_read& r = ( pass == 1 ? ppp[i].Left( ) : ppp[i].Right( ) );
               for ( int j = 0; j < r.isize( ); j++ )
               {    int v = r[j];
                    if ( edge_copyno[v] != 2 ) continue;
                    for ( int u = 0; u < twopos.isize( ); u++ )
                    {    if ( ps[ twopos[u] ] != v ) continue;
                         if ( twopos[u] < start || twopos[u] >= stop ) continue;
                         Bool changed = False;
                         Bool differ_low = False, differ_high = False;
                         int y;
                         for ( int x = twopos[u] - 1; x >= 0; x-- )
                         {    y = x - twopos[u] + j;
                              if ( y < 0 ) break;
                              if ( ps[x] != r[y] )
                              {    differ_low = True;
                                   if ( x > start )
                                   {    start = x;
                                        if ( pass == 1 ) p.SetStartL(start);
                                        else p.SetStartR(start);
                                        changed = True;    }
                                   break;    }    }
                         if ( !differ_low && y > 0 )
                         {    for ( int m = y-1; m >= 0; m-- )
                                   ...
                         for ( int x = twopos[u] + 1; x < ps.isize( ); x++ )
                         {    y = x - twopos[u] + j;
                              if ( y >= r.isize( ) ) break;
                              if ( ps[x] != r[y] )
                              {    differ_high = True;
                                   if ( x < stop )
                                   {    stop = x;
                                        if ( pass == 1 ) p.SetStopL(stop);
                                        else p.SetStopR(stop);
                                        changed = True;    }
                                   break;    }    }
          

void MergeCopyNumberTwoAlt( const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     vec<pp_pair>& ppp, const vec<int>& L, const vec<vrtx_t>& to_left_vertex, 
     const vec<vrtx_t>& to_right_vertex, const digraphE<int>& G, 
     const vec<int>& edge_copyno, vec<Bool>& remove )
{    for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          const pp_pair& p = ppp[i];
          for ( int pass = 1; pass <= 2; pass++ )
          {    const pp_read& r = ( pass == 1 ? p.Left( ) : p.Right( ) );
               for ( int j = 0; j < r.isize( ); j++ )
               {    int v = r[j];
                    if ( edge_copyno[v] != 2 ) continue;

*/
