/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "math/Functions.h"
#include "graph/Digraph.h"
#include "paths/BridgeGaps.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/MergeAndTighten.h"
#include "paths/MergeCopyNumberTwo.h"
#include "paths/PairedPair.h"
#include "paths/PairGraph.h"
#include "paths/simulation/Placement.h"

Bool CloseToOrigin( const pp_pair& p, const vec<Bool>& hstart_known,
     const vec<int>& hstart, const vec<Bool>& safe_unique, 
     const vec<int>& pppL, const int nhood_radius )
{    for ( int j = p.LeftSize( ) - 1; j >= 0; j-- )
     {    if ( hstart_known[ p.Left(j) ] && safe_unique[ p.Left(j) ] )
          {    int center = hstart[ p.Left(j) ];
               for ( int m = j; m < p.LeftSize( ); m++ )
                    center += pppL[ p.Left(m) ];
               if ( Abs(center) <= nhood_radius/2 ) return True;    }    }
     for ( int j = 0; j < p.RightSize( ); j++ )
     {    if ( hstart_known[ p.Right(j) ] && safe_unique[ p.Right(j) ] )
          {    int center = hstart[ p.Right(j) ];
               for ( int m = 0; m < j; m++ )
                    center -= pppL[ p.Right(m) ];
               if ( Abs(center) <= nhood_radius/2 ) return True;    }    }
     return False;    }

void MergeAndTighten( vec<pp_pair>& ppp, const vec<int>& pppL, 
     vec<Bool> safe_unique, const double dmult,
     const vec<vrtx_t>& to_right_vertex, const vec<vrtx_t>& to_left_vertex,
     const vec<Bool>& hstart_known, const vec<int>& hstart, 
     const int NHOOD_RADIUS, const int NHOOD_RADIUS_INTERNAL,
     const digraphE<int>& G, const vec<Bool>& editable, const vec<int>& edge_copyno,
     const vec<double>& edge_copyno_p, vec<Bool>& remove, const Bool verbose, 
     const Bool MERGE_COPY_NUMBER_TWO, const int SHOW_PAIR_PLACEMENTS, 
     const String& sub_dir, const HyperKmerPath& h, const KmerBaseBroker& kbb, 
     const vecbasevector& genome, const serfvec<placement>& locsv, 
     const Bool MERGE_COPY_NUMBER_TWO_VERBOSE )

{    
     // If an edge contains the same unique element twice, than that element is
     // declared unsafe.

     for ( int i = 0; i < ppp.isize( ); i++ )
     {    static vec<int> r;
          r = ppp[i].Left( );
          Sort(r);
          for ( int j = 1; j < r.isize( ); j++ )
          {    if ( r[j] == r[j-1] && safe_unique[ r[j] ] )
                    safe_unique[ r[j] ] = False;    }
          r = ppp[i].Right( );
          Sort(r);
          for ( int j = 1; j < r.isize( ); j++ )
          {    if ( r[j] == r[j-1] && safe_unique[ r[j] ] )
                    safe_unique[ r[j] ] = False;    }    }

     // Start main loop.

     while(1)
     {    int merges = 0;

          // Close edges where possible (changing dev to zero).  Note that we do
          // this even in the non-editable case!

          vec<pp_pair> ppp_old(ppp);
          int last_merges = merges;
          SimpleExtendPairedPairs( h, kbb, ppp, pppL, dmult, remove, 
               False, verbose );
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( remove[j] ) continue;
               if ( ppp_old[j] != ppp[j] ) 
               {    if (verbose) cout << "\nclosing " << ppp_old[j] << " to yield "
                         << ppp[j] << "\n";
                    ++merges;    }    }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after simple merging", 
                    sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove );    } 
          int max_loops = 100;
          last_merges = merges;
          ppp_old = ppp;
          static vec<Bool> remove_or_not_editable;
          remove_or_not_editable = remove;
          for ( int i = 0; i < editable.isize( ); i++ )
               if ( !editable[i] ) remove_or_not_editable[i] = True;
          merges += ClosePairedPairsDirectUnique( 
               ppp, h, max_loops, remove_or_not_editable );
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( remove[j] ) continue;
               if ( ppp_old[j].Dev( ) != ppp[j].Dev( ) ) 
               {    if (verbose) cout << "\nclosing " << ppp_old[j] << " to yield "
                         << ppp[j] << "\n";
                    ++merges;    }    }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after direct merging", 
                    sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove );    } 

          // Merge two edges if they share unique sequence on one side and can be
          // unambiguously merged.

          last_merges = merges;
          for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
          {    if ( ( i1 < editable.isize( ) && !editable[i1] ) || remove[i1] ) 
                    continue;
               const pp_pair& p1 = ppp[i1];
               static vec<int> l1, r1;
               l1 = p1.Left( ), r1 = p1.Right( );
               UniqueSort(l1), UniqueSort(r1);
               for ( int i2 = i1 + 1; i2 < ppp.isize( ); i2++ )
               {    if ( ( i2 < editable.isize( ) && !editable[i2] ) || remove[i2] )
                         continue;
                    const pp_pair& p2 = ppp[i2];
                    static vec<int> l2, r2;
                    l2 = p2.Left( ), r2 = p2.Right( );
                    UniqueSort(l2), UniqueSort(r2);
                    static vec<int> l, r;
                    l = Intersection( l1, l2 ), r = Intersection( r1, r2 );
                    int lunique = -1, runique = -1;
                    for ( int j = 0; j < l.isize( ); j++ )
                         if ( safe_unique[ l[j] ] ) lunique = l[j];
                    for ( int j = 0; j < r.isize( ); j++ )
                         if ( safe_unique[ r[j] ] ) runique = r[j];
                    if ( lunique < 0 && runique < 0 ) continue;
                    vec<pp_mpair> pnew;
                    int g = -1;
                    Bool fail = False;
                    for ( int pass = 1; pass <= 2; pass++ )
                    {    double dadd = ( pass == 1 ? 0.0 : 0.5 );
                         FindJoins( pp_mpair(p1), p2, pppL, dmult + dadd, pnew, 
                              1, 1, False );
                         vec<Bool> pnew_bad( pnew.size( ), False );
                         for ( int j = 0; j < pnew.isize( ); j++ )
                         {    if ( lunique >= 0 )
                              {    int c = 0;
                                   for ( int x = 0; x < pnew[j].LeftSize( ); x++ )
                                        if ( pnew[j].Left(x) == lunique ) ++c;
                                   if ( c != 1 ) pnew_bad[j] = True;    }
                              if ( runique >= 0 )
                              {    int c = 0;
                                   for ( int x = 0; x < pnew[j].RightSize( ); x++ )
                                        if ( pnew[j].Right(x) == runique ) ++c;
                                   if ( c != 1 ) pnew_bad[j] = True;    }    }
                         if ( pnew.isize( ) - Sum(pnew_bad) != 1 )
                         {    fail = True;
                              break;    }
                         for ( int j = 0; j < pnew.isize( ); j++ )
                              if ( !pnew_bad[j] ) g = j;    }
                    if (fail) continue;
                    pp_pair p = pnew[g];
     
                    // If join is on only one side, make sure that other side on the
                    // two paths couldn't be disjoint.
     
                    double maxdev = dmult * ( p1.Dev( ) + p2.Dev( ) );
                    if ( lunique >= 0 && runique < 0 )
                    {    int offset = Position( p1.Left( ), lunique ) 
                              - Position( p2.Left( ), lunique );
                         int protrusion2 = 0;
                         if ( p1.LeftSize( ) - offset <= p2.LeftSize( ) )
                         {    for ( int j = p1.LeftSize( ) - offset; 
                                   j < p2.LeftSize( ); j++ )
                              {    protrusion2 += pppL[ p2.Left(j) ];    }    }
                         else
                         {    for ( int j = offset + p2.LeftSize( ); 
                                   j < p1.LeftSize( ); j++ )
                              {    protrusion2 -= pppL[ p1.Left(j) ];    }    }
                         int gapdiff1 = -protrusion2, gapdiff2 = protrusion2;
                         for ( int j = 0; j < p1.RightSize( ); j++ )
                              gapdiff1 += pppL[ p1.Right(j) ];
                         double space1 
                              = maxdev - ( gapdiff1 - ( p2.Gap( ) - p1.Gap( ) ) );
                         if ( space1 >= 0 )
                         {    if ( G.ThisClose( 
                                   to_right_vertex[ p1.Right( ).back( ) ],
                                   to_left_vertex[ p2.Right( ).front( ) ], 
                                   int(floor(space1)) ) )
                              {    continue;    }    }
                         for ( int j = 0; j < p2.RightSize( ); j++ )
                              gapdiff2 += pppL[ p2.Right(j) ];
                         double space2
                              = maxdev - ( gapdiff2 - ( p1.Gap( ) - p2.Gap( ) ) );
                         if ( space2 >= 0 )
                         {    if ( G.ThisClose( 
                                   to_right_vertex[ p2.Right( ).back( ) ],
                                   to_left_vertex[ p1.Right( ).front( ) ], 
                                   int(floor(space2)) ) )
                              {    continue;    }    }    }
                    if ( lunique < 0 && runique >= 0 )
                    {    int offset = Position( p1.Right( ), runique ) 
                              - Position( p2.Right( ), runique );
                         int protrusion1 = 0;
                         if ( offset >= 0 )
                         {    for ( int j = 0; j < offset; j++ )
                                   protrusion1 += pppL[ p1.Right(j) ];    }
                         else
                         {    for ( int j = 0; j < -offset; j++ )
                                   protrusion1 -= pppL[ p2.Right(j) ];    }
                         int gapdiff1 = -protrusion1, gapdiff2 = protrusion1;
                         for ( int j = 0; j < p2.LeftSize( ); j++ )
                              gapdiff1 += pppL[ p2.Left(j) ];
                         double space1
                              = maxdev - ( gapdiff1 - ( p1.Gap( ) - p2.Gap( ) ) );
                         if ( space1 >= 0 )
                         {    if ( G.ThisClose( to_right_vertex[ p1.Left().back() ],
                                   to_left_vertex[ p2.Left( ).front( ) ],
                                   int(floor(space1)) ) )
                              {    continue;    }    }
                         for ( int j = 0; j < p1.LeftSize( ); j++ )
                              gapdiff2 += pppL[ p1.Left(j) ];
                         double space2 
                              = maxdev - ( gapdiff2 - ( p2.Gap( ) - p1.Gap( ) ) );
                         if ( space2 >= 0 )
                         {    if ( G.ThisClose( to_right_vertex[ p2.Left().back() ],
                                   to_left_vertex[ p1.Left( ).front( ) ],
                                   int(floor(space2)) ) )
                              {    continue;    }    }    }

                    // Merge.

                    if (verbose) 
                    {    int uniq = ( lunique >= 0 ? lunique : runique );
                         cout << "\n1merging (using unique " << BaseAlpha(uniq) 
                              << ") " << ppp[i1] << " and " << ppp[i2] 
                              << " to yield " << p << "\n";    }
                    ++merges;
                    remove[i2] = True;
                    ppp[i1] = p;    }    }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after 1merging", sub_dir, 
                    ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
                    locsv, remove );    } 

          // Merge pairs if one has been closed and the other overlaps it along
          // unique sequence.  In this case, the first pair is treated as a single
          // read:
          //     ----left1=right1-------
          //                ---left2------  ----right2---
          // or with right2 overlapping instead of left2.

          last_merges = merges;
          if ( merges == 0 )
          {
          static vec<Bool> close_to_origin;
          close_to_origin.resize( ppp.size( ) );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    close_to_origin[i] = CloseToOrigin( ppp[i], hstart_known, hstart, 
                    safe_unique, pppL, NHOOD_RADIUS_INTERNAL );    }
          for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
          {    if ( ( i1 < editable.isize( ) && !editable[i1] ) || remove[i1] ) 
                    continue;
               const pp_pair& p1 = ppp[i1];
               if ( p1.Dev( ) > 0 || p1.Left( ) != p1.Right( ) ) continue;
               static vec<int> x1;
               x1 = p1.Left( );
               UniqueSort(x1);
               for ( int i2 = 0; i2 < ppp.isize( ); i2++ )
               {    if ( ( i2 < editable.isize( ) && !editable[i2] ) || remove[i2] 
                         || i1 == i2 ) 
                    {    continue;    }
                    if ( !close_to_origin[i2] ) continue;
                    const pp_pair& p2 = ppp[i2];
                    static vec<int> l2, r2, l, r, offsets;
                    l2 = p2.Left( ), r2 = p2.Right( );
                    UniqueSort(l2), UniqueSort(r2);
                    l = Intersection( x1, l2 ), r = Intersection( x1, r2 );
                    int lunique = -1, runique = -1;
                    for ( int j = 0; j < l.isize( ); j++ )
                         if ( safe_unique[ l[j] ] ) lunique = l[j];
                    for ( int j = 0; j < r.isize( ); j++ )
                         if ( safe_unique[ r[j] ] ) runique = r[j];
                    int lunique0 = -1, runique0 = -1;
                    for ( int j = 0; j < l2.isize( ); j++ )
                         if ( safe_unique[ l2[j] ] ) lunique0 = l2[j];
                    for ( int j = 0; j < r2.isize( ); j++ )
                         if ( safe_unique[ r2[j] ] ) runique0 = r2[j];
                    double gap = p2.Gap( );
                    Bool replaced = False;
                    if ( lunique >= 0 && runique0 >= 0 )
                    {    GetOverlaps( p1.Right( ), p2.Left( ), offsets );
                         for ( int j = 0; j < offsets.isize( ); j++ )
                         {    pp_read left;
                              JoinReads( p1.Right( ), p2.Left( ), offsets[j], left );
                              if ( left.CountValue(lunique) != 1 ) continue;
                              int overhang1 
                                   = p1.RightSize( ) - offsets[j] - p2.LeftSize( );
                              for ( int u = 1; u <= overhang1; u++ )
                                   gap -= pppL[ p1.Right( p1.RightSize( ) - u ) ];
                              if ( gap < -dmult * p2.Dev( ) ) continue;
                              if (verbose) cout << "\n2merging " << ppp[i1] 
                                   << " and " << ppp[i2] << " to yield " 
                                   << pp_pair( left, p2.Right( ), gap, p2.Dev( ) )
                                   << "\n";
                              ++merges;
                              replaced = True;
                              remove[i2] = True;
                              ppp[i1] = pp_pair( left, p2.Right( ), gap, p2.Dev( ) );
                              break;    }    }
                    else if ( runique >= 0 && lunique0 >= 0 )
                    {    GetOverlaps( p1.Left( ), p2.Right( ), offsets );
                         for ( int j = 0; j < offsets.isize( ); j++ )
                         {    pp_read right;
                              JoinReads( p1.Left(), p2.Right(), offsets[j], right );
                              if ( right.CountValue(runique) != 1 ) continue;
                              for ( int u = 0; u < offsets[j]; u++ )
                                   gap -= pppL[ p1.Left(u) ];
                              if ( gap < -dmult * p2.Dev( ) ) continue;
                              pp_pair p( p2.Left( ), right, gap, p2.Dev() );
                              if (verbose) cout << "\n4merging " << ppp[i1] 
                                   << " and " << ppp[i2] << " to yield " << p 
                                   << "\n";
                              ++merges;
                              replaced = True;
                              remove[i2] = True;
                              ppp[i1] = p;
                              close_to_origin[i1] = CloseToOrigin( p, hstart_known, 
                                   hstart, safe_unique, pppL, 
                                   NHOOD_RADIUS_INTERNAL );
                              break;    }    }    
                    if (replaced) break;    }    }
               }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after 2+4merging", sub_dir, 
                    ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
                    locsv, remove );    }

          // Similar to previous, but now the picture is:
          //
          //     ----left1=right1----------------------------
          //                ---left2------  ----right2---
          // or with right2 overlapping instead of left2.
          // Also, use a slightly larger value for dmult.

          last_merges = merges;
          if ( merges == 0 )
          {
          static vec<Bool> close_to_origin;
          close_to_origin.resize( ppp.size( ) );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    close_to_origin[i] = CloseToOrigin( ppp[i], hstart_known, hstart, 
                    safe_unique, pppL, NHOOD_RADIUS_INTERNAL );    }
          for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
          {    if ( ( i1 < editable.isize( ) && !editable[i1] ) || remove[i1] ) 
                    continue;
               const pp_pair& p1 = ppp[i1];
               if ( p1.Dev( ) > 0 || p1.Left( ) != p1.Right( ) ) continue;
               static vec<int> x1;
               x1 = p1.Left( );
               UniqueSort(x1);
               for ( int i2 = 0; i2 < ppp.isize( ); i2++ )
               {    if ( ( i2 < editable.isize( ) && !editable[i2] ) || remove[i2] 
                         || i1 == i2 ) 
                    {    continue;    }
                    if ( !close_to_origin[i2] ) continue;
                    const pp_pair& p2 = ppp[i2];
                    static vec<int> l2, r2, l, r, offsets1, offsets2;
                    l2 = p2.Left( ), r2 = p2.Right( );
                    UniqueSort(l2), UniqueSort(r2);
                    l = Intersection( x1, l2 ), r = Intersection( x1, r2 );
                    int lunique = -1, runique = -1;
                    for ( int j = 0; j < l.isize( ); j++ )
                         if ( safe_unique[ l[j] ] ) lunique = l[j];
                    for ( int j = 0; j < r.isize( ); j++ )
                         if ( safe_unique[ r[j] ] ) runique = r[j];
                    Bool replaced = False;
                    if ( lunique < 0 && runique < 0 ) continue;
                    // A bit silly -- p1.Right = p1.Left:
                    GetOverlaps( p1.Right( ), p2.Left( ), offsets1 );
                    GetOverlaps( p1.Left( ), p2.Right( ), offsets2 );
                    for ( int j1 = 0; j1 < offsets1.isize( ); j1++ )
                    {    int o1 = offsets1[j1];
                         if ( o1 < 0 || o1 + p2.RightSize( ) > p1.LeftSize( ) )
                              continue;
                         for ( int j2 = 0; j2 < offsets2.isize( ); j2++ )
                         {    int o2 = offsets2[j2];
                              if ( o2 < 0 || o2 + p2.LeftSize( ) > p1.RightSize( ) )
                                   continue;
                              int start = o1 + p2.LeftSize( );
                              int stop = o2;
                              if ( !( start <= stop ) ) continue;
                              if (verbose)
                              {    cout << "\nCONSIDERING\n";
                                   cout << ppp[i1] << "\n";
                                   cout << "AND\n";
                                   cout << ppp[i2] << "\n";    }
                              int gap = 0;
                              for ( int u = start; u < stop; u++ )
                                   gap += pppL[ p1.Left(u) ];
                              cout << "gap = " << gap
                                   << ", p2.Gap( ) = " << setprecision(3)
                                   << p2.Gap( ) << ", p2.Dev( ) = "
                                   << setprecision(3) << p2.Dev( ) << "\n";
                              double gapdiff = Abs( double(gap) - p2.Gap( ) );
                              if (verbose)
                              {    PRINT(gapdiff);
                                   cout << setprecision(3)
                                        << double(gapdiff)/double( p2.Dev( ) )
                                        << "\n";    }
                              if ( gapdiff > ( dmult + 1.0 ) * p2.Dev( ) ) continue;
                              ++merges;
                              if (verbose) cout << "MERGING!\n";
                              replaced = True;
                              remove[i2] = True;
                              break;    }
                         if (replaced) break;    }    }    }

               }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after more merging", 
                    sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove );    }

          // If an edge contains the same unique element twice, than that element is
          // declared unsafe.  (Ditto previous instance.)
               
          last_merges = merges;
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    static vec<int> r;
               r = ppp[i].Left( );
               Sort(r);
               for ( int j = 1; j < r.isize( ); j++ )
               {    if ( r[j] == r[j-1] && safe_unique[ r[j] ] )
                         safe_unique[ r[j] ] = False;    }
               r = ppp[i].Right( );
               Sort(r);
               for ( int j = 1; j < r.isize( ); j++ )
               {    if ( r[j] == r[j-1] && safe_unique[ r[j] ] )
                         safe_unique[ r[j] ] = False;    }    }

          // If a read is extended by another read, and the extension includes a
          // unique edge, extend it.

          if ( merges == 0 )
          {

          for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
          {    if ( ( i1 < editable.isize( ) && !editable[i1] ) || remove[i1] ) 
                    continue;
               pp_pair& p1 = ppp[i1];
               for ( int pass1 = 1; pass1 <= 2; pass1++ )
               {    pp_read& x 
                         = ( pass1 == 1 ? p1.LeftMutable( ) : p1.RightMutable( ) );
                    for ( int i2 = 0; i2 < ppp.isize( ); i2++ )
                    {    if ( ( i2 < editable.isize( ) && !editable[i2] ) 
                              || i2 == i1 || remove[i2] ) 
                         {    continue;    }
                         const pp_pair& p2 = ppp[i2];
                         for ( int pass2 = 1; pass2 <= 2; pass2++ )
                         {    const pp_read& y 
                                   = ( pass2 == 1 ? p2.Left( ) : p2.Right( ) );
                              static vec<int> offsets;
                              GetOverlaps( x, y, offsets );
                              for ( int u = 0; u < offsets.isize( ); u++ )
                              {    int o = offsets[u];
                                   if ( o >= 0 && y.isize( ) + o <= x.isize( ) )
                                        continue;
                                   Bool have_unique = False;
                                   for ( int m = Max( 0, o ); 
                                        m < Min( x.isize( ), o + y.isize( ) ); m++ )
                                   {    if ( safe_unique[ x[m] ] )
                                        {    if (verbose)
                                             {    cout << "see unique "
                                                       << BaseAlpha( x[m] ) 
                                                       << "\n";    }
                                             have_unique = True;    }    }
                                   if ( !have_unique ) continue;
                                   pp_read xnew;
                                   JoinReads( x, y, o, xnew );
                                   if (verbose) cout << "\nBEFORE: " << p1 << "\n";
                                   double gap = p1.Gap( );
                                   if ( pass1 == 1 && o + y.isize( ) > x.isize( ) )
                                   {    for ( int z = x.isize( ); 
                                             z < o + y.isize( ); z++ )
                                        {    gap -= pppL[ y[ z - o ] ];    }    }
                                   if ( pass1 == 2 && o < 0 )
                                   {    for ( int z = 0; z < -o; z++ )
                                        {    gap -= pppL[ y[z] ];    }    }
                                   x = xnew;
                                   static vec<int> r;
                                   r = x;
                                   Sort(r);
                                   for ( int j = 1; j < r.isize( ); j++ )
                                   {    if ( r[j] == r[j-1] && safe_unique[ r[j] ] )
                                             safe_unique[ r[j] ] = False;    }
                                   p1.SetGap(gap);
                                   if (verbose) 
                                        cout << "EXTENDING USING " << p2 << "\n";
                                   if (verbose) cout << "AFTER:  " << p1 << "\n";
                                   ++merges;
                                   break;    }    }    }    }    }

          }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after even more merging", 
                    sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove );    }

          // If pair x contains unipath u exactly once, and 
          // prob( copynumber(u) = n ) >= 0.999, and u appears n times in pair y,
          // and for each of those n times x aligns to y and is subsumed by it, then
          // delete x.  Note that we could instead use the condition
          // prob( copynumber(u) <= n ) >= 0.999, but that would require a tad more
          // engineering to make the relevant probability data available here.
          //
          // Relaxed a bit.

          last_merges = merges;
          static vec<Bool> u_good;
          u_good.resize_and_set( edge_copyno.size( ), False );
          for ( int i = 0; i < edge_copyno.isize( ); i++ )
          {    if ( edge_copyno[i] >= 2 && edge_copyno_p[i] >= 0.999 )
                    u_good[i] = True;    }
          static vec< vec<int> > u_home, u_home_count;
          u_home.clear( ), u_home_count.clear( );
          u_home.resize( edge_copyno.size( ) );
          u_home_count.resize( edge_copyno.size( ) );
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( remove[i] ) continue;
               const pp_pair& p = ppp[i];
               Bool closed = IsClosed( p, pppL );
               Bool open = ( p.Gap( ) >= 3.0 * p.Dev( ) );
               if ( !open && !closed ) continue; // lazy
               static vec<int> us;
               us = p.Left( );
               if ( !closed ) us.append( p.Right( ) );
               Sort(us);
               for ( int j = 0; j < us.isize( ); j++ )
               {    int l;
                    for ( l = j + 1; l < us.isize( ); l++ )
                         if ( us[l] != us[j] ) break;
                    int u = us[j], n = l - j;
                    if ( !u_good[u] || edge_copyno[u] != n ) continue;
                    u_home[u].push_back(i), u_home_count[u].push_back(n);
                    u = l - 1;    }    }
          for ( int pi = 0; pi < ppp.isize( ); pi++ )
          {    if ( remove[pi] ) continue;
               const pp_pair& p = ppp[pi];
               Bool p_closed = IsClosed( p, pppL );
               Bool p_open = ( p.Gap( ) >= 3.0 * p.Dev( ) );
               if ( !p_open && !p_closed ) continue; // lazy
               static vec<int> us;
               us = p.Left( );
               if ( !p_closed ) us.append( p.Right( ) );
               Sort(us);
               for ( int j = 0; j < us.isize( ); j++ )
               {    int l;
                    for ( l = j + 1; l < us.isize( ); l++ )
                         if ( us[l] != us[j] ) break;
                    int u = us[j], n = l - j;
                    if ( n > 1 ) 
                    {    j = l - 1;
                         continue;    }
                    Bool on_left = Member( p.Left( ), u );
                    Bool on_right = Member( p.Right( ), u );
                    for ( int m = 0; m < u_home[u].isize( ); m++ )
                    {    int qi = u_home[u][m];
                         if ( remove[qi] ) continue;
                         const pp_pair& q = ppp[qi];
                         Bool q_closed = IsClosed( q, pppL );
                         Bool q_open = ( q.Gap( ) >= 3.0 * q.Dev( ) );
                         int qcount = u_home_count[u][m];
                         static vec<int> o1, o2;
                         if ( p_closed && q_closed )
                         {    GetOverlaps( p.Left( ), q.Left( ), o1, True );
                              if ( o1.isize( ) != qcount ) continue;    }
                         if ( p_closed && q_open )
                         {    GetOverlaps( p.Left( ), q.Left( ), o1, True );
                              GetOverlaps( p.Left( ), q.Right( ), o2, True );
                              if ( o1.isize( ) + o2.isize( ) != qcount ) 
                                   continue;    }
                         if ( p_open && q_closed )
                         {    static vec<int> ulocsq;
                              ulocsq.clear( );
                              for ( int l = 0; l < q.LeftSize( ); l++ )
                                   if ( q.Left(l) == u ) ulocsq.push_back(l);
                              Bool bad = False;
                              if (on_left)
                              {    int ulocp = Position( p.Left( ), u );
                                   for ( int l = 0; l < ulocsq.isize( ); l++ )
                                   {    int ulocq = ulocsq[l];
                                        int offset = ulocp - ulocq;
                                        if ( !ValidOffset( p.Left( ), q.Left( ), 
                                             offset ) ) 
                                        {    continue;    }
                                        if ( offset > 0 ) bad = True;
                                        if ( offset + q.LeftSize( ) < p.LeftSize( ) )
                                             bad = True;
                                        int q_to_right = 0;
                                        for ( int m = p.LeftSize( ) - offset;
                                             m < q.LeftSize( ); m++ )
                                        {    q_to_right += pppL[ q.Left(m) ];    }
                                        int p_to_right = int( floor( 
                                             p.Gap( ) + dmult * p.Dev( ) ) );
                                        for ( int m = 0; m < p.RightSize( ); m++ )
                                             p_to_right += pppL[ p.Right(m) ];
                                        if ( p_to_right > q_to_right )
                                             bad = True;    }    }
                              if (on_right)
                              {    int ulocp = Position( p.Right( ), u );
                                   for ( int l = 0; l < ulocsq.isize( ); l++ )
                                   {    int ulocq = ulocsq[l];
                                        int offset = ulocp - ulocq;
                                        if ( !ValidOffset( p.Right( ), q.Left( ), 
                                             offset ) ) 
                                        {    continue;    }
                                        if ( offset > 0 ) bad = True;
                                        if ( offset + q.LeftSize() < p.RightSize() )
                                             bad = True;
                                        int q_to_left = 0;
                                        for ( int m = 0; m < -offset; m++ )
                                        {    q_to_left += pppL[ q.Left(m) ];    }
                                        int p_to_left = int( floor( 
                                             p.Gap( ) + dmult * p.Dev( ) ) );
                                        for ( int m = 0; m < p.LeftSize( ); m++ )
                                             p_to_left += pppL[ p.Left(m) ];
                                        if ( p_to_left > q_to_left )
                                             bad = True;    }    }
                              if (bad) continue;    }
                         if ( p_open && q_open ) continue; // lazy
                         remove[pi] = True;
                         ++merges;
                         if (verbose)
                         {    cout << "\nCount test, u = " << BaseAlpha(u)
                                   << " (" << qcount << " times): eliminating " 
                                   << p << " using " << q << ".\n";    }
                         break;    }
                    if ( remove[pi] ) break;
                    j = l - 1;    }    }
          if ( SHOW_PAIR_PLACEMENTS >= 2 && merges > last_merges )
          {    PrintPairLocs( True, "in MergeAndTighten, after count test", 
                    sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, locsv, remove );    }

          if ( merges == 0 ) break;    }

     if (MERGE_COPY_NUMBER_TWO)
     {    double c2timer = WallClockTime( );
          if ( SHOW_PAIR_PLACEMENTS >= 1 )
          {    PrintPairLocs( True, "in MergeAndTighten, before calling "
                    "MergeCopyNumberTwo", sub_dir, ppp, h, kbb, genome, 
                    NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, locsv, remove );    } 
          MergeCopyNumberTwo( h, kbb, ppp, pppL, to_left_vertex, to_right_vertex, 
               G, edge_copyno, remove, MERGE_COPY_NUMBER_TWO_VERBOSE );    
          if ( SHOW_PAIR_PLACEMENTS >= 1 )
          {    PrintPairLocs( True, "in MergeAndTighten, after calling "
                    "MergeCopyNumberTwo", sub_dir, ppp, h, kbb, genome, 
                    NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, locsv, remove );    } 
          MergeAndTighten( ppp, pppL, safe_unique, dmult, to_right_vertex, 
               to_left_vertex, hstart_known, hstart, NHOOD_RADIUS, 
               NHOOD_RADIUS_INTERNAL, G, editable, edge_copyno, edge_copyno_p, 
               remove, verbose, False, SHOW_PAIR_PLACEMENTS, sub_dir, h, kbb, 
               genome, locsv, MERGE_COPY_NUMBER_TWO_VERBOSE );
          cout << TimeSince(c2timer) << " used in MergeCopyNumberTwo\n";    }    }
