/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// MakeRcDb: Given a paths file, make the corresponding rc and db.

#include "MainTools.h"
#include "paths/KmerPath.h"

// class TAG can be either a tagged_rpint or a big_tagged_rpint
template<class TAG> void BuildDbFile( const vecKmerPath& paths,
				      const vecKmerPath& paths_rc,
				      longlong nsegs,
				      String pathsdb_file ) {
  cout << "Building pathsdb" << flush;

  vec<TAG> segs;
  segs.reserve(nsegs);
  
  for ( int pass = 1; pass <= 2; pass++ ) {
    const vecKmerPath& v = ( pass == 1 ? paths : paths_rc );
    for ( int i = 0; i < v.size( ); i++ ) {
      int ind = ( pass == 1 ? i : -i-1 );
      for ( unsigned int j = 0; int(j) < v[i].NSegments( ); j++ ) {
	const KmerPathInterval& I = v[i].Segment(j);
	if( ! I.isGap() ) {
	  ForceAssertLe( j, TAG::POSITION_MAX );
	  segs.push_back( TAG( I.Start(), I.Length(), ind, j ) );
	}
      }
    }
  }

  cout << ", preparing" << flush;
  Prepare(segs);

  cout << ", writing" << flush;
  BinaryWrite3( pathsdb_file, segs );

  cout << ", done" << endl;
  return;
}



int main( int argc, char *argv[] )
{
  RunTime( );

  BeginCommandArguments;
  CommandArgument_String(PRE);
  CommandArgument_String(DATA);
  CommandArgument_String(RUN);
  CommandArgument_String_OrDefault(READS, "reads");
  CommandArgument_Int(K);
  CommandArgument_Bool_OrDefault(FORCE_BIG, False);
  EndCommandArguments;

  String run_dir = PRE + "/" + DATA + "/" + RUN;
  String KS = String(".k") + ToString(K);

  cout << "Loading paths" << flush;
  vecKmerPath paths( run_dir + "/" +READS+ ".paths" + KS );
  cout << ", done" << endl;

  int max_segs=0;
  longlong num_segs=0;

  cout << "Building paths_rc" << flush;
  vecKmerPath paths_rc;
  paths_rc.Reserve( paths.rawsize(), paths.size() );
  KmerPath temp;
  for(int i=0; i<paths.size(); i++) {
    temp = paths[i];
    max_segs = Max( max_segs, temp.NSegments() );
    num_segs += temp.NSegments();
    temp.Reverse();
    max_segs = Max( max_segs, temp.NSegments() );
    num_segs += temp.NSegments();
    paths_rc.push_back(temp);
  }
  cout << " and writing" << flush;
  paths_rc.WriteAll( run_dir + "/"+READS+".paths_rc" + KS );
  cout << ", done" << endl;

  bool big_reads = uint(max_segs) >= tagged_rpint::POSITION_MAX;
  bool big = big_reads || FORCE_BIG;

  if( big_reads )
    cout << "   NOTE: the longest KmerPath has " << max_segs << " segments."
	 << "\n   This is larger than the " <<  tagged_rpint::POSITION_MAX
	 << " limit for a tagged_rpint,"
	 << "\n   so building a vec<big_tagged_rpint> instead" << endl;
  else if( FORCE_BIG )
    cout << "FORCE_BIG=True, so building a vec<big_tagged_rpint>,"
	 << "\neven though there are no long paths which require it." << endl;

  String pathsdb_file = 
    run_dir + "/" + READS + ".paths" + ( big ? "db_big" : "db") + KS;

  if( big )
    BuildDbFile<big_tagged_rpint>(paths, paths_rc, num_segs, pathsdb_file );
  else
    BuildDbFile<tagged_rpint>(paths, paths_rc, num_segs, pathsdb_file );

}
