/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "paths/LocalizeReadsAnnex.h"

#include <unistd.h>
#include <strstream>
#include <iterator>
#include <algorithm>

#include "Alignment.h"
#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "math/Functions.h"
#include "math/HoInterval.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "ReadLocation.h"
#include "ReadPairing.h"
#include "TaskTimer.h"
#include "STLExtensions.h"
#include "lookup/LookAlign.h"
#include "lookup/QueryLookupTableCore.h"
#include "paths/AlignHyperKmerPath.h"
#include "paths/EvalUtils.h"
#include "paths/ExtendUnipathSeqs.h"
#include "paths/HyperKmerPath.h"
#include "paths/InternalMerge.h"
#include "paths/KmerBaseBroker.h"
#include "paths/KmerPath.h"
#include "paths/KmerPathDatabase.h"
#include "paths/KmerPathMuxSearcher.h"
#include "paths/PairGraph.h"
#include "paths/PairedPair.h"
#include "paths/ReadsToPathsCoreX.h"
#include "paths/SimpleLoop.h"
#include "paths/SubMuxGraphBuilder.h"
#include "paths/Unipath.h"
#include "paths/UnipathNhood.h"
#include "paths/UnipathSeqDatabase.h"
#include "paths/simulation/Placement.h"
#include "paths/simulation/SimTrueSequenceBroker.h"

// AlignNhoodStuffToReference.  Given sequences that may be from a neighborhood N,
// align to the reference, and report the alignments of those that are in fact 
// within N.

void AlignNhoodStuffToReference( const vecbasevector& seqs, int v, int NHOOD_RADIUS,
     const vecvec<placement>& locs, const vecbasevector& genome,
     const vec<int>& genome_path_lengths, vec<alignment_plus>& aligns )
{    int K = 96;
     for ( int i = 0; i < seqs.size( ); i++ )
     {    if ( seqs[i].size( ) == 0 ) continue;
          K = Min( K, seqs[i].isize( ) );    }
     if ( K < 24 ) K = 20;
     else if ( K < 48 ) K = 24;
     else if ( K < 96 ) K = 48;
     static vec<int> gs, starts;
     gs.clear( ), starts.clear( );
     static vecbasevector bases;
     bases = seqs;
     for ( int t = 0; t < locs[v].size( ); t++ )
     {    const placement& p = locs[v][t];
          int g = p.GenomeId( );
          gs.push_back(g);
          int start = Max( 0, p.pos( ) - NHOOD_RADIUS );
          starts.push_back(start);
          int stop = Min( genome_path_lengths[g], p.Pos( ) + NHOOD_RADIUS );
          static basevector b;
          b.SetToSubOf( genome[g], start, stop - start );
          bases.push_back_reserve(b);    }
     PerfectAligner pal( K, PerfectAligner::findProperOnly );
     pal.Align( bases, aligns, seqs.size( ) );
     for ( int u = 0; u < aligns.isize( ); u++ )
     {    alignment_plus& ap = aligns[u];
          static align a;
          a.UnpackFrom(ap.a);
          a.Setpos2( ap.a.pos2( ) + starts[ ap.Id2( ) ] );
          ap.a.Set(a);
          ap.SafeSetId2( gs[ ap.Id2( ) ] );    }    }

// PlacePairsOnPaths: given reads (given by paths, paths_rc), in pairs
// (given by pairs), try to place the pairs on another set of paths 
// (given by upaths, upaths_to_use, upathsdb), requiring that both ends are placed
// from end to end and that the pair stretch is at most 3 deviations.
// Return (u, start, stop), giving the start and stop kmer on upath u.  Multiple
// placements are allowed.

void PlacePairsOnPaths( 
     /* inputs: */
     const vec<read_pairing>& pairs, const vecKmerPath& paths,
     const vecKmerPath& paths_rc, const vecKmerPath& upaths, 
     const vec<Bool>& upaths_to_use, const vec<tagged_rpint>& upathsdb, const int K,
     /* outputs: */
     vec< vec<int> >& u, vec< vec<KmerPathLoc> >& start, 
     vec< vec<KmerPathLoc> >& stop )
{
     u.clear( ), u.resize( pairs.size( ) );
     start.clear( ), start.resize( pairs.size( ) );
     stop.clear( ), stop.resize( pairs.size( ) );
     for ( int i = 0; i < pairs.isize( ); i++ )
     {    const read_pairing& p = pairs[i];
          int id1 = p.id1, id2 = p.id2;
          const KmerPath &x1 = paths[id1], &x2 = paths_rc[id2];
          longlong kmer1 = x1.FirstSegment( ).Start( );
          static vec<longlong> places1, places2;
          Contains( upathsdb, kmer1, places1 );
          static vec<longlong> good_places1, good_places2;
          good_places1.clear( ), good_places2.clear( );
          
          // Find good placements for the first read.

          for ( int j = 0; j < places1.isize( ); j++ )
          {    const tagged_rpint& t = upathsdb[ places1[j] ];

               // Test for proper placement.

               Bool overlaps = ProperOverlapExt( paths[id1], 
                    upaths[ t.PathId( ) ], 0, t.PathPos( ) );
               if ( !overlaps ) continue;

               // Test for full placement.

               const KmerPath& q = upaths[ t.PathId( ) ];
               KmerPathLoc b( q, t.PathPos( ), kmer1 - q.Start( t.PathPos( ) ) );
               if ( x1.KmerCount( ) - 1 > DistMin( b, q.End( ) ) ) continue;

               // Record.

               good_places1.push_back( places1[j] );    }

          // Find good placements for the second read.

          longlong kmer2 = x2.FirstSegment( ).Start( );
          Contains( upathsdb, kmer2, places2 );
          for ( int j = 0; j < places2.isize( ); j++ )
          {    const tagged_rpint& t = upathsdb[ places2[j] ];

               // Test for proper placement.

               Bool overlaps = ProperOverlapExt( paths_rc[id2], 
                    upaths[ t.PathId( ) ], 0, t.PathPos( ) );
               if ( !overlaps ) continue;

               // Test for full placement.

               const KmerPath& q = upaths[ t.PathId( ) ];
               KmerPathLoc b( q, t.PathPos( ), kmer2 - q.Start( t.PathPos( ) ) );
               if ( x2.KmerCount( ) - 1 > DistMin( b, q.End( ) ) ) continue;

               // Record.

               good_places2.push_back( places2[j] );    }

          // Find good placements for the pair.

          for ( int j1 = 0; j1 < good_places1.isize( ); j1++ )
          {    for ( int j2 = 0; j2 < good_places2.isize( ); j2++ )
               {    const tagged_rpint& t1 = upathsdb[ good_places1[j1] ];
                    const tagged_rpint& t2 = upathsdb[ good_places2[j2] ];
                    if ( t1.PathId( ) != t2.PathId( ) ) continue;
                    if ( !upaths_to_use[ t1.PathId( ) ] ) continue;
                    const KmerPath& q = upaths[ t1.PathId( ) ];
                    KmerPathLoc l1( q, t1.PathPos( ), kmer1 - t1.Start( ) );
                    KmerPathLoc l2( q, t2.PathPos( ), kmer2 - t2.Start( ) );
                    int sep = DistMin( l1 + ( x1.KmerCount( ) - 1 ), l2 ) - (K-1);
                    double dev = double( Abs( p.sep - sep ) ) / double(p.sd);
                    if ( dev > 3.0 ) continue;
                    start[i].push_back( l1 );
                    stop[i].push_back( l2 + ( x2.KmerCount( ) - 1 ) );
                    u[i].push_back( t1.PathId( ) );    }    }    }    }

// PickCover.  Given a collection of intervals, choose a subset of them which
// covers, according to the following algorithm:
// 1. Start by picking the largest interval.
// 2. Repeat the following steps:
// (a) Remove from consideration any intervals that are contained in the union U of
// those picked already.
// (b) If no intervals remain, we're done.
// (c) Sort the intervals in decreasing order by their intersection with U.
// (d) Amongst the top decile, pick the interval with the largest amount outside U.
          
void PickCover( const vec<ho_interval>& I, vec<int>& C )
{    if ( I.empty( ) ) return; 
     vec<ho_interval> J = I;
     vec<int> from( I.size( ), vec<int>::IDENTITY );
     C.clear( );    
     vec<ho_interval> U;
     int M = 0, Mi = -1;
     for ( int i = 0; i < I.isize( ); i++ )
     {    if ( I[i].Length( ) > M )
          {    M = I[i].Length( ), Mi = i;    }    }
     C.push_back(Mi), U.push_back( I[Mi] );
     int N = 0;
     for ( int i = 0; i < I.isize( ); i++ )
          N = Max( I[i].Stop( ), N );
     while(1)
     {    static vec<int> nintersection, fromnew;
          static vec<ho_interval> Jnew, Unew;
          nintersection.clear( ), Jnew.clear( ), fromnew.clear( );
          for ( int i = 0; i < J.isize( ); i++ )
          {    int x = Overlap( J[i], U );
               if ( x == J[i].Length( ) ) continue;
               nintersection.push_back(-x);
               Jnew.push_back( J[i] ), fromnew.push_back( from[i] );    }
          J = Jnew, from = fromnew;
          if ( J.empty( ) ) return;
          SortSync( nintersection, J, from );
          const int div = 10;
          int B = 0, Bi = -1;
          int n = J.size( );
          for ( int i = 0; i < (n+div-1)/div; i++ )
          {    int out = J[i].Length( ) + nintersection[i];
               if ( out > B )
               {    B = out, Bi = i;    }    }
          C.push_back( from[Bi] ), U.push_back( J[Bi] );
          ExtractGivenCoverage( N, 1, U, Unew );
          U = Unew;    }    }

String VecSummary( const vec<int>& v )
{    ForceAssert( v.Ordered( ) );
     String summary;
     for ( int a = 0; a < v.isize( ); a++ ) 
     {    int b;
          for ( b = a + 1; b < v.isize( ); b++ )
               if ( v[b] != v[a] ) break;
          if ( a > 0 ) summary += " ";
          summary += ToString(v[a]) + "[" + ToString(b-a) + "]";
          a = b - 1;    }
     return summary;    }

void CheckCoverage( const placement& p, int K, const vec< pair<read_id_t,Bool> >& use, 
     genome_part_pos_t start, genome_part_pos_t stop, const vec<read_location>& readlocs, 
     const vec<int>& readlocs_index, const vec<nbases_t>& genome_path_lengths, 
     int NHOOD_RADIUS_INTERNAL, int& sim_tried, int& sim_covered,
     vec<ho_interval>& rcov )
{    int g = p.GenomeId( );
     static vec<ho_interval> cov;
     cov.clear( );
     for ( int j = 0; j < use.isize( ); j++ )
     {    read_id_t id = use[j].first;
          Bool rc = use[j].second;
          const read_location& rl = readlocs[ readlocs_index[id] ];
          if ( rl.Contig( ) != g ) continue;
          int rstart = Max( start, rl.Start( ) ) + K/2;
          int rstop = Min( stop, rl.Stop( ) + 1 ) - K/2;
          if ( rstart < rstop ) cov.push_back( ho_interval( rstart, rstop ) );    }
     ExtractGivenCoverage( stop, 1, cov, rcov );
     cout << PERCENT_RATIO( 4, Sum(rcov), stop - start - K ) << "\n";
     cout << "total number of reads within internal radius = " 
          << cov.size( ) << "\n";
     Bool covered = ( Sum(rcov) == stop - start - K );
     Bool near_end = True;
     if ( p.pos( ) >= NHOOD_RADIUS_INTERNAL &&
          p.Pos( ) + NHOOD_RADIUS_INTERNAL <= genome_path_lengths[g] )
     {    near_end = False;
          ++sim_tried;
          if (covered) ++sim_covered;    }
     if ( !covered && rcov.nonempty( ) )
     {    static vec< pair<int,int> > gaps;
          gaps.clear( );
          int first_cov = rcov.front( ).Start( ), last_cov = rcov.back( ).Stop( );
          if ( first_cov > start + K/2 )
               gaps.push_back( make_pair( first_cov - start - K/2, start + K/2 ) );
          if ( last_cov < stop - K/2 )
               gaps.push_back( make_pair( stop - K/2 - last_cov, last_cov ) );
          for ( int i = 0; i < rcov.isize( ) - 1; i++ )
               gaps.push_back( make_pair(
                    rcov[i+1].Start( ) - rcov[i].Stop( ), rcov[i].Stop( ) ) );
          cout << gaps.size( ) << " gaps, " << "the longest of which are:";
          ReverseSort(gaps);
          for ( int i = 0; i < Min( 5, gaps.isize( ) ); i++ )
          {    int glen = gaps[i].first, gstart = gaps[i].second;
               cout << " " << glen << "[" << g << "." << gstart 
                    << "-" << gstart + glen << "]";    }
          cout << "\n";    
          if (near_end) cout << "(near end of genome contig)" << "\n";    }    }

// ReportCoverage. Note: This generates internal structures scov and cov.  It is 
// not absolutely guaranteed that scov is a subset of cov.  The reason for this is 
// that a short-insert read may be accepted based on coverage by reads which are 
// from the wrong region of the genome.

void ReportCoverage( const serfvec<placement>& locsv, int K, 
     const vec< pair<int,Bool> >& use, const vec< pair<int,Bool> >& P,
     const vec<read_location>& readlocs, const vec<int>& readlocs_index, 
     const vec<int>& genome_path_lengths, int NHOOD_RADIUS_INTERNAL, int& sim_tried, 
     int& sim_covered, int& sim_tried_sh, int& sim_covered_sh )
{    cout << "\n";
     for ( int t = 0; t < locsv.size( ); t++ )
     {    cout << "checking coverage at location " << t+1 << "\n";
          const placement& p = locsv[t];
          int g = p.GenomeId( );
          int start = Max( 0, p.pos( ) - NHOOD_RADIUS_INTERNAL );
          int stop = Min( genome_path_lengths[g], p.Pos( ) + NHOOD_RADIUS_INTERNAL );
          static vec<ho_interval> cov, scov;
          cout << "checking coverage by primary read cloud\n";
          CheckCoverage( p, K, use, start, stop, readlocs, readlocs_index,
               genome_path_lengths, NHOOD_RADIUS_INTERNAL,
               sim_tried, sim_covered, cov );    
          cout << "now checking coverage by secondary read cloud\n";
          CheckCoverage( p, K, P, start, stop, readlocs, readlocs_index,
               genome_path_lengths, NHOOD_RADIUS_INTERNAL,
               sim_tried_sh, sim_covered_sh, scov );    }
     flush(cout);    }


void RenumberKmersInUnipaths( const vec<KmerPath>& localUnipaths, 
                              const KmerBaseBroker& localKBB,
                              const String& wrun_dir,
                              vecKmerPath& newUnipaths,
                              KmerBaseBroker& newlocalKBB )
{
  // Renumber kmers so that we know that most unipaths have one kmer path segment.
  newUnipaths.clear();
  newUnipaths.resize( localUnipaths.size() );
  
  map<longlong,longlong> oldToNewKmers;
  longlong nextNewKmer = halfway_kmer;
  for ( int u = 0; u < newUnipaths.size(); u++ ) {
    // Ensure that unipaths are separated by a kmer.
    ++nextNewKmer;
    KmerPath& newUnipath = newUnipaths[u];
    const KmerPath& oldUnipath = localUnipaths[u];
    for ( int seg = 0; seg < oldUnipath.NSegments(); ++seg ) {
      KmerPathInterval oldSeg = oldUnipath.Segment(seg);
      if ( is_palindrome( oldSeg.Start() ) )
        newUnipath.AddSegment( oldSeg );
      else {
        for ( longlong oldKmer = oldSeg.Start(); oldKmer <= oldSeg.Stop(); ++oldKmer ) {
          map<longlong,longlong>::iterator found =
            oldToNewKmers.find( oldKmer );
          if ( found == oldToNewKmers.end() ) {
            // Make sure segments in unipaths aren't too long.
            if ( ! newUnipath.IsEmpty() &&
                 newUnipath.LastSegment().Length() == KmerPathInterval::maxLength )
              ++nextNewKmer;
            oldToNewKmers.insert( make_pair( reverse_kmer(oldKmer), 
                                             reverse_kmer(nextNewKmer) ) );
            newUnipath.AddSegment( KmerPathInterval( nextNewKmer, nextNewKmer ) );
            ++nextNewKmer;
          } else {
            newUnipath.AddSegment( KmerPathInterval( found->second, found->second ) );
          }
        }
      }
    }
  }
  
  // We need to create the rcs of the unipaths for the KmerBaseBroker to work correctly.
  vecKmerPath newUnipathsRc = newUnipaths;
  for ( int u = 0; u < newUnipathsRc.size(); ++u )
    newUnipathsRc[u].Reverse();
  
  vec<tagged_rpint> newUnipathsDb;
  CreateDatabase( newUnipaths, newUnipathsRc, newUnipathsDb );

  // Save the new unipath info to wrun_dir.
  vecbasevector unipathBases;
  for ( unsigned int u = 0; u < localUnipaths.size(); ++u )
    unipathBases.push_back_reserve( localKBB.Seq( localUnipaths[u] ), 0, 2.0 );
  unipathBases.WriteAll( wrun_dir + "/unipaths.fastb" );
  newUnipaths.WriteAll( wrun_dir + "/unipaths.paths.k" + ToString(localKBB.GetK()) );
  newUnipathsRc.WriteAll( wrun_dir + "/unipaths.paths_rc.k" + ToString(localKBB.GetK()) );
  BinaryWrite2( wrun_dir + "/unipaths.pathsdb.k" + ToString(localKBB.GetK()), newUnipathsDb );
  
  newlocalKBB.Initialize( wrun_dir, localKBB.GetK(), "unipaths" );
}

void WalkLongInserts( vec<HyperKmerPath>& hypers,
                      const vec<pp_pair>& long_inserts, 
                      const vec<read_pairing>& long_insert_orig_pairs, 
                      const vec<HyperPairPlacement>& long_insert_hpairs,
                      const vec< vec<pp_closure> >& ppclosures,
                      const vecKmerPath& newUnipaths,
                      const KmerBaseBroker& newlocalKBB,
                      const int K, const int LONG_INSERT_WALK_K,
                      const int SD_MULT, const int MAX_PSEUDO,
                      const int search_limit,
                      const longlong answer_size_limit,
                      const int LONG_INSERT_MIN_READ_LENGTH,
                      const int LONG_INSERT_WALK_VERBOSITY, 
                      const Bool EVALUATE_INSERT_HYPER,
                        const vec<read_location>& readlocs,
                        const vec<int>& readlocs_index,
                        const String& data_dir,
                        const String& wdata_dir,
                        const vecbasevector& genome,
                      const int verbosity )
{
  longlong ppUnipathSeqsRawsize = 0;
  int ppUnipathSeqsSize = 0;
  
  set<pp_closure> uniqueClosures;
  for ( unsigned int i = 0; i < ppclosures.size(); ++i )
    for ( unsigned int j = 0; j < ppclosures[i].size(); ++j )
      uniqueClosures.insert( ppclosures[i][j] );
  
  for ( set<pp_closure>::iterator iUniq = uniqueClosures.begin(); 
        iUniq != uniqueClosures.end(); ++iUniq ) {
    ppUnipathSeqsRawsize += iUniq->size();
    ppUnipathSeqsSize += 1;
  }

  vecUnipathSeq walkingReads;
  walkingReads.Reserve( ppUnipathSeqsRawsize, ppUnipathSeqsSize );
  for ( set<pp_closure>::iterator iUniq = uniqueClosures.begin(); 
        iUniq != uniqueClosures.end(); ++iUniq )
    walkingReads.push_back( UnipathSeq( *iUniq ) );
  
  const int numWalkingReads = walkingReads.size();
  
  for ( int i = 0; i < long_inserts.isize(); ++i ) {
    const pp_pair &long_insert = long_inserts[i];
    ppUnipathSeqsRawsize += max( long_insert.Left().size(), long_insert.Right().size() );
    ppUnipathSeqsSize += 2;
  }
  
  vecUnipathSeq ppUnipathSeqsFw, ppUnipathSeqsRc;
  ppUnipathSeqsFw.Reserve( ppUnipathSeqsRawsize, ppUnipathSeqsSize );
  ppUnipathSeqsRc.Reserve( ppUnipathSeqsRawsize, ppUnipathSeqsSize );
  
  vec<read_pairing> longInsertPairs;
  vec<int> to_id;
  longInsertPairs.reserve( long_inserts.size() );
  
  for ( int i = 0; i < long_inserts.isize(); ++i ) {
    const pp_pair &long_insert = long_inserts[ i ];
    read_pairing long_insert_pair;
    
    long_insert_pair.id2 = ppUnipathSeqsRc.size();
    ppUnipathSeqsFw.push_back( UnipathSeq() );
    ppUnipathSeqsRc.push_back( UnipathSeq( long_insert.Right() ) );
    //cout << long_insert_pair.id2 << "rc:\t" << ppUnipathSeqsRc.back() << endl;
    
    long_insert_pair.id1 = ppUnipathSeqsFw.size();
    ppUnipathSeqsFw.push_back( UnipathSeq( long_insert.Left() ) );
    ppUnipathSeqsRc.push_back( UnipathSeq() );
    //cout << long_insert_pair.id1 << "fw:\t" << ppUnipathSeqsFw.back() << endl;
    
    long_insert_pair.sep = (int)round(long_insert.Gap());
    long_insert_pair.sd = (int)ceil(long_insert.Dev());
    longInsertPairs.push_back( long_insert_pair );
    
    const read_pairing& p = long_insert_orig_pairs[ i ];
    to_id.push_back( p.id2, p.id1 );
  }

  const int numLongInsertReads = ppUnipathSeqsFw.size();
  
  set<int> pairsToSkip;

  vecUnipathSeq extendedLongInsertReads;
  vec<Mux> extensions;

  UnipathSeqDatabase walkingReadsDb( walkingReads );

  for ( int dir = 0; dir < 2; ++dir ) {
    vecUnipathSeq& ppUnipathSeqs = ( dir == 0 ? ppUnipathSeqsFw : ppUnipathSeqsRc );
    ExtendUnipathSeqs( newUnipaths, walkingReads, ppUnipathSeqs, 
                       extendedLongInsertReads, extensions );
    for ( int i = 0; i < extendedLongInsertReads.size(); ++i ) {
      if ( extendedLongInsertReads[i].empty() ) continue;
      int length = 0;
      for ( int j = 0; j < extendedLongInsertReads[i].size(); ++j ) 
        length += newUnipaths[ extendedLongInsertReads[i][j] ].KmerCount();
      if ( length + K-1 < LONG_INSERT_MIN_READ_LENGTH ) {

        cout << i << ( dir == 0 ? "fw" : "rc" ) << " " << to_id[i]
             << " only extends to " << length + K-1 << "bp." << endl;

        // The problem with extending through the unipath graph is
        // that you can terminate prematurely due to reads that don't
        // align to the original read at all:
        //
        //    original   -------
        //    extender     --------------------
        //    spoiler      xxxxxxxx------xxxx ^
        //                              ^     |
        // The extension stops here ----'     |
        // But it could (arguably) stop here -'
        //
        // Therefore, before we throw out a pair, we try to extend it
        // using just the kmers that all of the aligning walking reads
        // share.

        length = 0;
        for ( int j = 0; j < ppUnipathSeqs[i].size(); ++j )
          length += newUnipaths[ ppUnipathSeqs[i][j] ].KmerCount();
        
        // Find all the walking reads that contain the first unipath
        // in the long insert read, and consider them "hits" if they
        // align properly to the long insert read.
        vec<UnipathSeqDatabase::Record> leftHits, leftExts;
        walkingReadsDb.Find( ppUnipathSeqs[i].front(), leftHits );
        for ( unsigned int k = 0; k < leftHits.size(); ++k ) {
          int readIndex = 0, hitIndex = leftHits[k].index;
          while ( 1 ) {
            if ( readIndex == ppUnipathSeqs[i].size() ||
                 hitIndex == walkingReads[ leftHits[k].seqId ].size() )
            {
              // We've reached the right end of one of the sequences
              // without hitting a disagreement, so consider this a
              // hit.
              leftExts.push_back( leftHits[k] );
              break;
            }
            // If they disagree, break.
            if ( ppUnipathSeqs[i][readIndex] != walkingReads[ leftHits[k].seqId ][hitIndex] )
              break;
            // Move on to the next segment.
            ++readIndex, ++hitIndex;
          }
        }

        // We walk backwards through all the extensions one unipath at
        // a time.  If they all agree, add that shared unipath's
        // length to the "length" of the extended long insert read.
        // Continue until there is a disagreement among the extensions
        // or one of the extensions terminates.
        if ( ! leftExts.empty() ) {
          while ( 1 ) {
            int sharedUnipath = -1;
            for ( unsigned int k = 0; k < leftExts.size(); ++k ) {
              int extIndex = --leftExts[k].index;
              // If any extension has run out, reset and stop.
              if ( extIndex < 0 ) {
                sharedUnipath = -1;
                break;
              }
              // Otherwise, save the unipath if it is the first one.
              int extUnipath = walkingReads[ leftExts[k].seqId ][ extIndex ];
              if ( sharedUnipath < 0 )
                sharedUnipath = extUnipath;
              // If it isn't the first, reset the shared unipath and break.
              else if ( sharedUnipath != extUnipath ) {
                sharedUnipath = -1;
                break;
              }
            }
            // If there is a unanimous extension, add its length to
            // the extended read's length.
            if ( sharedUnipath < 0 ) break;
            length += newUnipaths[ sharedUnipath ].KmerCount();
          }
        }

        // If we've extended enough, we can continue.
        if ( length + K-1 >= LONG_INSERT_MIN_READ_LENGTH ) {
          cout << "  Salvaged." << endl;
          continue;
        }

        // Now extend the long insert read to the right in the same fashion.
        vec<UnipathSeqDatabase::Record> rightHits, rightExts;
        walkingReadsDb.Find( ppUnipathSeqs[i].back(), rightHits );
        for ( unsigned int k = 0; k < rightHits.size(); ++k ) {
          int readIndex = ppUnipathSeqs[i].size() - 1, hitIndex = rightHits[k].index;
          while ( 1 ) {
            if ( readIndex < 0 || hitIndex < 0 )
            {
              rightExts.push_back( rightHits[k] );
              break;
            }
            if ( ppUnipathSeqs[i][readIndex] != walkingReads[ rightHits[k].seqId ][hitIndex] )
              break;
            --readIndex, --hitIndex;
          }
        }

        if ( ! rightExts.empty() ) {
          while ( 1 ) {
            int sharedUnipath = -1;
            for ( unsigned int k = 0; k < rightExts.size(); ++k ) {
              int extIndex = ++rightExts[k].index;
              if ( extIndex == walkingReads[ rightExts[k].seqId ].size() ) {
                sharedUnipath = -1;
                break;
              }
              int extUnipath = walkingReads[ rightExts[k].seqId ][ extIndex ];
              if ( sharedUnipath < 0 )
                sharedUnipath = extUnipath;
              else if ( sharedUnipath != extUnipath ) {
                sharedUnipath = -1;
                break;
              }
            }
            if ( sharedUnipath < 0 ) break;
            length += newUnipaths[ sharedUnipath ].KmerCount();
          }
        }

        if ( length + K-1 >= LONG_INSERT_MIN_READ_LENGTH ) {
          cout << "Salvaged." << endl;
          continue;
        }
       
        const int pairToSkip = i / 2;
        cout << "  Skipping pair " << pairToSkip << "." << endl;
        pairsToSkip.insert( pairToSkip );

        ForceAssert( longInsertPairs[pairToSkip].InvolvesId( i ) );
      }
    }
  }
  
  for ( set<pp_closure>::iterator iUniq = uniqueClosures.begin(); 
        iUniq != uniqueClosures.end(); ++iUniq ) {
    //cout << ppUnipathSeqsFw.size() << "fw:\t";
    ppUnipathSeqsFw.push_back( UnipathSeq( *iUniq ) );
    ppUnipathSeqsRc.push_back( UnipathSeq() );
    //cout << ppUnipathSeqsFw.back() << endl;
  }
  
  vecKmerPath ppKmerPathsFw, ppKmerPathsRc;
  ConvertUnipathSeqsToKmerPaths( ppUnipathSeqsFw, newUnipaths, ppKmerPathsFw );
  ConvertUnipathSeqsToKmerPaths( ppUnipathSeqsRc, newUnipaths, ppKmerPathsRc );

  // Now we trim down the extended long insert reads so they don't
  // include kmers outside the insert.  This is done because we can
  // sometimes make mistakes in the unipath structure at the edges of
  // the neighborhood, and these mistakes can be incorporated into the
  // long insert closures if we don't trim them.
  
  vec<int> originalLengthsFw( ppUnipathSeqsFw.size(), -1 );
  vec<int> originalLengthsRc( ppUnipathSeqsFw.size(), -1 );

  for ( int i = 0; i < longInsertPairs.isize(); ++i ) {
    read_pairing &p = longInsertPairs[i];
    KmerPath& fwRead = ppKmerPathsFw[p.id1];
    KmerPath& rcRead = ppKmerPathsRc[p.id2];

    originalLengthsFw[p.id1] = fwRead.KmerCount();
    int fwLeftTrimAmount = long_insert_hpairs[i].LeftExt1();
    int fwRightTrimAmount = long_insert_hpairs[i].RightExt1() + 1; // undercounted?
    KmerPathLoc trimmedFwStart = fwRead.Begin() + fwLeftTrimAmount;
    KmerPathLoc trimmedFwEnd   = fwRead.End()   - fwRightTrimAmount;
    KmerPath trimmedFwRead;
    fwRead.CopySubpath( trimmedFwStart, trimmedFwEnd, trimmedFwRead );
    fwRead = trimmedFwRead;

    originalLengthsRc[p.id2] = rcRead.KmerCount();
    int rcLeftTrimAmount = long_insert_hpairs[i].LeftExt2();
    int rcRightTrimAmount = long_insert_hpairs[i].RightExt2() + 1; // undercounted?
    KmerPathLoc trimmedRcBegin = rcRead.Begin() + rcLeftTrimAmount;
    KmerPathLoc trimmedRcEnd   = rcRead.End()   - rcRightTrimAmount;
    KmerPath trimmedRcRead;
    rcRead.CopySubpath( trimmedRcBegin, trimmedRcEnd, trimmedRcRead );
    rcRead = trimmedRcRead;

    p.sep += fwRightTrimAmount + rcLeftTrimAmount;
  }

  SubMuxGraphBuilder builder;
  builder.SetMinOverlap( LONG_INSERT_WALK_K - K + 1 );
  builder.BuildFromPaths( ppKmerPathsFw, ppKmerPathsRc, numLongInsertReads );
  
  //builder.BigPictureDot( "bigpicture.dot", numLongInsertReads );
  
  KmerPathMuxSearcher& searcher = builder.GetSearcher();
  searcher.SetVerbosity( LONG_INSERT_WALK_VERBOSITY );
  searcher.SetSearchLimit( search_limit );
  searcher.SetAnswerSizeLimit( answer_size_limit );

  hypers.clear();
  
  int numInsertsWalked = 0;
  for ( int i = 0; i < longInsertPairs.isize(); ++i ) {
    if ( pairsToSkip.count(i) ) continue;
    read_pairing &p = longInsertPairs[i];
    int id1 = p.id1, id2 = p.id2;
    int sep = p.sep, dev = p.sd;
    const int range = SD_MULT * dev;
    const int minAcceptableExtLength = sep + builder.GetReadLengths()[id1] - range;
    const int maxAcceptableExtLength = sep + builder.GetReadLengths()[id1] + range;
    TaskTimer searchTimer, closureTimer;
    double search_clock = WallClockTime( );
    MuxSearchResult result;
    
    cout << "\n------------------------------------------"
         << "--------------------------------------\n" << endl;
               
    cout << ++numInsertsWalked << ". "
         << "Walking from read " << to_id[id1] << " to read " << to_id[id2] << ".\n";
    // cout << "Walking from: "<< ppUnipathSeqsRc[id2] <<" ("<<originalLengthsRc[id2]<<" kmers)\n"
    //      << "     back to: "<< ppUnipathSeqsFw[id1] <<" ("<<originalLengthsFw[id1]<<" kmers)\n"
    //      << "          in: "<< minAcceptableExtLength <<" - "<< maxAcceptableExtLength <<"\n";

    if ( EVALUATE_INSERT_HYPER && ! readlocs.empty() ) {
      read_location rl1 = readlocs[ readlocs_index[ to_id[id1] ] ];
      read_location rl2 = readlocs[ readlocs_index[ to_id[id2] ] ];
      if ( rl1.Rc() )
        swap( rl1, rl2 );
      cout << "Actual location: " << rl1.Contig() << "." 
           << rl1.StartOnContig() << "-" << rl2.StopOnContig() << endl;
    }

    searcher.FindClosures( id1, id2, 
                           minAcceptableExtLength, maxAcceptableExtLength, 
                           result );
              
    if ( verbosity > 0 ) 
      cout << TimeSince(search_clock) << " used in search" << endl;

    Bool fail = False;
    if (result.hit_search_limit) {
      if ( verbosity > 0 )
        cout << "Found " << result.num_closures_found 
             << " closure events; DIRTY -- "
             << "abandoning insert" << endl;    
      fail = True;    
    }
    if ( MAX_PSEUDO > 0 && result.num_closures_found > MAX_PSEUDO ) {
      if ( verbosity > 0 )
        cout << "Number of pseudo-closures ("
             << result.num_closures_found 
             << ") exceeds MAX_PSEUDO." << endl;
      fail = True;    
    }
              
    if ( result.num_closures_found == 0 ) {
      if ( verbosity > 0 )
        cout << "No closures found." << endl;
      fail = True;
    }

    if ( !fail ) {
      HyperKmerPath insertHKP(K, vec<KmerPath>() ); // empty HKP with K set
      double merger_clock = WallClockTime( );
      
      // static int num = 0;
      // ++num;
      // String dotfile = "muxwalkgraph" + ToString(num) + ".dot";
      // cout << "Saving mux walk graph to " << dotfile << "." << endl;
      // ofstream dotstrm( dotfile.c_str() );
      // result.WalkGraph().Dot( dotstrm );
      // dotstrm.close();

      result.WalkGraph( ).MakeHyperKmerPath( &ppKmerPathsFw, &ppKmerPathsRc, 
                                             &builder.GetSubList(), insertHKP );
      insertHKP.TestValid( );
      //insertHKP.TestKmersGood( newlocalKBB );
      for ( int i = 0; i < insertHKP.EdgeObjectCount( ); i++ )
        ForceAssert( insertHKP.EdgeObject(i).GapFree( ) );
      cout << TimeSince(merger_clock) 
           << " used merging closures for one insert" << endl;
      hypers.push_back(insertHKP);    

      if ( EVALUATE_INSERT_HYPER ) {
        vec<look_align> aligns;
        vec< vec<int> > aligns_index;
        vec<TrustedPath> trusted_paths;
        AlignHyperKmerPath( insertHKP, &newlocalKBB, data_dir + "/genome", 
                            wdata_dir + "/run", aligns, aligns_index );
        FilterAligns( insertHKP, aligns, aligns_index, trusted_paths );
        insertHKP.PrintSummary( cout );
        cout << "\nComparison of insert closure HyperKmerPath to reference\n";
        copy( trusted_paths.begin(), trusted_paths.end(),
              ostream_iterator<TrustedPath>( cout, "\n" ) );
        PrintAlignedHyperKmerPath( cout, insertHKP, &newlocalKBB, genome, aligns,
                                   aligns_index, False, &trusted_paths );    
      }
    }

    // cin.ignore( 100, '\n' );
  }
}

/**
   Function: MergeNeighborhood

   Takes <HyperKmerPaths> obtained from walking long-insert pairs in
   one neighborhood, and merges them into a single HyperKmerPath for the
   neighborhood.

   Inputs:

      hypers - HyperKmerPaths, one for each long-insert pair, representing
         the possible sequences in the middle of that pair

   Output:

      nhoodHBV - a HyperKmerPath (in sequence space) representing the possible
         sequences of the entire neighborhood.
*/
void MergeNeighborhood( HyperBasevector& nhoodHBV, 
                        const vec<HyperKmerPath>& hypers,
                        const KmerBaseBroker& newlocalKBB,
                        const Bool PRINT_HYPER_BEFORE_FIRST_MERGE,
                        const Bool FIRST_MERGE,
                        const Bool PRINT_HYPER_AFTER_FIRST_MERGE,
                        const Bool PRINT_HYPER_BEFORE_SECOND_MERGE,
                        const Bool SECOND_MERGE,
                        const int MIN_OVERLAP,
                        const int MIN_PROPER_OVERLAP,
                        const Bool PRINT_HYPER_AFTER_SECOND_MERGE,
                        const Bool FIRST_DELOOP,
                        const vec< pair<read_id_t,Bool> >& secondaryCloud,
                        const vecbasevector& reads,
                        const vec<read_pairing>& pairs,
                        const vec<int>& pairs_index,
                        const Bool FIRST_DELOOP_VERBOSE,
                        const Bool PRINT_HYPER_AFTER_FIRST_DELOOP,
                        const Bool EVALUATE_NHOOD_HYPER,
                        const String& data_dir,
                        const String& wdata_dir,
                        const vecbasevector& genome )
{
  const int K = newlocalKBB.GetK();

  // Merge HyperKmerPaths and convert to base space.
            
  double clock2 = WallClockTime( );
  HyperKmerPath nhoodHKP( K, hypers );
  NegativeGapValidator ngv(&newlocalKBB);
  cout << TimeSince(clock2) << " used in WalkInserts - 2" << endl;
            
  //nhoodHKP.TestKmersGood( newlocalKBB );

  if (FIRST_MERGE) {    
    double merger_clock = WallClockTime( );
    if (PRINT_HYPER_BEFORE_FIRST_MERGE) { 
      cout << "\nHyperKmerPath before first merge:\n";
      nhoodHKP.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, True );
      cout << "\n";    
      //Ofstream( dotfile, "beforefirst.dot" );
      //nhoodHKP.PrintSummaryDOT0w( dotfile, True, True );
    }
    InternalMerge( nhoodHKP, ngv, 5000, 4000, False );
    InternalMerge( nhoodHKP, ngv, 5000, 3000, False );
    InternalMerge( nhoodHKP, ngv, 5000, 2000, False );
    InternalMerge( nhoodHKP, ngv, 5000, 1000, False );
    InternalMerge( nhoodHKP, ngv, 5000, 500, False );
    nhoodHKP.Zipper( );
    cout << "\n" << TimeSince(merger_clock) 
         << " used merging closures for this neighborhood" << endl;
    if (PRINT_HYPER_AFTER_FIRST_MERGE) {
      cout << "\nHyperKmerPath after first merge:\n";
      nhoodHKP.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, True );
      cout << "\n";
      //Ofstream( dotfile, "afterfirst.dot" );
      //nhoodHKP.PrintSummaryDOT0w( dotfile, True, True );
    }
  }
            
  if (PRINT_HYPER_BEFORE_SECOND_MERGE) {   
    cout << "\nHyperKmerPath before second merge:\n";
    nhoodHKP.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, True );
    cout << "\n";    
  }
            
  if (SECOND_MERGE) {   
    InternalMerge( nhoodHKP, ngv, MIN_OVERLAP, MIN_PROPER_OVERLAP, False );
    nhoodHKP.Zipper( );
    nhoodHKP.ReduceLoops( );
    nhoodHKP.CompressEdgeObjects( );
    nhoodHKP.RemoveDeadEdgeObjects( ); 
    nhoodHKP.RemoveEdgelessVertices( );    
  }
  if (PRINT_HYPER_AFTER_SECOND_MERGE) {
    cout << "\nHyperKmerPath after second merge:\n";
    nhoodHKP.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, True );
    cout << "\n";    
    //Ofstream( dotfile, "aftersecond.dot" );
    //nhoodHKP.PrintSummaryDOT0w( dotfile, True, True );
  }

  vec< basevector > edgeBases( nhoodHKP.Edges().size() );
  for ( unsigned int i = 0; i < edgeBases.size(); ++i )
    edgeBases[i] = newlocalKBB.Seq( nhoodHKP.Edges()[i] );

  // Try to disambiguate loops.  We find parts of the graph
  //        u----->v----->w (plus edge from v---->v)
  // where there is a loop bracketed by two edges.  Then we try to determine
  // the true number of times that the edge v---->v should occur, and if we
  // are successful, eliminate v and replace the three edges by a single
  // edge u---->w.
            
  if (FIRST_DELOOP)
  {    
    double dis_clock = WallClockTime( );
    vec<simple_loop> loops;
    GetSimpleLoops( nhoodHKP, loops );
    vecbasevector secondaryCloudReads_plus_edges;
    for ( int i = 0; i < secondaryCloud.isize( ); i++ ) {    
      basevector b = reads[ secondaryCloud[i].first ];
      if ( secondaryCloud[i].second ) b.ReverseComplement( );
      secondaryCloudReads_plus_edges.push_back_reserve(b);    
    }
    for ( unsigned int i = 0; i < edgeBases.size(); i++ )
      secondaryCloudReads_plus_edges.push_back_reserve( edgeBases[i] );
    PerfectAligner pal( K, PerfectAligner::findProperOnly );
    vec<alignment_plus> aligns;
    pal.Align( secondaryCloudReads_plus_edges, aligns, secondaryCloud.size( ) );
    vec< vec<int> > aligns_index( secondaryCloud.size( ) );
    for ( int i = 0; i < aligns.isize( ); i++ ) {
      if ( aligns[i].Rc2( ) ) continue;
      aligns_index[ aligns[i].Id1( ) ].push_back(i);    
    }
    vec< pair<int,int> > secondaryCloudByPair;
    for ( int i = 0; i < secondaryCloud.isize( ); i++ )
      secondaryCloudByPair.push_back( make_pair( pairs_index[ secondaryCloud[i].first ], i ) );
    Sort(secondaryCloudByPair);
    vec< vec< pair<int,int> > > sep_dev( loops.size( ) );
    for ( int i = 0; i < secondaryCloudByPair.isize( ); i++ ) {
      int j;
      for ( j = i + 1; j < secondaryCloudByPair.isize( ); j++ )
        if ( secondaryCloudByPair[j].first != secondaryCloudByPair[i].first ) break;
      if ( j - i == 2 && secondaryCloud[ secondaryCloudByPair[i].second ].second )
        swap( secondaryCloudByPair[i], secondaryCloudByPair[i+1] );
      if ( j - i == 2 ) {
        int id1 = secondaryCloudByPair[i].second, id2 = secondaryCloudByPair[i+1].second;
        const read_pairing& p = pairs[ secondaryCloudByPair[i].first ];
        if ( aligns_index[id1].solo() && aligns_index[id2].solo() ) {
          const alignment_plus& ap1 = aligns[ aligns_index[id1][0] ];
          const alignment_plus& ap2 = aligns[ aligns_index[id2][0] ];
          int uv = ap1.Id2( ), vw = ap2.Id2( );
          int li = -1;
          for ( int l = 0; l < loops.isize( ); l++ )
            if ( loops[l].uv == uv && loops[l].vw == vw ) 
              li = l;
          if ( li < 0 ) {
            i = j - 1;
            continue;
          }
          int sep = ( p.sep - ap2.a.pos2( ) + K - 1
                      - ( edgeBases[ ap1.Id2( ) ].size()
                          - ap1.a.Pos2( ) ) );
          if (FIRST_DELOOP_VERBOSE) PRINT3( li, sep, p.sd );
          sep_dev[li].push_back( make_pair( sep, p.sd ) );  
        }
      }
      i = j - 1;    
    }
    DisambiguateSimpleLoops( nhoodHKP, loops, sep_dev );
    cout << TimeSince(dis_clock) 
         << " used disambiguating loop multiplicities\n";    
  }
            
  if (PRINT_HYPER_AFTER_FIRST_DELOOP) {
    cout << "\nHyperKmerPath after first deloop:\n";
    nhoodHKP.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, True );
    cout << "\n";    
    //Ofstream( dotfile, "afterdeloop.dot" );
    //nhoodHKP.PrintSummaryDOT0w( dotfile, True, True );
  }

  // Evaluate.
            
  if (EVALUATE_NHOOD_HYPER) {
    vec<look_align> aligns;
    vec< vec<int> > aligns_index;
    vec<TrustedPath> trusted_paths;
    double eclock2 = WallClockTime( );
    AlignHyperKmerPath( nhoodHKP, &newlocalKBB, data_dir + "/genome", 
                        wdata_dir + "/run", aligns, aligns_index );
    FilterAligns( nhoodHKP, aligns, aligns_index, trusted_paths );
    cout << "\nComparison of neighborhood HyperKmerPath to reference\n";
    PrintAlignedHyperKmerPath( cout, nhoodHKP, &newlocalKBB, genome, aligns,
                               aligns_index, True, &trusted_paths );    
    cout << TimeSince(eclock2) << " used in WalkInserts - e2" 
         << endl;    
  }
            
  // Record result.
            
  double clock4 = WallClockTime( );
  nhoodHBV = HyperBasevector( nhoodHKP, newlocalKBB );
  cout << TimeSince(clock4) << " used in WalkInserts - 4" << endl;
}


void WalkInserts( const vec<int>& to_id, const vec< pair<int,Bool> >& use, 
     KmerPathMuxSearcher& searcher, int nps, const vec<Bool>& extended_far_enough, 
     const vec<int>& sleft_add, const vec<int>& pairs_sample, 
     const vec<read_pairing>& pairs, const vec<read_location>& readlocs, 
     const vec<int>& readlocs_index, const vecvec<placement>& locs, 
     int SD_MULT, const vec<int>& readLengths, int MAX_PSEUDO, int MAX_CLOSURES, 
     int verbosity, const vecKmerPath& pathsFw, const vecKmerPath& pathsRc, 
     const SubsumptionList* theSubList,
     Bool decompose_truth, Bool decompose_truth_brief, 
     SimTrueSequenceBroker& trueSeq, Bool USE_TRUTH, int v, KmerBaseBroker* kbb, 
     const vecbasevector& genome, vec<int>& insert_ids, int K, 
     int MIN_OVERLAP, int MIN_PROPER_OVERLAP, vec<HyperBasevector>& hyperbases,
     const vec< pair<int,Bool> >& P, const vec<read_pairing>& PAIRS,
     const vec<int>& PAIRS_INDEX, const vecbasevector& reads, int K_orig,
     const String& sub_dir, const String& wdata_dir, Bool FIRST_MERGE,
     Bool FIRST_DELOOP, const vecbasevector& sreads, Bool found_universal,
     Bool PRINT_HYPER_BEFORE_FIRST_MERGE, Bool PRINT_HYPER_AFTER_FIRST_MERGE,
     Bool PRINT_HYPER_BEFORE_SECOND_MERGE, Bool PRINT_HYPER_AFTER_SECOND_MERGE,
     Bool FIRST_DELOOP_VERBOSE, Bool EVALUATE_NHOOD_HYPER, const String& data_dir )
{    
     double clock1 = WallClockTime( );
     if (found_universal)
     {    cout << "Not walking, using universal closure instead.\n";
          vec<basevector> universe;
          universe.push_back( sreads[2*nps] );
          universe[0].ReverseComplement( );
          HyperBasevector hb( K, universe );
          hyperbases.push_back(hb);
          cout << TimeSince(clock1) << " used in WalkInserts - 1" << endl;
          return;    }
     static vec<int> reads_to_use_opener_dir, reads_to_use_closer_dir;
     MuxSearchPolicy* temp_policy = 0;
     MuxSearchPolicy* temp_policy2 = 0;
     static vec<CompletedInsertBasesOnly> inserts;
     inserts.clear( );
     static vec<HyperKmerPath> hypers;
     hypers.clear( );
     int icount = 1;
     cout << TimeSince(clock1) << " used in WalkInserts - 1" << endl;
     for ( int i = 0; i < nps; i++ )
     {    if ( !extended_far_enough[ 2*i ] ) continue;
          if ( !extended_far_enough[ 2*i + 1 ] ) continue;
          read_pairing p;
          p = pairs[i];
          int id1 = p.id1, id2 = p.id2;
          int rid1 = id1, rid2 = id2;
          rid1 = to_id[id1];
          rid2 = to_id[id2];
          static basevector truthb;
          truthb.Setsize(0);
          if ( readlocs.nonempty( ) )
          {    const read_location& rl1 = readlocs[ readlocs_index[id1] ];
               const read_location& rl2 = readlocs[ readlocs_index[id2] ];
               int m = rl1.Contig( );
               if ( m == rl2.Contig( ) )
               {    int start = Min( rl1.Start( ), rl2.Start( ) );
                    int stop = 1 + Max( rl1.Stop( ), rl2.Stop( ) );
                    stop = Min( stop, genome[m].isize( ) );
                    truthb.SetToSubOf( genome[m], start, stop - start );
                    truthb.ReverseComplement( );    }    }
          if ( verbosity > 0 )
          {    cout << "\n------------------------------------------"
                    << "--------------------------------------\n" << endl;
               cout << icount++ << ". Walking from read " << rid1 
                    << " to read " << rid2 << "." << endl;
               if ( readlocs.nonempty( ) )
               {    const read_location& rl1 = readlocs[ readlocs_index[id1] ];
                    const read_location& rl2 = readlocs[ readlocs_index[id2] ];
                    int g = rl1.Contig( );
                    if ( g == rl2.Contig( ) )
                    {    int low = Min( rl1.Start( ), rl2.Start( ) );
                         int high = Max( rl1.Stop( ), rl2.Stop( ) );
                         ho_interval h1( low, high );
                         const serfvec<placement>& p = locs[v];
                         int bestj = -1, mind = 1000000000;
                         Bool tie = False;
                         for ( int j = 0; j < p.size( ); j++ )
                         {    if ( g != p[j].GenomeId( ) ) continue;
                              ho_interval h2( p[j].pos( ), p[j].Pos( ) );
                              int d = Distance( h1, h2 );
                              if ( Abs( d - mind ) < 5000 ) tie = True;
                              if ( d < mind )
                              {    mind = d;
                                   bestj = j;    }    }
                         if ( !tie && bestj >= 0 )
                         {    if ( p[bestj].Fw( ) )
                              {    low -= p[bestj].pos( );
                                   high -= p[bestj].pos( );    }
                              else
                              {    low -= p[bestj].Pos( );
                                   high -= p[bestj].Pos( );    }
                              cout << "actual location: ["
                                   << low << "," << high << "], length = "
                                   << high - low << endl;    }    }    }    }
          int sep = p.sep, dev = p.sd;
          const int range = SD_MULT * dev;
          const int minAcceptableExtLength = sep + readLengths[id2] - range;
          const int maxAcceptableExtLength = sep + readLengths[id2] + range;
          TaskTimer searchTimer, closureTimer;
          double search_clock = WallClockTime( );
          MuxSearchResult result;

          reads_to_use_opener_dir.clear( ), reads_to_use_closer_dir.clear( );

          reads_to_use_opener_dir.push_back( id2 );
          reads_to_use_closer_dir.push_back( id1 );
          for ( int ii = 2*nps; ii < sreads.size( ); ii++ )
               reads_to_use_opener_dir.push_back(ii);
               
          temp_policy2 = new MSP_TheseOrientedReadsOnly( 
               reads_to_use_opener_dir, reads_to_use_closer_dir );
          searcher.AddPolicy(temp_policy2);

          searcher.FindClosures( id2, id1, minAcceptableExtLength, 
               maxAcceptableExtLength, result );
          searcher.RemovePolicy(temp_policy2);
          delete temp_policy2;
          if ( verbosity > 0 ) 
               cout << TimeSince(search_clock) << " used in search" << endl;
          Bool fail = False;
          if (result.hit_search_limit) 
          {    if ( verbosity > 0 )
               {    cout << "Found " << result.num_closures_found 
                         << " closure events; DIRTY -- "
                         << "abandoning insert" << endl;    }
               fail = True;    }
          if ( MAX_PSEUDO > 0 && result.num_closures_found > MAX_PSEUDO )
          {    if ( verbosity > 0 )
               {    cout << "Number of pseudo-closures ("
                         << result.num_closures_found 
                         << ") exceeds MAX_PSEUDO." << endl;    }
               fail = True;    }
          /*
          double closure_clock=0.;
          if ( !fail )
          {    closure_clock = WallClockTime( );
	       // result.WalkGraph().Summary(cout); // for debugging
               result.CalculateAllClosures( &pathsFw, &pathsRc, MAX_CLOSURES );
               if ( result.hit_closure_limit )
               {    fail = True;
                    cout << TimeSince(closure_clock) 
                         << " used building paths" << endl;
		    cout << "Hit MAX_CLOSURES=" << MAX_CLOSURES 
		         << " limit" << endl;
		    if ( result.all_closures.size() < MAX_CLOSURES )
		    {    cout << "Actually found only " 
                              << result.all_closures.size( ) << " distinct closures,"
                              << " but found them with multiplicity"
			      << endl;    }    }    }
          if ( !fail )
          {    int id1_ext = sleft_add[ 2*i ], id2_ext = sleft_add[ 2*i + 1 ];
               if ( verbosity > 0 ) 
               {    cout << TimeSince(closure_clock) 
                         << " used building paths" << endl;
                    cout << "Found " << result.all_closures.size()
                         << " closures" << endl;
                    static vec<int> csizes;
                    csizes.clear( );
                    for( vecKmerPath::const_iterator j 
                         = result.all_closures.begin( );
                         j != result.all_closures.end( ); j++ )
                    {    csizes.push_back( 
                              j->KmerCount( ) - id1_ext - id2_ext );    }
                    Sort(csizes);
                    if ( csizes.nonempty( ) )
                    {    cout << "Closure sizes: " << VecSummary(csizes) 
                              << "\n";    }    }
               if ( kbb != 0 && truthb.size( ) > 0 )
               {    Bool found_truth = False;
                    for( vecKmerPath::const_iterator j = result.all_closures.begin();
                         j != result.all_closures.end(); j++ )
                    {    if ( kbb->Seq(*j) == truthb ) found_truth = True;    }
                    if ( !found_truth && !result.all_closures.empty( ) )
                         cout << "Warning: only false closures found!\n";    
                    if ( !found_truth && result.all_closures.size( ) > 0
                         && result.all_closures.size( ) <= 20 )
                    {    truthb.Print( cout, "truth" );
                         int cc = 1;
                         for( vecKmerPath::const_iterator j
                              = result.all_closures.begin();
                              j != result.all_closures.end(); j++ )
                         {    kbb->Seq(*j).Print(
                                   cout, "closure" + ToString(cc++) );    }    }    }
               if ( kbb != 0 )
               {    static vecbasevector bases;
                    bases.clear( );
                    for( vecKmerPath::const_iterator j = result.all_closures.begin();
                         j != result.all_closures.end(); j++ )
                    {    basevector b = kbb->Seq(*j), b2;
                         b2.SetToSubOf( b, id1_ext, b.isize( ) - id1_ext - id2_ext );
                         bases.push_back(b2);    }
                    CompletedInsertBasesOnly C( rid1, rid2,
                         minAcceptableExtLength - id1_ext - id2_ext,
                         maxAcceptableExtLength - id1_ext - id2_ext, 1, bases );
                    inserts.push_back(C);    }    }
          */
          if ( !fail )
          {    HyperKmerPath h(K, vec<KmerPath>() ); // empty HKP with K set
               double merger_clock = WallClockTime( );
               result.WalkGraph( ).MakeHyperKmerPath( &pathsFw, &pathsRc, 
						      theSubList, h );
               h.TestValid( );
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
                    ForceAssert( h.EdgeObject(i).GapFree( ) );
               /*
               int fd = OpenForWrite( "closure_" + ToString(i) + ".hkp" );
               BinaryWrite( fd, h );
               Close( fd );
               */
               cout << TimeSince(merger_clock) 
                    << " used merging closures for one insert" << endl;
               hypers.push_back(h);    }
          if ( ( decompose_truth || decompose_truth_brief ) &&
               readlocs.nonempty( ) && trueSeq.KmerPathMode( ) ) 
          {    trueSeq.SetInsert( id1, id2 );
               KmerPath truth;
               trueSeq.GetTruth(truth);
               cout << "Truth decomposition = ";
               if (decompose_truth) DecomposePath(truth);    
               else DecomposePath( truth, True );    }    }

     // Merge HyperKmerPaths for the inserts and write the merged HyperKmerPath
     // for the neighborhood.

     if ( kbb != 0 )
     {    
          // Merge HyperKmerPaths and convert to base space.

          double clock2 = WallClockTime( );
          HyperKmerPath h( K, hypers );
          NegativeGapValidator ngv(kbb);
          cout << TimeSince(clock2) << " used in WalkInserts - 2" << endl;
          if (FIRST_MERGE)
          {    double merger_clock = WallClockTime( );
               if (PRINT_HYPER_BEFORE_FIRST_MERGE)
               {    cout << "\nHyperKmerPath before first merge:\n";
                    h.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
                    cout << "\n";    }
               InternalMerge( h, ngv, 5000, 4000, False );
               InternalMerge( h, ngv, 5000, 3000, False );
               InternalMerge( h, ngv, 5000, 2000, False );
               InternalMerge( h, ngv, 5000, 1000, False );
               InternalMerge( h, ngv, 5000, 500, False );
               cout << "\n" << TimeSince(merger_clock) 
                    << " used merging closures for this neighborhood" << endl;
               if (PRINT_HYPER_AFTER_FIRST_MERGE)
               {    cout << "\nHyperKmerPath after first merge:\n";
                    h.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
                    cout << "\n";    }    }
          double clock3 = WallClockTime( );
          HyperBasevector hb( h, *kbb );
          for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
               ForceAssertGt( hb.EdgeObject(i).size( ), 0 );

          // Lower K, merge again, convert back to base space.

          hb.LowerK(K_orig);
          if ( hb.EdgeObjectCount( ) == 0 )
          {    hyperbases.push_back(hb);
               return;    }
          vecbasevector bases;
          for ( int i = 0; i < hb.EdgeObjectCount( ); i++ )
               bases.push_back_reserve( hb.EdgeObject(i) );
          String qrun_dir = wdata_dir + "/qrun";
          Mkdir777(qrun_dir);
          bases.WriteAll( qrun_dir + "/reads.fastb" );
          vecKmerPath paths;
          ReadsToPathsCoreY( bases, K_orig, bases.actual_rawsize(), paths );
          paths.WriteAll( qrun_dir + "/reads.paths.k" + ToString(K_orig) );

          // Scoped for immediate destruction.
          {
            vecKmerPath pathsrc(paths);
            for ( int pathid = 0; pathid < pathsrc.size(); ++pathid )
              pathsrc[pathid].Reverse();
            pathsrc.WriteAll( qrun_dir + "/reads.paths_rc.k" + ToString(K_orig) );
            KmerPathDatabase pathsdb( paths, pathsrc );
            pathsdb.Write( qrun_dir + "/reads.pathsdb.k" + ToString(K_orig) );
          }
          KmerBaseBroker* kbb2 = new KmerBaseBroker( qrun_dir, K_orig );
          NegativeGapValidator ngv2(kbb2);
          static vec<KmerPath> these_paths;
          these_paths.clear( );
          for ( int j = 0; j < hb.EdgeObjectCount( ); j++ )
               these_paths.push_back( paths[j] );
          HyperKmerPath h2( K_orig, hb, these_paths );
          if (PRINT_HYPER_BEFORE_SECOND_MERGE)
          {    cout << "\nHyperKmerPath before second merge:\n";
               h2.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
               cout << "\n";    }
          if (FIRST_MERGE)
          {    InternalMerge( h2, ngv2, MIN_OVERLAP, MIN_PROPER_OVERLAP, False );
               h2.ReduceLoops( );
               h2.CompressEdgeObjects( );
               h2.RemoveDeadEdgeObjects( ); 
               h2.RemoveEdgelessVertices( );    }
          if (PRINT_HYPER_AFTER_SECOND_MERGE)
          {    cout << "\nHyperKmerPath after second merge:\n";
               h2.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
               cout << "\n";    }
          h2.Reverse( );
          HyperBasevector hb2( h2, *kbb2 );
          cout << TimeSince(clock3) << " used in WalkInserts - 3" << endl;

          // Try to disambiguate loops.  We find parts of the graph
          //        u----->v----->w (plus edge from v---->v)
          // where there is a loop bracketed by two edges.  Then we try to determine
          // the true number of times that the edge v---->v should occur, and if we
          // are successful, eliminate v and replace the three edges by a single
          // edge u---->w.

          if (FIRST_DELOOP)
          {    double dis_clock = WallClockTime( );
               vec<simple_loop> loops;
               GetSimpleLoops( h2, loops );
               vecbasevector Preads_plus_edges;
               for ( int i = 0; i < P.isize( ); i++ )
               {    basevector b = reads[ P[i].first ];
                    if ( P[i].second ) b.ReverseComplement( );
                    Preads_plus_edges.push_back_reserve(b);    }
               for ( int i = 0; i < hb2.EdgeObjectCount( ); i++ )
                    Preads_plus_edges.push_back_reserve( hb2.EdgeObject(i) );
               PerfectAligner pal( K_orig, PerfectAligner::findProperOnly );
               vec<alignment_plus> aligns;
               pal.Align( Preads_plus_edges, aligns, P.size( ) );
               vec< vec<int> > aligns_index( P.size( ) );
               for ( int i = 0; i < aligns.isize( ); i++ )
               {    if ( aligns[i].Rc2( ) ) continue;
                    aligns_index[ aligns[i].Id1( ) ].push_back(i);    }
               vec< pair<int,int> > Pp;
               for ( int i = 0; i < P.isize( ); i++ )
                    Pp.push_back( make_pair( PAIRS_INDEX[ P[i].first ], i ) );
               Sort(Pp);
               vec< vec< pair<int,int> > > sep_dev( loops.size( ) );
               for ( int i = 0; i < Pp.isize( ); i++ )
               {    int j;
                    for ( j = i + 1; j < Pp.isize( ); j++ )
                         if ( Pp[j].first != Pp[i].first ) break;
                    if ( j - i == 2 && P[ Pp[i].second ].second )
                         swap( Pp[i], Pp[i+1] );
                    if ( j - i == 2 )
                    {    int id1 = Pp[i].second, id2 = Pp[i+1].second;
                         const read_pairing& p = PAIRS[ Pp[i].first ];
                         if ( aligns_index[id1].solo() && aligns_index[id2].solo() )
                         {    const alignment_plus& ap1 
                                   = aligns[ aligns_index[id1][0] ];
                              const alignment_plus& ap2 
                                   = aligns[ aligns_index[id2][0] ];
                              int uv = ap1.Id2( ), vw = ap2.Id2( );
                              int li = -1;
                              for ( int l = 0; l < loops.isize( ); l++ )
                                   if ( loops[l].uv == uv && loops[l].vw == vw ) 
                                        li = l;
                              if ( li < 0 ) 
                              {    i = j - 1;
                                   continue;    }
                              int sep = p.sep - ap2.a.pos2( ) + K_orig - 1
                                   - ( hb2.EdgeLength( ap1.Id2( ) ) 
                                        - ap1.a.Pos2( ) );
                              if (FIRST_DELOOP_VERBOSE) PRINT3( li, sep, p.sd );
                              sep_dev[li].push_back( 
                                   make_pair( sep, p.sd ) );    }    }
                    i = j - 1;    }
               DisambiguateSimpleLoops( h2, loops, sep_dev );
               cout << TimeSince(dis_clock) 
                    << " used disambiguating loop multiplicities\n";    }

          // Evaluate.

          if (EVALUATE_NHOOD_HYPER)
          {    vec<look_align> aligns;
               vec< vec<int> > aligns_index;
               vec<TrustedPath> trusted_paths;
               AlignHyperKmerPath( h2, kbb2, data_dir + "/genome", 
                    wdata_dir + "/run", aligns, aligns_index );
               double eclock2 = WallClockTime( );
               FilterAligns( h2, aligns, aligns_index, trusted_paths );
               cout << "\nComparison of neighborhood HyperKmerPath to reference\n";
               PrintAlignedHyperKmerPath( cout, h2, kbb2, genome, aligns,
                    aligns_index, True, &trusted_paths );    
               cout << TimeSince(eclock2) << " used in WalkInserts - e2" 
                    << endl;    }

          // Record result.

          double clock4 = WallClockTime( );
          h2.Reverse( );
          HyperBasevector hb2p( h2, *kbb2 );
          hyperbases.push_back(hb2p);    
          cout << TimeSince(clock4) << " used in WalkInserts - 4" << endl;    }    }

void PropagateStarts( const pp_pair& p, const HyperKmerPath& h,
     const vec<Bool>& hstart_known, const vec<Bool>& unique_edge, 
     const vec<int>& hstart, Bool& start_known, vec<int>& lstart, vec<int>& rstart )
{    lstart.resize( p.LeftSize( ) ), rstart.resize( p.RightSize( ) );
     Bool lstart_known = False, rstart_known = False;
     for ( int j = 0; j < p.LeftSize( ); j++ )
     {    int id = p.Left(j);
          if ( hstart_known[id] && unique_edge[id] )
          {    lstart_known = True, lstart[j] = hstart[id];
               for ( int m = j-1; m >= 0; m-- )
                    lstart[m] = lstart[m+1] - h.EdgeLength( p.Left(m) );
               for ( int l = j+1; l < p.LeftSize( ); l++ )
               {    if ( hstart_known[ p.Left(l) ] && unique_edge[ p.Left(l) ] )
                    {    lstart[l] = hstart[ p.Left(l) ];    }
                    else 
                    {    lstart[l] = lstart[l-1]
                              + h.EdgeLength( p.Left(l-1) );    }    }    }    }
     for ( int j = p.RightSize( ) - 1; j >= 0; j-- )
     {    int id = p.Right(j);
          if ( hstart_known[id] && unique_edge[id] )
          {    rstart_known = True, rstart[j] = hstart[id];
               for ( int m = j+1; m < p.RightSize( ); m++ )
                    rstart[m] = rstart[m-1] + h.EdgeLength( p.Right(m-1) );
               for ( int l = j-1; l >= 0; l-- )
               {    if ( hstart_known[ p.Right(l) ] && unique_edge[ p.Right(l) ] )
                    {    rstart[l] = hstart[ p.Right(l) ];    }
                    else 
                    {    rstart[l] = rstart[l+1]
                              - h.EdgeLength( p.Right(l) );    }    }    }    }
     start_known = lstart_known || rstart_known;
     if ( lstart_known && !rstart_known )
     {    rstart[0] = lstart.back( ) + h.EdgeLength( p.Left( p.LeftSize( ) - 1 ) )
               + int( round( p.Gap( ) ) );
          for ( int j = 1; j < p.RightSize( ); j++ )
               rstart[j] = rstart[j-1] + h.EdgeLength( p.Right(j-1) );    }
     if ( !lstart_known && rstart_known )
     {    lstart.back( ) = rstart[0] - int( round( p.Gap( ) ) )
               - h.EdgeLength( p.Left( p.LeftSize( ) - 1 ) );
          for ( int j = p.LeftSize( ) - 2; j >= 0; j-- )
               lstart[j] = lstart[j+1] - h.EdgeLength( p.Left(j) );    }    }

void FindFalseReads( const vec<pp_pair>& ppp, const HyperKmerPath& h,
     const vecbasevector& genome, const KmerBaseBroker& kbb, 
     int v, int NHOOD_RADIUS, const vecvec<placement>& locs,
     const vec<int>& genome_path_lengths, Bool verbose )
{    static vec<pp_read> reads;
     static vecbasevector bases;
     reads.clear( ), bases.clear( );
     for ( int u = 0; u < ppp.isize( ); u++ )
     {    for ( int pass = 1; pass <= 2; pass++ )
          {    const pp_read& r = ( pass == 1 ? ppp[u].Left( ) : ppp[u].Right( ) );
               reads.push_back(r);    }    }
     UniqueSort(reads);
     for ( int u = 0; u < reads.isize( ); u++ )
     {    static KmerPath p;
          p.Clear( );
          for ( int v = 0; v < reads[u].isize( ); v++ )
               p.Append( h.EdgeObject( reads[u][v] ) );
          bases.push_back_reserve( kbb.Seq(p) );    }
     double fclock = WallClockTime( );
     static vec<alignment_plus> aligns;
     AlignNhoodStuffToReference( bases, v, NHOOD_RADIUS, locs, genome,
          genome_path_lengths, aligns );
     cout << TimeSince(fclock) << " used aligning to find "
          << "false pp_reads" << endl;
     vec<Bool> aligned( reads.size( ), False );
     for ( int u = 0; u < aligns.isize( ); u++ )
     {    const alignment_plus& ap = aligns[u];
          int id1 = ap.Id1( );
          if ( ap.a.pos1( ) == 0 && ap.a.Pos1( ) == bases[id1].isize( ) )
               aligned[id1] = True;    }
     for ( int x = 0; x < reads.isize( ); x++ )
     {    if ( aligned[x] ) continue;
          int u = x/2, pass = x % 2;
          if (verbose)
          {    cout << "Warning: " << reads[x] << " does not align "
                    << "perfectly to reference.\n";    }    }    }

void TrimPairs( int v, const vec<int>& ulen, const HyperKmerPath& h, 
     vec<pp_pair>& ppp, const vec<int>& pppL, const vec<Bool>& hstart_known, 
     const vec<int>& hstart, vec<Bool>& unique_edge, 
     const int NHOOD_RADIUS_INTERNAL, const double TRIM_MULTIPLIER, 
     Bool& found_universal, Bool BASIC_DEBUG )
{
     // Propagate predicted start positions.  Then trim pairs to remove stuff that
     // is too far from the origin of the neighboorhood.  (Note that trimming is not
     // yet carried out.)  Also, if there is a pair that is a closure, that contains 
     // a unique element, and that is predicted to cover the entire 
     // neighborhood, delete all the other pairs.

     int min_universal_radius = int( floor( 0.7 * double(NHOOD_RADIUS_INTERNAL) ) );
     cout << "\nTrimming pairs:\n";
     int universal = -1;
     found_universal = False;
     vec<Bool> to_remove( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    pp_pair& p = ppp[i];
          Bool start_known;
          vec<int> lstart, rstart;
          PropagateStarts( p, h, hstart_known, unique_edge, hstart, 
               start_known, lstart, rstart );
          if ( IsClosed( p, pppL ) )
          {    Bool have_unique = False;
               for ( int j = 0; j < p.LeftSize( ); j++ )
                    if ( unique_edge[ p.Left(j) ] ) have_unique = True;
               int right = rstart.back( ) + pppL[ p.Right( ).back( ) ];
               if ( have_unique && start_known
                    && lstart.front( ) <= -min_universal_radius
                    && right >= min_universal_radius + ulen[v] )
               {    cout << "UNIVERSAL: pair " << i << "\n";
                    found_universal = True;
                    universal = i;    }    }
          int nlim = int( floor( double(NHOOD_RADIUS_INTERNAL) * TRIM_MULTIPLIER ) );
          if ( !start_known ) continue;
          vec<int> lstop( p.LeftSize( ) ), rstop( p.RightSize( ) );
          for ( int j = 0; j < p.LeftSize( ); j++ )
               lstop[j] = lstart[j] + h.EdgeLength( p.Left(j) );
          for ( int j = 0; j < p.RightSize( ); j++ )
               rstop[j] = rstart[j] + h.EdgeLength( p.Right(j) );
          if ( lstop[0] >= -nlim && rstop[0] >= -nlim
               && lstart.back( ) <= nlim && rstart.back( ) <= nlim + ulen[v] )
          {    continue;    }
          if ( lstop.back( ) < -nlim || rstart[0] > nlim + ulen[v]
               || lstart[0] > nlim + ulen[v] || rstop.back( ) < -nlim )
          {    to_remove[i] = True;
               if (BASIC_DEBUG)
               {    cout << "\nNeed to delete pair " << i << ":\n";
                    cout << p << "\n";    }    }
          else
          {    int left_remove_left = 0, left_remove_right = 0;
               int right_remove_left = 0, right_remove_right = 0;
               double gap = p.Gap( );
               for ( int j = 0; j < p.LeftSize( ); j++ )
               {    if ( lstop[j] < -nlim ) ++left_remove_left;
                    else break;    }
               for ( int j = p.LeftSize( ) - 1; j >= 0; j-- )
               {    if ( lstart[j] > nlim + ulen[v] ) ++left_remove_right;
                    else break;    }
               for ( int j = 0; j < p.RightSize( ); j++ )
               {    if ( rstop[j] < -nlim ) ++right_remove_left;
                    else break;    }
               for ( int j = p.RightSize( ) - 1; j >= 0; j-- )
               {    if ( rstart[j] > nlim + ulen[v] ) ++right_remove_right;
                    else break;    }
               if ( IsClosed( p, pppL ) )
               {    left_remove_left = Max( left_remove_left, right_remove_left );
                    right_remove_left = Max( left_remove_left, right_remove_left );
                    left_remove_right = Max( left_remove_right, right_remove_right );
                    right_remove_right 
                         = Max( left_remove_right, right_remove_right );    }
               for ( int j = 0; j < left_remove_right; j++ )
                    gap += pppL[ p.Left( p.LeftSize( ) - j - 1 ) ];
               for ( int j = 0; j < right_remove_left; j++ )
                    gap += pppL[ p.Right(j) ];
               if (BASIC_DEBUG)
               {    cout << "\nMay need to trim pair " << i << ":\n";
                    cout << "remove " << left_remove_left << " units from "
                         << "left of left read;\n";
                    cout << "remove " << left_remove_right 
                         << " units from " << "right of left read;\n";
                    cout << "remove " << right_remove_left 
                         << " units from " << "left of right read;\n";
                    cout << "remove " << right_remove_right 
                         << " units from " << "right of right read.\n";    }
               pp_read left = p.Left( ), right = p.Right( );
               if ( ( left_remove_left + left_remove_right >= left.isize( ) )
                    || ( right_remove_left + right_remove_right >= right.isize( ) ) )
               {    cout << "Danger: attempted to illegally trim\n";
                    PRINT(p);
                    PRINT3( left, left_remove_left, left_remove_right );
                    PRINT3( right, right_remove_left, right_remove_right );
                    to_remove[i] = True;
                    if ( universal == i ) universal = -1;
                    continue;    }
               left.resize( left.size( ) - left_remove_right );
               left.ReverseMe( );
               left.resize( left.size( ) - left_remove_left );
               left.ReverseMe( );
               right.resize( right.size( ) - right_remove_right );
               right.ReverseMe( );
               right.resize( right.size( ) - right_remove_left );
               right.ReverseMe( );
               pp_pair pnew( left, right, gap, p.Dev( ) );    
               p = pnew;    }    }
     cout << endl;
     if ( universal >= 0 )
     {    pp_pair p = ppp[universal];
          ppp.resize(1);
          ppp[0] = p;    }
     else EraseIf( ppp, to_remove );     }

Bool cmp_align2( const alignment_plus& ap1, const alignment_plus& ap2 )
{    if ( ap1.Id2( ) < ap2.Id2( ) ) return True;
     if ( ap1.Id2( ) > ap2.Id2( ) ) return False;
     if ( ap1.a.pos2( ) < ap2.a.pos2( ) ) return True;
     return False;    }

// Add in unipaths of predicted copy number one that are linked to by copy number
// one unipaths, regardless of their lengths and how many of them there are.
// NO! CHANGED! Now looks for copy number <= copy number(v).

void AddCopyNumberOne( const int v, vec<ustart>& processed, 
     const vec<int>& predicted_copyno, const digraphE<sepdev>& Gplus, 
     const vec<int>& ulen, const int NHOOD_RADIUS, const double MAX_DEV )
{    static vec<ustart> processed_extra;
     processed_extra.clear( );
     for ( int i = 0; i < processed.isize( ); i++ )
     {    int w = processed[i].Uid( );
          if ( predicted_copyno[w] > predicted_copyno[v] ) continue;
          int wstart = processed[i].Start( );
          vec<int> wdev = processed[i].Dev( );
          for ( int i = 0; i < Gplus.From(w).isize( ); i++ )
          {    int x = Gplus.From(w)[i];
               if ( predicted_copyno[x] > predicted_copyno[v] ) continue;
               int wxsep = Gplus.EdgeObjectByIndexFrom( w, i ).Sep( );
               int xstart = wstart + ulen[w] + wxsep;
               if ( xstart > ulen[v] + NHOOD_RADIUS ) continue;
               Bool used = False;
               for ( int j = 0; j < processed.isize( ); j++ )
               {    if ( x == processed[j].Uid( ) )
                    {    used = True;
                         break;    }    }
               if (used) continue;
               for ( int j = 0; j < processed_extra.isize( ); j++ )
               {    if ( x == processed_extra[j].Uid( ) )
                    {    used = True;
                         break;    }    }
               if (used) continue;
               static vec<int> dev;
               dev = wdev;
               dev.push_back( Gplus.EdgeObjectByIndexFrom( w, i ).Dev( ) );
               processed_extra.push_back( ustart( x, xstart, dev ) );
               if ( processed_extra.back( ).MeanDev( ) > MAX_DEV )
                    processed_extra.resize( processed_extra.isize( ) - 1 );    }
          for ( int i = 0; i < Gplus.To(w).isize( ); i++ )
          {    int x = Gplus.To(w)[i];
               if ( predicted_copyno[x] > predicted_copyno[v] ) continue;
               int xwsep = Gplus.EdgeObjectByIndexTo( w, i ).Sep( );
               int xstart = wstart + - xwsep - ulen[x];
               if ( xstart + ulen[x] < -NHOOD_RADIUS ) continue;
               Bool used = False;
               for ( int j = 0; j < processed.isize( ); j++ )
               {    if ( x == processed[j].Uid( ) )
                    {    used = True;
                         break;    }    }
               if (used) continue;
               for ( int j = 0; j < processed_extra.isize( ); j++ )
               {    if ( x == processed_extra[j].Uid( ) )
                    {    used = True;
                         break;    }    }
               if (used) continue;
               static vec<int> dev;
               dev = wdev;
               dev.push_back( Gplus.EdgeObjectByIndexTo( w, i ).Dev( ) );
               processed_extra.push_back( ustart( x, xstart, dev ) );
               if ( processed_extra.back( ).MeanDev( ) > MAX_DEV )
                    processed_extra.resize( processed_extra.isize( ) - 1 );    }    }
     for ( int j = 0; j < processed_extra.isize( ); j++ )
          processed.push_back( processed_extra[j] );
     Sort(processed);    }

// Print placement of truth on graph.

void PrintTruthPlacement( const int v, const vecvec<placement>& locs, 
     const vecbasevector& genome, const int NHOOD_RADIUS_INTERNAL,
     const HyperKmerPath& h, const KmerBaseBroker& kbbnb,
     const vec<int>& edge_copyno )
{    cout << "\nTruth for this neighborhood:\n";
     int K = h.K( );
     for ( int i = 0; i < locs[v].size( ); i++ )
     {    cout << i+1 << ". ";
          const placement& p = locs[v][i];
          const basevector& g = genome[ p.GenomeId( ) ];
          int start = Max( 0, p.pos( ) - NHOOD_RADIUS_INTERNAL );
          int stop = Min( g.isize( ), p.pos( ) + NHOOD_RADIUS_INTERNAL );
          static vecbasevector X;
          X.clear( );
          for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
               X.push_back_reserve( kbbnb.Seq( h.EdgeObject(i) ) );
          static basevector b;
          b.SetToSubOf( g, start, stop - start );
          X.push_back_reserve(b);
          PerfectAligner pal( K, PerfectAligner::findProperOnly );
          vec<alignment_plus> aligns;
          pal.Align( X, aligns, h.EdgeObjectCount( ) );
          sort( aligns.begin( ), aligns.end( ), cmp_align2 );
          int pos2 = 0;
          Bool first_print = True;
          for ( int j = 0; j < aligns.isize( ); j++ )
          {    const alignment_plus& ap = aligns[j];
               int id1 = ap.Id1( );
               if ( ap.Rc2( ) ) continue;
               if ( ap.a.pos2( ) < pos2 ) continue;
               if ( ap.a.pos2( ) > pos2 )
               {    if ( !first_print ) cout << "\n.";
                    cout << "?.\n";
                    first_print = True;
                    pos2 = ap.a.Pos2( ) - K + 1;
                    continue;    }
               if ( ap.a.pos2( ) > 0 && ap.a.pos1( ) > 0 ) continue;
               if ( ap.a.Pos2( ) < b.isize( )
                    && ap.a.Pos1( ) < X[id1].isize( ) )
               {    continue;    }
               pos2 = ap.a.Pos2( ) - K + 1;
               if ( !first_print ) cout << ".";
               cout << BaseAlpha( aligns[j].Id1( ) );
               first_print = False;    }
          if ( pos2 != b.isize( ) - K + 1 ) cout << "\n.?";
          cout << "\n";    }
     cout << "where:\n";
     vec<String> s;
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
     {    static String x;
          x = BaseAlpha(i) + "[" + ToString( h.EdgeLength(i) );
          if ( edge_copyno[i] >= 0 ) x += ",c=" + ToString( edge_copyno[i] );
          x += "]";
          s.push_back(x);    }
     PrintInColumns(s);    }

// Trace elements of ppp back to the read pairs which could (in principle) 
// originate them.

void TracePairs( const vecbasevector& sreads, const vec<pp_pair>& ppp,
     const HyperKmerPath& h, const KmerBaseBroker& kbbnb, 
     const vec<read_location>& slocs, const vec<read_pairing>& spairs )
{    vecbasevector all(sreads);
     vec<pp_read> preads;
     for ( int u = 0; u < ppp.isize( ); u++ )
     {    static KmerPath p;
          p.Clear( );
          for ( int m = 0; m < ppp[u].LeftSize( ); m++ )
               p.Append( h.EdgeObject( ppp[u].Left(m) ) );
          all.push_back_reserve( kbbnb.Seq(p) );
          preads.push_back( ppp[u].Left( ) );
          p.Clear( );
          for ( int m = 0; m < ppp[u].RightSize( ); m++ )
               p.Append( h.EdgeObject( ppp[u].Right(m) ) );
          p.Reverse( );
          all.push_back_reserve( kbbnb.Seq(p) );
          preads.push_back( ppp[u].Right( ) );    }
     int K = h.K( );
     PerfectAligner pal( K, PerfectAligner::findProperOnly );
     vec<alignment_plus> aligns;
     pal.Align( all, aligns, sreads.size( ) );
     vec< vec<int> > alindex( sreads.size( ) );
     for ( int i = 0; i < aligns.isize( ); i++ )
     {    const alignment_plus& ap = aligns[i];
          if ( ap.Rc2( ) ) continue;
          if ( ap.a.pos1( ) > 0 || ap.a.Pos1( ) < sreads[ ap.Id1( ) ].isize( ) )
               continue;
          alindex[ ap.Id1( ) ].push_back(i);    }
     for ( int id = 0; id < sreads.size( ); id++ )
     {    static vec<Bool> to_remove;
          to_remove.resize_and_set( alindex[id].size( ), False );
          for ( int j1 = 0; j1 < alindex[id].isize( ); j1++ )
          {    const alignment_plus& ap1 = aligns[ alindex[id][j1] ];
               const pp_read& r1 = preads[ ap1.Id2( ) ];
               for ( int j2 = 0; j2 < alindex[id].isize( ); j2++ )
               {    const alignment_plus& ap2 = aligns[ alindex[id][j2] ];
                    const pp_read& r2 = preads[ ap2.Id2( ) ];
                    if ( j1 == j2 || r1 == r2 ) continue;
                    static vec<int> offsets;
                    GetOverlaps( r1, r2, offsets );
                    for ( int u = 0; u < offsets.isize( ); u++ )
                    {    int o = offsets[u];
                         if ( o < 0 ) continue;
                         if ( o + r2.isize( ) > r1.isize( ) ) continue;
                         to_remove[j1] = True;    }    }    }
          EraseIf( alindex[id], to_remove );    }
     vec< vec<int> > source( ppp.isize( ) );
     for ( int u = 0; u < spairs.isize( ); u++ )
     {    int id1 = spairs[u].id1, id2 = spairs[u].id2;
          for ( int j1 = 0; j1 < alindex[id1].isize( ); j1++ )
          {    const alignment_plus& ap1 = aligns[ alindex[id1][j1] ];
               int ID1 = ap1.Id2( );
               if ( ID1 % 2 != 0 ) continue;
               for ( int j2 = 0; j2 < alindex[id2].isize( ); j2++ )
               {    const alignment_plus& ap2 = aligns[ alindex[id2][j2] ];
                    int ID2 = ap2.Id2( );
                    if ( ID2 != ID1 + 1 ) continue;
                    int p = ID1/2;
                    source[p].push_back(u);    }    }    }
     vec<int> slocs_index( sreads.size( ), -1 );
     for ( int u = 0; u < slocs.isize( ); u++ )
          slocs_index[ slocs[u].ReadId( ) ] = u;
     cout << "\nSources of paired pairs:\n";
     for ( int u = 0; u < ppp.isize( ); u++ )
     {    static vec< pair<read_location,read_location> > plocs;
          plocs.clear( );
          cout << "[" << u << "]:\n";
          for ( int j = 0; j < source[u].isize( ); j++ )
          {    int pid = source[u][j];
               int id1 = spairs[pid].id1, id2 = spairs[pid].id2;
               const read_location& rl1 = slocs[ slocs_index[id1] ];
               const read_location& rl2 = slocs[ slocs_index[id2] ];
               plocs.push_back( make_pair( rl1, rl2 ) );    }
          Sort(plocs);
          for ( int j = 0; j < plocs.isize( ); j++ )
          {    const read_location &rl1 = plocs[j].first, &rl2 = plocs[j].second;
               cout << rl1.Contig( ) << "." << rl1.Start( ) << "-" << rl1.Stop( ) 
                    << " ---> " << rl2.Contig( ) << "." << rl2.Start( ) 
                    << "-" << rl2.Stop( ) << "\n";    }    }    }

// Print pairs, showing predicted placements.

void PrintPairs( const vec<pp_pair>& ppp, const vec<Bool>& hstart_known,
     const vec<int>& hstart, const vec<Bool>& unique_edge )
{    cout << "\nPairs with placements:\n";
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    cout << "\n[" << i << "] ";
          const pp_pair& p = ppp[i];
          for ( int j = 0; j < p.LeftSize( ); j++ )
          {    if ( j > 0 ) cout << ".";
               if ( j > 0 && j % 15 == 0 ) cout << "\n";
               int id = p.Left(j);
               cout << BaseAlpha(id);
               if ( hstart_known[id] && unique_edge[id] ) 
                    cout << "[" << hstart[id] << "]";    }
          cout << " --- ";
          RightPrecisionOut( cout, p.Gap( ), 2 );
          cout << " +/- ";
          RightPrecisionOut( cout, p.Dev( ), 2 );
          cout << " --> ";
          for ( int j = 0; j < p.RightSize( ); j++ )
          {    if ( j > 0 ) cout << ".";
               if ( j > 0 && j % 15 == 0 ) cout << "\n";
               int id = p.Right(j);
               cout << BaseAlpha(id);
               if ( hstart_known[id] && unique_edge[id] ) 
                    cout << "[" << hstart[id] << "]";    }
          cout << "\n";    }    }

// Show coverage of reference by closures and find inserts having no true closure.

void CheckReferenceCoverage( const int v, const vec< vec<pp_closure> >& ppclosures,
     const HyperKmerPath& h, const KmerBaseBroker& kbbnb, const int NHOOD_RADIUS,
     const vecvec<placement>& locs, const vecbasevector& genome,
     const vec<int>& genome_path_lengths, const Bool BASIC_DEBUG,
     const vec<pp_pair>& ppp, const int SHOW_PAIR_PLACEMENTS, 
     const String& sub_dir )
{    if (BASIC_DEBUG) cout << "\nCoverage of reference by paired pair closures:\n";
     static vec<pp_closure> clo, clo2;
     static vec< pair<int,int> > cloi;
     clo.clear( ), clo2.clear( ), cloi.clear( );
     for ( int i = 0; i < ppclosures.isize( ); i++ )
     {    for ( int j = 0; 
               j < ppclosures[i].isize( ); j++ )
          {    clo.push_back( ppclosures[i][j] );    
               cloi.push_back( make_pair(i,j) );    }    }
     SortSync( clo, cloi );
     static vec< vec< pair<int,int> > > clov;
     clo2.clear( ), clov.clear( );
     for ( int i = 0; i < clo.isize( ); i++ )
     {    int j;
          for ( j = i + 1; j < clo.isize( ); j++ )
               if ( clo[j] != clo[i] ) break;
          clo2.push_back( clo[i] );
          static vec< pair<int,int> > v;
          v.clear( );
          for ( int u = i; u < j; u++ )
               v.push_back( cloi[u] );
          clov.push_back(v);
          i = j - 1;    }
     clo = clo2;
     static vecbasevector bases;
     bases.clear( );
     for ( int u = 0; u < clo.isize( ); u++ )
     {    const pp_closure& r = clo[u];
          static KmerPath p;
          p.Clear( );
          for ( int v = 0; v < r.isize( ); v++ )
               p.Append( h.EdgeObject( r[v] ) );
          bases.push_back_reserve( kbbnb.Seq(p) );    }
     vec<alignment_plus> aligns;
     AlignNhoodStuffToReference( bases, v, NHOOD_RADIUS, locs, 
          genome, genome_path_lengths, aligns );
     vec<Bool> have_true( ppp.size( ), False );
     for ( int u = 0; u < aligns.isize( ); u++ )
     {    const alignment_plus& ap = aligns[u];
          int id1 = ap.Id1( ), id2 = ap.Id2( );
          if ( ap.a.pos1( ) == 0 && ap.a.Pos1( ) == bases[id1].isize( ) )
          {    if (BASIC_DEBUG)
               {    cout << "\n" << clo[id1] << " " << id2 << "." 
                         << ap.a.pos2( ) << "-" << ap.a.Pos2( );    }
               for ( int x = 0; x < clov[id1].isize( ); x++ )
               {    int i = clov[id1][x].first;
                    int j = clov[id1][x].second;
                    have_true[i] = True;    
                    if (BASIC_DEBUG) cout << " [" << i << "," << j << "]";    }
               if (BASIC_DEBUG) cout << "\n";    }    }
     cout << "\n";
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    int pairlocs_stage = GetAndIncrementPairlocsStage( );
          String outfile = sub_dir + "/pairlocs/" + ToString(pairlocs_stage);
          String title = "short-insert closures";
          String report = "STAGE " + ToString(pairlocs_stage) + ", " 
               + title + "\n\n";
          Ofstream( out, outfile );
          cout << "\n" << report;
          out << report << "Closure locations:\n\n";
          vec<pair_placement> places;
          for ( int u = 0; u < aligns.isize( ); u++ )
          {    const alignment_plus& ap = aligns[u];
               int id1 = ap.Id1( ), id2 = ap.Id2( );
               if ( ap.pos1( ) == 0 && ap.Pos1( ) == bases[id1].isize( ) )
               {    for ( int x = 0; x < clov[id1].isize( ); x++ )
                    {    int i = clov[id1][x].first;
                         places.push_back( pair_placement( i, id2, ap.pos2( ), 
                              ap.Pos2( ), ap.pos2( ), ap.Pos2( ), 
                              ap.Rc2( ) ) );    }    }    }
          Sort(places);
          for ( int i = 0; i < places.isize( ); i++ )
               out << places[i] << " [CLOSED]\n";    }
     if (BASIC_DEBUG)
     {    for ( int u = 0; u < ppp.isize( ); u++ )
          {    if ( ppclosures[u].empty() )
               {    cout << "Warning: no closures found for " 
                         << ppp[u] << ".\n";    }
               else if ( !have_true[u] )
               {    cout << "Warning: only false closures found for " 
                         << ppp[u] << ".\n";    }    }    }    }

void ProcessUniversalClosure( const vec<pp_pair>& ppp, const HyperKmerPath& h,
     const vec<read_pairing>& spairs, const vec<Bool>& primary,
     const vecKmerPath& xpaths, const vecKmerPath& xpaths_rc,
     const KmerBaseBroker& kbbnb, const Bool EVALUATE_NHOOD_HYPER,
     const String& wrun_dir, const String& data_dir,
     vec<HyperBasevector>& hyperbases )
{    int K = h.K( );
     static KmerPath p;
     static vec<int> pstarts;
     p.Clear( ), pstarts.clear( );
     pstarts.push_back(0);
     for ( int m = 0; m < ppp[0].LeftSize( ); m++ )
          p.Append( h.EdgeObject( ppp[0].Left(m) ) );
     int sum = 0;
     for ( int j = 0; j < p.NSegments( ); j++ )
     {    sum += p.Length(j);
          pstarts.push_back(sum);    }

     // Place read cloud pairs that are primary AND secondary on the universal 
     // closure and then trim it back so it does not extend beyond the end of all 
     // pair placements.  Note that it would make better sense to use
     // primary read cloud pairs.

     static vecKmerPath pv;
     pv.clear( );
     pv.push_back_reserve(p);
     static vec<tagged_rpint> pdb;
     CreateDatabase( pv, pdb );
     Bool first = True;
     pair<int,int> left, right;
     for ( int i = 0; i < spairs.isize( ); i++ )
     {    int id1 = spairs[i].id1, id2 = spairs[i].id2;
          if ( !primary[id1] ) continue;
          const KmerPath &p1 = xpaths[id1], &p2 = xpaths_rc[id2];
          static vec< pair<int,int> > starts, stops;
          starts.clear( ), stops.clear( );
          longlong x = p1.Start(0);
          static vec<longlong> con;
          Contains( pdb, x, con );
          for ( int j = 0; j < con.isize( ); j++ )
          {    const tagged_rpint& t = pdb[ con[j] ];
               int pp = t.PathPos( );
               if ( !ProperOverlapExt( p1, p, 0, pp ) ) continue;
               starts.push_back( 
                    make_pair( pp, x - t.Start( ) ) );    }
          int n2 = p2.NSegments( );
          x = p2.Stop(n2-1);
          Contains( pdb, x, con );
          for ( int j = 0; j < con.isize( ); j++ )
          {    const tagged_rpint& t = pdb[ con[j] ];
               int pp = t.PathPos( );
               if ( !ProperOverlapExt( p2, p, n2-1, pp ) ) continue;
               stops.push_back( 
                    make_pair( pp, x - t.Start( ) ) );    }
          static vec< pair< double, pair<int,int> > > start_stops;
          start_stops.clear( );
          for ( int j1 = 0; j1 < starts.isize( ); j1++ )
          {    for ( int j2 = 0; j2 < stops.isize( ); j2++ )
               {    int start = pstarts[ starts[j1].first ]
                         + starts[j1].second;
                    int stop = pstarts[ stops[j2].first ]
                         + stops[j2].second;
                    int sep = ( stop - p2.KmerCount( ) + 1 )
                         - ( start + p1.KmerCount( ) - 1 ) - K;
                    double dev =
                         double( Abs( sep - spairs[i].sep ) )
                              / double( spairs[i].sd );
                    if ( dev > 3.0 ) continue;
                    start_stops.push_back( make_pair(
                         dev, make_pair(j1,j2) ) );    }    }
          static vec<Bool> remove;
          remove.resize_and_set( start_stops.size( ), False );
          for ( int u1 = 0; u1 < start_stops.isize( ); u1++ )
          {    int j1 = start_stops[u1].second.first;
               int j2 = start_stops[u1].second.second;
               for ( int u2 = 0; u2 < start_stops.isize( ); u2++ )
               {    if ( u1 == u2 ) continue;
                    int j1x = start_stops[u2].second.first;
                    int j2x = start_stops[u2].second.second;
                    if ( j1 == j1x && stops[j2x] > stops[j2] )
                         remove[u2] = True;
                    if ( j2 == j2x && starts[j1x] < starts[j1] )
                         remove[u2] = True;    }    }
          EraseIf( start_stops, remove );
          for ( int j = 0; j < start_stops.isize( ); j++ )
          {    pair<int,int> start =
                    starts[ start_stops[j].second.first ];
               pair<int,int> stop =
                    stops[ start_stops[j].second.second ];
               if (first)
               {    left = start, right = stop;
                    first = False;
                    continue;    }
               left = Min( left, start );
               right = Max( right, stop );    }    }    
     if ( !first )
     {    static KmerPath pnew;
          pnew.Clear( );
          KmerPathLoc pleft( p, left.first, left.second );
          KmerPathLoc pright( p, right.first, right.second );
          p.CopySubpath( pleft, pright, pnew );
          p = pnew;    }
     
     // Save the universal closure.

     p.Reverse( );
     vec<basevector> universe;
     universe.push_back( kbbnb.Seq(p) );
     if (EVALUATE_NHOOD_HYPER)
     {    String hyper_fasta = wrun_dir + "/nhood.hyper.fasta";
          {    Ofstream( out, hyper_fasta );
               kbbnb.Seq(p).Print( out, "nhood_hyper" );    }
          cout << "\nComparison of neighborhood HyperKmerPath to "
               << "reference:\n";
          QueryLookupTableCore( "K=12 QUIET=True L=" 
               + data_dir + "/genome.lookup VISUAL=True NH=True "
               + "SEQS=" + hyper_fasta );    }
     HyperBasevector hb( K, universe );
     hyperbases.push_back(hb);    }

void ReportTrueCoverageGaps( const vecbasevector& genome, 
     const vec<read_location>& readlocs, const int K,
     const vec<Bool>& is_short_pair_read )
{    cout << "True coverage gaps, by all reads:\n";
     vec< vec<ho_interval> > cov( genome.size( ) );
     for ( int i = 0; i < readlocs.isize( ); i++ )
     {    const read_location& rl = readlocs[i];
          int rstart = rl.Start( ) + K/2, rstop = rl.Stop( ) + 1 - K/2;
          int m = rl.Contig( );
          if ( rstart < rstop )
               cov[m].push_back( ho_interval( rstart, rstop ) );    }
     for ( int m = 0; m < genome.size( ); m++ )
     {    static vec< pair<ho_interval,int> > cov2;
          CondenseIntervals( genome[m].size( ), cov[m], cov2 );
          for ( int j = 0; j < cov2.isize( ); j++ )
          {    if ( cov2[j].second == 0 )
               {    const ho_interval& h = cov2[j].first;
                    cout << m << "." << h.Start( )
                         << "-" << h.Stop( ) << "\n";    }    }    }
     cout << "True coverage gaps, by reads in short-insert pairs:\n";
     vec< vec<ho_interval> > covsh( genome.size( ) );
     for ( int i = 0; i < readlocs.isize( ); i++ )
     {    const read_location& rl = readlocs[i];
          if ( !is_short_pair_read[ rl.ReadId( ) ] ) continue;
          int rstart = rl.Start( ) + K/2, rstop = rl.Stop( ) + 1 - K/2;
          int m = rl.Contig( );
          if ( rstart < rstop )
               covsh[m].push_back( ho_interval( rstart, rstop ) );    }
     for ( int m = 0; m < genome.size( ); m++ )
     {    static vec< pair<ho_interval,int> > cov2;
          CondenseIntervals( genome[m].size( ), covsh[m], cov2 );
          for ( int j = 0; j < cov2.isize( ); j++ )
          {    if ( cov2[j].second == 0 )
               {    const ho_interval& h = cov2[j].first;
                    cout << m << "." << h.Start( )
                         << "-" << h.Stop( ) << "\n";    }    }    }    }

// EdgeCopyNumber: Guess what the copy number of the edges are.  Default 
// (no knowledge) value is -1.  Note that copy number > 10 is not looked at.
// Note that unipaths < EDGE_MIN are not seen.  Also we enforce a hard-coded lower 
// bound on the probability of edge uniqueness.

void EdgeCopyNumber( const HyperKmerPath& h, const vec<int>& fw_reads_orig,
     const vec<int>& rc_reads_orig, const vecKmerPath& zpaths,
     const vecKmerPath& zpaths_rc, const vec<read_location_short>& ulocs,
     const vecvec<int>& ulocs_indexr, const vec<int>& predicted_copyno,
     const vec<double>& predicted_copyno_p, int EDGE_MIN, 
     const vec<int>& readLengths, const vec<int>& ulen,
     const vec<tagged_rpint>& edgedb,
     /* outputs: */ vec<int>& edge_copyno, vec<double>& edge_copyno_p,
     vec<Bool>& unique_edge )
{
     const double min_p = 0.98;

     // 1. We find the kmer ranges [start,stop) on oriented reads 
     // id in P that contain a globally unique kmer.

     const int max_copy_number = 10;
     static vec< vec<KmerPathInterval> > qx;
     static vec< vec<double> > qxp;
     qx.clear( ), qxp.clear( );
     qx.resize( max_copy_number + 1 ), qxp.resize( max_copy_number + 1 );
     for ( int pass = 1; pass <= 2; pass++ )
     {    const vec<int>& O = ( pass == 1 ? fw_reads_orig : rc_reads_orig );
          const vecKmerPath& Z = ( pass == 1 ? zpaths : zpaths_rc );
          for ( int i = 0; i < O.isize( ); i++ )
          {    int id = O[i];
               for ( int j = 0; j < ulocs_indexr[id].size( ); j++ )
               {    const read_location_short& rl = ulocs[ ulocs_indexr[id][j] ];
                    int u = rl.Contig( );
                    if ( ulen[u] < EDGE_MIN ) continue;
                    int pc = predicted_copyno[u];
                    if ( pc > max_copy_number ) continue;
                    if ( predicted_copyno_p[u] < min_p ) continue;
                    if ( pass == 1 && rl.Rc( ) ) continue;
                    if ( pass == 2 && rl.Fw( ) ) continue;
                    int start = Max( 0, -rl.Start( ) );
                    int stop = Min( readLengths[id] - h.K( ) + 1,
                         -rl.Start( ) + ulen[u] );
                    KmerPathLoc Start = Z[i].Begin( ), Stop = Z[i].Begin( );
                    Start = Start + start, Stop = Stop + (stop - 1);
                    static KmerPath p;
                    p.Clear( );
                    Z[i].CopySubpath( Start, Stop, p );
                    p.Reverse( );
                    for ( int x = 0; x < p.NSegments( ); x++ )
                    {    qx[pc].push_back( p.Segment(x) );    
                         qxp[pc].push_back( 
                              predicted_copyno_p[u] );    }    }    }    }
     edge_copyno.resize_and_set( h.EdgeObjectCount( ), -1 );
     edge_copyno_p.resize_and_set( h.EdgeObjectCount( ), -1 );
     for ( int c = 1; c <= max_copy_number; c++ )
     {    for ( int i = 0; i < qx[c].isize( ); i++ )
          {    static vec<longlong> places;
               Contains( edgedb, qx[c][i], places );
               for ( int j = 0; j < places.isize( ); j++ )
               {    const tagged_rpint& t = edgedb[ places[j] ];
                    int e = t.PathId( );
                    if ( edge_copyno[e] == -1 || c < edge_copyno[e]
                         || ( c == edge_copyno[e] && qxp[c][i] > edge_copyno_p[e] ) )
                    {    edge_copyno[e] = c;    
                         edge_copyno_p[e] = qxp[c][i];    }    }    }    }
     unique_edge.resize_and_set( h.EdgeObjectCount( ), False );
     cout << "\nEdges that appear to be globally unique:";
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
     {    if ( edge_copyno[i] == 1 ) 
          {    ostrstream o;
               o << setiosflags(ios::fixed) << setprecision(1) << setw(4) 
                    << 100.0 * edge_copyno_p[i] << resetiosflags(ios::fixed) << ends;
               String uniqp( o.str( ) );
               cout << " " << BaseAlpha(i);
               if ( !uniqp.Contains( "100", 0 ) ) cout << "(" << uniqp << "%" << ")";
               unique_edge[i] = True;    }    }    }

/**
   Local function: GetSubpathKmerAdjacencies

   Given a <KmerPath> and a range of kmers within it, add the <kmer adjacencies> found in
   this range to the specified set of kmer adjacencies.   Used by <GetGenomicKmerAdjacencies()>
   and <GetLocalizedReadsKmerAdjacencies()>.

   We look at kmers _starting_ at positions from fromLoc to toLoc.
*/
void GetSubpathKmerAdjacencies(// input
			       const KmerPath& kmerPath,
			       const KmerPathLoc& fromLoc,
			       const KmerPathLoc& toLoc,
			       // output
			       KmerAdjSet& subpathKmerAdjs ) {
  ForceAssert( fromLoc.GetPathPtr()->NSegments() == kmerPath.NSegments()  &&
	       toLoc.GetPathPtr()->NSegments() == kmerPath.NSegments() );
  kmer_id_t prevKmerId = NULL_KMER_ID;
  Bool firstIter = True;
  KmerPathLoc prevLoc;
  for ( KmerPathLoc curKmerLoc = fromLoc; curKmerLoc <= toLoc
	  &&  ( firstIter || curKmerLoc > prevLoc ) ; curKmerLoc.Increment() ) {
    kmer_id_t curKmerId = curKmerLoc.GetKmer();
    if ( prevKmerId != NULL_KMER_ID )
      subpathKmerAdjs.insert( make_pair( prevKmerId, curKmerId ) );
    prevKmerId = curKmerId;
    firstIter = False;
    prevLoc = curKmerLoc;
  }
  
}  // GetSubpathKmerAdjacences()
			        
/**
   Function: GetGenomicKmerAdjacencies

   For a given neighborhood, take its alignment to the reference, and get the set of
   <kmer adjacencies> on that strand of the reference.  If the neighborhood has several
   alignments to the reference, get the unified set of kmer adjacencies from all
   of its placements on the reference (since a <HyperKmerPath> that we'll eventually
   reconstruct from the neighborhood will represent the union of possible genome sequences
   at all these placements).
*/
void GetGenomicKmerAdjacencies(// input
			       unipath_id_t seed, nbases_t K, const vecKmerPath& genome_paths, const vecKmerPath& genome_paths_rc,
			       const vecKmerPathIndex& genome_paths_idx,
			       const vecvec<placement>& unipathPlacementsOnGenome,
			       const vec<nkmers_t>& unipathLengthsInKmers,
			       nbases_t NHOOD_RADIUS_INTERNAL,
			       
			       // output
			       KmerAdjSet& genomicKmerAdjs ) {

  genomicKmerAdjs.clear();

  for ( int seedPlacementNum = 0; seedPlacementNum < unipathPlacementsOnGenome[seed].size(); seedPlacementNum++ ) {

    // Get the boundaries of the neighborhood on the reference:
    // start of seed minus NHOOD_RADIUS_INTERNAL,
    // through end of seed plus NHOOD_RADIUS_INTERNAL.
    const placement& seedPlacement = unipathPlacementsOnGenome[seed][seedPlacementNum];
    nbases_t seedLen = unipathLengthsInKmers[seed] + K - 1;
    //  cout << " seed=" << seed << " seedPlacement.pos()=" << seedPlacement.pos() <<
    //    " seedPlacement.Pos()=" << seedPlacement.Pos() << " K=" << K << " unipathLengthsInKmers[seed]=" << unipathLengthsInKmers[seed] <<
    //    " seedLen=" << seedLen << endl;
    ForceAssert( seedPlacement.Pos() - seedPlacement.pos() /* + 1 */ == seedLen );
    genome_part_id_t genomePartId = seedPlacement.GenomeId();
    const KmerPath& genomePartPath = (seedPlacement.Orient() == ORIENT_FW ? genome_paths : genome_paths_rc)[genomePartId];
    ForceAssert( genomePartPath.GapFree() );
    nbases_t genomePartLen = genomePartPath.KmerCount() + K - 1;
    ForceAssert( 0 <= seedPlacement.pos()  &&  seedPlacement.pos() <= seedPlacement.Pos() &&
		 seedPlacement.Pos() < genomePartLen );
    genome_part_pos_t seedStartOnGenomePart = seedPlacement.pos();
    genome_part_pos_t seedEndOnGenomePart = seedPlacement.Pos();
    genome_part_pos_t neighborhoodStartOnGenomePart = Max( seedStartOnGenomePart - NHOOD_RADIUS_INTERNAL, 0 );
    genome_part_pos_t neighborhoodEndOnGenomePart = Min( seedEndOnGenomePart + NHOOD_RADIUS_INTERNAL, genomePartLen-1 );
    
    cout << " neighbStart=" << neighborhoodStartOnGenomePart <<
      " neighbEnd=" << neighborhoodEndOnGenomePart << endl;

    cout << "genomePartId=" << genomePartId << " genome_paths.size=" << genome_paths.size() << endl;
    KmerPathLoc neighborhoodFirstKmerLoc =
      genome_paths_idx.FindLoc( genomePartId, neighborhoodStartOnGenomePart, seedPlacement.Orient()  );
    ForceAssert( neighborhoodFirstKmerLoc.GetPathPtr() != NULL );
    ForceAssert( neighborhoodFirstKmerLoc.GetPathPtr()->NSegments() == genomePartPath.NSegments() );
    KmerPathLoc neighborhoodLastKmerLoc =
      genome_paths_idx.FindLoc( genomePartId, neighborhoodEndOnGenomePart - K + 1, seedPlacement.Orient()  );
    ForceAssert( neighborhoodLastKmerLoc.GetPathPtr() != NULL );
    ForceAssert( neighborhoodLastKmerLoc.GetPathPtr()->NSegments() == genomePartPath.NSegments() );
    
    GetSubpathKmerAdjacencies(// input
			      genomePartPath, neighborhoodFirstKmerLoc, neighborhoodLastKmerLoc,
			      // output
			      genomicKmerAdjs );
    
  }
  
}  // GetGenomicKmerAdjacencies()


				

/**
   Function: GetLocalizedReadsKmerAdjacencies

   For all reads <localized> to a given <neighborhood>, gather the set of <kmer adjacencies>
   represented by these reads.  This set should approximate the set returned by
   <GetGenomicKmerAdjacencies()>, and we can measure how well the approximation works.
   This in turn gives us an idea of how well localization of reads to neighborhoods is working.
*/
void GetLocalizedReadsKmerAdjacencies(// input
				      unipath_id_t neighborhoodSeed, nbases_t K,
				      const vecKmerPath& paths,
				      const vecKmerPath& paths_rc,
				      const vec< pair<read_id_t,orient_t> >& readsLocalizedToNeighborhood,
				      const vec<Bool> *readsToInclude,
			       
				      // output
				      KmerAdjSet& localizedReadsKmerAdjs ) {

  localizedReadsKmerAdjs.clear();
  for ( int i = 0; i < readsLocalizedToNeighborhood.isize(); i++ ) {
    const pair<read_id_t, orient_t>& readInfo = readsLocalizedToNeighborhood[i];
    read_id_t readId = readInfo.first;

    if ( !readsToInclude  ||  (*readsToInclude)[readId] ) {
      orient_t readOrientation = readInfo.second;

      const KmerPath& path = (readOrientation == ORIENT_FW ? paths : paths_rc)[readId];
      
      GetSubpathKmerAdjacencies(// input
				path, path.Begin(), path.End(),
				// output
				localizedReadsKmerAdjs );
    }
  }  // for each read localized to the neighborhood
  
}  // GetLocalizedReadsKmerAdjacencies()

/**
   Function: ComputeKmerAdjacencyPredictionStats

   For a given neighborhood, gather stats on how well the reads localized to that
   neighborhood predict the <kmer adjacencies> actually occurring in the neighborhood.

   The cell of <PredictionStats> showing how many kmer adjacencies are not in the genome
   and were correctly predicted not to be in the genome is left at zero, since we don't
   have the full list of all possible kmer adjacencies.  The remaining three cells are
   correctly filled in.
*/
void ComputeKmerAdjacencyPredictionStats( // inputs:
					  const KmerAdjSet& adjacenciesInLocalizedReads,
					  const KmerAdjSet& adjacenciesInGenomicNeighborhood,
					  // outputs:
					  PredictionStats& kmerAdjPredictionStats ) {

  cout << " comparing " << adjacenciesInLocalizedReads.size() << " predicted adjacencies to "
       << adjacenciesInGenomicNeighborhood.size() << " actual adjacencies." << endl;

  kmerAdjPredictionStats.Reset( "genomic kmer adjacency", "non-genomic kmer adjacency" );

  vec< pair< kmer_id_t, kmer_id_t > > predictedYes_actualYes;
  set_intersection( adjacenciesInLocalizedReads.begin(), adjacenciesInLocalizedReads.end(),
		    adjacenciesInGenomicNeighborhood.begin(), adjacenciesInGenomicNeighborhood.end(),
		    back_inserter( predictedYes_actualYes ) );
  
  vec< pair< kmer_id_t, kmer_id_t > > predictedYes_actualNo;
  set_difference( adjacenciesInLocalizedReads.begin(), adjacenciesInLocalizedReads.end(),
		  adjacenciesInGenomicNeighborhood.begin(), adjacenciesInGenomicNeighborhood.end(),
		  back_inserter( predictedYes_actualNo ) );
  
  vec< pair< kmer_id_t, kmer_id_t > > predictedNo_actualYes;
  set_difference(
		  adjacenciesInGenomicNeighborhood.begin(), adjacenciesInGenomicNeighborhood.end(),
		  adjacenciesInLocalizedReads.begin(), adjacenciesInLocalizedReads.end(),
		  back_inserter( predictedNo_actualYes ) );

  kmerAdjPredictionStats( PREDICTED_YES, ACTUAL_YES ) = predictedYes_actualYes.size();
  kmerAdjPredictionStats( PREDICTED_YES, ACTUAL_NO ) = predictedYes_actualNo.size();
  kmerAdjPredictionStats( PREDICTED_NO, ACTUAL_YES ) = predictedNo_actualYes.size();
  
  cout << " resulting prediction is: " << endl;
  kmerAdjPredictionStats.Print();
  
}  // ComputeKmerAdjacencyPredictionStats()

/**
   Function: GetSpecificityOfReadLocalization

   For simulated reads localized to a given neighborhood,
   get the percentage of reads that actually came from that
   neighborhood.  This measures the specificity (but not the
   sensitivity) of localization.

   Returns a pair of integers: first is the number of reads in the set
   that originate from the neighborhood, second is the total number
   of reads.
*/
void GetSpecificityOfReadLocalization(// input
				      unipath_id_t seed, nbases_t K, const vecKmerPath& genome_paths, const vecKmerPath& genome_paths_rc,
					       const vecvec<placement>& unipathPlacementsOnGenome,
					       const vec<nkmers_t>& unipathLengthsInKmers,
					       nbases_t NHOOD_RADIUS_INTERNAL,
					       const vec<read_location>& readlocs,
					       const vec<int>& readlocs_index,
					       const vec< pair<read_id_t,orient_t> >& readsLocalizedToNeighborhood,
				      const vec<Bool> *readsToInclude,
				      // output
				      int& readsActuallyFromNeighborhood,
				      int& totalReadsInSet
					       ) {

  //
  // Step: Get neighborhood boundaries
  //
  // For each placement of the seed unipath, get the boundaries of the neighborhood
  // at that placement.  There should be at most a few placements, since seeds are
  // low-copy-number.
  //

  int nplacements = unipathPlacementsOnGenome[seed].size();
  vec<genome_part_id_t> genomePartId(nplacements);
  vec<genome_part_pos_t> neighborhoodStartOnGenomePart(nplacements);
  vec<genome_part_pos_t> neighborhoodEndOnGenomePart(nplacements);
  for ( int seedPlacementNum = 0; seedPlacementNum < nplacements; seedPlacementNum++ ) {

    // Get the boundaries of the neighborhood on the reference:
    // start of seed minus NHOOD_RADIUS_INTERNAL,
    // through end of seed plus NHOOD_RADIUS_INTERNAL.
    const placement& seedPlacement = unipathPlacementsOnGenome[seed][seedPlacementNum];
    nbases_t seedLen = unipathLengthsInKmers[seed] + K - 1;
    //  cout << " seed=" << seed << " seedPlacement.pos()=" << seedPlacement.pos() <<
    //    " seedPlacement.Pos()=" << seedPlacement.Pos() << " K=" << K << " unipathLengthsInKmers[seed]=" << unipathLengthsInKmers[seed] <<
    //    " seedLen=" << seedLen << endl;
    ForceAssert( seedPlacement.Pos() - seedPlacement.pos() /* + 1 */ == seedLen );
    genomePartId[seedPlacementNum] = seedPlacement.GenomeId();
    const KmerPath& genomePartPath = (seedPlacement.Orient() == ORIENT_FW ? genome_paths : genome_paths_rc)[ genomePartId[seedPlacementNum] ];
    ForceAssert( genomePartPath.GapFree() );
    nbases_t genomePartLen = genomePartPath.KmerCount() + K - 1;
    ForceAssert( 0 <= seedPlacement.pos()  &&  seedPlacement.pos() <= seedPlacement.Pos() &&
		 seedPlacement.Pos() < genomePartLen );
    genome_part_pos_t seedStartOnGenomePart = seedPlacement.pos();
    genome_part_pos_t seedEndOnGenomePart = seedPlacement.Pos();
    neighborhoodStartOnGenomePart[seedPlacementNum] = Max( seedStartOnGenomePart - NHOOD_RADIUS_INTERNAL, 0 );
    neighborhoodEndOnGenomePart[seedPlacementNum] = Min( seedEndOnGenomePart + NHOOD_RADIUS_INTERNAL, genomePartLen-1 );
  }


  //
  // Step: Count reads that were actually from the neighborhood
  //
  // Take the true origin of each simulated read, and see if it comes from
  // within one of the possible locations of the neighborhood
  // (each formed around a possible placement of the seed).
  
  readsActuallyFromNeighborhood = 0;
  totalReadsInSet = 0;

  for ( int readNum = 0; readNum < readsLocalizedToNeighborhood.isize(); readNum++ ) {
    const pair<read_id_t, orient_t>& readInfo = readsLocalizedToNeighborhood[readNum];
    read_id_t readId = readInfo.first;
    if ( !readsToInclude  ||  (*readsToInclude)[readId] ) {
      totalReadsInSet++;

      const read_location& readLoc = readlocs[ readlocs_index[readId] ];
      ForceAssert( readLoc.ReadId() == readId );
      for ( int seedPlacementNum = 0; seedPlacementNum < nplacements; seedPlacementNum++ ) {
	if ( readLoc.Contig() == genomePartId[seedPlacementNum] &&
	     neighborhoodStartOnGenomePart[seedPlacementNum] <= readLoc.StartOnContig()  &&
	     readLoc.StopOnContig() <= neighborhoodEndOnGenomePart[seedPlacementNum] ) {
	  readsActuallyFromNeighborhood++;
	  break;
	}  // if the read is from the neighborhood
      }  // for each seed placement -- for each possible location of neighborhood on genome
    }  // if this read is part of the read set whose specificity-of-localization we're evaluating
  }  // for each read

}  // GetSpecificityOfReadLocalization()

					  
/**
   Function: EvaluateKmerAdjPredictionForNeighborhood

   Print stats on how accurately we localize reads to neighborhoods, by seeing how well the
   set of kmer adjacencies in reads localized to each neighborhood matches up with the
   true set of kmer adjacencies in that neighborhood (from the <reference>).

*/
void EvaluateKmerAdjPredictionForNeighborhood( // input
					      unipath_id_t v,  // neighborhood seed
					      unsigned int seeds_processed,
					      nbases_t K,
					      const vecKmerPath& genome_paths, const vecKmerPath& genome_paths_rc,				      
					      const vecKmerPathIndex& genome_paths_idx,
					      const vecKmerPath& paths,
					      const vecKmerPath& paths_rc,
					      const vecvec<placement>& unipathPlacementsOnGenome,
					      const vec<nkmers_t>& unipathLengthsInKmers,
					      nbases_t NHOOD_RADIUS_INTERNAL,
					      const vec<int>& predicted_copyno,
					      const String& run_dir,
					      const vec<Bool>& is_short_pair_read,
					      Bool USING_READLOCS,
					      const vec<read_location>& readlocs,
					      const vec<int>& readlocs_index,
					      const vec< pair<read_id_t,orient_t> >& primaryReadCloud,
					      const vec< pair<read_id_t,orient_t> >& secondaryReadCloud ) {

  const int NRINGS = 10;
  const int RING_SIZE_PERC = 5;
  
  KmerAdjSet actualAdjs;
  cout << "evaluating kmer adjacencies for seed " << v << " actualCopyNum=" << unipathPlacementsOnGenome[v].size() <<
    " predictedCopyNum=" << predicted_copyno[v] << 
    " primaryReadCloud.size=" << primaryReadCloud.size() <<
    " secondaryReadCloud.size=" << secondaryReadCloud.size() << endl;
  GetGenomicKmerAdjacencies( v, K, genome_paths, genome_paths_rc, genome_paths_idx,
			     unipathPlacementsOnGenome, unipathLengthsInKmers, NHOOD_RADIUS_INTERNAL,
			     actualAdjs );
  cout << "got " << actualAdjs.size() << " actual adjacencies." << endl;

  vec<KmerAdjSet> actualAdjsInclRings(NRINGS);
  for ( int ring = 1; ring <= NRINGS; ring++ ) {
    GetGenomicKmerAdjacencies( v, K, genome_paths, genome_paths_rc, genome_paths_idx,
			       unipathPlacementsOnGenome, unipathLengthsInKmers,
			       nbases_t( double(NHOOD_RADIUS_INTERNAL) * (1.0 + ( double(RING_SIZE_PERC) / 100.0 ) * double(ring)) )  ,
			       actualAdjsInclRings[ring-1] );
  }

  String predStatsFileName = run_dir + "/adjpredict.dat." + ToString(getpid());
  ofstream predStats( predStatsFileName.c_str(), ios_base::out | ios_base::app );
  static Bool wrotePredictionStatsHeader = False;
  
  if ( !wrotePredictionStatsHeader ) {
    predStats << "\" " << Date() << " kmer adjacencency prediction stats: NRINGS=" << NRINGS << " RING_SIZE_PERC=" <<
      RING_SIZE_PERC << "% \"" << endl;
    
    wrotePredictionStatsHeader = True;
  }

  // Write out the record header for this neighborhood: its size and number of placements on the genome.
  predStats << seeds_processed << ", " << v << ", " << (unipathLengthsInKmers[v] + K - 1) << ", " << unipathPlacementsOnGenome[v].size()
	    << ", " << predicted_copyno[v];

  //
  // We do three iterations:
  //    iteration 0 - gather stats for the primary read cloud
  //    iteration 1 - gather stats for just the short reads in the primary read cloud
  //    iteration 2 - gather stats for the secondary read cloud.
  //
  for ( int whichReadSet = 0; whichReadSet < 3; whichReadSet++ ) {
    KmerAdjSet predictedAdjs;
    GetLocalizedReadsKmerAdjacencies( v, K, paths, paths_rc,
				      whichReadSet < 2 ? primaryReadCloud : secondaryReadCloud,
				      whichReadSet == 1 ? &is_short_pair_read : NULL,
				      predictedAdjs );
    cout << "got " << predictedAdjs.size() << " predicted adjacencies." << endl;
    PredictionStats kmerAdjPredictStats;
    ComputeKmerAdjacencyPredictionStats( predictedAdjs, actualAdjs, kmerAdjPredictStats );
    cout << " computed prediction stats." << endl;
    
    predStats << ", " << v << ", "
	      << kmerAdjPredictStats(PREDICTED_YES, ACTUAL_YES) << ", "
	      << kmerAdjPredictStats(PREDICTED_NO, ACTUAL_YES) << ", "
	      << kmerAdjPredictStats(PREDICTED_YES, ACTUAL_NO) << "  ";
    
    
    for ( int ring = 1; ring <= NRINGS; ring++ ) {
      vec< pair< kmer_id_t, kmer_id_t > > predictedYes_actualNo;
      set_difference( predictedAdjs.begin(), predictedAdjs.end(),
		      actualAdjsInclRings[ring-1].begin(), actualAdjsInclRings[ring-1].end(),
		      back_inserter( predictedYes_actualNo ) );
      
      predStats << ", " << predictedYes_actualNo.size() << " ";
    }
    
  }  // for each read set

  //
  // Step: Write specificity of read localization
  //
  // For each read set, write out the size of the set, and the number of reads in the set
  // that actually did come from the neighborhood.
  if ( USING_READLOCS ) {
    for ( int whichReadSet = 0; whichReadSet < 3; whichReadSet++ ) {
      predStats << ", -1 ";
      int readsActuallyFromNeighborhood = 0;
      int totalReadsInSet = 0;
      for (int ring = 0; ring <= NRINGS; ring++ ) {
	GetSpecificityOfReadLocalization( v, K, genome_paths, genome_paths_rc, unipathPlacementsOnGenome,
					  unipathLengthsInKmers,
					  nbases_t( double(NHOOD_RADIUS_INTERNAL) * (1.0 + ( double(RING_SIZE_PERC) / 100.0 ) * double(ring)) ) ,
					  readlocs, readlocs_index,
					  whichReadSet < 2 ? primaryReadCloud : secondaryReadCloud,
					  whichReadSet == 1 ? &is_short_pair_read : NULL,
					  readsActuallyFromNeighborhood,
					  totalReadsInSet );
	if ( ring == 0 )
	  predStats << ", " << totalReadsInSet;
	predStats << ", " << readsActuallyFromNeighborhood;
      }
      
    }
  } else
    cout << "NOT USING READLOCS" << endl;
  
  
  predStats << endl;
  
  
  cout << "wrote kmer adj data for seed " << v << endl;
  
}  // EvaluateKmerAdjPredictionForNeighborhood()

			       

