/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/**
   Program: LocalizeReads
  
   This is a main assembly module.

   On a high level, it does the following things:

   1. Localize reads to "neighborhoods".  We pick "seeds" of unique sequence spread along the
   genome, and for each seed identify a set ("cloud") of reads that likely came from the
   vicinity of the seed.  We pick seeds so that each neighborhood overlaps with some surrounding
   neighborhoods (so that we can later merge the reconstructions of overlapping neighborhoods
   into reconstructions of longer genome stretches).

      - identify clusters of <CN1 unipaths> lying near each other
      - for each such cluster, identify <paired reads> with at least one read
        landing on a unipath in the cluster.  This 'localizes' the reads,
	i.e. gathers together all reads from the unique region of the genome
	from which the CN1 unipaths come.  Each cluster of CN1 unipaths together
	with reads localized to that cluster is one <neighborhood>.


   2. Assemble each neighborhood into a <HyperKmerPath> representing all possible
   sequences of the neighborhood that are consistent with the reads.

       2.1.   Assemble short contigs, and contig pairs (with known gap size),
        from overlapping short insert pairs.

       2.2.   For each long-insert pair localized to the neighborhood,
       construct a HyperKmerPath representing what's in the middle.

           Use short contigs and contig pairs constructed in step 2.1
	   to walk from one end of a long-insert pair to the other.
	   If multiple walks (linearizations) are possible, represent that
	   as ambiguities in the constructed HyperKmerPath.


       2.3.  Merge overlapping HyperKmerPaths built from long-insert pairs,
       into a single HyperKmerPath for the neighborhood.
       
   
       - construct local read unipaths from reads in the neighborhood, make them
       into a graph that represents the possible paths through the neighborhood
       (each path representing one possibility for what the actual genomic neighborhood
       looks like)
       - merge the local unipath graph from the neighborhoods together into one global
       graph whose connected components describe the possible sequences of various
       genome regions -- ideally, one connected component per <genome part>, but in practice
       there are several.


   3. Merge HyperKmerPaths reconstructed from overlapping neighborhoods into HyperKmerPaths covering
   larger contiguous regions of the genome.

       Which neighborhood reconstructions to merge, is determined based on sequence overlap.

   The above description omits many, many important details.

   Input to this program is set up by <RunAssemblyThroughUnipaths>.

   See also the following auxiliary modules:
  
   EvalHyper - evaluate HyperKmerPath, as done at tail end of LocalizeReads;
   CleanHyper - clean and evaluate HyperKmerPath, as at tail end of LocalizeReads;
   LocalizeMerge - merge and evaluate output of parallel LocalizeReads runs.
  
   Basic syntax:
   DATA, RUN, SUBDIR  - required arguments
   USE_TRUTH=True     - set this for simulated data unless you want to "play dumb"
   PARALLEL_BATCHES=n - if you want LocalizeReads to parallelize its main part
                   = {box1,box2,...,boxn} - to parallelize on multiple machines.
                   = {LSF}*n - to parallelize on up to n blades on the LSF
                   = {LSF*n,box2*m} - to parallelize on n blades and m machines.
                                            (or other ParseSet syntax)
   LSF_HOST        = name of LSF host machine (default is lead)
   LSF_ARGS        = bsub args used when submitting LSF jobs
                     (for example: LSF_ARGS="-q priority")
  
   What this module does:
  
   1. Define a subset of the unipaths, called the "normal" ones, which are those
   whose length is at least MIN_KMERS and whose predicted copy number is at most
   MAX_COPY_NUMBER.
   2. Form the digraph whose vertices are the normal unipaths and which has an
   edge from x to y and an edge from rc(y) to rc(x) if there is a read pair whose 
   first read overlaps x and whose second read overlaps rc(y), where x != y.
   The edge is marked to show the expected separation and deviation in kmers 
   between x and y.
   3. For each vertex v, build a "neighborhood" of it, consisting of those vertices
   w which are connected to v, subject to the following limitations:
   (a) predicted location of w must be within NHOOD_RADIUS of v;
   (b) number of such vertices w must not exceed MAX_PATHS_IN_NHOOD;
   (c) deviation of composite edge must not exceed MAX_DEV.
   This neighborhood is not entirely well-defined, as it depends on the order of
   operations.
   4. Find the reads S which appear to be in the neighborhood.
   5. Find the short-insert reads P which align fully to reads in S.
   6. Walk these short inserts.
   7. Choose a handful of long-inserts from the neighborhood, and walk their inserts
   using the completions of the short inserts as reads.
  
   You need to have first run Unipather, UnipathCoverage, and UnipathLocs.  This 
   code does not work if the read paths have gaps.
  
   Known defects:
   - In order to accommodate local duplications, a neighborhood should be allowed to
     have a given unipath in more than one location.  To facilitate this, the graph
     itself would need to have multiple edges between two given vertices.
   - In building a neighborhood, we should always start with the unprocessed member
     having the smallest deviation.
  
   Arguments which control which unipaths are used as seeds:
   - SEEDS: this has one of two formats:
           - default - attempt to pick seeds intelligently;
           - a list of integers in ParseIntSet format.
   - MAX_SEEDS (default = -1): if set, terminate after this many seeds
   - MAX_COPY_NUMBER (default = 2): if estimated copy number exceeds this, exclude
   - MIN_KMERS_IN_SEED (default = 100): don't use seeds smaller than this.
   - SEED_POST_FILTER: intersect with this set after everything else.
  
   Arguments that control how the unipath neighborhood is built:
   - MAX_COPY_NUMBER_OTHER (default = 0): if set, allows more unipaths to be used
   - MIN_KMERS (default = 25): don't use unipaths smaller than this
   - MAX_PATHS_IN_NHOOD (default = 400): after this many, stop building
   - BUILD_NHOOD_FW_ONLY (default = False): only walk in forward direction
   - FILTER_NHOOD (default = True): to remove unipaths we think don't belong.
  
   Arguments as in MuxSearchInsert that get passed to all instances of mux searching:
   - search_limit (default = 200)
   - answer_size_limit (default = 200)
   - MAX_PSEUDO (default = 200) [also MAX_PSEUDO_LONG, for long inserts]
   - MAX_CLOSURES (default = 1500)
   - SD_MULT (default = 3).
  
   Arguments which control verbosity of mux searching:
   - verbosity (default = 1)
   - SHORT_INSERT_WALK_DETAILS (default = False)
   - decompose_truth (default = False)
   - decompose_truth_brief (default = False).
  
   Arguments which control which long-insert pairs are walked:
   - PAIRS_SAMPLE (default = random:25): this has one of two formats:
          - a list of read ids in ParseIntSet format;
          - random:n, where n is a positive integer.
   - MIN_LONG_INSERT (default = 800).
  
   Files outputted as "assembly":
     - closures.bases, that is a vec<HyperBasevector> (intermediate)
     - hyper.initial (intermediate)
     - hyper, that is a HyperKmerPath representing the assembly graph
     - hyper.aligns, alignments of the HyperKmerPath edges to the genome
     - dot, that is a representation of the assembly graph in DOT format.
  
   Some other options:
  
   USE_TRUTH                - Use reference sequence to evaluate performance.
  
   SHOW_LONG_READS          - Print the ids of the long insert reads chosen to walk.
  
   MIN_OVERLAP              - Minimum overlap for merger by InternalMerge.
   MIN_OVERLAP_FINAL        - Same, but applies to final merge.
  
   MIN_PROPER_OVERLAP       - Minimum proper overlap for merger by InternalMerge.
   MIN_PROPER_OVERLAP_FINAL - Same, but applies to final merge.
  
   DUMP_CLOUD               - Print ids and orientations of reads in neighborhood,
                              to the specified file.
  
   INSTANCE                 - Appended to SUBDIR.
  
   RANDOM_SEED              - Initialize random number generator with this seed.
  
   UNIPATH_SEED_STATS       - Give information about seeds that were not selected.
  
   K_ADD_OUT                - Add this value to K for certain purposes.
  
   TEST_GRAPH_EDGES         - Determine which initial local pairs are invalid in
                              the sense that their sequence representation has no
                              perfect placement on the genome.
   TRACE_GRAPH_EDGES        - For each short-insert "pair edge", find the true 
                              locations of real pairs that could give rise to it.
  
   BASIC_DEBUG              - Turn on more verbose output.

   SPECIAL_WALK="g.a1-b1,a2-b2:d%"
   This causes a fake pair to be created from truth data and walked.  The pair is 
   created by taking genome contig g and making bases a1-b1 into the left read,
   bases a2-b2 into the right read, and assigning the pair separation a2-b1 +/- d%.
   The base ranges are expanded if needed so they comprise complete sequences of
   local unipaths.  Special experimental code is used to walk the pair.  Then exit.
  
   SPECIAL_WALK_ORIG=True: Use the original pairs (pairlocs/0) to walk the 
   pairs in SPECIAL_WALK, not the condensed pairs.

   Term: seed

   A <normal unipath> around which we build a neighborhood.

   Term: nhood unipath

   A unipath from the neighborhood we're talking about.

   Term: primary read cloud

   Set of read pairs where at least one read of the pair is estimated to lie within
   the <inner radius> of the <nhood seed>.

   Term: secondary read cloud

   Set of read pairs where both reads of each pair contain only kmers from reads in the
   primary read cloud.

   Synonyms: LocalizeReads synonyms

      nhood - See <neighborhood>
      nhood seed, neighborhood seed - See <seed>

   Aspect: LocalizeReads parallelization

   Once the <neighborhood seeds> have been identified, the work of building the neighborhoods can be
   parallelized.  The master copy of LocalizeReads makes a list of all the neighborhood seeds
   (work units), then starts up subordinate copies of LocalizeReads on various processors of the current machine
   and/or on other machines via ssh.  Each subordinate copy takes "the next available" seed from the list of
   seeds and works on it.  The master copy waits for all subordinate copies to finish, then assembles
   their results.
*/

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

// MakeDepend: CC_OPT1

#include <strstream>
#include <fstream>
#include <map>
#include <sys/wait.h>

#include "Alignment.h"
#include "Basevector.h"
#include "Bitvector.h"
#include "Equiv.h"
#include "FastIfstream.h"
#include "Feudal.h"
#include "FeudalMimic.h"
#include "MainTools.h"
#include "PackAlign.h"
#include "ParseSet.h"
#include "ReadLocation.h"
#include "ReadPairing.h"
#include "Set.h"
#include "TaskTimer.h"
#include "graph/Digraph.h"
#include "lookup/LookAlign.h"
#include "math/Functions.h"
#include "math/HoInterval.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "paths/PdfEntry.h"
#include "paths/AlignHyperKmerPath.h"
#include "paths/BridgeGaps.h"
#include "paths/CompletedInsert.h"
#include "paths/EvalUtils.h"
#include "paths/ExtendUnipathSeqs.h"
#include "paths/FindClosures.h"
#include "paths/Hospital.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/KmerPath.h"
#include "paths/KmerPathMuxSearcher.h"
#include "paths/KmerPathOnHyper.h"
#include "paths/LocalizeReadsAnnex.h"
#include "paths/LocalizeReadsAnnex2.h"
#include "paths/LocalizeReadsTail.h"
#include "paths/MergeTwoGreedy.h"
#include "paths/MuxGraph.h"
#include "paths/MuxSearchPolicy.h"
#include "paths/MuxSetup.h"
#include "paths/PairedPair.h"
#include "paths/PairedWalk.h"
#include "paths/PairGraph.h"
#include "paths/ReadFillDatabase.h"
#include "paths/ReadsToPathsCore.h"
#include "paths/ReadsToPathsCoreX.h"
#include "paths/SharedCounter.h"
#include "paths/SimpleLoop.h"
#include "paths/SubMuxGraphBuilder.h"
#include "paths/Unipath.h"
#include "paths/UnipathNhood.h"
#include "paths/UnipathSeeds.h"
#include "paths/UnipathSeqBuilder.h"
#include "paths/simulation/Placement.h"
#include "paths/simulation/SimTrueSequenceBroker.h"
#include "random/Random.h"

#define TEST_EXIT_AT(tag)                                                         \
{    if ( EXIT_AT == #tag )                                                       \
     {    cout << "\nTerminating at " << #tag << " by request.\n";                \
          cout << "\nTime for entire run: " << TimeSince(all_clock) << endl;      \
          if ( PARALLEL_CONTROL != "" ) {                                         \
          Ofstream( slave_finished, sub_dir + "/SLAVE_FINISHED"); }               \
          exit(0);    }    }

#define PRINT_PAIR_LOCS(message)                                                  \
{    if ( SHOW_PAIR_PLACEMENTS >= 1 )                                             \
     {    PrintPairLocs( True, "in LocalizeReads, " message, sub_dir, ppp, h,     \
               kbbnb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL,                \
               locs[v] );    }    }

#define CHANGE_TIMER( OLD_TIMER, ACTIVITY, NEW_TIMER )                    \
     cout << "\n" << TimeSince(OLD_TIMER) << " " << ACTIVITY << endl;     \
     double NEW_TIMER = WallClockTime( );

int main( int argc, char *argv[] )
{
  
     RunTime( );

     BeginCommandArguments;
     CommandArgument_String(PRE);
     CommandArgument_String(DATA);
     CommandArgument_String(RUN);
     CommandArgument_Int_OrDefault_Doc(K, 0, "kmer size used for read paths and unipaths");
     CommandArgument_IntSet_OrDefault_Doc(PATH_KS, "", "additional kmer sizes for which read paths have been created");
     CommandArgument_Int_OrDefault(MAX_COPY_NUMBER, 2);
     CommandArgument_Int_OrDefault(MAX_COPY_NUMBER_OTHER, 0);
     CommandArgument_Int_OrDefault(MIN_KMERS, 25);
     CommandArgument_Int_OrDefault(MIN_KMERS_IN_SEED, 100);
     CommandArgument_Bool_OrDefault(USE_TRUTH, False);
     CommandArgument_Int_OrDefault(NHOOD_RADIUS, 20000);
     CommandArgument_Int_OrDefault(NHOOD_RADIUS_INTERNAL, 10000);
     CommandArgument_Int_OrDefault(MAX_PATHS_IN_NHOOD, 400);
     CommandArgument_Int_OrDefault(MAX_TO_CUT, 4000);
     CommandArgument_Int_OrDefault(MAX_DEV, 2000);
     CommandArgument_Int_OrDefault(MAX_SHORT_INSERT_SEP, 500);
     CommandArgument_Bool_OrDefault(DUMP_GRAPH, False);
     CommandArgument_String_OrDefault(PAIRS_SAMPLE, "random:25");
     CommandArgument_String_OrDefault(SEEDS, "");
     CommandArgument_Bool_OrDefault(FW_SEEDS_ONLY, USE_TRUTH);
     CommandArgument_Bool_OrDefault(BUILD_NHOOD_FW_ONLY, False);
     CommandArgument_Bool_OrDefault(REACH_FW_ONLY, False);
     CommandArgument_Bool_OrDefault(FILTER_NHOOD, True);
     CommandArgument_Int_OrDefault(MIN_LONG_INSERT, 800);
     CommandArgument_Int_OrDefault(MIN_LONG_INSERT_SHORTS, 800);
     CommandArgument_Bool_OrDefault(SHORT_INSERT_WALK_DETAILS, False);
     CommandArgument_Bool_OrDefault(FIND_SHORT_INSERTS, True);
     CommandArgument_Bool_OrDefault(USE_TRUE_READS_SHORT, False);
     CommandArgument_Int_OrDefault(MAX_SHORT_INSERTS_TO_WALK, 0);
     CommandArgument_Int_OrDefault(MAX_PSEUDO, 200);
     CommandArgument_Int_OrDefault(MAX_PSEUDO_LONG, 50000);
     CommandArgument_Int_OrDefault(MAX_CLOSURES, 10000);
     CommandArgument_String_OrDefault(PREFIX, "pathshq");
     CommandArgument_UnsignedInt_OrDefault(verbosity, 1);
     CommandArgument_Double_OrDefault(search_limit, 200);
     CommandArgument_Double_OrDefault(answer_size_limit, 200.0);
     CommandArgument_UnsignedInt_OrDefault(SD_MULT, 3);
     CommandArgument_String_OrDefault(PATHSHQ, "pathshq");
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_PRE, 2);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_MAX_EXT, 0);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_VERBOSITY, 0);
     CommandArgument_Bool_OrDefault(decompose_truth, False);
     CommandArgument_Bool_OrDefault(decompose_truth_brief, False);
     CommandArgument_String(SUBDIR);
     CommandArgument_Bool_OrDefault(LONG_PRINT_FASTA, False);
     CommandArgument_Int_OrDefault(MAX_SEEDS, -1);
     CommandArgument_Int_OrDefault(MIN_LOCAL_UNIPATH_TO_USE, 200);
     CommandArgument_Bool_OrDefault(SHOW_LONG_READS, False);
     CommandArgument_Int_OrDefault(SIMULATION_TRIES, 20);
     CommandArgument_String_OrDefault(DUMP_CLOUD, "");
     CommandArgument_Bool_OrDefault(MERGE_BY_INSERT_FIRST, True);
     CommandArgument_Int_OrDefault(MIN_OVERLAP, 3500);
     CommandArgument_Int_OrDefault(MIN_PROPER_OVERLAP, 700);
     CommandArgument_Int_OrDefault(MIN_OVERLAP_FINAL, 3500);
     CommandArgument_Int_OrDefault(MIN_PROPER_OVERLAP_FINAL, 400);
     CommandArgument_Bool_OrDefault(PRINT_EASY_GRAPH, False);
     CommandArgument_String_OrDefault(INSTANCE, "");
     CommandArgument_String_OrDefault(SEED_POST_FILTER, "");
     CommandArgument_UnsignedInt_OrDefault(RANDOM_SEED, 0);
     CommandArgument_Bool_OrDefault(UNUSED_SEED_STATS, False);
     CommandArgument_Bool_OrDefault(PRINT_TRUE_COVERAGE_GAPS, False);
     CommandArgument_Bool_OrDefault(EXTEND_PAIRS, False);
     CommandArgument_Bool_OrDefault(EXTRA_LONGS, True);
     CommandArgument_Int_OrDefault(MAX_SEED_DIST, 4000);
     CommandArgument_Bool_OrDefault(GLOBAL_CLEAN, SEEDS.empty( ));
     CommandArgument_Int_OrDefault(K_ADD_IN, 0);
     CommandArgument_Int_OrDefault(K_ADD_OUT, 0);
     CommandArgument_Bool_OrDefault(SHOW_PAIR_ALIGNS, False);
     CommandArgument_Bool_OrDefault(FINAL_MERGE, True);
     CommandArgument_Bool_OrDefault(FIRST_MERGE, True);
     CommandArgument_Bool_OrDefault(SECOND_MERGE, True);
     CommandArgument_Bool_OrDefault(FIRST_DELOOP, False);
     CommandArgument_Bool_OrDefault(LONG_INSERT_CHOICE_VERBOSE, False);
     CommandArgument_Double_OrDefault(DMULT, 3.0);
     CommandArgument_Double_OrDefault(DMULT_ADD0, 0.0);
     CommandArgument_Bool_OrDefault(FILTER_ALIGNS, True);
     CommandArgument_Double_OrDefault(TRIM_MULTIPLIER, 1.0);
     CommandArgument_Bool_OrDefault(TEST_GRAPH_EDGES, False);
     CommandArgument_Bool_OrDefault(CHEAT_BY_DELETING_INVALID_EDGES, False);
     CommandArgument_Bool_OrDefault(DELETE_HANGING_ENDS, True);
     CommandArgument_Bool_OrDefault(DELETE_POORLY_COVERED_PAIRS, False);
     CommandArgument_Bool_OrDefault(PRECOMPUTE_CLOSURE_LENGTHS, False);
     CommandArgument_Int_OrDefault(EDGE_MIN, 5);
     CommandArgument_Double_OrDefault(BIG_NHOOD_MULTIPLIER, 1.0);
     CommandArgument_Bool_OrDefault(TRACE_GRAPH_EDGES, False);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_MAX_PROCESSED, 1000);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_MAX_UNPROCESSED, 1000);
     CommandArgument_Int_OrDefault(FINDCLOSURES_MAX_PSEUDO_CLOSURES, 100);
     CommandArgument_Int_OrDefault(FINDCLOSURES_MAX_CLOSURES, 100);
     CommandArgument_Bool_OrDefault(ACCEPT_INCOMPLETE_CLOSURE_SETS, True);
     CommandArgument_Bool_OrDefault(DUMP_TRUE_SEED_LOCS, False);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_ROUND2_MAX, 500);
     CommandArgument_Int_OrDefault(PAIRED_PAIRS_MAX_TO_CLOSE_NONUNIQUE, 700);
     CommandArgument_Bool_OrDefault(MERGE_PAIRED_PAIRS_VERBOSE, False);
     CommandArgument_Bool_OrDefault(POORLY_COVERED_PAIRS_VERBOSE, False);
     CommandArgument_Bool_OrDefault(BITS_VERBOSE, False);
     CommandArgument_Bool_OrDefault(REDUCE_VERBOSE, False);
     CommandArgument_Int_OrDefault(MAX_OPENS, 1000);
     CommandArgument_Int_OrDefault(MAX_OPENS0, 1000);
     CommandArgument_Int_OrDefault(MAX_OPENS1, 1000);
     CommandArgument_Bool_OrDefault(TRY_HARD, False);
     CommandArgument_Int_OrDefault(MAX_OPENS1_TRY_HARD, 5000);
     CommandArgument_Int_OrDefault(MAX_OPENS_EASY, 1000);
     CommandArgument_Bool_OrDefault(PRINT_HYPER_BEFORE_FIRST_MERGE, False);
     CommandArgument_Bool_OrDefault(PRINT_HYPER_AFTER_FIRST_MERGE, False);
     CommandArgument_Bool_OrDefault(PRINT_HYPER_BEFORE_SECOND_MERGE, False);
     CommandArgument_Bool_OrDefault(PRINT_HYPER_AFTER_SECOND_MERGE, False);
     CommandArgument_Bool_OrDefault(FIRST_DELOOP_VERBOSE, False);
     CommandArgument_Bool_OrDefault(PRINT_HYPER_AFTER_FIRST_DELOOP, False);
     CommandArgument_Bool_OrDefault(BASIC_DEBUG, False);
     CommandArgument_Bool_OrDefault(PRINT_PRIMARY_CLOUD, False);
     CommandArgument_Bool_OrDefault(PRINT_SECONDARY_CLOUD, False);
     CommandArgument_Bool_OrDefault_Doc(EVAL_KMER_ADJS, False, "only measure quality of localization, then quit");
     CommandArgument_Bool_OrDefault(EVALUATE_NHOOD_HYPER, False);
     CommandArgument_Bool_OrDefault(BUILD_GRAPH_VERBOSE, False);
     CommandArgument_Bool_OrDefault(TRANSITIVE_FILL_IN, True);
     CommandArgument_Int_OrDefault(TRANSITIVE_FILL_IN_VERBOSITY, 0);
     CommandArgument_String_OrDefault(PARALLEL_BATCHES, "");
     CommandArgument_String_OrDefault(PARALLEL_CONTROL, "");
     CommandArgument_String_OrDefault(LSF_HOST, "lead");
     CommandArgument_String_OrDefault(LSF_ARGS, "");
     CommandArgument_Int_OrDefault(MIN_COMPONENT, 5000);
     CommandArgument_Int_OrDefault(MIN_EDGE_MULTIPLICITY, 3);
     CommandArgument_Int_OrDefault(SEED_VERBOSITY, 0);
     CommandArgument_Bool_OrDefault(FORCE, False);
     CommandArgument_String_OrDefault(EXIT_AT, "");
     CommandArgument_Bool_OrDefault(MERGE_COPY_NUMBER_TWO, False);
     CommandArgument_Bool_OrDefault(GENERATE_PAIR_GRAPH, False);
     CommandArgument_Bool_OrDefault(BRIDGE_VERBOSE, False);
     CommandArgument_Int_OrDefault(SHOW_PAIR_PLACEMENTS, 0);
     CommandArgument_Bool_OrDefault(REMOVE_SOLO, True);
     CommandArgument_Bool_OrDefault(COUT, True);
     CommandArgument_Int_OrDefault(SIMPLE_WALK_VERBOSITY, 0);
     CommandArgument_Int_OrDefault(EASY_CLOSE_VERBOSITY, 0);
     CommandArgument_Bool_OrDefault(MERGE_COPY_NUMBER_TWO_VERBOSE, False);
     CommandArgument_Bool_OrDefault(JUST_PLOT_LOCAL_KMER_FREQS, False);
     CommandArgument_Int_OrDefault(MIN_COPY_NUMBER_TO_CLOSE, 0);
     CommandArgument_Int_OrDefault(MAX_NODES, 0);
     CommandArgument_Int_OrDefault(MAX_NODES_TRY_HARD, 20000);
     CommandArgument_Bool_OrDefault(DIPLOID, False);
     CommandArgument_Bool_OrDefault(DISAMBIGUATE_SIMPLE_LOOPS_VERBOSE, False);
     CommandArgument_Bool_OrDefault(PULL_VERBOSE, False);
     CommandArgument_Bool_OrDefault(APPEND_LONG_PAIRS, False);
     CommandArgument_Int_OrDefault(LONG_INSERT_WALK_VERBOSITY, 0);
     CommandArgument_Int_OrDefault(LONG_INSERT_WALK_K, 96);
     CommandArgument_Int_OrDefault(LONG_INSERT_MIN_READ_LENGTH, 96);
     CommandArgument_Bool_OrDefault(EVALUATE_INSERT_HYPER, False);
     CommandArgument_Bool_OrDefault(TRACK_GLOBAL_CLEAN_CHANGES, False);
     CommandArgument_Bool_OrDefault(TREAT_SHORT_AS_LONG, False);
     CommandArgument_Bool_OrDefault(SHORTEST_MERGE, False);
     CommandArgument_Bool_OrDefault(ONELINE_UNIPATH_OUTPUT, False);
     CommandArgument_Bool_OrDefault(EXTEND_CLOSURES, False);

     CommandArgument_Bool_OrDefault(WALK_OFF_CLOSURES, False);
     CommandArgument_Int_OrDefault(MAX_WALK_OFF, 1000);
     CommandArgument_Int_OrDefault(MAX_WALK_OFF_OPENS, 1000);
     CommandArgument_Int_OrDefault(MAX_WALK_OFF_NODES, 5000);
     CommandArgument_Int_OrDefault(WALK_OFF_VERBOSITY, 1);

     CommandArgument_String_OrDefault(SPECIAL_WALK, "");
     CommandArgument_Bool_OrDefault(SPECIAL_WALK_ORIG, False);

     CommandArgument_Bool_OrDefault(ADD_GLOBAL, False);
     CommandArgument_Bool_OrDefault(ADD_VALIDATED_SHORTS, False);
     CommandArgument_Bool_OrDefault(NEW_POORLY_COVERED, False);
     CommandArgument_Bool_OrDefault(NEW_UNIQUE_EXTEND, False);
     EndCommandArguments;

     // Timer for entire run. 

     double all_clock = WallClockTime( );

     // If slave process then get instance number from counter file

     int instance = 0;
     if ( PARALLEL_CONTROL != "" ) 
     {    String instance_file = PRE + "/" + DATA + "/" + RUN + "/ASSEMBLIES/" 
               + SUBDIR + "/current_instance";
          int pc_instance_fd = open( instance_file.c_str() , O_RDWR, 0664 );
          if ( pc_instance_fd < 0 ) FatalErr( "Could not open current_instance." );
          IncrementSharedCounter( pc_instance_fd, instance);
          INSTANCE = "/" + ToString(instance);    }

     // Set up directories.

     SUBDIR += INSTANCE;
     String data_dir = PRE + "/" + DATA;
     String run_dir = PRE + "/" + DATA + "/" + RUN;
     String sub_dir = run_dir + "/ASSEMBLIES/" + SUBDIR;
     String wdata_dir = sub_dir;
     String wrun_dir = sub_dir + "/run";
     if ( !IsDirectory(run_dir) )
     {    cout << "The run_dir (" << run_dir << ") does not exist.\n";
          cout << "Abort.\n";
          exit(1);    }
     Mkdir777( run_dir + "/ASSEMBLIES" ), Mkdir777(wdata_dir), Mkdir777(wrun_dir);

     // Capture standard output, put in LocalizeReads.out.

     String logfile = sub_dir + "/LocalizeReads.out";
     {    Ofstream( cout0, logfile );
          PrintCommandPretty(cout0);    }
     String pipe_command = ( COUT ? "tee -a " : "cat >> " ) + logfile;
     procbuf outpipe( pipe_command.c_str( ), ios::app );
     ostream pipeout( &outpipe );
     std::cout.rdbuf( pipeout.rdbuf( ) );
     cout << "Assembly subdirectory =\n" << RealPath(sub_dir) << "\n\n";

     // Figure out what K is.

     if ( K == 0 ) {
       vec<String> paths_files = AllFilesWithPrefix( "reads.paths.k", run_dir );
       vec<int> Kvalues;
       for ( int i = 0; i < paths_files.isize( ); i++ )
	 {    if ( paths_files[i].After( "reads.paths.k" ).IsInt( ) )
	   Kvalues.push_back( 
			     paths_files[i].After( "reads.paths.k" ).Int( ) );    }
       if ( Kvalues.solo( ) ) K = Kvalues[0];
     }
     if ( K == 0 ) FatalErr( "Can't figure out K value from reads.paths files." );

     // Load paths built using longer kmers, if given
     UniqueSort( PATH_KS );
     
     vec< vecKmerPath > extra_paths( PATH_KS.size() );
     vec< vecKmerPath > extra_paths_rc( PATH_KS.size() );
     vec< vec<tagged_rpint> > extra_paths_db( PATH_KS.size() );
     String reads_base = run_dir + "/reads";
     for ( int i = 0; i < PATH_KS.isize(); i++ ) {
       extra_paths[i].ReadAll( reads_base + ".paths.k" + ToString( PATH_KS[i] ) );
       extra_paths_rc[i].ReadAll( reads_base + ".paths_rc.k" + ToString( PATH_KS[i] ) );
       BinaryRead2( reads_base + ".pathsdb.k" + ToString( PATH_KS[i] ), extra_paths_db[i] );
     }
     
     // Don't allow overwrite by subset.

     String LAST_SEEDS;
     if ( IsRegularFile( sub_dir + "/SEEDS" ) )
     {    Ifstream( seedsin, sub_dir + "/SEEDS" );
          seedsin >> LAST_SEEDS;
          if ( !FORCE && LAST_SEEDS == "" && SEEDS != "" )
          {    cout << "I'm worried that you may be accidentally overwriting a "
                    << "full assembly with a\npartial one.  ";
               cout << "If you really want to do this, rerun with FORCE=True.\n";
               outpipe.close( );
               exit(1);    }    }
     {    Ofstream( seedsout, sub_dir + "/SEEDS" );
          seedsout << SEEDS << endl;    }

     // Parse SEEDS argument.

     vec<unipath_id_t> seeds;
     if ( SEEDS != "" ) ParseIntSet( SEEDS, seeds );

     if ( FW_SEEDS_ONLY && ! USE_TRUTH ) {
       cout << "The option FW_SEEDS_ONLY=True can only be used if USE_TRUTH=True." << endl;
       exit(1);
     }

     // Parse SPECIAL_WALK argument.

     int special_g, special_a1, special_b1, special_a2, special_b2;
     double special_d;
     if ( SPECIAL_WALK != "" )
     {    ForceAssert(USE_TRUTH);
          ForceAssert( seeds.solo( ) );
          special_g = SPECIAL_WALK.Before( "." ).Int( );
          special_a1 = SPECIAL_WALK.Between( ".", "-" ).Int( );
          special_b1 = SPECIAL_WALK.Between( "-", "," ).Int( );
          special_a2 = SPECIAL_WALK.Between( ",", "-" ).Int( );
          special_b2 = SPECIAL_WALK.After( "," ).Between( "-", ":" ).Int( );
          special_d = SPECIAL_WALK.Between( ":", "%" ).Double( );    }

     // Do some argument consistency checks.

     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    if ( SEEDS == "" || seeds.size( ) > 1 )
          {    cout << "The SHOW_PAIR_PLACEMENTS options produces lots of files "
                    << "for each seed.  Therefore\nuse of it with more than one "
                    << "seed is discouraged.  If you really want to run with\n"
                    << "more than one seed, add an option to override this test.\n";
               exit(1);    }    }
     if ( PARALLEL_BATCHES != "" )
     {    String C = command.TheCommand( );
          if ( C.Contains( " INSTANCE=" ) || C.Contains( " COUT=" )
               || C.Contains( " EXIT_AT=" ) || C.Contains( " SEEDS=" )
               || C.Contains( " SEED_POST_FILTER=" )
               || C.Contains( " PARALLEL_CONTROL=" ) )
          {    cout << "If PARALLEL_BATCHES is specified, "
                    << "none of the following options should be used:\n";
               cout << "     COUT\n";
               cout << "     EXIT_AT\n";
               cout << "     INSTANCE\n";
               cout << "     SEED_POST_FILTER\n";
               cout << "     SEEDS\n";
               cout << "     PARALLEL_CONTROL\n";
               cout << "Abort.\n";
               exit(1);    }    }
     if (PRINT_TRUE_COVERAGE_GAPS) ForceAssert(USE_TRUTH);
     if (DUMP_TRUE_SEED_LOCS)
     {    String C = command.TheCommand( );
          if ( C.Contains( "SEEDS=" ) )
          {    cout << "If DUMP_TRUE_SEED_LOCS is specified, SEEDS should "
                    << "not be.\n";
               cout << "Abort.\n";
               exit(1);    }    }
     if ( PARALLEL_CONTROL != "" ) ForceAssert( SEEDS != "" );

     // More sanity tests, etc.

     String KS = ToString(K);
     if (USE_TRUTH) ForceAssert( IsRegularFile( data_dir + "/genome.fastb" ) );
     String muxgraph_file = run_dir + "/reads." + PREFIX + "_muxgraph.k" + KS;
     String unilocs_file = run_dir + "/reads.unilocs." + KS + ".10.1";
     ForceAssert( IsRegularFile(unilocs_file) );
     ForceAssertLe( MAX_COPY_NUMBER, 10 );

     vecKmerPath genome_paths, genome_paths_rc;

     if (USE_TRUTH) 
     {    String gpaths = "/genome.paths.k" + KS;
          // If there is no gpaths file specific to the RUN dir, we create
          // a link in the RUN dir to the gpaths file in the DATA dir.
          if ( ! IsRegularFile( run_dir + gpaths ) )
          {    ForceAssert( IsRegularFile( data_dir + gpaths ) );
               SymlinkForce( data_dir + gpaths, run_dir + gpaths );     }
          // Always link to the gpaths file (or link) in the RUN dir, so that
          // the USE_TRUTH code works correctly for error-corrected data.
          SymlinkForce( run_dir + gpaths, wdata_dir + gpaths );

	  genome_paths.ReadAll( run_dir + "/genome.paths.k" + ToString(K) );
	  genome_paths_rc.ReadAll( run_dir + "/genome.paths_rc.k" + ToString(K) );
	  
     }

     if ( SHOW_PAIR_PLACEMENTS > 0 ) PRINT_EASY_GRAPH = True;
     
     // Read in unipaths and pairing data.

     MakeMappedMastervec( run_dir + "/reads.unipaths.k" + KS, 
          vecKmerPath, unipaths );

     int nreads = MastervecFileObjectCount( run_dir + "/reads.fastb" );
     int nuni = unipaths.size( );
     if ( seeds.nonempty( ) && seeds.back( ) >= nuni )
     {    cout << "You've requested seed " << seeds.back( ) << ".  There are "
               << "only " << nuni << " unipaths." << endl;
          cout << "Abort." << endl;
          exit(1);    }
     vec<read_pairing> pairs;
     ReadPairsFile( run_dir + "/reads.pairto", pairs );

     vec<pair_id_t> pairs_index( nreads, -1 );
     for ( int i = 0; i < pairs.isize( ); i++ )
          pairs_index[ pairs[i].id1 ] = pairs_index[ pairs[i].id2 ] = i;
     vec<read_id_t> partner( nreads, -1 );
     for ( int i = 0; i < pairs.isize( ); i++ )
     {    partner[ pairs[i].id1 ] = pairs[i].id2;
          partner[ pairs[i].id2 ] = pairs[i].id1;    }

     // Set up for mux searching.

     int search_limitx = ifloor( search_limit * 1000000.0 );
     longlong answer_size_limitx = ifloor( answer_size_limit * 1000.0 );
     BREAD2( run_dir + "/reads.lengths", vec<int>, readLengths );

     // Load genome (if known) and in any case determine its size.

     vecbasevector genome;
     if (USE_TRUTH) genome.ReadAll( data_dir + "/genome.fastb" );
     else ForceAssert( IsRegularFile( data_dir + "/genome.size" ) );
     if ( IsRegularFile( data_dir + "/genome.size" ) )
          Cp2( data_dir + "/genome.size", wdata_dir );
     else
     {    longlong genome_size = 0;
          for ( int i = 0; i < genome.size( ); i++ )
               genome_size += genome[i].size( );
          Ofstream( gout, wdata_dir + "/genome.size" );
          gout << genome_size << endl;    }
     longlong genome_size = StringOfFile( wdata_dir + "/genome.size", 1 ).Int( );

     // Find the short inserts and compute some statistics about them.

     vec<Bool> is_short_pair( pairs.size( ) ), is_short_pair_read( nreads, False );
     longlong short_insert_bases = 0, short_insert_reads = 0;
     longlong short_insertlength_sum = 0, short_insertdev_sum = 0;
     for ( int i = 0; i < pairs.isize( ); i++ )
     {    const read_pairing& p = pairs[i];
          int id1 = p.id1, id2 = p.id2;
          is_short_pair[i] = ( p.sep <= MAX_SHORT_INSERT_SEP );
          if ( is_short_pair[i] )
          {    is_short_pair_read[id1] = True;
               is_short_pair_read[id2] = True;    
               short_insert_reads += 2;
               short_insert_bases += readLengths[id1] + readLengths[id2];
               short_insertlength_sum 
                    += p.sep + readLengths[id1] + readLengths[id2];    
               short_insertdev_sum += p.sd;    }    }
     double short_insert_coverage = double(short_insert_bases)/double(genome_size);
     double short_insert_readlength 
          = double(short_insert_bases)/double(short_insert_reads);
     double short_insertlength 
          = double(short_insertlength_sum) / double(short_insert_reads/2);
     double short_insertdev 
          = double(short_insertdev_sum) / double(short_insert_reads/2);

     // Load truth data.

     vec<read_location> readlocs;
     vec<int> readlocs_index, genome_path_lengths;
     vecvec<placement> locs;
     Bool USING_READLOCS = USE_TRUTH && ( BASIC_DEBUG || TRACE_GRAPH_EDGES 
          || PRINT_TRUE_COVERAGE_GAPS || PRINT_PRIMARY_CLOUD 
          || PRINT_SECONDARY_CLOUD || EVAL_KMER_ADJS );
     if (USE_TRUTH)
     {    if (USING_READLOCS)
          {    READX( run_dir + "/reads.ref.locs", readlocs );
               readlocs_index.resize(nreads);
               for ( int i = 0; i < readlocs.isize( ); i++ )
               {    const read_location& rl = readlocs[i];
                    readlocs_index[ rl.ReadId( ) ] = i;    }    }
          genome_path_lengths.resize( genome.size( ) );
          for ( int i = 0; i < genome.size( ); i++ )
               genome_path_lengths[i] = genome[i].size( ) + 1 - K;
          locs.ReadAll( run_dir + "/reads.unipaths.k" + KS + ".locs" );    }

     if ( ONELINE_UNIPATH_OUTPUT ) istring::UseOneLineOutput();

     // Find true gaps in coverage.

     if (PRINT_TRUE_COVERAGE_GAPS)
          ReportTrueCoverageGaps( genome, readlocs, K, is_short_pair_read );

     // Find involution which maps a unipath to its reverse complement.  Compute
     // length of each unipath.  Read in output of UnipathCoverage.

     vec<unipath_id_t> to_rc;
     vec<nkmers_t> ulen(nuni);
     MakeMappedVec( run_dir + "/reads.unipathsdb.k" + KS, tagged_rpint, unipathsdb );
     
     UnipathInvolution( unipaths, unipathsdb, to_rc );
     for ( int i = 0; i < nuni; i++ )
          ulen[i] = unipaths[i].KmerCount( );
     vecvec<pdf_entry> cp;
     cp.ReadAll( run_dir + "/reads.unipaths.predicted_count.k" + KS );

     // Define the normal unipaths.

     vec<Bool> normal(nuni, False), normalplus(nuni, False);
     vec<int> predicted_copyno(nuni, -1);
     vec<double> predicted_copyno_p(nuni, -1);
     for ( int i = 0; i < nuni; i++ )
     {    int copyno = -1;
          double maxp = 0;
          for ( int j = 0; j < cp[i].size( ); j++ )
          {    if ( cp[i][j].second > maxp )
               {    copyno = cp[i][j].first;
                    maxp = cp[i][j].second;    }    }
          predicted_copyno[i] = copyno;
          predicted_copyno_p[i] = maxp;
          if ( copyno > MAX_COPY_NUMBER ) continue;
          normalplus[i] = True;
          if ( ulen[i] < MIN_KMERS ) continue;
          normal[i] = True;    }

     // Load the read placements on unipaths.

     BREAD2( unilocs_file, vec<read_location_short>, ulocs );
     vecvec<int> ulocs_indexr;
     ulocs_indexr.ReadAll( unilocs_file + ".indexr" );

     vec<int> uindex( nuni + 1, -1 );
     uindex[0] = 0;
     for ( int i = 0; i < ulocs.isize( ); i++ )
          uindex[ ulocs[i].Contig( ) + 1 ] = i + 1;
     for ( int i = 1; i <= nuni; i++ )
          if ( uindex[i] < 0 ) uindex[i] = uindex[i-1];

     // Load unipath graphs.

     digraphE<sepdev> G, Gplus;
     digraphE<fsepdev> FG;
     BinaryRead( run_dir + "/reads.unipathgraph.k" + KS, G );
     BinaryRead( run_dir + "/reads.unipathgraphplus.k" + KS, Gplus );
     BinaryRead( run_dir + "/reads.unipathgraph_fp.k" + KS, FG );

     // Select seeds, launch parallel processes if requested.

     vec<unipath_id_t> seedsx;
     if ( SEEDS == "" )
     {    
          // Generate seeds.

       cout << "generating seeds..." << endl;
          UnipathSeeds( MIN_KMERS_IN_SEED, ulen, normal, predicted_copyno, G, FG,
               USE_TRUTH, locs, UNUSED_SEED_STATS, DUMP_TRUE_SEED_LOCS,
               MAX_SEED_DIST, genome, seeds, SEED_VERBOSITY );
       cout << "done generating seeds." << endl;
	  
          if (DUMP_TRUE_SEED_LOCS) exit(0);

          // Filter seeds.

          int nseeds = 0;
          for ( int vi = 0; vi < seeds.isize( ); vi++ )
          {    unipath_id_t v = seeds[vi];
               if ( G.From(v).empty( ) && G.To(v).empty( ) ) continue;
               if ( ulen[v] < MIN_KMERS_IN_SEED ) continue;
         
               // For simulated data, we skip the rc case.
        
               if ( USE_TRUTH && FW_SEEDS_ONLY && !locs[v].empty( ) )
               {    Bool have_fw = False;
                    for ( int u = 0; u < locs[v].size( ); u++ )
                    {    const placement& p = locs[v][u];
                         if ( !p.Rc( ) ) have_fw = True;    }
                    if ( !have_fw ) continue;    }
       
               if ( MAX_SEEDS >= 0 && ++nseeds > MAX_SEEDS ) break;
               seedsx.push_back(v);    }
          seeds = seedsx;
          if ( SEED_POST_FILTER != "" )
          {    vec<unipath_id_t> filter;
               ParseIntSet( SEED_POST_FILTER, filter );
               seeds = Intersection( seeds, filter );    }
     
          // If using truth, sort seeds according to their position along the
          // genome.
     
          if (USE_TRUTH)
          {    vec< pair<placement,unipath_id_t> > seeds_pos;
               for ( int i = 0; i < seeds.isize( ); i++ )
               {    unipath_id_t v = seeds[i];
                    for ( int u = 0; u < locs[v].size( ); u++ )
                         seeds_pos.push_back( make_pair( locs[v][u], v ) );    }
               Sort(seeds_pos);
               seeds.clear( );
               for ( int i = 0; i < seeds_pos.isize( ); i++ )
               {    if ( Member( seeds, seeds_pos[i].second ) ) continue;
                    seeds.push_back( seeds_pos[i].second );    }    }

          // Parallelize if requested.

          if ( PARALLEL_BATCHES != "" )
          {    Destroy( uindex, ulocs, normal, normalplus, predicted_copyno );
               Destroy( predicted_copyno_p, to_rc, ulen, readlocs, readlocs_index );
               Destroy( genome_path_lengths, pairs, pairs_index, partner );
               Destroy( ulocs_indexr, cp, locs, genome );
               Destroy( readLengths );
               LocalizeReadsParallelLaunch( PARALLEL_BATCHES, seeds, command, 
                    data_dir, run_dir, sub_dir, LSF_ARGS, LSF_HOST, K, USE_TRUTH,
                    MIN_OVERLAP_FINAL, MIN_PROPER_OVERLAP_FINAL, GLOBAL_CLEAN, 
                    TRACK_GLOBAL_CLEAN_CHANGES, SHOW_PAIR_ALIGNS, FILTER_ALIGNS, 
                    FINAL_MERGE, BASIC_DEBUG, DIPLOID, 
                    DISAMBIGUATE_SIMPLE_LOOPS_VERBOSE, PULL_VERBOSE, MIN_COMPONENT, 
                    MAX_SHORT_INSERT_SEP, unilocs_file, wrun_dir,
                    NEW_POORLY_COVERED );
               cout << "\nelapsed time for entire "
                    "LocalizeReads run = " << TimeSince(all_clock) << endl;
               exit(0);    }    }

     // Load paths and reads.

     MakeMappedMastervec( run_dir + "/reads.paths.k" + KS, vecKmerPath, paths );
     MakeMappedMastervec( run_dir + "/reads.paths_rc.k" + KS, 
          vecKmerPath, paths_rc );
     MakeMappedVec( run_dir + "/reads.pathsdb.k" + KS, tagged_rpint, pathsdb );
     MakeMappedMastervec( run_dir + "/reads.fastb", vecbasevector, reads );
     KmerBaseBroker* kbb = new KmerBaseBroker(run_dir, K, paths, paths_rc, pathsdb);
     KmerBaseBroker* gkbb = kbb;
     
     // Set up file descriptor for parallel control.

     int pc_fd = -1;
     if ( PARALLEL_CONTROL != "" )
     {    pc_fd = open( PARALLEL_CONTROL.c_str( ), O_RDWR, 0664 );
          if ( pc_fd < 0 ) FatalErr( "Could not open PARALLEL_CONTROL." );
	  Ofstream( slave_started, sub_dir + "/SLAVE_STARTED"); }

     // For each unipath, find its neighborhood.

     vec<int> zero_dev;
     zero_dev.push_back(0);
     int sim_tried = 0, sim_covered = 0, sim_tried_sh = 0, sim_covered_sh = 0;
     vec<HyperBasevector> hyperbases;
     hyperbases.reserve( seeds.size( ) );
     cout << TimeSince(all_clock) << " used in initialization\n";

     unsigned int seeds_processed = 0;
     map< unipath_id_t, PredictionStats > kmerAdjPredictionStats;

     vecKmerPathIndex genome_paths_idx( genome_paths, genome_paths_rc );

     cout << " *** processing " << seeds.isize() << " seeds." << endl;
     
     for ( int vi = 0; vi < seeds.isize( ); vi++ )
       {  
       
          // If this process is part of a parallel run, get next seed index.

          if ( PARALLEL_CONTROL != "" )
          {    IncrementSharedCounter( pc_fd, vi, seeds.size( ) );
               if ( vi == seeds.isize( ) ) break;    }

	  if ( !EVAL_KMER_ADJS )
	    ForceAssertEq( seeds_processed, hyperbases.size() );
          ++seeds_processed;

          // Get id of seed, start neighborhood clock.

          unipath_id_t v = seeds[vi];
          double nhood_all_clock = WallClockTime( );
	  cout << "looking at seed " << v << "( " << seeds_processed << " of " <<
	    seeds.isize() << ")" << ", MemUsage=" << MemUsage() << "k" << endl;

          // Reset random number generator.

          if ( RANDOM_SEED == 0 ) RANDOM_SEED = 1694384237;
          srandomx(RANDOM_SEED);

          // Generate neighborhood of v and print it.

          double nhood_clock = WallClockTime( );
          static vec<ustart> processed;
          FindUnipathNhood( v, G, ulen, cp, predicted_copyno, paths, ulocs, uindex,
               pairs, pairs_index, FILTER_NHOOD, MAX_COPY_NUMBER_OTHER, 
               NHOOD_RADIUS, MAX_TO_CUT, MAX_DEV, MAX_PATHS_IN_NHOOD, 
               BUILD_NHOOD_FW_ONLY, SIMULATION_TRIES, processed );
          PrintNhood( v, processed, ulen, USE_TRUTH, &locs );
          if (DUMP_GRAPH)
          {    cout << "\n";
               for ( int j = 0; j < processed.isize( ); j++ )
               {    int w = processed[j].Uid( );
                    {    for ( int r = 0; r < G.From(w).isize( ); r++ )
                         {    int x = G.From(w)[r];
                              for ( int u = 0; u < processed.isize( ); u++ )
                              {    if ( processed[u].Uid( ) != x ) continue;
                                   cout << j+1 << " --> " << u+1 
                                        << "\n";    }    }    }    }    }
          CHANGE_TIMER( nhood_clock, "used building neighborhood", hhclock );

          // Add in unipaths of predicted copy number one that are linked to by 
          // copy number one unipaths, regardless of their lengths and how many of 
          // them there are.
          // NO! CHANGED! Now looks for copy number <= copy number(v).

          AddCopyNumberOne( v, processed, predicted_copyno, Gplus, ulen, 
               NHOOD_RADIUS, MAX_DEV );

          // Populate neighborhood with reads.  Also create larger cloud of reads.

          static vec< pair<read_id_t,orient_t > > use, use_big;
          static vec<pos_rel_to_seed_t> usestart, usestart_big;
	  static vec<pair_id_t> pairs_to_use, pairs_to_use_big;
          PopulateNhoodWithReads( v, processed, K, ulen, pairs, pairs_index, paths,
               ulocs, uindex, NHOOD_RADIUS_INTERNAL, MAX_DEV, REACH_FW_ONLY, 
               use, usestart, pairs_to_use );
          if (PRINT_PRIMARY_CLOUD)
          {    cout << "\nPrimary cloud:\n";
               vec<read_id_t> usefirst;
               for ( int i = 0; i < use.isize( ); i++ )
                 usefirst.push_back( use[i].first );
               for ( int i = 0; i < use.isize( ); i++ )
               {    if ( use[i].second ) continue;
                    read_id_t id1 = use[i].first;
                    read_id_t id2 = partner[id1];
                    const read_location& rl1 = readlocs[ readlocs_index[id1] ];
                    const read_location& rl2 = readlocs[ readlocs_index[id2] ];
                    cout << rl1.Contig( ) << "." << rl1.Start( ) << "-"
                         << rl1.Stop( )+1 << " ===> " << rl2.Start( ) << "-"
                         << rl2.Stop( )+1 << " [" << id1 << "," << id2 << "]";
                    int i2 = BinPosition( usefirst, id2 );
                    vec<int> seed_starts;
                    for ( int l = 0; l < locs[v].size(); ++l )
                      seed_starts.push_back( locs[v][l].pos() );
                    UniqueSort( seed_starts );
                    for ( unsigned int s = 0; s < seed_starts.size(); ++s ) {
                      if ( s == 0 ) cout << "  (";
                      else cout << ", ";
                      cout << seed_starts[s] + usestart[i]
                           << " ===> " 
                           << seed_starts[s] + usestart[i2];
                    }
                    cout << ")\n";
               }
               cout << "\n";    }

          // Plot kmer-frequencies.

          if ( JUST_PLOT_LOCAL_KMER_FREQS )
          {    JustPlotLocalKmerFrequencies( use, reads, run_dir );
               continue;    }


          // Find the <secondary read cloud>: short-insert read pairs such that each of its reads could be
          // completely contained in a contig created from the used reads S.

          static vec< pair<read_id_t,orient_t> > P;
          if (FIND_SHORT_INSERTS)
	    {    GetShortInsertReads( use, pathsdb, paths, paths_rc,
				      PATH_KS, extra_paths_db, extra_paths, extra_paths_rc,
				      partner,
				      is_short_pair_read, P );    }

	  if ( EVAL_KMER_ADJS ) {
	    EvaluateKmerAdjPredictionForNeighborhood( v, seeds_processed, K, genome_paths, genome_paths_rc,
						      genome_paths_idx, paths, paths_rc,
						      locs, ulen, NHOOD_RADIUS_INTERNAL,
						      predicted_copyno,
						      run_dir, is_short_pair_read,
						      USING_READLOCS,
						      readlocs, readlocs_index,
						      use, P );

	    continue;  // to next seed; only when all we're doing is measuring quality of localization.
	  }
	  

          // Check for coverage if data is simulated.  Print the gaps, which are
          // half-open intervals on the genome.

          if (USING_READLOCS)
          {    ReportCoverage( locs[v], K, use, P, readlocs, readlocs_index,
                    genome_path_lengths, NHOOD_RADIUS_INTERNAL, sim_tried,
                    sim_covered, sim_tried_sh, sim_covered_sh );    }
          TEST_EXIT_AT(ReportCoverage);

          // Dump read cloud.

          static vec< pair<int,Bool> > use_plus_P;
          use_plus_P = use;
          use_plus_P.append(P);
          UniqueSort(use_plus_P);
          if ( DUMP_CLOUD != "" )
          {    Ofstream( out, DUMP_CLOUD );
               for ( int i = 0; i < use_plus_P.isize( ); i++ )
                    out << use_plus_P[i].first << " " 
                         << int( use_plus_P[i].second ) << "\n";    }
          CHANGE_TIMER( hhclock, "used woofing", bitclock );

          // Find bits of reads in use that are in nhood, and not covered by 
          // reads in P.  Each bit is two kmers long.  Convert to basevectors.

          static vec<basevector> extra_sreads;
          static vec<Bool> extra_sreads_pushed_funny;
          static vec<KmerPath> extra_sreadsp;
          static vec< vec<placement> > nulocs;
          {    vecKmerPath epaths;
               int NP = use_plus_P.size( );
               for ( int i = 0; i < NP; i++ )
               {    int id = use_plus_P[i].first;
                    if ( !use_plus_P[i].second ) 
                         epaths.push_back_reserve( paths[id] );
                    else epaths.push_back_reserve( paths_rc[id] );    }
               vec<tagged_rpint> epathsdb;
               CreateDatabase( epaths, epathsdb );
               vec<int> to_P(NP), to_use(NP);
               for ( int i = 0; i < NP; i++ )
               {    to_P[i] = BinPosition( P, use_plus_P[i] );
                    to_use[i] = BinPosition( use, use_plus_P[i] );    }
               extra_sreads.clear( ), extra_sreads_pushed_funny.clear( );
               extra_sreadsp.clear( ), nulocs.clear( );
               for ( int z = 0; z < NP; z++ )
               {    if ( to_P[z] >= 0 ) continue;
                    int j = to_use[z];
                    if ( usestart[j] < -NHOOD_RADIUS_INTERNAL 
                         || usestart[j] > NHOOD_RADIUS_INTERNAL )
                    {    continue;    }
                    const KmerPath& q1 = epaths[z];
                    int seg1 = 0, pos1 = 0, count = 0;
                    while(1)
                    {    int seg1p = seg1, pos1p = pos1 + 1;
                         if ( pos1p == q1.Length(seg1p) )
                         {    ++seg1p;
                              if ( seg1p == q1.NSegments( ) ) break;
                              pos1p = 0;    }
                         Bool hit = False;
                         longlong x = q1.Start(seg1) + pos1, 
                              xp = q1.Start(seg1p) + pos1p;
                         static vec<longlong> places;
                         Contains( epathsdb, x, places );
                         for ( int b = 0; b < places.isize( ); b++ )
                         {    const tagged_rpint& t = epathsdb[ places[b] ];
                              int id = t.ReadId( );
                              if ( to_P[id] < 0 ) continue;
                              const KmerPath& q2 = epaths[id];
                              int seg2 = t.PathPos( ), pos2 = x - t.Start( );
                              int seg2p = seg2, pos2p = pos2 + 1;
                              if ( pos2p == q2.Length(seg2p) )
                              {    ++seg2p;
                                   if ( seg2p == q2.NSegments( ) ) continue;
                                   pos2p = 0;    }
                              if ( xp == q2.Start(seg2p) + pos2p )
                              {    hit = True;
                                   break;    }    }
                         if ( !hit )
                         {    static KmerPath p;
                              p.Clear( );
                              p.AddSegment( x, x );
                              p.AddSegment( xp, xp );
                              if (BITS_VERBOSE)
                              {    cout << "adding bit " << kbb->Seq(p).ToString( ) 
                                        << "\n";    }
                              extra_sreads.push_back( kbb->Seq(p) );
                              extra_sreadsp.push_back(p);
                              extra_sreads_pushed_funny.push_back(False);
                              static vec<placement> these_locs;
                              these_locs.clear( );
                              // for ( int j = 0; j < locs[w].size( ); j++ )
                              // {    const placement& z = locs[w][j];
                              //      these_locs.push_back(
                              //           placement( z.GenomeId( ), 
                              //                z.pos( ) + count,
                              //                z.pos( ) + count + 1 + K, 
                              //                z.Rc( ) ) );   }
                              nulocs.push_back(these_locs);    }
                         ++count;
                         ++pos1;
                         if ( pos1 == q1.Length(seg1) )
                         {    ++seg1;
                              if ( seg1 == q1.NSegments( ) ) break;
                              pos1 = 0;    }    }    }    }

          // Find global unipaths that are touched by reads in P but which either
          // have kmers not in P or kmer junctions not in P.  Add them to the bits
          // just generated.  Also global unipath junctures.
          // NO: FOR NOW JUST ADD ALL GLOBAL UNIPATHS THAT ARE TOUCHED.

          if (ADD_GLOBAL)
          {    set<int> global_unipaths_hit_s;
               for ( int i = 0; i < P.isize( ); i++ )
               {    const KmerPath& x = paths[ P[i].first ];
                    for ( int j = 0; j < x.NSegments( ); j++ )
                    {    static vec<longlong> con;
                         Contains( unipathsdb, x.Segment(j), con );
                         for ( int l = 0; l < con.isize( ); l++ )
                         {    global_unipaths_hit_s.insert( 
                                   unipathsdb[ con[l] ].ReadId( ) );    }    }    }
               vec<int> global_unipaths_hit;
               for ( set<int>::iterator i = global_unipaths_hit_s.begin( );
                    i != global_unipaths_hit_s.end( ); ++i )
               {    global_unipaths_hit.push_back(*i);    }
               for ( int i = 0; i < global_unipaths_hit.isize( ); i++ )
               {    int u = global_unipaths_hit[i];
                    extra_sreads.push_back( kbb->Seq(unipaths[u]) );
                    extra_sreadsp.push_back( unipaths[u] );
                    extra_sreads_pushed_funny.push_back(False);
                    static vec<placement> these_locs;
                    these_locs.clear( );
                    nulocs.push_back(these_locs);    }    }
          CHANGE_TIMER( bitclock, "finding extra bits", uclock );

          // Find reads in use, not in P, that do not introduce new kmers.  

          vecKmerPath sofar, sofarrc;
          for ( int j = 0; j < P.isize( ); j++ )
          {    if ( !P[j].second ) sofar.push_back_reserve( paths[ P[j].first ] );
               else sofar.push_back_reserve( paths_rc[ P[j].first ] );    }
          for ( int j = 0; j < extra_sreadsp.isize( ); j++ )
               sofar.push_back_reserve( extra_sreadsp[j] );
          vec<tagged_rpint> sofardb;
          CreateDatabase( sofar, sofardb );
          vec< pair< int, pair<int,int> > > for_extra_spairs;
          vec<read_pairing> extra_spairs;
          vec<int> extra_spairs_pairs_index;
          vec<int> spairs_pairs_index;
          for ( int j = 0; j < use.isize( ); j++ )
          {    // if ( usestart[j] < -NHOOD_RADIUS_INTERNAL 
               //      || usestart[j] > NHOOD_RADIUS_INTERNAL )
               // {    continue;    }
               if ( BinMember( P, use[j] ) ) continue;
               int uid = use[j].first;
               const KmerPath& p = ( use[j].second ? paths_rc[uid] : paths[uid] );
               if ( !SubContig( p, sofar, sofarrc, sofardb ) ) continue;
               for_extra_spairs.push( pairs_index[uid], 
                    make_pair( j, extra_sreads.size( ) ) );
               extra_sreads.push_back( kbb->Seq(p) );
               // extra_sreads.push_back( reads[uid] );
               extra_sreads_pushed_funny.push_back( use[j].second );
               static vec<placement> these_locs;
               nulocs.push_back(these_locs);    }
          Sort(for_extra_spairs);
          for ( int j = 0; j < for_extra_spairs.isize( ); j++ )
          {    int k;
               for ( k = j + 1; k < for_extra_spairs.isize( ); k++ )
               {    if ( for_extra_spairs[k].first != for_extra_spairs[j].first )
                         break;    }
               if ( k - j == 2 )
               {    int use_index = for_extra_spairs[j].second.first;
                    if ( use[use_index].second ) 
                         swap( for_extra_spairs[j], for_extra_spairs[j+1] );    }
               if ( k - j == 2 )
               {    int pi = for_extra_spairs[j].first;
                    read_pairing p = pairs[pi];
                    int use_index = for_extra_spairs[j].second.first;
                    int extra_sreads_pos = for_extra_spairs[j].second.second;
                    int nid1 = for_extra_spairs[j].second.second + P.isize( );
                    int nid2 = for_extra_spairs[j+1].second.second + P.isize( );
                    if ( p.id1 != use[use_index].first ) swap( nid1, nid2 );
                    p.id1 = nid1, p.id2 = nid2;
                    extra_spairs.push_back(p);
                    extra_spairs_pairs_index.push_back(pi);    }
               j = k - 1;    }
          cout << TimeSince(uclock) << " used finding reads from use" << endl;

          // Walk inserts in P using only the reads from P.  This yields a set of
          // closures for the short inserts in the neighborhood.

          Bool found_universal = False, found_universal_strong = False;
          static vec<basevector> closures;
          closures.clear( );
          vec< vec<pp_closure> > ppclosures;
          vec<pp_read> chunks;
          vec<basevector> ppclosures_unique;
          static vec<HyperPairPlacement> hpairs, hpairs_unsorted, extra_hpairs;
          static vec<int> hpairs_source, extra_hpairs_source;
          static vec<pp_pair> extra_ppp_nonreduced;
          static vec<pp_pair> ppp_nonreduced;
          static vec<int> pppL;
          HyperKmerPath h;
          KmerBaseBroker kbbnb;
          vecKmerPath xpaths, xpaths_rc;
          vec<tagged_rpint> xpathsdb;
          vecbasevector sreads;
          {    int nsreads = P.size( ) + extra_sreads.size( );
               double jjjclock1 = WallClockTime( );
               int K2 = K;
               int nbases = 0;
               for ( int i = 0; i < P.isize( ); i++ )
                    nbases += readLengths[ P[i].first ];
               for ( int i = 0; i < extra_sreads.isize( ); i++ )
                    nbases += extra_sreads[i].size( );
               sreads.clear( );
               sreads.Reserve( nbases/16 + nsreads, nsreads );
               vec<read_pairing> spairs;
               static vec<read_location> slocs;
               static vec<int> fw_reads, rc_reads;
               slocs.clear( ), fw_reads.clear( ), rc_reads.clear( );

               // Create reads from short insert reads.

               vec< pair<read_location, read_location> > cloud2;
               static vec<int> fw_reads_orig, rc_reads_orig;
               fw_reads_orig.clear( ), rc_reads_orig.clear( );
               static vec<Bool> primary;
               primary.clear( );
               for ( int i = 0; i < P.isize( ); i++ )
               {    if ( P[i].second ) continue;
                    Bool prim = BinMember( use, P[i] );
                    int pi = pairs_index[ P[i].first ];
                    read_pairing p = pairs[pi];
                    if ( p.id1 != P[i].first ) swap( p.id1, p.id2 );
                    int nid1 = sreads.size( );
                    rc_reads.push_back(nid1);
                    rc_reads_orig.push_back(p.id1);
                    sreads.push_back( reads[p.id1] );
                    int nid2 = sreads.size( );
                    fw_reads.push_back(nid2);
                    fw_reads_orig.push_back(p.id2);
                    sreads.push_back( reads[p.id2] );
                    primary.push_back( prim, prim );
                    if (USING_READLOCS)
                    {    read_location rl1 = readlocs[ readlocs_index[p.id1] ];
                         read_location rl2 = readlocs[ readlocs_index[p.id2] ];
                         rl1.SetReadId(nid1);
                         rl2.SetReadId(nid2);
                         slocs.push_back( rl1, rl2 );
                         if (PRINT_SECONDARY_CLOUD)
                              cloud2.push_back( make_pair( rl1, rl2 ) );    }
                    p.id1 = nid1, p.id2 = nid2;
                    spairs.push_back(p);
                    spairs_pairs_index.push_back(pi);    }
               CHANGE_TIMER( jjjclock1, "getting ready 1", jjjclock2 );

               // Create reads from extra reads.

               for ( int i = 0; i < extra_sreads.isize( ); i++ )
               {    int nid = sreads.size( );
                    rc_reads.push_back(nid);
                    primary.push_back(True);
                    sreads.push_back( extra_sreads[i] );
                    if (USING_READLOCS)
                    {    
                         // Note that a unipath can have multiple locations.  This
                         // may cause trouble somewhere.  Also, it is conceivable
                         // that a unipath might have no locations at all.
                         // (Not sure if this comment makes sense anymore.)

                         for ( int j = 0; j < nulocs[i].isize( ); j++ )
                         {    const placement& p = nulocs[i][j];
                              int g = p.GenomeId( );
                              read_location rl( nid, p.Pos( ) - p.pos( ), g,
                                   p.pos( ), p.Rc( ), genome[g].size( ) );
                              slocs.push_back(rl);    }    }    }
               cout << TimeSince(jjjclock2) << " getting ready 2\n";

               // Write files.

               int restarted = 0, restartedt = 0;
               int fw_reads_size_before_restart = fw_reads.size( );
               int rc_reads_size_before_restart = rc_reads.size( );
               restart:
               double jjjclock2b = WallClockTime( );
               if (USING_READLOCS)
               {    Sort(slocs);
                    WriteLocs( wrun_dir + "/reads.ref.locs", slocs, 
                         genome.size( ), nsreads );    }
               CHANGE_TIMER( jjjclock2b, "getting ready 2b", jjjclock3a );
               ReadsToPathsCoreY( sreads, K2, genome_size, xpaths, xpaths_rc,
                    xpathsdb );
               CHANGE_TIMER( jjjclock3a, "used getting ready at ReadsToPathsCore",
                    jjjclock3 );

               // Build zunipaths, in which each local unipath is split into its 
               // global components.  (I'm not sure this comment makes sense.)

               kbbnb.Initialize( K2, sreads, xpaths, xpaths_rc, xpathsdb );
               vecKmerPath zpaths, zpaths_rc, zunipaths;
               zpaths.Reserve( xpaths.rawsize(), xpaths.size() );
               zpaths_rc.Reserve( xpaths_rc.rawsize(), xpaths_rc.size() );
               for ( int i = 0; i < fw_reads.isize( ); i++ )
                    zpaths.push_back( xpaths[ fw_reads[i] ] );
               for ( int i = 0; i < rc_reads.isize( ); i++ )
                    zpaths_rc.push_back( xpaths_rc[ rc_reads[i] ] );
               vec<tagged_rpint> zpathsdb, zunipathsdb;
               CreateDatabase( zpaths, zpaths_rc, zpathsdb );
               Unipath( zpaths, zpaths_rc, zpathsdb, zunipaths, zunipathsdb );

               // Remove unipaths that are supported by only one read.

               vec<Bool> solo_unipath( zunipaths.size( ), False );
               for ( int i = 0; i < zunipaths.size( ); i++ )
               {    const KmerPath& u = zunipaths[i];
                    static vec<int> support;
                    support.clear( );
                    for ( int j = 0; j < u.NSegments( ); j++ )
                    {    const KmerPathInterval& I = u.Segment(j);
                         static vec<longlong> con;
                         Contains( zpathsdb, I, con );
                         for ( int l = 0; l < con.isize( ); l++ )
                         {    const tagged_rpint& t = zpathsdb[ con[l] ];
                              int id = t.PathId( ), rid = t.ReadId( );
                              if ( id >= 0 && rid >= fw_reads_orig.isize( ) )
                                   continue;
                              if ( id < 0 && rid >= rc_reads_orig.isize( ) )
                                   continue;
                              if ( support.empty( )
                                   || ( support.nonempty( ) && support[0] != id ) )
                              {    support.push_back(id);    }
                              if ( support.size( ) >= 2 ) break;    }
                         if ( support.size( ) >= 2 ) break;    }
                    if ( support.size( ) <= 1 ) solo_unipath[i] = True;    }
               vecKmerPath zunipaths2;
               for ( int i = 0; i < zunipaths.size( ); i++ )
               {    if ( !solo_unipath[i] ) 
                         zunipaths2.push_back_reserve( zunipaths[i] );    }
               zunipaths = zunipaths2;
               CreateDatabase( zunipaths, zunipathsdb );

               // Build unipath graph.

               digraph A;
               BuildUnipathAdjacencyGraph( zpaths, zpaths_rc, zpathsdb,
                    zunipaths, zunipathsdb, A );
               BuildUnipathAdjacencyHyperKmerPath( K, A, zunipaths, h );
               h.Reverse( );
               const int MIN_KMERS_IN_COMPONENT_TO_KEEP = 100;
               h.RemoveSmallComponents( MIN_KMERS_IN_COMPONENT_TO_KEEP );

               // Merge at unipath junctures where there is no branching.  Then
               // rebuild unipath graph.

               h.RemoveUnneededVertices( );
               h.RemoveDeadEdgeObjects( );
               h.Reverse( );
               zunipaths.clear( );
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
                    zunipaths.push_back_reserve( h.EdgeObject(i) );
               CreateDatabase( zunipaths, zunipathsdb );
               BuildUnipathAdjacencyGraph( zpaths, zpaths_rc, zpathsdb,
                    zunipaths, zunipathsdb, A );
               BuildUnipathAdjacencyHyperKmerPath( K, A, zunipaths, h );
               h.Reverse( );

               // Determine the approximate start point of each h edge,
               // relative to the origin of the neighborhood (start of v).

               vecbasevector usereads_edges;
               for ( int i = 0; i < use.isize( ); i++ )
               {    int id = use[i].first;
                    if ( !use[i].second )
                         usereads_edges.push_back_reserve( reads[id] );
                    else
                    {    static basevector r;
                         r = reads[id];
                         r.ReverseComplement( );
                         usereads_edges.push_back_reserve(r);    }    }
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
               {    usereads_edges.push_back_reserve( 
                         kbbnb.Seq( h.EdgeObject(i) ) );    }
               PerfectAligner xpal( K, PerfectAligner::findProperOnly );
               vec<alignment_plus> xaligns;
               CHANGE_TIMER( jjjclock3, "used in first part of finding unipaths",
                    xpclock );
               xpal.Align( usereads_edges, xaligns, use.size( ) );
               CHANGE_TIMER( xpclock, 
                    "used aligning reads to determine edge start points", u2clock );
               vec< vec<int> > hstarts( h.EdgeObjectCount( ) );
               for ( int i = 0; i < xaligns.isize( ); i++ )
               {    const alignment_plus& ap = xaligns[i];
                    if ( ap.Rc2( ) ) continue;
                    int id1 = ap.Id1( ), id2 = ap.Id2( );
                    hstarts[id2].push_back( 
                         usestart[id1] + ap.a.pos1( ) - ap.a.pos2( ) );    }
               vec<Bool> hstart_known( h.EdgeObjectCount( ), False );
               vec<int> hstart( h.EdgeObjectCount( ) );
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
               {    Sort( hstarts[i] );
                    if ( hstarts[i].nonempty( ) )
                    {    hstart_known[i] = True;
                         hstart[i] = hstarts[i][ hstarts[i].size( )/2 ];    }    }
               /*
               cout << "Predicted edge starts:\n";
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
               {    cout << "[" << i << "]: ";
                    if ( hstart_known[i] ) cout << hstart[i] << "\n";
                    else cout << "?\n";    }
               */

               // Output graph.

               vec<look_align> aligns;
               vec< vec<int> > aligns_index;
               cout << TimeSince(u2clock) << " used in second part of "
                    << "finding unipaths" << endl;
               if ( PRINT_EASY_GRAPH && USE_TRUTH )
               {    AlignHyperKmerPath( h, &kbbnb, data_dir + "/genome", 
                         wrun_dir, aligns, aligns_index );
                    double tclock = WallClockTime( );
                    for ( int v = 0; v < h.N( ); v++ )
                    {    for ( int j = 0; j < h.From(v).isize( ); j++ )
                         {    int w = h.From(v)[j];
                              int i = h.EdgeObjectIndexByIndexFrom( v, j );
                              Bool perfect = False;
                              for ( int u = 0; u < aligns_index[i].isize( ); u++ )
                              {    const look_align& la 
                                        = aligns[ aligns_index[i][u] ];
                                   if ( la.Errors( ) == 0 && la.FullLength( ) )
                                        perfect = True;    }
                              if ( !perfect )
                              {    const KmerPath& p = h.EdgeObject(i);
                                   cout << "Warning: edge " << BaseAlpha(i) 
                                        << " from " << v << " to " << w 
                                        << " of length " << p.KmerCount( );
                                   if ( aligns_index[i].empty( ) )
                                        cout << " not aligned:\n";
                                   else cout << " has no perfect alignment:\n";
                                   kbbnb.Seq(p).Print( cout, 
                                        BaseAlpha(i) );    }    }    }
                    cout << TimeSince(tclock) << " used getting truth" << endl;    }
               double u3clock = WallClockTime( );
               if (PRINT_EASY_GRAPH)
               {    static int counter(0);
                    String easy_fn = sub_dir + "/easy/" + ToString(counter) + ".dot";
                    cout << "\nEasy graph for neighborhood of seed " << v << ":\n";
                    if ( seeds.solo( ) )
                    {    Mkdir777( sub_dir + "/easy" );
                         cout << "(see also " << easy_fn << ")\n";    }
                    {    Ofstream( eout, sub_dir + "/current_easy_graph" );
                         if (USE_TRUTH)
                         {    PrintAlignedHyperKmerPath( cout, h, &kbbnb, genome, 
                                   aligns, aligns_index );
                              PrintAlignedHyperKmerPath( eout, h, &kbbnb, genome, 
                                   aligns, aligns_index );    }
                         else 
                         {    h.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
                              h.PrintSummaryPlus( 
                                   eout, 0, 0, 0, 0, 0, False );    }    }
                    if ( seeds.solo( ) )
                    {    Ofstream( dot, easy_fn );
                         counter++;
                         h.PrintSummaryDOT0w( dot, False, False, True );    }    }
               CHANGE_TIMER( u3clock,
                    "used in third part of finding unipaths", uclock );

               // Determine which edges contain kmers that appear to be 
               // globally unique.  Note that we might rather know if they are
               // locally unique.  Also get higher copy number (up to 10).
                    
               vec<Bool> unique_edge;
               vec<int> edge_copyno;
               vec<double> edge_copyno_p;
               vec<tagged_rpint> edgedb;
               h.MakeEdgeDatabase(edgedb);
               EdgeCopyNumber( h, fw_reads_orig, rc_reads_orig, zpaths, zpaths_rc, 
                    ulocs, ulocs_indexr, predicted_copyno, predicted_copyno_p,
                    EDGE_MIN, readLengths, ulen, edgedb, edge_copyno, 
                    edge_copyno_p, unique_edge );
               CHANGE_TIMER( uclock, "used finding edge copy numbers", u4clock );
               cout << endl;

               // Set up edge data structures, print placement of truth on graph.

               vec<vrtx_t> to_right_vertex;
               h.ToRight(to_right_vertex);
               if (USE_TRUTH)
               {    PrintTruthPlacement( v, locs, genome, NHOOD_RADIUS_INTERNAL, h, 
                         kbbnb, edge_copyno );    }
     
               // Show placements of read pairs on graph.

               vecKmerPath jpaths;
               vec<read_pairing> jpairs;
               cout << TimeSince(u4clock) << " used in fourth part of "
                    << "finding unipaths" << endl;
               PlacePairs( spairs, PRINT_SECONDARY_CLOUD, cloud2, xpaths, xpaths_rc, 
                    h, edgedb, to_right_vertex, hpairs, hpairs_source,
                    unique_edge, K_ADD_IN, K_ADD_OUT, BASIC_DEBUG,
                    NEW_UNIQUE_EXTEND );
               hpairs_unsorted = hpairs;
               for ( int i = 0; i < extra_sreads_pushed_funny.isize( ); i++ )
               {    if ( extra_sreads_pushed_funny[i] )
                    {    int id = i + P.isize( );
                         swap( xpaths[id], xpaths_rc[id] );    }    }
               PlacePairs( extra_spairs, False, cloud2, xpaths, xpaths_rc, 
                    h, edgedb, to_right_vertex, extra_hpairs, extra_hpairs_source,
                    unique_edge, K_ADD_IN, K_ADD_OUT, BASIC_DEBUG,
                    NEW_UNIQUE_EXTEND );
               for ( int i = 0; i < extra_sreads_pushed_funny.isize( ); i++ )
               {    if ( extra_sreads_pushed_funny[i] )
                    {    int id = i + P.isize( );
                         swap( xpaths[id], xpaths_rc[id] );    }    }
               double trusted_paths_clock = WallClockTime( );
               vecKmerPath trustedpaths;
               vec<int> tpi;
               for ( int i = 0; i < fw_reads_size_before_restart; i++ )
               {    if ( primary[ fw_reads[i] ] )
                    {    trustedpaths.push_back_reserve( xpaths_rc[ fw_reads[i] ] );
                         tpi.push_back(i);    }    }
               int trpart1 = trustedpaths.size( );
               for ( int i = 0; i < rc_reads_size_before_restart; i++ )
               {    if ( primary[ rc_reads[i] ] )
                         trustedpaths.push_back_reserve( 
                              xpaths[ rc_reads[i] ] );    }
               CHANGE_TIMER( trusted_paths_clock, "used making trusted paths",
                    translation_clock );
               vec<pp_read> trusteds;
               TranslateTrustedPaths( trustedpaths, h, trusteds, NEW_UNIQUE_EXTEND );
               if (BASIC_DEBUG) cout << "\nTRUSTED PAIRS:\n";
               vec<read_pairing> trspairs;
               for ( int i = 0; i < tpi.isize( ); i++ )
                    trspairs.push_back( spairs[ tpi[i] ] );
               static vec<HyperPairPlacement> trhpairs;
               static vec<int> trhpairs_source;
               static vec< pair<read_location, read_location> > cloud2_null;
               cout << TimeSince(translation_clock) 
                    << " used translating trusted paths" << "\n";
               PlacePairs( trspairs, False, cloud2_null, xpaths, xpaths_rc, h, 
                    edgedb, to_right_vertex, trhpairs, trhpairs_source, unique_edge,
                    K_ADD_IN, K_ADD_OUT, BASIC_DEBUG, NEW_UNIQUE_EXTEND );
               double u4cclock = WallClockTime( );
               Sort(trhpairs);
               vec<pp_pair> trusted_pairs;
               for ( int i = 0; i < trhpairs.isize( ); i++ )
               {    int j;
                    for ( j = i + 1; j < trhpairs.isize( ); j++ )
                    {    if ( trhpairs[j].Edges1( ) != trhpairs[i].Edges1( ) ) break;
                         if ( trhpairs[j].Edges2( ) != trhpairs[i].Edges2( ) ) 
                              break;    }
                    static vec<HyperPairPlacement> hp;
                    hp.clear( );
                    for ( int u = i; u < j; u++ )
                         hp.push_back( trhpairs[u] );
                    vec<int> hpmult;
                    Reduce( hp, hpmult );
                    if ( j - i > 1 )
                    {    for ( int u = 0; u < hp.isize( ); u++ )
                         {    trusted_pairs.push_back( pp_pair( 
                                   pp_read( hp[u].Edges1( ) ),
                                   pp_read( hp[u].Edges2( ) ), 
                                   hp[u].Sep( ), hp[u].Dev( ) ) );    }    }
                    i = j - 1;    }
               if (BASIC_DEBUG)
               {    for ( int i = 0; i < trusted_pairs.isize( ); i++ )
                         cout << trusted_pairs[i] << "\n";    }
               cout << TimeSince(u4cclock) << " used on clock 4c" << endl;

               // Compute extra_ppp.  It consists of the pp_pairs derived from 
               // the pairs in the primary read cloud that are not in the secondary
               // read cloud.  

               vec<pp_pair> extra_ppp;
               extra_ppp_nonreduced.clear( );
               for ( int u = 0; u < extra_hpairs.isize( ); u++ )
               {    extra_ppp_nonreduced.push( pp_read( extra_hpairs[u].Edges1( ) ),
                         pp_read( extra_hpairs[u].Edges2( ) ), 
                         extra_hpairs[u].Sep( ), extra_hpairs[u].Dev( ) );    }
               ppp_nonreduced.clear( );
               for ( int u = 0; u < hpairs.isize( ); u++ )
               {    ppp_nonreduced.push( pp_read( hpairs[u].Edges1( ) ),
                         pp_read( hpairs[u].Edges2( ) ), 
                         hpairs[u].Sep( ), hpairs[u].Dev( ) );    }
               vec<HyperPairPlacement> extra_hpairs_sorted( extra_hpairs );
               Sort(extra_hpairs_sorted);
               for ( int i = 0; i < extra_hpairs_sorted.isize( ); i++ )
               {    int j;
                    for ( j = i + 1; j < extra_hpairs_sorted.isize( ); j++ )
                    {    if ( extra_hpairs_sorted[j].Edges1( ) 
                              != extra_hpairs_sorted[i].Edges1( ) ) break;
                         if ( extra_hpairs_sorted[j].Edges2( ) 
                              != extra_hpairs_sorted[i].Edges2( ) ) break;    }
                    static vec<HyperPairPlacement> hp;
                    hp.clear( );
                    for ( int u = i; u < j; u++ )
                         hp.push_back( extra_hpairs_sorted[u] );
                    vec<int> hpmult;
                    Reduce( hp, hpmult );
                    for ( int u = 0; u < hp.isize( ); u++ )
                    {    extra_ppp.push( pp_read( hp[u].Edges1( ) ),
                              pp_read( hp[u].Edges2( ) ), 
                              hp[u].Sep( ), hp[u].Dev( ) );    }
                    i = j - 1;    }
               if (BASIC_DEBUG)
               {    cout << "\nEXTRA PAIRS:\n";
                    for ( int i = 0; i < extra_ppp.isize( ); i++ )
                         cout << "[" << i << "] " << extra_ppp[i] << "\n";    }

               // Try to bridge gaps in trusted pairs.

               equiv_rel e;
               h.ComponentRelation(e);
               vec<basevector> bridges;
               vec<vrtx_t> to_left_vertex;
               h.ToLeft(to_left_vertex);
               for ( int i = 0; i < trusted_pairs.isize( ); i++ )
               {    const pp_pair& p = trusted_pairs[i];
                    if (BRIDGE_VERBOSE)
                    {    cout << "\ncontemplating trusted pair bridge:\n";
                         PRINT(p);    }
                    if ( p.Gap( ) < -10 ) continue;
                    double overlap = -(p.Gap( ) - K + 1), dev = p.Dev( );
                    int low = iceil( overlap - 4.0 * dev );
                    int high = ifloor( overlap + 4.0 * dev );
                    if (BRIDGE_VERBOSE) PRINT2( low, high );
                    if ( low <= 0 ) continue;
                    int v = to_right_vertex[ p.Left(0) ];
                    int w = to_left_vertex[ p.Right(0) ];
                    int c1 = e.ClassId(v), c2 = e.ClassId(w);
                    if (BRIDGE_VERBOSE) PRINT2( c1, c2 );
                    if ( c1 == c2 && !h.Sink(v) && !h.Source(w) ) continue;
                    int from = p.Left( ).back( ), to = p.Right( ).front( );
                    const KmerPath &f = h.EdgeObject(from), &t = h.EdgeObject(to);
                    basevector fb = kbbnb.Seq(f), tb = kbbnb.Seq(t);
                    if (BRIDGE_VERBOSE)
                    {    PRINT2( BaseAlpha(from), BaseAlpha(to) );
                         fb.Print( cout, "from" );
                         tb.Print( cout, "to" );    }
                    vec<int> overlaps;
                    for ( int j = low; j <= high; j++ )
                         if ( Overlap( fb, tb, j ) ) overlaps.push_back(j);
                    if (BRIDGE_VERBOSE)
                    {    PRINT( overlaps.size( ) );
                         CompactPrint( cout, overlaps );
                         cout << "\n";    }
                    if ( overlaps.size( ) != 1 ) continue;
                    int over = overlaps[0];
                    if ( over > K ) continue;
                    basevector bridge( 2 * K - over );
                    for ( int j = 0; j < K; j++ )
                    {    bridge.Set( j, fb[ fb.size( ) - K + j ] );
                         bridge.Set( j + K - over, tb[j] );    }
                    bridge.ReverseComplement( );
                    bridges.push_back(bridge);    }
               if ( restartedt <= 1 && bridges.nonempty( ) )
               {    for ( int i = 0; i < bridges.isize( ); i++ )
                    {    sreads.push_back_reserve( bridges[i] );
                         fw_reads.push_back(nsreads++);    }
                    cout << "restarting after adding "
                         << bridges.size( ) << " bridges from trusted pairs\n";
                    ++restartedt;
                    goto restart;    }

               // Generate pairs.

               double u4cclockx = WallClockTime( );
               UniqueSort(trusteds);
               if (BASIC_DEBUG)
               {    cout << "\nTRUSTED PATHS:\n";
                    for ( int i = 0; i < trusteds.isize( ); i++ )
                         if ( trusteds[i].size( ) > 0 ) cout << trusteds[i] << "\n";
                    cout << "\n";    }
               Sort(hpairs);
               vec<pp_pair> ppp, ppp_orig;
               pppL.clear( );
               for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
                    pppL.push_back( h.EdgeObject(i).KmerCount( ) );
               BinaryOverwrite( sub_dir + "/pairsL", pppL );
               cout << TimeSince(u4cclockx) << " used on clock 4cx" << endl;
               vec<int> pppmult;
               vec<Bool> conflicts_tr;
               MakePairedPairs1( sub_dir, hpairs, unique_edge, edge_copyno, 
                    hstart_known, hstart, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
                    short_insert_coverage, short_insert_readlength, 
                    short_insertlength, short_insertdev, trusteds, trusted_pairs, 
                    cout, ppp, pppL, TEST_GRAPH_EDGES, genome, kbbnb, 
                    ( SHOW_PAIR_PLACEMENTS >= 1 ? locs[v] : serfvec<placement>( ) ),
                    PRECOMPUTE_CLOSURE_LENGTHS, REDUCE_VERBOSE, 
                    MERGE_PAIRED_PAIRS_VERBOSE, SHOW_PAIR_PLACEMENTS, pppmult, 
                    conflicts_tr );
               ppp_orig = ppp;
               if ( USE_TRUTH && TEST_GRAPH_EDGES ) 
               {    FindFalseReads( ppp, h, genome, kbbnb, v, NHOOD_RADIUS, locs, 
                         genome_path_lengths, BASIC_DEBUG );    }

               // Trace elements of ppp back to the read pairs which could 
               // (in principle) originate them.

               double xxxclock = WallClockTime( );
               if ( USE_TRUTH && TRACE_GRAPH_EDGES )
                    TracePairs( sreads, ppp, h, kbbnb, slocs, spairs );
               BinaryOverwrite(sub_dir + "/pairs", ppp);

               // Attempt to bridge gaps.

               if ( restarted <= 1 )
               {    vec<basevector> bridges;
                    vec< vec<pp_closure> > ppclosures( ppp.size( ) );
                    BridgeGaps( ppp, ppclosures, h, kbbnb, bridges, BRIDGE_VERBOSE );
                    if ( bridges.nonempty( ) )
                    {    for ( int i = 0; i < bridges.isize( ); i++ )
                         {    sreads.push_back_reserve( bridges[i] );
                              fw_reads.push_back(nsreads++);    }
                         cout << "restarting after adding "
                              << bridges.size( ) << " bridges\n";
                         ++restarted;
                         cout << TimeSince(xxxclock) << " used babbling" << "\n";
                         goto restart;    }    }

               // Finish MakePairedPairs.

               cout << TimeSince(xxxclock) << " used babbling" << "\n";
               MakePairedPairs2( h, sub_dir, hpairs, unique_edge, edge_copyno, 
                    edge_copyno_p, hstart_known, hstart, NHOOD_RADIUS, 
                    NHOOD_RADIUS_INTERNAL, short_insert_coverage, 
                    short_insert_readlength, short_insertlength, short_insertdev, 
                    trusteds, trusted_pairs, cout, ppp, pppL, TEST_GRAPH_EDGES, 
                    genome, kbbnb, 
                    ( SHOW_PAIR_PLACEMENTS >= 1 ? locs[v] : serfvec<placement>( ) ),
                    CHEAT_BY_DELETING_INVALID_EDGES, DELETE_HANGING_ENDS, 
                    DELETE_POORLY_COVERED_PAIRS, POORLY_COVERED_PAIRS_VERBOSE,
                    MERGE_PAIRED_PAIRS_VERBOSE, MERGE_COPY_NUMBER_TWO, 
                    MERGE_COPY_NUMBER_TWO_VERBOSE, REMOVE_SOLO,
                    SHOW_PAIR_PLACEMENTS, pppmult, conflicts_tr );

               // Merge paired pairs.

               PRINT_PAIR_LOCS( "before first call to MergePairedPairs" );
               if (APPEND_LONG_PAIRS) ppp.append(extra_ppp);
               MergePairedPairs( ppp, h, unique_edge, edge_copyno, edge_copyno_p,
                    hstart, hstart_known, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
                    DMULT, pppL, cout, MERGE_PAIRED_PAIRS_VERBOSE, 
                    MERGE_COPY_NUMBER_TWO, MERGE_COPY_NUMBER_TWO_VERBOSE,
                    SHOW_PAIR_PLACEMENTS, sub_dir, kbbnb, genome,
                    ( SHOW_PAIR_PLACEMENTS ? locs[v] : serfvec<placement>( ) ) );
               double xxx2clock = WallClockTime( );
               PRINT_PAIR_LOCS( "before call to MergeCopyNumberTwoGreedy" );
               if (MERGE_COPY_NUMBER_TWO)
               {    vec<Bool> remove( ppp.size( ), False );
                    digraphE<int> G( h, pppL );
                    MergeCopyNumberTwoGreedy( h, kbbnb, ppp, pppL, to_left_vertex,
                         to_right_vertex, G, edge_copyno, remove,
                         MERGE_COPY_NUMBER_TWO_VERBOSE );
                    EraseIf( ppp, remove );    }
               if (GENERATE_PAIR_GRAPH)
               {    cout << "\nPair graph:\n";
                    digraph PG;
                    MakeGraphFromPairs( ppp, pppL, DMULT, PG );
                    for ( int x = 0; x < PG.N( ); x++ )
                    {    cout << "[" << x << "] -->";
                         for ( int j = 0; j < PG.From(x).isize( ); j++ )
                              cout << " [" << PG.From(x)[j] << "]";
                         cout << "\n";    }
                    Ofstream( out, sub_dir + "/pairs.dot" );
                    if ( !USE_TRUTH ) OutputPairsGraph( out, PG );
                    else
                    {    OutputTruthLabeledPairsGraph( out, PG, ppp, pppL, DMULT,
                              h, kbbnb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL,
                              locs[v], genome_path_lengths );    }    }
               TEST_EXIT_AT(MergePairedPairs.1);

               // Print pairs, showing predicted placements.

               if (BASIC_DEBUG) PrintPairs( ppp, hstart_known, hstart, unique_edge );
               PRINT_PAIR_LOCS( "before first TrimPairs" );
               TrimPairs( v, ulen, h, ppp, pppL, hstart_known, hstart, unique_edge, 
                    NHOOD_RADIUS_INTERNAL, TRIM_MULTIPLIER, found_universal,
                    BASIC_DEBUG );
               PRINT_PAIR_LOCS( "after first TrimPairs" );
               int min_universal_radius 
                    = int( floor( 0.7 * double(NHOOD_RADIUS_INTERNAL) ) );
               int universal = -1;
               cout << TimeSince(xxx2clock) << " used doing some stuff" << "\n";

               // Do first pass of closing.  If all the closures from a pair
               // contain unique edges and are subsumed by closures from
               // (certain!) others, then that pair can be deleted.  If a pair
               // has a unique closure, replace the pair by that closure.

               for ( int count_pre = 1; 
                    !found_universal && count_pre <= PAIRED_PAIRS_PRE; count_pre++ )
               {    
                    // Set aside pairs having a long gap or high deviation.

                    vec<pp_pair> ppp_short, ppp_long;
                    if (APPEND_LONG_PAIRS)
                    {    for ( int i = 0; i < ppp.isize( ); i++ )
                         {    if ( ppp[i].Gap( ) 
                                   <= 2.0 * double(MAX_SHORT_INSERT_SEP)
                                   && ppp[i].Dev( ) <= 200.0 )
                              {    ppp_short.push_back( ppp[i] );    }
                              else ppp_long.push_back( ppp[i] );    }
                         ppp = ppp_short;
                         cout << "\nTO CLOSE SHORT PAIRS:\n";
                         for ( int i = 0; i < ppp.isize( ); i++ )
                              cout << ppp[i] << "\n";
                         cout << "\n";    }

                    vec< vec<pp_closure> > ppclosures0, ppclosures0e;
                    double dmult = 3.0;
                    vec< vec<double> > devs0, devs0e;
                    vec<Bool> pairs_to_close( ppp.size( ), False );
                    for ( int j = 0; j < ppp.isize( ); j++ )
                    {    const pp_pair& p = ppp[j];
                         for ( int m = 0; m < p.LeftSize( ); m++ )
                         {    if ( unique_edge[ p.Left(m) ] )
                                   pairs_to_close[j] = True;    }
                         for ( int m = 0; m < p.RightSize( ); m++ )
                         {    if ( unique_edge[ p.Right(m) ] 
                                   && p.Gap( ) <= MAX_SHORT_INSERT_SEP ) 
                                   pairs_to_close[j] = True;    }    }
                    cout << "\npreclosing " << Sum(pairs_to_close)
                         << " pairs" << endl;
                    SimpleExtendPairedPairs( h, kbbnb, ppp, pppL, DMULT );
                    PRINT_PAIR_LOCS( "after SimpleExtendPairedPairs" );
                    vec<Bool> cfail, cfaile;

                    double close_pairs_timer0 = WallClockTime( );
                    ClosePairedPairsEasy( MAX_SHORT_INSERT_SEP, h, ppp, 
                         pairs_to_close, pppL, DMULT + DMULT_ADD0, ppclosures0e, 
                         devs0e, PAIRED_PAIRS_MAX_EXT, cfaile,
                         PAIRED_PAIRS_MAX_PROCESSED, PAIRED_PAIRS_MAX_UNPROCESSED, 
                         EASY_CLOSE_VERBOSITY, SIMPLE_WALK_VERBOSITY, 
                         MAX_OPENS_EASY, MAX_NODES, ACCEPT_INCOMPLETE_CLOSURE_SETS );
                    for ( int j = 0; j < cfaile.isize( ); j++ )
                         if ( !cfaile[j] ) pairs_to_close[j] = False;
                    CHANGE_TIMER( close_pairs_timer0,
                         "closing paired pairs (0)", close_pairs_timer1 );
                    ClosePairedPairs( ppp, pairs_to_close, pppL, 
                         DMULT + DMULT_ADD0, ppclosures0, devs0, 
                         PAIRED_PAIRS_MAX_EXT, cfail, PAIRED_PAIRS_MAX_PROCESSED,
                         PAIRED_PAIRS_MAX_UNPROCESSED, PAIRED_PAIRS_VERBOSITY,
                         SIMPLE_WALK_VERBOSITY, MAX_OPENS0, MAX_NODES, 
                         ACCEPT_INCOMPLETE_CLOSURE_SETS );
                    CHANGE_TIMER( close_pairs_timer1,
                         "closing paired pairs (1)", clock9 );
                    for ( int j = 0; j < cfaile.isize( ); j++ )
                    {    if ( !cfaile[j] )
                         {    ppclosures0[j] = ppclosures0e[j];
                              devs0[j] = devs0e[j];
                              pairs_to_close[j] = True;
                              cfail[j] = False;    }    }
                    TEST_EXIT_AT(PairedPairs);
                    for ( int j = 0; j < ppp.isize( ); j++ )
                    {    if ( ppclosures0[j].solo( ) && !cfail[j] )
                         {    const pp_closure& c = ppclosures0[j][0];
                              int len = 0;
                              for ( int m = 0; m < c.isize( ); m++ )
                                   len += pppL[ c[m] ];
                              if (BASIC_DEBUG)
                              {    cout << "PREMERGING: " << ppp[j] << " =====(" 
                                        << setprecision(3) << devs0[j][0] 
                                        << " devs)=====> ";    }
                              ppp[j] = pp_pair( c, c, -len, 0.0 );    
                              if (BASIC_DEBUG) cout << ppp[j] << "\n";    }    }
                    PRINT_PAIR_LOCS( "before deleting" );
                    vec<Bool> to_delete( ppp.size( ), False );
                    for ( int j1 = 0; j1 < ppp.isize( ); j1++ )
                    {    if ( cfail[j1] ) continue;
                         if ( ppclosures0[j1].empty( ) ) goto next_pair;
                         for ( int u1 = 0; u1 < ppclosures0[j1].isize( ); u1++ )
                         {    const pp_closure& c1 = ppclosures0[j1][u1];
                              Bool have_unique = False;
                              for ( int x = 0; x < c1.isize( ); x++ )
                              {    if ( unique_edge[ c1[x] ] ) 
                                   {    have_unique = True;
                                        break;    }    }
                              if ( !have_unique ) goto next_pair;    }
                         for ( int u1 = 0; u1 < ppclosures0[j1].isize( ); u1++ )
                         {    const pp_closure& c1 = ppclosures0[j1][u1];
                              for ( int j2 = 0; j2 < ppp.isize( ); j2++ )
                              {    if ( j2 == j1 || to_delete[j2] ) continue;
                                   if ( cfail[j2] ) continue;
                                   if ( !IsClosed( ppp[j2], pppL ) ) continue;
                                   if ( ppp[j2].Left( ) != ppp[j2].Right( ) )
                                        continue;
                                   for ( int u2 = 0; 
                                        u2 < ppclosures0[j2].isize( ); u2++ )
                                   {    const pp_closure& c2 = ppclosures0[j2][u2];
                                        static vec<int> offsets;
                                        GetOverlaps( c1, c2, offsets );
                                        for ( int m = 0; m < offsets.isize( ); m++ )
                                        {    if ( offsets[m] <= 0 
                                                  && offsets[m] + c2.isize( )
                                                       >= c1.isize( ) )
                                             {    goto c1_in_c2;    }    }    }    }
                              goto next_pair;
                              c1_in_c2: continue;    }
                         cout << "DELETING: " << ppp[j1] << "\n"; // QQQQQQQQQQ
                         to_delete[j1] = True;
                         next_pair: continue;    }

                    // Restore pairs having a long gap.

                    if (APPEND_LONG_PAIRS) ppp.append(ppp_long);

                    for ( int i = 0; i < ppp.isize( ); i++ )
                    {    pp_pair& p = ppp[i];
                         Bool start_known;
                         vec<int> lstart, rstart;
                         PropagateStarts( p, h, hstart_known, unique_edge, 
                              hstart, start_known, lstart, rstart );
                         if ( IsClosed( p, pppL ) )
                         {    Bool have_unique = False;
                              for ( int j = 0; j < p.LeftSize( ); j++ )
                              {    if ( unique_edge[ p.Left(j) ] ) 
                                        have_unique = True;    }
                              if ( have_unique && start_known
                                   && lstart.front( ) <= -min_universal_radius
                                   && rstart.back( ) 
                                        >= min_universal_radius + ulen[v] )
                              {    cout << "UNIVERSAL: pair " << i << "\n";
                                   found_universal = True;
                                   universal = i;    }    }    }
                    if ( universal >= 0 )
                    {    pp_pair p = ppp[universal];
                         ppp.resize(1);
                         ppp[0] = p;    }
                    else EraseIf( ppp, to_delete );
                    if ( ppp.size( ) == 1 )
                    {    cout << "Only one pair left after PAIRED_PAIRS_PRE!\n";
                         cout << "Call it UNIVERSAL.\n";
                         found_universal = True;    }
                    PRINT_PAIR_LOCS( "before second call to MergePairedPairs" );
                    MergePairedPairs( ppp, h, unique_edge, edge_copyno, 
                         edge_copyno_p, hstart, hstart_known, NHOOD_RADIUS, 
                         NHOOD_RADIUS_INTERNAL, DMULT, pppL, cout, 
                         MERGE_PAIRED_PAIRS_VERBOSE, MERGE_COPY_NUMBER_TWO, 
                         MERGE_COPY_NUMBER_TWO_VERBOSE, SHOW_PAIR_PLACEMENTS, 
                         sub_dir, kbbnb, genome, ( SHOW_PAIR_PLACEMENTS >= 1
                              ? locs[v] : serfvec<placement>( ) ) );
                    PRINT_PAIR_LOCS( "after second call to MergePairedPairs" );
                    TEST_EXIT_AT(MergePairedPairs.2);
                    TrimPairs( v, ulen, h, ppp, pppL, hstart_known, hstart, 
                         unique_edge, NHOOD_RADIUS_INTERNAL, TRIM_MULTIPLIER, 
                         found_universal, BASIC_DEBUG );
                    cout << TimeSince(clock9) << " on uni stuff" << "\n";
                    PRINT_PAIR_LOCS( "after second TrimPairs" );

                    // Print pairs, showing predicted placements.

                    if (BASIC_DEBUG) 
                         PrintPairs( ppp, hstart_known, hstart, unique_edge );    }

               // If we found a universal closure, use it.  Here we apply a more
               // stringent definition of universal - probably should do so earlier
               // too.  Or maybe it's not more stringent.

               found_universal_strong = found_universal && IsClosed( ppp[0], pppL );
               if (found_universal_strong)
               {    double usclock = WallClockTime( );
                    ProcessUniversalClosure( ppp, h, spairs, primary, xpaths, 
                         xpaths_rc, kbbnb, EVALUATE_NHOOD_HYPER, wrun_dir, data_dir,
                         hyperbases );
                    cout << TimeSince(usclock) 
                         << " used finalizing universal\n";    
                    continue;
               }

               // Set up for paired pairs computation.

               double yyyclock = WallClockTime( );
               MakePairedPairsAncillary( ppp, h, jpaths, jpairs );
               double dmult = 3.0;
               vecKmerPath jpaths_rc = jpaths;
               for ( int i = 0; i < jpaths_rc.size( ); i++ )
                    jpaths_rc[i].Reverse( );
               vec<tagged_rpint> jpathsdb;
               CreateDatabase( jpaths, jpaths_rc, jpathsdb );
               Bool using_paired_pairs = True;
               cout << TimeSince(yyyclock) << " used doing some other stuff" << "\n";
               if (EXTEND_PAIRS)
               {    cout << "\nEXTENSIONS:\n";
                    vec<pp_pair> extensions;
                    ExtendPairedPairs( ppp, pppL, dmult, extensions );
                    ppp = extensions;    }

               // Check for "false" pp_reads.

               if (USE_TRUTH) 
               {    FindFalseReads( ppp, h, genome, kbbnb, v, NHOOD_RADIUS, locs, 
                         genome_path_lengths, BASIC_DEBUG );    }

               // EXPERIMENT: add an artificial pair

               // pp_pair special( ppp[0].Right( ), ppp[8].Left( ), 797, 10 );
               // ppp.push_back(special);

               // Set aside pairs having a long gap or high deviation.

               if (APPEND_LONG_PAIRS)
               {    vec<pp_pair> ppp_short, ppp_long;
                    for ( int i = 0; i < ppp.isize( ); i++ )
                    {    if ( ppp[i].Gap( ) <= 2.0 * double(MAX_SHORT_INSERT_SEP)
                              && ppp[i].Dev( ) <= 200.0 )
                         {    ppp_short.push_back( ppp[i] );    }
                         else ppp_long.push_back( ppp[i] );    }
                    ppp = ppp_short;    }

               // Close paired pairs.

               cout << "\nCLOSING " << ppp.size( ) << " PAIRED PAIRS:" << endl;

               vec< vec<double> > devs;
               vec<Bool> pairs_to_close( ppp.size( ), True );
               if ( MIN_COPY_NUMBER_TO_CLOSE > 0 )
               {    pairs_to_close.clear( );
                    for ( int i = 0; i < ppp.isize( ); i++ )
                    {    const pp_pair& p = ppp[i];
                         Bool have_low = False;
                         for ( int j = 0; j < p.LeftSize( ); j++ )
                         {    int cn = edge_copyno[ p.Left(j) ];
                              if ( cn >= 1 && cn <= MIN_COPY_NUMBER_TO_CLOSE )
                                   have_low = True;    }
                         for ( int j = 0; j < p.RightSize( ); j++ )
                         {    int cn = edge_copyno[ p.Right(j) ];
                              if ( cn >= 1 && cn <= MIN_COPY_NUMBER_TO_CLOSE )
                                   have_low = True;    }
                         if ( IsClosed( p, pppL ) ) have_low = True;
                         pairs_to_close.push_back(have_low);    }    }
               else if ( ppp.isize( ) > PAIRED_PAIRS_MAX_TO_CLOSE_NONUNIQUE )
               {    pairs_to_close.clear( );
                    for ( int i = 0; i < ppp.isize( ); i++ )
                    {    const pp_pair& p = ppp[i];
                         Bool have_unique_or_closed = False;
                         for ( int j = 0; j < p.LeftSize( ); j++ )
                         {    if ( unique_edge[ p.Left(j) ] ) 
                                   have_unique_or_closed = True;    }
                         for ( int j = 0; j < p.RightSize( ); j++ )
                         {    if ( unique_edge[ p.Right(j) ] ) 
                                   have_unique_or_closed = True;    }
                         if ( IsClosed( p, pppL ) ) have_unique_or_closed = True;
                         pairs_to_close.push_back(have_unique_or_closed);    }    }

               vec<Bool> cfail, cfaile;
               vec< vec<pp_closure> > ppclosurese;
               vec< vec<double> > devse;
               double close_pairs_timer2 = WallClockTime( );
               ClosePairedPairsEasy( MAX_SHORT_INSERT_SEP, h, ppp, pairs_to_close, 
                    pppL, DMULT, ppclosurese, devse, PAIRED_PAIRS_MAX_EXT, cfaile, 
                    PAIRED_PAIRS_MAX_PROCESSED, PAIRED_PAIRS_MAX_UNPROCESSED, 
                    EASY_CLOSE_VERBOSITY, SIMPLE_WALK_VERBOSITY, MAX_OPENS_EASY,
                    MAX_NODES, ACCEPT_INCOMPLETE_CLOSURE_SETS );
               for ( int j = 0; j < cfaile.isize( ); j++ )
               {    if ( ppclosurese[j].empty( ) ) cfaile[j] = True;
                    if ( !cfaile[j] ) pairs_to_close[j] = False;    }
               ClosePairedPairs( ppp, pairs_to_close, pppL, DMULT, ppclosures, devs, 
                    PAIRED_PAIRS_MAX_EXT, cfail, PAIRED_PAIRS_MAX_PROCESSED, 
                    PAIRED_PAIRS_MAX_UNPROCESSED, PAIRED_PAIRS_VERBOSITY, 
                    SIMPLE_WALK_VERBOSITY, MAX_OPENS1, MAX_NODES, 
                    ACCEPT_INCOMPLETE_CLOSURE_SETS );

               /*
               FindClosures( ppp, pairs_to_close, ppclosures, h.Edges(), h.K(),
                             DMULT, ppclosures, cfail, MAX_OPENS1, 10, 10 );
               devs.resize( ppclosures.size() );
               */
               
               // EXPERIMENT:

               /*
               vec< vec<pp_closure> > eclosures;
               Bool create_closures_if_fail = True;
               vec<Bool> efail;
               vec<Bool> epairs_to_close(pairs_to_close);
               for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( ppp[i].Left( ).Length(pppL) < 500 )
                         epairs_to_close[i] = False;    }
               int everbosity = 2;
               int EOPENS = 100000;
               PairedWalkRight( ppp, epairs_to_close, pppL, DMULT, eclosures,
                    EOPENS, create_closures_if_fail, efail, everbosity, 
                    MAX_NODES );
               cout << "CLOSURE RESULTS:\n";
               for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( !epairs_to_close[i] ) continue;
                    cout << "\n[" << i << "] " << ppp[i] << "\n";
                    PRINT2( ppclosures[i].size( ), eclosures[i].size( ) );
                    ppclosures[i].append( eclosures[i] );
                    UniqueSort( ppclosures[i] );    }
               */

               // Used PairedWalkRight to extend closures.

               chunks.clear( );
               if (WALK_OFF_CLOSURES)
               {    vec<pp_closure> templates;
                    for ( int i = 0; i < ppclosures.isize( ); i++ )
                         templates.append( ppclosures[i] );
                    vec< vec<pp_closure> > eclosures1, eclosures2;
                    PairedWalkRight( ppp, templates, MAX_WALK_OFF, pppL, DMULT,
                         ppclosures, eclosures1, chunks,
                         MAX_WALK_OFF_OPENS, WALK_OFF_VERBOSITY, 
                         MAX_WALK_OFF_NODES );
                    PairedWalkLeft( ppp, templates, MAX_WALK_OFF, pppL, DMULT,
                         ppclosures, eclosures2, chunks,
                         MAX_WALK_OFF_OPENS, WALK_OFF_VERBOSITY, 
                         MAX_WALK_OFF_NODES );
                    static vecbasevector bases;
                    bases.clear( );
                    for ( int u = 0; u < chunks.isize( ); u++ )
                    {    const pp_read& r = chunks[u];
                         static KmerPath p;
                         p.Clear( );
                         for ( int v = 0; v < r.isize( ); v++ )
                              p.Append( h.EdgeObject( r[v] ) );
                         bases.push_back_reserve( kbbnb.Seq(p) );    }
                    vec<alignment_plus> aligns;
                    AlignNhoodStuffToReference( bases, v, NHOOD_RADIUS, locs,
                         genome, genome_path_lengths, aligns );
                    vec< vec<int> > aligns_index( chunks.size( ) );
                    for ( int i = 0; i < aligns.isize( ); i++ )
                         aligns_index[ aligns[i].Id1( ) ].push_back(i);
                    cout << "\nChunks found by walking off closures:\n";
                    for ( int i = 0; i < chunks.isize( ); i++ )
                    {    cout << "\n[" << i+1 << "]\n" << chunks[i] << "\n";
                         for ( int j = 0; j < aligns_index[i].isize( ); j++ )
                         {    const alignment_plus& a = aligns[ aligns_index[i][j] ];
                              cout << a.Id2( ) << "." << a.pos2( )
                                   << "-" << a.Pos2( ) << "\n";    }    }
                    cout << "\n";    }

               // Process SPECIAL_WALK argument.
               // Garbage: not implemented yet, may never be.

               /*
               if ( SPECIAL_WALK != "" )
               {    ForceAssertGe( special_g, 0 );
                    ForceAssertLt( special_g, genome.size( ) );
                    ForceAssertGe( special_a1, 0 );
                    ForceAssertLt( special_a1, special_b1 );
                    ForceAssertLe( special_b1, special_a2 );
                    ForceAssertLt( special_a2, special_b2 );
                    ForceAssertLe( special_b2, genome[special_g].isize( ) );
                    ForceAssertGe( special_d, 0.0 );
                    pp_pair special( ppclosures[special_n1][special_n2], 
                         ppclosures[special_n2][special_k2],
                         special_s + 20 - 1, special_d );
                    ppp_orig.push_back(special);
                    vec< vec<pp_closure> > eclosures;
                    Bool create_closures_if_fail = True;
                    vec<Bool> efail;
                    vec<Bool> epairs_to_close( ppp_orig.size( ), False );
                    epairs_to_close.back( ) = True;
                    int everbosity = 4;
                    int EOPENS = 100000;
                    PairedWalkRight( ppp_orig, epairs_to_close, pppL, DMULT, 
                         eclosures, EOPENS, create_closures_if_fail, efail, 
                         everbosity, MAX_NODES );
                    exit(0);    }
               */

               // For inserts with no closures, try again using higher DMULT.

               static vec<Bool> pairs_to_close2;
               pairs_to_close2 = pairs_to_close;
               for ( int j = 0; j < pairs_to_close.isize( ); j++ )
               {    if ( ppclosures[j].nonempty( ) || cfail[j] )
                         pairs_to_close2[j] = False;    }
               if ( Sum(pairs_to_close2) > 0 )
               {    static vec< vec<pp_closure> > ppclosures_save;
                    ppclosures_save = ppclosures;
                    static vec< vec<double> > devs_save;
                    devs_save = devs;
                    static vec<Bool> cfail_save;
                    cfail_save = cfail;
                    ClosePairedPairs( ppp, pairs_to_close2, pppL, DMULT + 0.5, 
                         ppclosures, devs, PAIRED_PAIRS_MAX_EXT, cfail, 
                         PAIRED_PAIRS_MAX_PROCESSED, PAIRED_PAIRS_MAX_UNPROCESSED,
                         PAIRED_PAIRS_VERBOSITY, SIMPLE_WALK_VERBOSITY, MAX_OPENS1,
                         MAX_NODES, ACCEPT_INCOMPLETE_CLOSURE_SETS );
                    for ( int j = 0; j < pairs_to_close.isize( ); j++ )
                    {    if ( pairs_to_close[j] && !pairs_to_close2[j] )
                         {    ppclosures[j] = ppclosures_save[j];
                              devs[j] = devs_save[j];    
                              cfail[j] = cfail_save[j];   }    }    }

               // For selected inserts with no closures, try again with larger
               // MAX_OPENS1.  We do the following:
               // 1. Find all copy number one unipaths.
               // 2. Find those that do not occur in any closure yet found.
               // 3. For each, find all pairs containing it.
               // 4. Pick pairs having lowest gap*dev.

               if (TRY_HARD)
               {    vec<Bool> used( h.EdgeObjectCount( ), False );
                    for ( int j = 0; j < pairs_to_close.isize( ); j++ )
                    {    for ( int l = 0; l < ppclosures[j].isize( ); l++ )
                         {    for ( int m = 0; m < ppclosures[j][l].isize( ); m++ )
                                   used[ ppclosures[j][l][m] ] = True;    }    }
                    vec<Bool> ppp_use( ppp.size( ), False );
                    for ( int e = 0; e < h.EdgeObjectCount( ); e++ )
                    {    if ( used[e] || !unique_edge[e] ) continue;
                         vec<int> epairs;
                         vec<double> sepdev;
                         for ( int i = 0; i < ppp.isize( ); i++ )
                         {    if ( Member( ppp[i].Left( ), e )
                                   || Member( ppp[i].Right( ), e ) )
                         {    epairs.push_back(i);    
                                   sepdev.push_back( 
                                        ppp[i].Gap( ) * ppp[i].Dev( ) );    }    }
                         if ( sepdev.nonempty( ) )
                         {    SortSync( sepdev, epairs );
                              int n;
                              for ( n = 1; n < sepdev.isize( ); n++ )
                                   if ( sepdev[n] > sepdev[n-1] ) break;
                              for ( int j = 0; j < n; j++ )
                              {    cout << "To recover copy-number-one edge "
                                        << BaseAlpha(e) << ", trying to close pair "
                                        << epairs[j] << ".\n";
                                   ppp_use[ epairs[j] ] = True;    }    }    }
                    pairs_to_close2 = pairs_to_close;
                    for ( int j = 0; j < pairs_to_close.isize( ); j++ )
                    {    if ( ppclosures[j].nonempty( ) ) pairs_to_close2[j] = False;
                         if ( !ppp_use[j] ) pairs_to_close2[j] = False;    }
                    if ( Sum(pairs_to_close2) > 0 )
                    {    static vec< vec<pp_closure> > ppclosures_save;
                         ppclosures_save = ppclosures;
                         static vec< vec<double> > devs_save;
                         devs_save = devs;
                         static vec<Bool> cfail_save;
                         cfail_save = cfail;
                         ClosePairedPairs( ppp, pairs_to_close2, pppL, DMULT + 0.5, 
                              ppclosures, devs, PAIRED_PAIRS_MAX_EXT, cfail, 
                              PAIRED_PAIRS_MAX_PROCESSED, 
                              PAIRED_PAIRS_MAX_UNPROCESSED, PAIRED_PAIRS_VERBOSITY, 
                              SIMPLE_WALK_VERBOSITY, MAX_OPENS1_TRY_HARD, 
                              MAX_NODES_TRY_HARD, ACCEPT_INCOMPLETE_CLOSURE_SETS );
                         for ( int j = 0; j < pairs_to_close.isize( ); j++ )
                         {    if ( pairs_to_close[j] && !pairs_to_close2[j] )
                              {    ppclosures[j] = ppclosures_save[j];
                                   devs[j] = devs_save[j];    
                                   cfail[j] = cfail_save[j];   }    }    }    }

               // Muck with closures.

               cout << TimeSince(close_pairs_timer2) 
                    << " closing paired pairs (2)" << "\n";
               for ( int j = 0; j < cfaile.isize( ); j++ )
               {    if ( !cfaile[j] )
                    {    ppclosures[j] = ppclosurese[j];
                         devs[j] = devse[j];
                         pairs_to_close[j] = True;
                         cfail[j] = False;    }    }

               if ( !ACCEPT_INCOMPLETE_CLOSURE_SETS )
               {    for ( int j = 0; j < ppp.isize( ); j++ )
                         if ( cfail[j] ) ppclosures[j].clear( );    }

               // Make closures into pairs and try closing again.

               vec<pp_pair> ppp2;
               pairs_to_close.clear( );
               vec<int> source;
               for ( int i = 0; i < ppp.isize( ); i++ )
               {    if ( ppclosures[i].empty( ) && !cfail[i] )
                    {    const pp_pair& p = ppp[i];
                         ppp2.push_back(p);
                         Bool have_unique = False;
                         for ( int j = 0; j < p.LeftSize( ); j++ )
                              if ( unique_edge[ p.Left(j) ] ) have_unique = True;
                         for ( int j = 0; j < p.RightSize( ); j++ )
                              if ( unique_edge[ p.Right(j) ] ) have_unique = True;
                         pairs_to_close.push_back(have_unique);
                         source.push_back(i);    }
                    else
                    {    for ( int j = 0; j < ppclosures[i].isize( ); j++ )
                         {    pp_read r = ppclosures[i][j];
                              int len = 0;
                              for ( int u = 0; u < r.isize( ); u++ )
                                   len += pppL[ r[u] ];
                              pp_pair p( r, r, -len, 0 );
                              ppp2.push_back(p);
                              pairs_to_close.push_back(False);    
                              source.push_back(i);    }    }    }
               /*
               if ( ppp2.isize( ) < PAIRED_PAIRS_ROUND2_MAX )
               {    vec< vec<pp_closure> > ppclosures2;
                    cout << "round 2: using " << ppp2.size( ) << " pairs" << endl;
                    PRINT_PAIR_LOCS( "before round 2" );
                    double close_pairs_timer3 = WallClockTime( );
                    ClosePairedPairs( ppp2, pairs_to_close, pppL, DMULT, 
                         ppclosures2, devs, PAIRED_PAIRS_MAX_EXT, cfail, 
                         PAIRED_PAIRS_MAX_PROCESSED, PAIRED_PAIRS_MAX_UNPROCESSED,
                         PAIRED_PAIRS_VERBOSITY, SIMPLE_WALK_VERBOSITY, MAX_OPENS,
                         MAX_NODES, ACCEPT_INCOMPLETE_CLOSURE_SETS );
                    cout << TimeSince(close_pairs_timer3) 
                         << " closing paired pairs (3)" << "\n";
                    if ( !ACCEPT_INCOMPLETE_CLOSURE_SETS )
                    {    for ( int j = 0; j < ppp2.isize( ); j++ )
                              if ( cfail[j] ) ppclosures2[j].clear( );    }

                    for ( int i = 0; i < ppp2.isize( ); i++ )
                    {    if ( pairs_to_close[i] )
                              ppclosures[ source[i] ] = ppclosures2[i];    }    }
               */

               // Try to extend closures.

               if (EXTEND_CLOSURES)
               {    ExtendClosures( ppp, pppL, ppclosures, DMULT, 
                         MAX_SHORT_INSERT_SEP );    }

               if (BASIC_DEBUG)
               {    for ( int i = 0; i < ppclosures.isize( ); i++ )
                    {    cout << "\nclosures of [" << i << "] " << ppp[i] << "\n";
                         for ( int j = 0; j < ppclosures[i].isize( ); j++ )
                         {    cout << "[" << j+1 << "] " 
                                   << ppclosures[i][j] << "\n";    }    }
                    cout << endl;    }

               // Show coverage of reference by closures and find
               // inserts having no true closure.

               double rclock = WallClockTime( );
               if (USE_TRUTH)
               {    CheckReferenceCoverage( v, ppclosures, h, kbbnb, NHOOD_RADIUS,
                         locs, genome, genome_path_lengths, BASIC_DEBUG, ppp,
                         SHOW_PAIR_PLACEMENTS, sub_dir );    }
               cout << TimeSince(rclock)
                    << " used finding closures on reference\n";

               // Attempt to bridge gaps.

               if ( restarted <= 1 )
               {    vec<basevector> bridges;
                    BridgeGaps( ppp, ppclosures, h, kbbnb, bridges, BRIDGE_VERBOSE );
                    if ( bridges.nonempty( ) )
                    {    for ( int i = 0; i < bridges.isize( ); i++ )
                         {    sreads.push_back_reserve( bridges[i] );
                              fw_reads.push_back(nsreads++);    }
                         cout << "restarting after adding "
                              << bridges.size( ) << " bridges\n";
                         ++restarted;
                         goto restart;    }    }

               vec<pp_closure> uniqs;
               const int max_for_semi_unique = 4;
               for ( int j = 0; j < ppclosures.isize( ); j++ )
               {    if ( ppclosures[j].isize( ) <= max_for_semi_unique )
                    {    for ( int l = 0; l < ppclosures[j].isize( ); l++ )
                              uniqs.push_back( ppclosures[j][l] );    }    }
               UniqueSort(uniqs);
               for ( int j = 0; j < uniqs.isize( ); j++ )
               {    static KmerPath p;
                    p.Clear( );
                    for ( int u = 0; u < uniqs[j].isize( ); u++ )
                         p.Append( h.EdgeObject( uniqs[j][u] ) );
                    ppclosures_unique.push_back( kbbnb.Seq(p) );    }
               for ( int j = 0; j < ppclosures.isize( ); j++ )
               {    for ( int u = 0; u < ppclosures[j].isize( ); u++ )
                    {    static KmerPath p;
                         p.Clear( );
                         for ( int t = 0; t < ppclosures[j][u].isize( ); t++ )
                              p.Append( h.EdgeObject( ppclosures[j][u][t] ) );
                         closures.push_back( kbbnb.Seq(p) );    }    }
               for ( int j = 0; j < chunks.isize( ); j++ )
               {    static KmerPath p;
                    p.Clear( );
                    for ( int t = 0; t < chunks[j].isize( ); t++ )
                         p.Append( h.EdgeObject( chunks[j][t] ) );
                    closures.push_back( kbbnb.Seq(p) );    }

               // Summarize results.

               cout << endl << "found " << closures.size( )
                    << " closures of short inserts" << endl;    
          }

          // Place long inserts on ppclosure_unique.

          cout << Date( ) << ": starting new stuff" << endl;
          vecbasevector zoo;
          for ( int i = 0; i < ppclosures_unique.isize( ); i++ )
               zoo.push_back( ppclosures_unique[i] );
          vec<read_pairing> zpairs;
          for ( int i = 0; i < pairs_to_use.isize( ); i++ )
          {    read_pairing p = pairs[ pairs_to_use[i] ];
               int id1 = p.id1, id2 = p.id2;
               p.id1 = zoo.size( );
               zoo.push_back_reserve( reads[id1] );
               p.id2 = zoo.size( );
               zoo.push_back_reserve( reads[id2] );    
               zpairs.push_back(p);    }
          vecKmerPath zpaths, zpaths_rc;
          vec<tagged_rpint> zpathsdb;
          ReadsToPathsCoreY( zoo, K, genome_size, zpaths, zpaths_rc, zpathsdb );
          
          KmerBaseBroker zookbb;
          zookbb.Initialize( K, zoo, zpaths, zpaths_rc, zpathsdb );

          vecKmerPath closure_paths;
          for ( int i = 0; i < ppclosures_unique.isize( ); i++ )
               closure_paths.push_back_reserve( zpaths[i] );
          vec<tagged_rpint> closure_pathsdb;
          CreateDatabase( closure_paths, closure_pathsdb );
          
          vec<Bool> closure_paths_to_use( closure_paths.size( ), True );
          vec< vec<int> > u;
          vec< vec<KmerPathLoc> > start, stop;
          PlacePairsOnPaths( zpairs, zpaths, zpaths_rc, closure_paths, 
               closure_paths_to_use, closure_pathsdb, K, u, start, stop );
          vec< vec<ho_interval> > zcov( closure_paths.size( ) );
          vec< vec<int> > zcovu( closure_paths.size( ) );
          for ( int i = 0; i < u.isize( ); i++ )
          {    for ( int j = 0; j < u[i].isize( ); j++ )
               {    int z = u[i][j];
                    const KmerPathLoc &from = start[i][j], &to = stop[i][j];
                    int low = DistMin( closure_paths[z].Begin( ), from );
                    int high = DistMin( closure_paths[z].Begin( ), to ) + 1;
                    zcov[z].push_back( ho_interval( low, high ) );    
                    zcovu[z].push_back(i);    }    }

          // Find closure paths that are "validated" by long inserts.  What this 
          // does now is not very smart.

          vec<basevector> validated_shorts;
          vec<KmerPath> validated_shorts_KP;
          if (ADD_VALIDATED_SHORTS)
          {    for ( int i = 0; i < closure_paths.size( ); i++ )
               {    vec<ho_interval> zzz;
                    for ( int j = 0; j < zcov[i].isize( ); j++ )
                    {    if ( zcov[i][j].Length( ) >= MIN_LONG_INSERT_SHORTS )
                              zzz.push_back( zcov[i][j] );    }
                    Sort(zzz);
                    if ( zzz.empty( ) ) continue;
                    int low = zzz[0].Start( ), high = zzz[0].Stop( );
                    for ( int j = 1; j < zzz.isize( ); j++ )
                    {    if ( zzz[j].Start( ) > high - 100 ) break;
                         high = Max( high, zzz[j].Stop( ) );    }
                    basevector b;
                    b.SetToSubOf( ppclosures_unique[i], low, high - low + K - 1 );
                    validated_shorts.push_back(b);
                    KmerPathLoc start = closure_paths[i].Begin() + low;
                    KmerPathLoc stop  = closure_paths[i].Begin() + (high-1);
                    KmerPath kp;
                    closure_paths[i].CopySubpath( start, stop, kp );
                    ForceAssertEq( kp.KmerCount() + K - 1, b.isize() );
                    validated_shorts_KP.push_back( kp );
               }    
          }

          // Use closure paths to prepare to pick long inserts.

          vec<int> zkeep;
          for ( int i = 0; i < closure_paths.size( ); i++ )
          {    vec<int> C;
               PickCover( zcov[i], C );
               // if (LONG_INSERT_CHOICE_VERBOSE)
               // {    cout << "\n";
               //      PRINT3( i, C.size( ), closure_paths[i].KmerCount( ) );    }
               for ( int j = 0; j < C.isize( ); j++ )
               {    int x = C[j];
                    if ( zcov[i][x].Length( ) >= 1000 )
                    {    if (LONG_INSERT_CHOICE_VERBOSE)
                         {    cout << "zkeep: selecting: " 
                                   << pairs[ pairs_to_use[ zcovu[i][x] ] ].id1 << "/"
                                   << pairs[ pairs_to_use[ zcovu[i][x] ] ].id2 
                                   << "\n";    }
                         zkeep.push_back( zcovu[i][x] );    }
                    // if (LONG_INSERT_CHOICE_VERBOSE)
                    // {    cout << zcovu[i][x] << " --> " 
                    //           << zcov[i][x] << "\n";    }
               }    }
          UniqueSort(zkeep);
          PRINT( zkeep.size( ) );
          cout << Date( ) << ": done" << endl;

          // Select a few long inserts to be walked.

          static vec<int> pairs_sample;
          pairs_sample.clear( );
          vec<int> usefirst;
          for ( int j = 0; j < use.isize( ); j++ )
               usefirst.push_back( use[j].first );
          if ( PAIRS_SAMPLE.Contains( "random:", 0 ) )
          {    int n = PAIRS_SAMPLE.After( "random:" ).Int( );
               ForceAssertGt( n, 0 );
               int fails = 0;
               
               while( pairs_sample.isize( ) < n && fails < 100 )
               {    ++fails;
                    int x = randomx( ) % pairs_to_use.size( );
                    int pi = pairs_to_use[x];
                    if ( Member( pairs_sample, pi ) ) continue;
                    const read_pairing& p = pairs[pi];
                    if ( p.sep + readLengths[p.id1] + readLengths[p.id2] 
                         < MIN_LONG_INSERT )
                    {    continue;    }
                    int u1 = BinPosition( usefirst, p.id1 );
                    int u2 = BinPosition( usefirst, p.id2 );
                    ForceAssertGe( u1, 0 );
                    ForceAssertGe( u2, 0 );
                    int outer = NHOOD_RADIUS_INTERNAL;
                    if ( usestart[u1] < -outer || usestart[u1] > ulen[v] + outer )
                         continue;
                    if ( usestart[u2] < -outer || usestart[u2] > ulen[v] + outer ) 
                         continue;
                    if (LONG_INSERT_CHOICE_VERBOSE)
                         PRINT4( p.id1, p.id2, usestart[u1], usestart[u2] );
                    fails = 0;
                    pairs_sample.push_back(pi);    }
               if (EXTRA_LONGS)
               {    for ( int i = 0; i < zkeep.isize( ); i++ )
                         pairs_sample.push_back( pairs_to_use[ zkeep[i] ] );
                    UniqueSort(pairs_sample);    }    }

          else if ( PAIRS_SAMPLE == "cover" ) {
            vec<ho_interval> covers;
            vec<int> pair_ids;
            for ( unsigned int ptui = 0; ptui < pairs_to_use.size(); ++ptui ) {
              int pi = pairs_to_use[ptui];
              const read_pairing& p = pairs[pi];
              int id1 = p.id1, id2 = p.id2;
              if ( readLengths[id1] + p.sep + readLengths[id2] < MIN_LONG_INSERT )
                continue;    
              int u1 = BinPosition( usefirst, id1 );
              int u2 = BinPosition( usefirst, id2 );
              ForceAssertGe( u1, 0 );
              ForceAssertGe( u2, 0 );
              if ( use[u1].second ) {
                if ( use[u2].second ) continue;
                swap( u1, u2 );
                swap( id1, id2 );
              }
              else
                if ( ! use[u2].second ) continue;
              int start1 = usestart[u1], end1 = usestart[u1] + readLengths[id1];
              int start2 = usestart[u2], end2 = usestart[u2] + readLengths[id2];
              if ( end1 >= start2 ) continue;
              int outer = NHOOD_RADIUS_INTERNAL;
              if ( end1 < -outer || start1 > ulen[v] + outer ) continue;
              if ( end2 < -outer || start2 > ulen[v] + outer ) continue;
              covers.push_back( ho_interval( end1, start2 ) );
              pair_ids.push_back( pi );
            }
            int minStart = covers.front().Start();
            for ( unsigned int coveri = 0; coveri < covers.size(); ++coveri )
              minStart = min( minStart, covers[coveri].Start() );
            for ( unsigned int coveri = 0; coveri < covers.size(); ++coveri )
              covers[coveri].Shift( -minStart );
            vec<int> picks;
            PickCover( covers, picks );
            sort( picks.begin(), picks.end() );
            for ( unsigned int picki = 0; picki < picks.size(); ++picki ) {
              int pi = pair_ids[ picks[picki] ];
              if ( LONG_INSERT_CHOICE_VERBOSE ) {
                const read_pairing& p = pairs[pi];
                int u1 = BinPosition( usefirst, p.id1 );
                int u2 = BinPosition( usefirst, p.id2 );
                PRINT4( p.id1, p.id2, usestart[u1], usestart[u2] );
              }
              pairs_sample.push_back( pi );
            }
          }
          else 
          {    static vec<int> readids;
               ParseIntSet( PAIRS_SAMPLE, readids );
               for ( int i = 0; i < readids.isize( ); i++ )
               {    int pi = pairs_index[ readids[i] ];
                    ForceAssertGe( pi, 0 );
                    if ( !BinMember( pairs_to_use, pi ) ) continue;
                    if (LONG_INSERT_CHOICE_VERBOSE) {
                         const read_pairing& p = pairs[pi];
                         int u1 = BinPosition( usefirst, p.id1 );
                         int u2 = BinPosition( usefirst, p.id2 );
                         PRINT4( p.id1, p.id2, usestart[u1], usestart[u2] );
                    }
                    pairs_sample.push_back(pi);    }    }
          int nps = pairs_sample.size( );
          vec<Bool> extended_far_enough( 2*nps, True );
          if (SHOW_LONG_READS)
          {    cout << "\nusing PAIRS_SAMPLE=\"{";
               for ( int i = 0; i < pairs_sample.isize( ); i++ )
               {    if ( i > 0 ) cout << ",";
                    cout << pairs[ pairs_sample[i] ].id1;    }
               cout << "}\"\n\n";    }

          // Translate pairs_sample into pp_pairs and read_pairings.

          vec<pp_pair> long_inserts;
          vec<read_pairing> long_insert_orig_pairs;
          vec<HyperPairPlacement> long_insert_hpairs;
          static vec<int> pairs_to_spairs, pairs_to_extra_spairs;
          if ( pairs_to_spairs.empty( ) )
               pairs_to_spairs.resize( pairs.size( ), -1 );
          if ( pairs_to_extra_spairs.empty( ) )
               pairs_to_extra_spairs.resize( pairs.size( ), -1 );
          for ( int i = 0; i < extra_hpairs_source.isize( ); i++ )
          {    pairs_to_extra_spairs[
                    extra_spairs_pairs_index[ extra_hpairs_source[i] ] ] = i;    }
          for ( int i = 0; i < hpairs_source.isize( ); i++ )
               pairs_to_spairs[ spairs_pairs_index[ hpairs_source[i] ] ] = i;
          for ( int i = 0; i < pairs_sample.isize( ); i++ )
          {    int e = pairs_to_extra_spairs[ pairs_sample[i] ];
               if ( e >= 0 ) 
               {    const pp_pair& p = extra_ppp_nonreduced[e];
                    {    long_inserts.push_back( extra_ppp_nonreduced[e] );
                         long_insert_orig_pairs.push_back( 
                              pairs[ pairs_sample[i] ] );
                         long_insert_hpairs.push_back( 
                              extra_hpairs[e] );    }    }    }

          if (TREAT_SHORT_AS_LONG)
          {    for ( int i = 0; i < pairs_sample.isize( ); i++ )
               {    int e = pairs_to_spairs[ pairs_sample[i] ];
                    if ( e >= 0 ) 
                    {    const pp_pair& p = ppp_nonreduced[e];
                         {    long_inserts.push_back( ppp_nonreduced[e] );
                              long_insert_orig_pairs.push_back( 
                                   pairs[ pairs_sample[i] ] );
                              long_insert_hpairs.push_back( 
                                   hpairs_unsorted[e] );    }    }    }    }

          for ( int i = 0; i < extra_hpairs_source.isize( ); i++ )
          {    pairs_to_extra_spairs[
                    extra_spairs_pairs_index[ extra_hpairs_source[i] ] ] = -1;    }
          for ( int i = 0; i < hpairs_source.isize( ); i++ )
               pairs_to_spairs[ spairs_pairs_index[ hpairs_source[i] ] ] = -1;

          // At this point we have the following data:
          // - HyperKmerPath h - the edges are the local unipaths
          // - KmerBaseBroker kbbnb - to get to base space
          // - pppL - lengths of local unipaths, in kmers (deducible from h)
          // - vec< vec<pp_closure> > ppclosures - the closures of the short inserts
          // - vec<pp_pair> extra_ppp_nonreduced - the long insert pairs
          // - vec<pp_pair> long_inserts - long insert pairs to be walked
          // - vec<read_pairing> long_insert_orig_pairs - original pairs 
          //                     whence long_inserts came

          if (BASIC_DEBUG) PRINT( long_inserts.size( ) );

          hyperbases.push_back( HyperBasevector(K) );

          if ( ! long_inserts.empty() ) {
            
            vecKmerPath newLocalUnipaths;
            KmerBaseBroker newLocalKBB;
            RenumberKmersInUnipaths( h.Edges(), kbbnb, wrun_dir, 
                  newLocalUnipaths, newLocalKBB );

            vec<HyperKmerPath> hypers;
            WalkLongInserts( hypers, long_inserts, long_insert_orig_pairs, 
                 long_insert_hpairs, ppclosures, newLocalUnipaths, newLocalKBB, K, 
                 LONG_INSERT_WALK_K, SD_MULT, MAX_PSEUDO_LONG,
                 search_limitx, answer_size_limitx,
                 LONG_INSERT_MIN_READ_LENGTH, LONG_INSERT_WALK_VERBOSITY, 
                 EVALUATE_INSERT_HYPER, readlocs, readlocs_index, data_dir, 
                 wdata_dir, genome, verbosity );

            // {
            //   HyperKmerPath allInserts( K, hypers );
            //   ofstream dotout( "allinserts.dot" );
            //   allInserts.PrintSummaryDOT0w( dotout );
            // }

            MergeNeighborhood( hyperbases.back(), hypers, newLocalKBB,
                                PRINT_HYPER_BEFORE_FIRST_MERGE,
                               FIRST_MERGE,
                                PRINT_HYPER_AFTER_FIRST_MERGE,
                                PRINT_HYPER_BEFORE_SECOND_MERGE,
                               SECOND_MERGE,
                                 MIN_OVERLAP, MIN_PROPER_OVERLAP,
                                PRINT_HYPER_AFTER_SECOND_MERGE,
                               FIRST_DELOOP,
                                 P, reads, pairs, pairs_index, FIRST_DELOOP_VERBOSE,
                                PRINT_HYPER_AFTER_FIRST_DELOOP,
                               EVALUATE_NHOOD_HYPER,
                                 data_dir, wdata_dir, genome );    }

          if (ADD_VALIDATED_SHORTS)
          {    HyperBasevector hv( K, validated_shorts );
               vec<HyperBasevector> stuff;
               stuff.push_back( hyperbases.back( ), hv );
               hyperbases.back( ).SetToDisjointUnionOf(stuff);    
               HyperKmerPath stuffHKP( K, validated_shorts_KP );
               if (EVALUATE_NHOOD_HYPER) {
                 vec<look_align> aligns;
                 vec< vec<int> > aligns_index;
                 vec<TrustedPath> trusted_paths;
                 double eclock2 = WallClockTime( );
                 AlignHyperKmerPath( stuffHKP, &zookbb, data_dir + "/genome", 
                                     wdata_dir + "/run", aligns, aligns_index );
                 FilterAligns( stuffHKP, aligns, aligns_index, trusted_paths );
                 cout << "\nComparison of validated shorts to reference\n";
                 PrintAlignedHyperKmerPath( cout, stuffHKP, &zookbb, genome, aligns,
                                            aligns_index, True, &trusted_paths );    
               }
               
          }
            
          cout << "total nhood time = " << TimeSince(nhood_all_clock) << endl;    
       } // for each seed, assemble the seed's neighborhood into a HyperKmerPath

     if ( EVAL_KMER_ADJS ) {
       cout << "only evaluated kmer adjacencies, stopping..." << endl;
       exit(0);
     }
       

     BinaryOverwrite( sub_dir + "/closures.bases", hyperbases );
     Remove( sub_dir + "/current_easy_graph" );
     TEST_EXIT_AT(hyperbases);
     LocalizeReadsTail( sub_dir, hyperbases, K, KS, wrun_dir, 
          MIN_OVERLAP_FINAL, MIN_PROPER_OVERLAP_FINAL, GLOBAL_CLEAN,
          SHOW_PAIR_ALIGNS, pairs, pairs_index, partner, reads, gkbb, unipaths, 
          ulocs, ulocs_indexr, MAX_SHORT_INSERT_SEP, FILTER_ALIGNS, genome,
          FINAL_MERGE, USE_TRUTH, data_dir, nreads, BASIC_DEBUG, predicted_copyno,
          MIN_COMPONENT, readlocs, readlocs_index, DIPLOID, False, False, False,
          TRACK_GLOBAL_CLEAN_CHANGES, SHORTEST_MERGE, NEW_POORLY_COVERED );

     // Summarize coverage of neighborhoods if simulated data.
     
     if (USING_READLOCS)
     {    cout << "\nSUMMARY: ";
          if ( sim_tried > 0 )
          {    cout << PERCENT_RATIO( 3, sim_covered, sim_tried )
                    << " of neighborhoods are covered\n";    }
          else cout << " 0 of 0 are covered\n";
          if ( sim_tried_sh > 0 )
          {    cout << PERCENT_RATIO( 3, sim_covered_sh, sim_tried_sh )
                    << " of neighborhoods are covered by short inserts\n";    }
          else cout << " 0 of 0 are covered by short inserts\n";    }
     
     // Bye.


     if ( PARALLEL_CONTROL != "" ) {
       Ofstream( slave_finished, sub_dir + "/SLAVE_FINISHED"); 
     }

     if ( !GLOBAL_CLEAN ) cout << "\nWarning: GLOBAL_CLEAN not run.\n";
     
     cout << "\nTime for entire run: " << TimeSince(all_clock) << endl;    }
