/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef KMER_PATH_ON_HYPER_H
#define KMER_PATH_ON_HYPER_H

#include "CoreTools.h"
#include "ReadLocation.h"
#include "ReadPairing.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerPath.h"
#include "paths/PairedPair.h"
#include "paths/simulation/Placement.h"

// A KmerPathOnHyper represents a placement of a KmerPath p on a HyperKmerPath, 
// extending to both ends of p.  Such a placement is recorded as a sequence of 
// edges, together with a left extension and right extension, representing the 
// number of kmers that the edge sequence extends p by.

class KmerPathOnHyper {

     public:

     KmerPathOnHyper( ) { }
     KmerPathOnHyper( const vec<int>& edge, int left_ext, int right_ext )
          : edge_(edge), left_ext_(left_ext), right_ext_(right_ext) { }

     int NEdges( ) const { return edge_.size( ); }
     int Edge( int j ) const { return edge_[j]; }
     const vec<int>& Edges( ) const { return edge_; }
     int LeftExt( ) const { return left_ext_; }
     int RightExt( ) const { return right_ext_; }
     void ExtendLeft( int e, int e_length )
     {    edge_.push_front(e);
          left_ext_ += e_length;    }
     void ExtendRight( int e, int e_length )
     {    edge_.push_back(e);
          right_ext_ += e_length;    }

     // UniqueExtend extends the edge sequence as far to the left and right as
     // one can go unambiguously in the graph.  However, stop if cycling is
     // encountered.

     void UniqueExtend( const HyperKmerPath& h, const vec<vrtx_t>& to_left_vertex,
          const vec<vrtx_t>& to_right_vertex, const Bool NEW_UNIQUE_EXTEND = False );

     // Convert edge sequence into a dot-separated string of letters.

     String Alpha( ) const;

     private:

     vec<int> edge_;
     int left_ext_;
     int right_ext_;

};

void TranslateTrustedPaths( const vecKmerPath& paths, const HyperKmerPath&,
     vec<pp_read>& placements, const Bool NEW_UNIQUE_EXTEND = False );

// A HyperPairPlacement describes the placement of a read pair on a HyperKmerPath,
// as represented by two sequences of edges, and the predicted separation of the
// two sequences in kmers.

class HyperPairPlacement {

     public:

     HyperPairPlacement( ) { }

     HyperPairPlacement( const HyperKmerPath& h, const KmerPathOnHyper& r1, 
          const KmerPathOnHyper& r2, const read_pairing& p );

     const HyperKmerPath& H( ) const { return *h_; }
     const vec<int>& Edges1( ) const { return edges1_; }
     const vec<int>& Edges2( ) const { return edges2_; }
     int Edges1( int j ) const { return edges1_[j]; }
     int Edges2( int j ) const { return edges2_[j]; }
     double Sep( ) const { return sep_; }
     double Dev( ) const { return dev_; }
     int LeftExt1( ) const { return left_ext1_; }
     int RightExt1( ) const { return right_ext1_; }
     int LeftExt2( ) const { return left_ext2_; }
     int RightExt2( ) const { return right_ext2_; }

     void ExtendLeft( int e, int elen )
     {    edges2_.push_front(e);
          left_ext2_ += elen;
          sep_ -= elen;    }
     void ExtendRight( int e, int elen )
     {    edges1_.push_back(e);
          right_ext1_ += elen;
          sep_ -= elen;    }

     KmerPath EdgePath1F( ) const;
     KmerPath EdgePath2R( ) const;

     friend Bool operator<( 
          const HyperPairPlacement& p1, const HyperPairPlacement& p2 )
     {    if ( p1.edges1_ < p2.edges1_ ) return True;
          if ( p1.edges1_ > p2.edges1_ ) return False;
          if ( p1.edges2_ < p2.edges2_ ) return True;
          if ( p1.edges2_ > p2.edges2_ ) return False;
          return False;    }

     friend ostream& operator<<( ostream& o, const HyperPairPlacement& p );

     // Reduce: given a list of HyperPairPlacements, for the same edge data, and
     // having the same deviations, combine those placements that appear to arise
     // from the same mean separation.  The algorithm is a weak attempt at
     // one-dimensional clustering.

     friend void Reduce( vec<HyperPairPlacement>& r, vec<int>& rmult );

     private:

     const HyperKmerPath* h_;
     vec<int> edges1_, edges2_;
     double sep_, dev_;
     int left_ext1_, left_ext2_, right_ext1_, right_ext2_;

};

// PlacePairs: Convert read_pairings into HyperPairPlacements.

void PlacePairs( const vec<read_pairing>& pairs, Bool PRINT_SECONDARY_CLOUD, 
     const vec< pair<read_location,read_location> >& pairs_locs,
     const vecKmerPath& paths,
     const vecKmerPath& paths_rc, const HyperKmerPath& h, 
     const vec<tagged_rpint>& edgedb, const vec<vrtx_t>& to_right_vertex,
     vec<HyperPairPlacement>& hpairs, vec<int>& hpairs_source,
     const vec<Bool>& unique_edge, int k_add_in = 0, int k_add_out = 0,
     Bool verbose = False, const Bool NEW_UNIQUE_EXTEND = False );

void PlaceTrustedPairs( const vec<read_pairing>& pairs, const vecKmerPath& paths,
     const vecKmerPath& paths_rc, const HyperKmerPath& h,        
     const vec<tagged_rpint>& edgedb, const vec<vrtx_t>& to_right_vertex,
     const vec<Bool>& uniform, vec<pp_pair>& pppt );

void MakePairedPairs1(
          
     /* input:   */   const String& sub_dir,
                      const vec<HyperPairPlacement>&  hpairs,  // must be sorted!
                      vec<Bool>& unique_edge,   /* Edited! */
                      const vec<int>& edge_copyno,
                      const vec<Bool>& hstart_known,
                      const vec<int>& hstart,
                      const int NHOOD_RADIUS,
                      const int NHOOD_RADIUS_INTERNAL,
                      const double short_insert_coverage,
                      const double short_insert_readlength,
                      const double short_insertlength,
                      const double short_insertdev,
                      const vec<pp_read>& trusteds,
                      const vec<pp_pair>& trusted_pairs,
          
     /* outputs: */   ostream& out,             // for reporting
                      vec<pp_pair>& ppp,        // the pp_pairs
                      vec<int>& pppL,           // edge lengths
                      
     /* optional: */  Bool test_valid,             // compare to genome
                      const vecbasevector& genome, // the genome 
                      const KmerBaseBroker& kbb,   // to compare to genome
                      const serfvec<placement>& locsv,
                      Bool PRECOMPUTE_CLOSURE_LENGTHS,
                      Bool REDUCE_VERBOSE,
                      Bool MERGE_PAIRED_PAIRS_VERBOSE,
                      int SHOW_PAIR_PLACEMENTS,

     /* other: */     vec<int>& pppmult,
                      vec<Bool>& conflicts_tr

          );

void MakePairedPairs2(
          
     /* input:   */   const HyperKmerPath& h,
                      const String& sub_dir,
                      const vec<HyperPairPlacement>&  hpairs,  // must be sorted!
                      vec<Bool>& unique_edge,   /* Edited! */
                      const vec<int>& edge_copyno,
                      const vec<double>& edge_copyno_p,
                      const vec<Bool>& hstart_known,
                      const vec<int>& hstart,
                      const int NHOOD_RADIUS,
                      const int NHOOD_RADIUS_INTERNAL,
                      const double short_insert_coverage,
                      const double short_insert_readlength,
                      const double short_insertlength,
                      const double short_insertdev,
                      const vec<pp_read>& trusteds,
                      const vec<pp_pair>& trusted_pairs,
          
     /* outputs: */   ostream& out,             // for reporting
                      vec<pp_pair>& ppp,        // the pp_pairs
                      vec<int>& pppL,           // edge lengths
                      
     /* optional: */  Bool test_valid,             // compare to genome
                      const vecbasevector& genome, // the genome 
                      const KmerBaseBroker& kbb,   // to compare to genome
                      const serfvec<placement>& locsv,
                      Bool CHEAT_BY_DELETING_INVALID_EDGES,
                      Bool DELETE_HANGING_ENDS,
                      Bool DELETE_POORLY_COVERED_PAIRS,
                      Bool POORLY_COVERED_PAIRS_VERBOSE,
                      Bool MERGE_PAIRED_PAIRS_VERBOSE,
                      Bool MERGE_COPY_NUMBER_TWO,
                      Bool MERGE_COPY_NUMBER_TWO_VERBOSE,
                      Bool REMOVE_SOLO,
                      int SHOW_PAIR_PLACEMENTS,

     /* other: */     vec<int>& pppmult,
                      vec<Bool>& conflicts_tr

          );

void MakePairedPairsAncillary(

     /* input:   */  const vec<pp_pair>& ppp,  // the pp_pairs
                     const HyperKmerPath& h,   // the HyperKmerPath

     /* outputs: */  vecKmerPath& paths,       // KmerPaths of reads in the pairs
                     vec<read_pairing>& pairs  // the pairs
          );

void MergePairedPairs(

     /* both:     */  vec<pp_pair>& ppp,        // the pp_pairs

     /* input:    */  const HyperKmerPath& h,
                      const vec<Bool>& unique_edge,
                      const vec<int>& edge_copyno,
                      const vec<double>& edge_copyno_p,
                      const vec<int>& hstart,
                      const vec<Bool>& hstart_known,
                      const int nhood_radius,
                      const int nhood_radius_internal,
                      const double dmult,
                      vec<int>& pppL,           // edge lengths

     /* output:   */  ostream& out,             // for reporting

     /* control:  */  const Bool verbose,
                      const Bool MERGE_COPY_NUMBER_TWO,
                      const Bool MERGE_COPY_NUMBER_TWO_VERBOSE,

     /* for pair  */  const int SHOW_PAIR_PLACEMENTS,
     /* placement */  const String& sub_dir,
     /* printing: */  const KmerBaseBroker& kbb,
                      const vecbasevector& genome,
                      const serfvec<placement>& locsv

          );

#endif
