/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include <strstream>

#include "CoreTools.h"
#include "Equiv.h"
#include "math/Functions.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "ReadLocation.h"
#include "ReadPairing.h"
#include "VecOverlap.h"
#include "graph/DigraphTemplate.h"
#include "paths/BridgeGaps.h"
#include "paths/FindClosures.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerPath.h"
#include "paths/KmerPathOnHyper.h"
#include "paths/MergeAndTighten.h"
#include "paths/MergeCopyNumberTwo.h"
#include "paths/PairGraph.h"
#include "paths/PairedPair.h"
#include "paths/PoorlyCoveredPairs.h"
#include "paths/UniqueExtend.h"
#include "paths/simulation/Placement.h"

// not used but possibly useful
inline basevector Seq( 
const pp_read& p, const HyperKmerPath& h, const KmerBaseBroker& kbb )
{    KmerPath x; 
     for ( int i = 0; i < p.isize( ); i++ )
          x.Append( h.EdgeObject( p[i] ) );
     return kbb.Seq(x);    }

class partial_placement { // Used only in PlaceKmerPath, below.

     public:

     partial_placement( ) { }
     partial_placement( const KmerPathLoc& pos_on_p, int v, 
          const vec<int>& edges_so_far, int left_ext ) : pos_on_p(pos_on_p), 
          v(v), edges_so_far(edges_so_far), left_ext(left_ext) { }

     KmerPathLoc pos_on_p;    // position on p, matching with:
     int v;                   // vertex of HyperKmerPath
     vec<int> edges_so_far;   // to left of pos_on_p
     int left_ext;            // how much these extend p to left

};

// Given a gap-free KmerPath p, find all its placements on a gap-free
// HyperKmerPath h.  It needs two prebuilt structures: edgedb (made from h's
// edges), and to_right_vertex, that maps each edge to its right vertex.

void PlaceKmerPath( const KmerPath& p, const HyperKmerPath& h,
     const vec<tagged_rpint>& edgedb, const vec<vrtx_t>& to_right_vertex, 
     const vec<int>& edgelengths, vec<KmerPathOnHyper>& places,
     const vec< vec<int> >& left_sub_edgelengths = vec< vec<int> >( ) )
{    longlong x = p.FirstSegment( ).Start( );
     static vec<longlong> xplaces;
     static vec<partial_placement> unprocessed;
     places.clear( ), unprocessed.clear( );
     Contains( edgedb, x, xplaces );
     int plen = p.KmerCount( );
     for ( int i = 0; i < xplaces.isize( ); i++ )
     {    const tagged_rpint& t = edgedb[ xplaces[i] ];
          int e = t.PathId( );
          int left_ext = x - t.Start( );
          if ( left_sub_edgelengths.nonempty( ) )
               left_ext += left_sub_edgelengths[e][ t.PathPos( ) ];
          else
          {    for ( int j = 0; j < t.PathPos( ); j++ )
                    left_ext += h.EdgeObject(e).Length(j);    }
          int to_right = edgelengths[e] - left_ext;
          static vec<int> edge(1);
          edge[0] = e;
          if ( plen <= to_right )
          {    places.push_back( KmerPathOnHyper( 
                    edge, left_ext, to_right - plen ) );    }
          else
          {    unprocessed.push_back( partial_placement( p.Begin( ) + to_right,
                    to_right_vertex[e], edge, left_ext ) );    }    }
     while( unprocessed.nonempty( ) )
     {    partial_placement X = unprocessed.back( );
          unprocessed.resize( unprocessed.size( ) - 1 );
          for ( int i = 0; i < h.From(X.v).isize( ); i++ )
          {    int w = h.From(X.v)[i];
               int e = h.EdgeObjectIndexByIndexFrom( X.v, i );
               const KmerPathLoc& start = X.pos_on_p;
               static vec<int> edge;
               edge = X.edges_so_far;
               int left_ext = X.left_ext;
               longlong x = start.GetKmer( );
               if ( x != h.EdgeObject(e).Start(0) ) continue;
               if ( !ProperOverlapExt( p, h.EdgeObject(e), start.GetIndex( ), 0 ) )
                    continue;
               int kmers_on_p_to_right = plen - start.GetLoc( );
               for ( int j = 0; j < start.GetIndex( ); j++ )
                    kmers_on_p_to_right -= p.Length(j);
               edge.push_back(e);
               int elen = edgelengths[e];
               if ( kmers_on_p_to_right <= elen )
               {    places.push_back( KmerPathOnHyper( edge, left_ext,
                         elen - kmers_on_p_to_right ) );    }
               else
               {    unprocessed.push_back( partial_placement( start + elen, w,
                         edge, left_ext ) );    }    }    }    }

void KmerPathOnHyper::UniqueExtend( const HyperKmerPath& h, 
     const vec<vrtx_t>& to_left_vertex, const vec<vrtx_t>& to_right_vertex,
     const Bool NEW_UNIQUE_EXTEND )
{    vrtx_t w = to_right_vertex[ edge_.front( ) ];
     vrtx_t v = -1;
     for ( int i = 0; i < h.To(w).isize( ); i++ )
     {    int e = h.EdgeObjectIndexByIndexTo( w, i );
          if ( e == edge_[0] )
          {    v = h.To(w)[i];
               break;    }    }
     static vec<vrtx_t> added;
     added.clear( );
     while( h.To(v).size( ) == 1 )
     {    if (NEW_UNIQUE_EXTEND)
          {    if ( h.To(v)[0] == v ) break;    }
          int e = h.EdgeObjectIndexByIndexTo( v, 0 );
          edge_.insert( edge_.begin( ), e );
          left_ext_ += h.EdgeObject(e).KmerCount( );
          added.push_back(v);
          v = h.To(v)[0];    
          if ( Member( added, v ) ) break;    }
     w = to_right_vertex[ edge_.back( ) ];
     added.clear( );
     while( h.From(w).size( ) == 1 )
     {    if (NEW_UNIQUE_EXTEND)
          {    if ( h.From(w)[0] == w ) break;    }
          int e = h.EdgeObjectIndexByIndexFrom( w, 0 );
          edge_.push_back(e);
          right_ext_ += h.EdgeObject(e).KmerCount( );
          added.push_back(w);
          w = h.From(w)[0];    
          if ( Member( added, w ) ) break;    }    }

String KmerPathOnHyper::Alpha( ) const
{    String answer;
     for ( int j = 0; j < NEdges( ); j++ )
     {    if ( j > 0 ) answer += ".";
          answer += BaseAlpha( Edge(j) );    }
     return answer;    }

void Reduce( vec<HyperPairPlacement>& r, vec<int>& rmult )
{    
     // Sanity checks.

     int N = r.size( );
     ForceAssert( N > 0 );
     for ( int i = 1; i < N; i++ )
     {    ForceAssert( r[i].edges1_ == r[0].edges1_ );
          ForceAssert( r[i].edges2_ == r[0].edges2_ );    }
     double mindev = r[0].Dev( ), maxdev = r[0].Dev( );
     for ( int i = 0; i < N; i++ )
     {    mindev = Min( mindev, r[i].Dev( ) );
          maxdev = Max( maxdev, r[i].Dev( ) );    }
     double dev = ( mindev + maxdev ) / 2.0;

     // Form clusters by first finding the maximum deviation d and then splitting
     // wherever there is a gap of > 4d.  However, partially disintegrate clusters 
     // if they appear to be heterogeneous.

     static vec< vec<int> > clusters;
     {    static vec< pair<double, int> > sep_ind;
          sep_ind.resize(N);
          double d = 0.0;
          for ( int i = 0; i < N; i++ )
          {    d = Max( d, r[i].Dev( ) );
               sep_ind[i] = make_pair( r[i].Sep( ), i );    }
          Sort(sep_ind);
          clusters.clear( );
          for ( int i = 0; i < N; i++ )
          {    int j;
               for ( j = i + 1; j < N; j++ )
                    if ( sep_ind[j].first - sep_ind[j-1].first > 4.0 * d ) break;
               static vec<int> c;
               c.clear( );
               for ( int u = i; u < j; u++ )
                    c.push_back( sep_ind[u].second );

               // Test for heterogeneity: take middle 2/3 of data.  Is gap more
               // than 4 devs?  If so break into 3 dev wide clusters.

               int n = j - i;
               double gap = sep_ind[ i + (5*n)/6 ].first - sep_ind[ i + n/6 ].first;
               if ( gap/d > 4.0 )
               {    static vec<int> b;
                    b.clear( );
                    double start = 0.0;
                    for ( int u = i; u < j; u++ )
                    {    if ( b.empty( ) ) start = sep_ind[u].first;
                         else if ( sep_ind[u].first - start > 3.0 * d )
                         {    clusters.push_back(b);
                              b.clear( );
                              start = sep_ind[u].first;    }
                         b.push_back( sep_ind[u].second );    }
                    if ( b.nonempty( ) ) clusters.push_back(b);    }
               else clusters.push_back(c);

               i = j - 1;    }    }

     // Create merged HyperPairPlacements.

     static vec<HyperPairPlacement> rnew;
     rnew.clear( ), rmult.clear( );
     for ( int i = 0; i < clusters.isize( ); i++ )
     {    vec<int>& c = clusters[i];
          Sort(c);
          int n = c.size( );
          rmult.push_back(n);
          double s = 0.0, dd = 0.0;
          for ( int i = 0; i < n; i++ )
               s += r[ c[i] ].Sep( );
          s /= double(n);
          for ( int i = 0; i < n; i++ )
          {    double m = r[ c[i] ].Sep( ) - s;
               dd += m * m;    }
          double d0 = sqrt( dd / double(n) );
          double d1;
          double dev = 0;
          for ( int i = 0; i < n; i++ )
               dev += r[ c[i] ].Dev( ) / double(n);
          if ( d0 <= dev ) d1 = dev;
          else d1 = dev + 1.5 * ( d0 - dev );
          HyperPairPlacement p;
          p.h_ = &r[0].H( );
          p.edges1_ = r[0].edges1_;
          p.edges2_ = r[0].edges2_;
          p.sep_ = s;
          p.dev_ = d1 / sqrt( double(n) );

          // If the median is too far away, fix.

          p.dev_ = Max( p.dev_, Abs( r[ c[n/2] ].Sep( ) - s ) / 2.0 );

          rnew.push_back(p);    }
     r = rnew;    }
     
KmerPath HyperPairPlacement::EdgePath1F( ) const
{    KmerPath p;
     for ( int i = 0; i < edges1_.isize( ); i++ )
          p.Append( h_->EdgeObject( edges1_[i] ) );
     return p;    }

KmerPath HyperPairPlacement::EdgePath2R( ) const
{    KmerPath p;
     for ( int i = 0; i < edges2_.isize( ); i++ )
          p.Append( h_->EdgeObject( edges2_[i] ) );
     p.Reverse( );
     return p;    }

HyperPairPlacement::HyperPairPlacement( const HyperKmerPath& h, 
     const KmerPathOnHyper& r1, const KmerPathOnHyper& r2, const read_pairing& p )
{    h_ = &h;
     edges1_ = r1.Edges( );
     edges2_ = r2.Edges( );
     sep_ = p.sep - r1.RightExt( ) - r2.LeftExt( ) + h.K( ) - 1;
     dev_ = p.sd;
     left_ext1_ = r1.LeftExt( );
     right_ext1_ = r1.RightExt( );
     left_ext2_ = r2.LeftExt( );
     right_ext2_ = r2.RightExt( );    }

ostream& operator<<( ostream& o, const HyperPairPlacement& p )
{    for ( int i = 0; i < p.edges1_.isize( ); i++ )
     {    if ( i > 0 ) o << ".";
          o << BaseAlpha( p.edges1_[i] );    }
     o << " --- ";
     RightPrecisionOut( o, p.sep_, 2 );
     o << " +/- ";
     RightPrecisionOut( o, p.dev_, 2 );
     o << " --> ";
     for ( int i = 0; i < p.edges2_.isize( ); i++ )
     {    if ( i > 0 ) o << ".";
          o << BaseAlpha( p.edges2_[i] );    }
     return o;    }

void TranslateTrustedPaths( const vecKmerPath& paths, const HyperKmerPath& h,
     vec<pp_read>& trusteds, const Bool NEW_UNIQUE_EXTEND )
{    vec<vrtx_t> to_left_vertex, to_right_vertex;
     h.ToLeft(to_left_vertex), h.ToRight(to_right_vertex);
     vec<tagged_rpint> edgedb;
     h.MakeEdgeDatabase(edgedb);
     trusteds.clear( );
     static vec<int> edgelengths;
     edgelengths.clear( );
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          edgelengths.push_back( h.EdgeLength(i) );
     vec< vec<int> > left_sub_edgelengths( h.EdgeObjectCount( ) );
     for ( int e = 0; e < h.EdgeObjectCount( ); e++ )
     {    left_sub_edgelengths[e].resize( h.EdgeObject(e).NSegments( ) );
          left_sub_edgelengths[e][0] = 0;
          for ( int j = 1; j < left_sub_edgelengths[e].isize( ); j++ )
          {    left_sub_edgelengths[e][j] = left_sub_edgelengths[e][j-1] 
                    + h.EdgeObject(e).Length(j-1);    }    }
     for ( int i = 0; i < paths.size( ); i++ )
     {    const KmerPath& p = paths[i];
          static vec<KmerPathOnHyper> places;
          PlaceKmerPath( p, h, edgedb, to_right_vertex, edgelengths, places,
               left_sub_edgelengths );
          static pp_read ppr;
          ppr.clear( );
          if ( places.size( ) != 1 ) trusteds.push_back(ppr);  
          else
          {    places[0].UniqueExtend( h, to_left_vertex, to_right_vertex,
                    NEW_UNIQUE_EXTEND );
               static vec<int> r;
               r = places[0].Edges( );
               for ( int i = 0; i < r.isize( ); i++ )
                    ppr.push_back( r[i] );
               trusteds.push_back(ppr);    }    }    }

void PlacePairs( const vec<read_pairing>& pairs, Bool PRINT_SECONDARY_CLOUD,
     const vec< pair<read_location,read_location> >& pairs_locs,
     const vecKmerPath& paths, const vecKmerPath& paths_rc, const HyperKmerPath& h,
     const vec<tagged_rpint>& edgedb, const vec<vrtx_t>& to_right_vertex,     
     vec<HyperPairPlacement>& hpairs, vec<int>& hpairs_source,
     const vec<Bool>& unique_edge, int k_add_in, int k_add_out, Bool verbose,
     const Bool NEW_UNIQUE_EXTEND )
{    hpairs.clear( ), hpairs_source.clear( );
     double clock1 = WallClockTime( );
     vec<vrtx_t> to_left_vertex( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.From(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( i, j );
               to_left_vertex[e] = i;    }    } 
     equiv_rel(e);
     h.ComponentRelation(e);
     vec<int> L; 
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          L.push_back( h.EdgeObject(i).KmerCount( ) );
     digraphE<int> G( h, L );
     static vec<pp_read> reads;
     reads.clear( );
     vec< vec<int> > left_sub_edgelengths( h.EdgeObjectCount( ) );
     for ( int e = 0; e < h.EdgeObjectCount( ); e++ )
     {    left_sub_edgelengths[e].resize( h.EdgeObject(e).NSegments( ) );
          left_sub_edgelengths[e][0] = 0;
          for ( int j = 1; j < left_sub_edgelengths[e].isize( ); j++ )
          {    left_sub_edgelengths[e][j] = left_sub_edgelengths[e][j-1] 
                    + h.EdgeObject(e).Length(j-1);    }    }
     cout << TimeSince(clock1) << " used in PlacePairs, part 1\n";
     if ( k_add_in > 0 || k_add_out > 0 )
     {    for ( int i = 0; i < pairs.isize( ); i++ )
          {    const KmerPath& p1 = paths_rc[ pairs[i].id2 ];
               const KmerPath& p2 = paths[ pairs[i].id1 ];
               static vec<KmerPathOnHyper> places1, places2;
               PlaceKmerPath( p1, h, edgedb, to_right_vertex, L, places1,
                    left_sub_edgelengths );
               PlaceKmerPath( p2, h, edgedb, to_right_vertex, L, places2,
                    left_sub_edgelengths );
               for ( int j = 0; j < places1.isize( ); j++ )
                    places1[j].UniqueExtend( h, to_left_vertex, to_right_vertex,
                         NEW_UNIQUE_EXTEND );
               for ( int j = 0; j < places2.isize( ); j++ )
                    places2[j].UniqueExtend( h, to_left_vertex, to_right_vertex,
                         NEW_UNIQUE_EXTEND );
               for ( int pass = 1; pass <= 2; pass++ )
               {    vec<KmerPathOnHyper>& places = ( pass == 1 ? places1 : places2 );
                         for ( int j = 0; j < places.isize( ); j++ )
                    {    reads.push_back( 
                              pp_read( places[j].Edges( ) ) );    }    }    }    }
     double xclock = WallClockTime( );
     UniqueSort(reads);
     vec< vec<int> > ireads( reads.size( ) );
     for ( int i = 0; i < reads.isize( ); i++ )
          ireads[i] = reads[i];
     vec_overlap<int> over(ireads);
     static vec< pair<String,String> > suspect_print;
     suspect_print.clear( );
     vec<String> encodings;
     cout << TimeSince(xclock) << " used in PlacePairs, part x\n";
     double clock3 = WallClockTime( );
     for ( int i = 0; i < pairs.isize( ); i++ )
     {    const KmerPath& p1 = paths_rc[ pairs[i].id2 ];
          const KmerPath& p2 = paths[ pairs[i].id1 ];
          static vec<KmerPathOnHyper> places1, places2;
          PlaceKmerPath( p1, h, edgedb, to_right_vertex, L, places1,
               left_sub_edgelengths );
          PlaceKmerPath( p2, h, edgedb, to_right_vertex, L, places2,
               left_sub_edgelengths );

          // Check for weird pathology where link is inconsistent with graph
          // structure.  

          /*
          if ( places1.size( ) == 1 && places2.size( ) == 1 )
          {    const KmerPathOnHyper &p1 = places1[0], &p2 = places2[0];
               int e1 = p1.Edges( ).front( ), e2 = p2.Edges( ).back( );
               int v = to_right_vertex[e2], w = to_left_vertex[e1];
               int dist = pairs[i].sep + h.K( ) + 4 * pairs[i].sd;
               if ( e1 != e2 && e.Equiv( v, w ) && !G.ThisClose( v, w, dist ) )
               {    if ( !Member( suspect_print, 
                         make_pair( p2.Alpha( ), p1.Alpha( ) ) ) )
                    {    if (verbose)
                         {    cout << "Suspect problem with edge "
                                   << p2.Alpha( ) << " ---> " << p1.Alpha( ) 
                                   << "\n";    }
                         suspect_print.push_back(
                              make_pair( p2.Alpha( ), p1.Alpha( ) ) );    }
                    if (PRINT_SECONDARY_CLOUD) 
                         encodings.push_back( "(untranslated)" );
                    continue;    }    }
          */

          // Extend uniquely through graph.

          for ( int j = 0; j < places1.isize( ); j++ )
               places1[j].UniqueExtend( h, to_left_vertex, to_right_vertex,
                    NEW_UNIQUE_EXTEND );
          for ( int j = 0; j < places2.isize( ); j++ )
               places2[j].UniqueExtend( h, to_left_vertex, to_right_vertex,
                    NEW_UNIQUE_EXTEND );

          // Apply k_add tests.

          const int max_adds = 20;
          const int max_read = 50000;
          for ( int dpass = 1; dpass <= 2; dpass++ )
          {    int k_add = ( dpass == 1 ? k_add_in : k_add_out );
               if ( k_add == 0 ) continue;
               for ( int pass = 1; pass <= 2; pass++ )
               {    vec<KmerPathOnHyper>& places = ( pass == 1 ? places1 : places2 );
                    for ( int j = 0; j < places.isize( ); j++ )
                    {    static vec<int> added_left, added_right;
                         added_left.clear( ), added_right.clear( );
                         while(1)
                         {    pp_read x( places[j].Edges( ) );
                              static vec<int> left, right;
                              left.clear( ), right.clear( );

                              vec<int> ix(x);
                              vec< pair<int,int> > overlaps, OVERLAPS;
                              over.GetOverlaps( ix, overlaps, False );
                              for ( int u = 0; u < overlaps.isize( ); u++ )
                              {    int r = overlaps[u].first, o = overlaps[u].second;
                                   int OVER = 0;
                                   int start = Max( 0, o );
                                   int stop = Min( ix.isize( ), 
                                        o + reads[r].isize( ) );
                                   for ( int z = start; z < stop; z++ )
                                        OVER += L[ ix[z] ];
                                   if ( OVER >= k_add + 1 )
                                        OVERLAPS.push_back( overlaps[u] );    }

                              for ( int z = 0; z < OVERLAPS.isize( ); z++ )
                              {    int u = OVERLAPS[z].first, o = OVERLAPS[z].second;
                                   if ( o < 0 ) left.push_back( reads[u][-o-1] );
                                   if ( o + reads[u].isize( ) > x.isize( ) )
                                        right.push_back( reads[u][ 
                                             x.isize( ) - o ] );    }

                              UniqueSort(left), UniqueSort(right);
                              int left_kmers = 0, right_kmers = 0;
                              if ( left.solo( ) )
                              {    left_kmers = L[ left[0] ];
                                   for ( int u = 0; u < places[j].NEdges( ); u++ )
                                        left_kmers += L[ places[j].Edge(u) ];    }
                              if ( right.solo( ) )
                              {    right_kmers = L[ right[0] ];
                                   for ( int u = 0; u < places[j].NEdges( ); u++ )
                                        right_kmers += L[ places[j].Edge(u) ];    }
                              Bool extended = False;
                              if ( left.solo( ) && pass == dpass
                                   && added_left.isize( ) <= max_adds
                                   && left_kmers <= max_read )
                              {    extended = True;
                                   places[j].ExtendLeft( left[0], L[ left[0] ] );
                                   added_left.push_back( left[0] );    }
                              if ( right.solo( ) && pass != dpass 
                                   && added_right.isize( ) <= max_adds
                                   && right_kmers <= max_read )
                              {    extended = True;
                                   places[j].ExtendRight( right[0], L[ right[0] ] );
                                   added_right.push_back( right[0] );    }
                              if ( !extended ) break;    }    }    }    }

          if ( places1.size( ) != 1 || places2.size( ) != 1 )
          {    if (PRINT_SECONDARY_CLOUD) encodings.push_back( "(untranslated)" );
               continue;    }

          if ( places1.size( ) == 1 && places2.size( ) == 1 )
          {    hpairs.push_back( HyperPairPlacement( h, places2[0],
                    places1[0], pairs[i] ) );
               hpairs_source.push_back(i);
               if (PRINT_SECONDARY_CLOUD)
               {    ostrstream out;
                    out << hpairs.back( ) << ends;
                    encodings.push_back( out.str( ) );    }    }
          else if (PRINT_SECONDARY_CLOUD) 
          {    encodings.push_back( "(untranslated)" );    }    }

     // Print secondary cloud.

     if (PRINT_SECONDARY_CLOUD)
     {    cout << "\nSecondary cloud:\n";
          vec< pair<read_location,read_location> > cloud2 = pairs_locs;
          SortSync( cloud2, encodings );
          for ( int i = 0; i < cloud2.isize( ); i++ )
          {    const read_location& rl1 = cloud2[i].first;
               const read_location& rl2 = cloud2[i].second;
               cout << rl1.Contig( ) << "." << rl1.Start( ) << "-"
                    << rl1.Stop( ) + 1 << " ===> " << rl2.Start( ) << "-" 
                    << rl2.Stop( ) + 1 << "    " << encodings[i] << "\n";    }
          cout << "\n";    }
     cout << TimeSince(clock3) << " used in PlacePairs, part 3\n";    

     // For each unipath A of copy number one, find the unipaths that follow it,
     // together with their multiplicities.  If there is more than one possibility,
     // and one is very much more frequent than the rest, delete the rest.  Ditto
     // for the unipaths before A.

     vec<int> unique_edges;
     for ( int u = 0; u < h.EdgeObjectCount( ); u++ )
          if ( unique_edge[u] ) unique_edges.push_back(u);
     vec< vec< pair< vec<int>, int > > > before_unique( unique_edges.size( ) );
     vec< vec< pair< vec<int>, int > > > after_unique( unique_edges.size( ) );
     for ( int i = 0; i < hpairs.isize( ); i++ )
     {    const HyperPairPlacement& h = hpairs[i];
          for ( int pass = 1; pass <= 2; pass++ )
          {    const vec<int>& E = ( pass == 1 ? h.Edges1( ) : h.Edges2( ) );
               for ( int j = 0; j < E.isize( ); j++ )
               {    int u = E[j];
                    int p = BinPosition( unique_edges, u );
                    if ( p < 0 ) continue;
                    if ( j > 0 )
                    {    before_unique[p].push_back(
                              make_pair( Reverse( RangeOf( E, 0, j ) ), i ) );    }
                    if ( j < E.isize( ) - 1 )
                    {    after_unique[p].push_back( 
                              make_pair( RangeOf( 
                                   E, j+1, E.size( ) ), i ) );    }    }    }    }
     vec<Bool> reject( hpairs.size( ), False );
     for ( int i = 0; i < unique_edges.isize( ); i++ )
     {    for ( int pass = 1; pass <= 2; pass++ )
          {    const vec< pair< vec<int>, int > >& A
                    = ( pass == 1 ? before_unique[i] : after_unique[i] );
               int maxtail = 0;
               for ( int j = 0; j < A.isize( ); j++ )
                    maxtail = Max( maxtail, A[j].first.isize( ) );
               for ( int j = 0; j < maxtail; j++ )
               {    vec< pair<int,int> > votes;
                    for ( int k = 0; k < A.isize( ); k++ )
                    {    if ( A[k].first.isize( ) > j )
                         {    votes.push_back( make_pair( 
                                   A[k].first[j], A[k].second ) );    }    }
                    Sort(votes);
                    vec< pair<int,int> > freq;
                    for ( int k = 0; k < votes.isize( ); k++ )
                    {    int l;
                         for ( l = k + 1; l < votes.isize( ); l++ )
                              if ( votes[l].first != votes[k].first ) break;
                         freq.push_back( make_pair( l - k, votes[k].first ) );
                         k = l - 1;    }
                    if ( freq.solo( ) ) continue;
                    ReverseSort(freq);
                    if ( freq[0].first >= 10 * freq[1].first )
                    {    for ( int k = 0; k < votes.isize( ); k++ )
                         {    if ( votes[k].first != freq[0].second )
                                   reject[ votes[k].second ] = True;    }    }
                    break;    }    }    }
     EraseIf( hpairs, reject );
     EraseIf( hpairs_source, reject );    }

void PlaceTrustedPairs( const vec<read_pairing>& pairs, const vecKmerPath& paths,
     const vecKmerPath& paths_rc, const HyperKmerPath& h,        
     const vec<tagged_rpint>& edgedb, const vec<vrtx_t>& to_right_vertex,     
     const vec<Bool>& uniform, vec<pp_pair>& pppt )
{    static vec<pp_pair> pppt0;
     pppt.clear( ), pppt0.clear( );
     static vec<int> edgelengths;
     edgelengths.clear( );
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          edgelengths.push_back( h.EdgeLength(i) );
     vec< vec<int> > left_sub_edgelengths( h.EdgeObjectCount( ) );
     for ( int e = 0; e < h.EdgeObjectCount( ); e++ )
     {    left_sub_edgelengths[e].resize( h.EdgeObject(e).NSegments( ) );
          left_sub_edgelengths[e][0] = 0;
          for ( int j = 1; j < left_sub_edgelengths[e].isize( ); j++ )
          {    left_sub_edgelengths[e][j] = left_sub_edgelengths[e][j-1] 
                    + h.EdgeObject(e).Length(j-1);    }    }
     for ( int i = 0; i < pairs.isize( ); i++ )
     {    const KmerPath& p1 = paths_rc[ pairs[i].id2 ];
          const KmerPath& p2 = paths[ pairs[i].id1 ];
          static vec<KmerPathOnHyper> places1, places2;
          PlaceKmerPath( p1, h, edgedb, to_right_vertex, edgelengths, places1,
               left_sub_edgelengths );
          PlaceKmerPath( p2, h, edgedb, to_right_vertex, edgelengths, places2,
               left_sub_edgelengths );
          if ( places1.size( ) != 1 || places2.size( ) != 1 ) continue;
          HyperPairPlacement hp( h, places2[0], places1[0], pairs[i] );
          pp_pair p( pp_read( hp.Edges1( ) ), pp_read( hp.Edges2( ) ), 
               hp.Sep( ), hp.Dev( ) );
          int uleft1 = p.LeftSize( ), uleft2 = -1;
          for ( int j = 0; j < p.LeftSize( ); j++ )
          {    if ( uniform[ p.Left(j) ] )
               {    uleft1 = j;
                    break;    }    }
          for ( int j = p.LeftSize( ) - 1; j >= 0; j-- )
          {    if ( uniform[ p.Left(j) ] )
               {    uleft2 = j;
                    break;    }    }
          if ( !( uleft1 <= uleft2 ) ) continue;
          int uright1 = p.RightSize( ), uright2 = -1;
          for ( int j = 0; j < p.RightSize( ); j++ )
          {    if ( uniform[ p.Right(j) ] )
               {    uright1 = j;
                    break;    }    }
          for ( int j = p.RightSize( ) - 1; j >= 0; j-- )
          {    if ( uniform[ p.Right(j) ] )
               {    uright2 = j;
                    break;    }    }
          if ( !( uright1 <= uright2 ) ) continue;
          double gap = p.Gap( );
          for ( int j = uleft2 + 1; j < p.LeftSize( ); j++ )
               gap += h.EdgeLength( p.Left(j) );
          for ( int j = 0; j < uright1; j++ )
               gap += h.EdgeLength( p.Right(j) );
          static pp_read left, right;
          left.SetToRangeOf( p.Left( ), uleft1, uleft2 + 1 );
          right.SetToRangeOf( p.Right( ), uright1, uright2 + 1 );
          pp_pair pnew( left, right, gap, p.Dev( ) );
          pppt0.push_back(pnew);    }
     Sort(pppt0);
     for ( int i = 0; i < pppt0.isize( ); i++ )
     {    int j;
          for ( j = i + 1; j < pppt0.isize( ); j++ )
          {    if ( pppt0[j].Left( ) != pppt0[i].Left( ) ) break;
               if ( pppt0[j].Right( ) != pppt0[i].Right( ) ) break;    }
          static vec<pp_pair> x;
          x.clear( );
          for ( int u = i; u < j; u++ )
               x.push_back( pppt0[u] );

          // Copied from Reduce, bad bad bad.

          static vec< vec<int> > clusters;
          {    static vec< pair<double, int> > sep_ind;
               sep_ind.resize( x.size( ) );
               double d = 0.0;
               for ( int i = 0; i < x.isize( ) ; i++ )
               {    d = Max( d, x[i].Dev( ) );
                    sep_ind[i] = make_pair( x[i].Gap( ), i );    }
               Sort(sep_ind);
               clusters.clear( );
               for ( int i = 0; i < x.isize( ); i++ )
               {    int j;
                    for ( j = i + 1; j < x.isize( ); j++ )
                         if ( sep_ind[j].first - sep_ind[j-1].first > 4.0 * d ) 
                              break;
                    static vec<int> c;
                    c.clear( );
                    for ( int u = i; u < j; u++ )
                         c.push_back( sep_ind[u].second );
                    int n = j - i;
                    double gap 
                         = sep_ind[ i + (5*n)/6 ].first - sep_ind[ i + n/6 ].first;
                    if ( gap/d > 4.0 )
                    {    static vec<int> b;
                         b.clear( );
                         double start = 0.0;
                         for ( int u = i; u < j; u++ )
                         {    if ( b.empty( ) ) start = sep_ind[u].first;
                              else if ( sep_ind[u].first - start > 3.0 * d )
                              {    clusters.push_back(b);
                                   b.clear( );
                                   start = sep_ind[u].first;    }
                              b.push_back( sep_ind[u].second );    }
                         if ( b.nonempty( ) ) clusters.push_back(b);    }
                    else clusters.push_back(c);
                    i = j - 1;    }    }
          for ( int z = 0; z < clusters.isize( ); z++ )
          {    const vec<int>& c = clusters[z];
               int n = c.size( );
               double s = 0.0, d = 0.0;
               for ( int i = 0; i < n; i++ )
               {    s += x[ c[i] ].Gap( );
                    d += x[ c[i] ].Dev( );    }
               s /= double(n);
               d /= double(n);
               pppt.push_back( pp_pair( x[0].Left( ), x[0].Right( ), 
                    s, d / sqrt( double(n) ) ) );    }
          i = j - 1;    }    }

// MergeIdenticalEdges: where possible, merge pp_pairs whose left and right sides
// are identical.

// probably belongs in PairedPair.cc:
void MergeIdenticalEdges( vec<pp_pair>& ppp, vec<int>& pppmult )
{    vec<Bool> remove( ppp.size( ), False );
     for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
     {    if ( remove[i1] ) continue;
          pp_pair& p1 = ppp[i1];
          for ( int i2 = i1 + 1; i2 < ppp.isize( ); i2++ )
          {    if ( remove[i2] ) continue;
               const pp_pair& p2 = ppp[i2];
               if ( p1.Left( ) != p2.Left( ) || p1.Right( ) != p2.Right( ) ) 
                    continue;
               double d1 = p1.Dev( ), d2 = p2.Dev( );
               double g1 = p1.Gap( ), g2 = p2.Gap( );
               double dmult = 3.0;
               // code also in PairedPair.cc/FindJoins, should share:
               double g, d;
               if ( d1 == 0 && d2 == 0 )
               {    if ( g1 != g2 ) continue;
                    g = g1;
                    d = d1;    }
               else if ( d1 == 0 ) 
               {    g = g1;
                    d = d1;
                    if ( Abs( g2 - g ) > dmult * d2 ) continue;    }
               else if ( d2 == 0 )
               {    g = g2;
                    d = d2;
                    if ( Abs( g1 - g ) > dmult * d1 ) continue;    }
               else
               {    double w1 = 1.0/(d1*d1), w2 = 1.0/(d2*d2);
                    g = ( w1*g1 + w2*g2 ) / (w1+w2);
                    d = sqrt( w1*w1*d1*d1 + w2*w2*d2*d2 ) / (w1+w2);
                    if ( Abs( g1 - g ) > dmult * d1 ) continue;
                    if ( Abs( g2 - g ) > dmult * d2 ) continue;    }
               p1.SetGap(g), p1.SetDev(d);
               remove[i2] = True;    }    }
     EraseIf( ppp, remove );
     EraseIf( pppmult, remove );    }

void DeletePairsContraTrustedPairs( vec<pp_pair>& ppp, vec<Bool>& solo,
     vec<Bool>& ppp_trusted, const vec<int>& pppL, const vec<Bool>& unique_edge,
     digraphE<int>& G, const vec<vrtx_t>& to_right_vertex,
     const vec<vrtx_t>& to_left_vertex, vec<Bool>& pppvalid, Bool test_valid )
{    double dmult = 3.85;

     // Find trusted links between components in G.

     equiv_rel e;
     G.ComponentRelation(e);
     vec< pair<int,int> > trusted_component_links;
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !ppp_trusted[i] ) continue;
          const pp_pair& p = ppp[i];
          int c1 = e.ClassId( to_left_vertex[ p.Left(0) ] );
          int c2 = e.ClassId( to_left_vertex[ p.Right(0) ] );
          if ( c1 != c2 )
               trusted_component_links.push_back( make_pair( c1, c2 ) );    }
     UniqueSort(trusted_component_links);

     // Do the rest.

     vec<Bool> ppp_untrusted( ppp.size( ), False );
     vec<int> pppleftlen( ppp.size( ), 0 ), ppprightlen( ppp.size( ), 0 );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          for ( int j = 0; j < p.LeftSize( ); j++ )
               pppleftlen[i] += pppL[ p.Left(j) ];
          for ( int j = 0; j < p.RightSize( ); j++ )
               ppprightlen[i] += pppL[ p.Right(j) ];    }
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !ppp_trusted[i] ) continue;
          const pp_pair& p = ppp[i];
          int b1 = e.ClassId( to_left_vertex[ p.Left(0) ] );
          int b2 = e.ClassId( to_left_vertex[ p.Right(0) ] );
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( j == i ) continue;
               if ( ppp_trusted[j] ) continue;
               const pp_pair& q = ppp[j];
               int c1 = e.ClassId( to_left_vertex[ q.Left(0) ] );
               int c2 = e.ClassId( to_left_vertex[ q.Right(0) ] );
               static vec<int> loffsets, roffsets;
               GetOverlaps( p.Left( ), q.Left( ), loffsets );
               GetOverlaps( p.Right( ), q.Right( ), roffsets );
               Bool have_unique = False;
               for ( int u = 0; u < loffsets.isize( ); u++ )
               {    int lo = loffsets[u];
                    Bool have_unique = False;
                    for ( int v = Max( 0, lo ); 
                         v < Min( p.LeftSize( ), lo + q.LeftSize( ) ); v++ )
                    {    if ( unique_edge[ p.Left(v) ] ) have_unique = True;    }
                    if ( !have_unique ) continue;
                    int Lo = 0;
                    if ( lo >= 0 )
                    {    for ( int z = 0; z < lo; z++ )
                              Lo += pppL[ p.Left(z) ];    }
                    else
                    {    for ( int z = 0; z < -lo; z++ )
                              Lo -= pppL[ q.Left(z) ];    }
                    double Ro = Lo + pppleftlen[j] + q.Gap( )
                         - pppleftlen[i] - p.Gap( );
                    int Rolow 
                         = int( ceil( Ro - dmult * p.Dev( ) - dmult * q.Dev( ) ) );
                    int Rohigh 
                         = int( floor( Ro + dmult * p.Dev( ) + dmult * q.Dev( ) ) );
                    if ( Rolow <= -ppprightlen[j] )
                    {    int gap = -Rolow - ppprightlen[j];
                         if ( c1 != c2 && BinMember( trusted_component_links,
                              make_pair( c1, c2 ) ) )
                         {    continue;    }
                         if ( b1 != b2 ) continue;
                         if ( G.ThisClose( to_right_vertex[ q.Right( ).back( ) ],
                              to_left_vertex[ p.Right( ).front( ) ], gap ) )
                         {    continue;    }    }
                    if ( Rohigh >= ppprightlen[i] )
                    {    int gap = Rohigh - ppprightlen[i];
                         if ( c1 != c2 && BinMember( trusted_component_links,
                              make_pair( c1, c2 ) ) )
                         {    continue;    }
                         if ( b1 != b2 ) continue;
                         if ( G.ThisClose( to_right_vertex[ p.Right( ).back( ) ],
                              to_left_vertex[ q.Right( ).front( ) ], gap ) )
                         {    continue;    }    }
                    Bool found_right_overlap = False;
                    for ( int v = 0; v < roffsets.isize( ); v++ )
                    {    int ro = roffsets[v];
                         int Ro = 0;
                         if ( ro >= 0 )
                         {    for ( int z = 0; z < ro; z++ )
                                   Ro += pppL[ p.Right(z) ];    }
                         else
                         {    for ( int z = 0; z < -ro; z++ )
                                   Ro -= pppL[ q.Right(z) ];    }
                         if ( Ro >= Rolow && Ro <= Rohigh ) 
                              found_right_overlap = True;    }
                    if ( !found_right_overlap )
                    {    cout << q;
                         if ( test_valid && pppvalid[j] )
                              cout << " [WARNING: VALID!]";
                         cout << " conflicts with trusted " << p;
                         if ( test_valid && !pppvalid[i] )
                              cout << " [WARNING: INVALID!]";
                         cout << " and should be deleted\n";    
                         ppp_untrusted[j] = True;    }    }
               for ( int u = 0; u < roffsets.isize( ); u++ )
               {    int ro = roffsets[u];
                    Bool have_unique = False;
                    for ( int v = Max( 0, ro ); 
                         v < Min( p.RightSize( ), ro + q.RightSize( ) ); v++ )
                    {    if ( unique_edge[ p.Right(v) ] ) have_unique = True;    }
                    if ( !have_unique ) continue;
                    int Ro = 0;
                    if ( ro >= 0 )
                    {    for ( int z = 0; z < ro; z++ )
                              Ro += pppL[ p.Right(z) ];    }
                    else
                    {    for ( int z = 0; z < -ro; z++ )
                              Ro -= pppL[ q.Right(z) ];    }
                    double Lo = Ro + pppleftlen[i] + p.Gap( )
                         - pppleftlen[j] - q.Gap( );
                    int Lolow 
                         = int( ceil( Lo - dmult * p.Dev( ) - dmult * q.Dev( ) ) );
                    int Lohigh 
                         = int( floor( Lo + dmult * p.Dev( ) + dmult * q.Dev( ) ) );
                    if ( Lolow <= -pppleftlen[j] )
                    {    int gap = -Lolow - pppleftlen[j];
                         if ( c1 != c2 && BinMember( trusted_component_links,
                              make_pair( c1, c2 ) ) )
                         {    continue;    }
                         if ( b1 != b2 ) continue;
                         if ( G.ThisClose( to_right_vertex[ q.Left( ).back( ) ],
                              to_left_vertex[ p.Left( ).front( ) ], gap ) )
                         {    continue;    }    }
                    if ( Lohigh >= pppleftlen[i] )
                    {    int gap = Lohigh - pppleftlen[i];
                         if ( c1 != c2 && BinMember( trusted_component_links,
                              make_pair( c1, c2 ) ) )
                         {    continue;    }
                         if ( b1 != b2 ) continue;
                         if ( G.ThisClose( to_right_vertex[ p.Left( ).back( ) ],
                              to_left_vertex[ q.Left( ).front( ) ], gap ) )
                         {    continue;    }    }
                    Bool found_left_overlap = False;
                    for ( int v = 0; v < loffsets.isize( ); v++ )
                    {    int lo = loffsets[v];
                         int Lo = 0;
                         if ( lo >= 0 )
                         {    for ( int z = 0; z < lo; z++ )
                                   Lo += pppL[ p.Left(z) ];    }
                         else
                         {    for ( int z = 0; z < -lo; z++ )
                                   Lo -= pppL[ q.Left(z) ];    }
                         if ( Lo >= Lolow && Lo <= Lohigh ) 
                              found_left_overlap = True;    }
                    if ( !found_left_overlap )
                    {    cout << q;
                         if ( test_valid && pppvalid[j] )
                              cout << " [WARNING: VALID!]";
                         cout << " conflicts with trusted " << p;
                         if ( test_valid && !pppvalid[i] )
                              cout << " [WARNING: INVALID!]";
                         cout << " and should be deleted\n";    
                         ppp_untrusted[j] = True;    }    }    }    }
     EraseIf( ppp, ppp_untrusted );
     EraseIf( solo, ppp_untrusted );
     EraseIf( ppp_trusted, ppp_untrusted );    }

void MakePairedPairs1(

     /* input:   */   const String& sub_dir,
                      const vec<HyperPairPlacement>&  hpairs,  // must be sorted!
                      vec<Bool>& unique_edge,  /* Edited! */
                      const vec<int>& edge_copyno,
                      const vec<Bool>& hstart_known,
                      const vec<int>& hstart,
                      const int NHOOD_RADIUS,
                      const int NHOOD_RADIUS_INTERNAL,
                      const double short_insert_coverage,
                      const double short_insert_readlength,
                      const double short_insertlength,
                      const double short_insertdev,
                      const vec<pp_read>& trusteds,
                      const vec<pp_pair>& trusted_pairs,

     /* outputs: */   ostream& out,                // for reporting
                      vec<pp_pair>& ppp,           // the pp_pairs
                      vec<int>& pppL,              // edge lengths

     /* optional: */  Bool test_valid,             // compare to genome
                      const vecbasevector& genome, // the genome
                      const KmerBaseBroker& kbb,   // to compare to genome
                      const serfvec<placement>& locsv,
                      Bool PRECOMPUTE_CLOSURE_LENGTHS,
                      Bool REDUCE_VERBOSE,
                      Bool MERGE_PAIRED_PAIRS_VERBOSE,
                      int SHOW_PAIR_PLACEMENTS,

     /* other: */     vec<int>& pppmult,
                      vec<Bool>& conflicts_tr

          )

{    double clock1 = WallClockTime( );
     ppp.clear( ), pppL.clear( );
     if ( hpairs.empty( ) ) return;
     static vec<int> pppm;
     pppm.clear( );
     const HyperKmerPath& h = hpairs[0].H( );
     for ( int i = 0; i < h.EdgeObjectCount( ); i++ )
          pppL.push_back( h.EdgeObject(i).KmerCount( ) );
     pppmult.clear( );
     for ( int i = 0; i < hpairs.isize( ); i++ )
     {    int j;
          for ( j = i + 1; j < hpairs.isize( ); j++ )
          {    if ( hpairs[j].Edges1( ) != hpairs[i].Edges1( ) ) break;
               if ( hpairs[j].Edges2( ) != hpairs[i].Edges2( ) ) break;    }
          static vec<HyperPairPlacement> hp;
          hp.clear( );
          for ( int u = i; u < j; u++ )
               hp.push_back( hpairs[u] );
          vec<int> hpmult;
          if (REDUCE_VERBOSE)
          {    cout << "\n" << pp_read( hpairs[i].Edges1( ) ) << " --> "
                    << pp_read( hpairs[i].Edges2( ) ) << "\n";
               cout << "before:\n";
               for ( int u = 0; u < hp.isize( ); u++ )
               {    RightPrecisionOut( cout, hp[u].Sep( ), 2 );
                    cout << " +/- ";
                    RightPrecisionOut( cout, hp[u].Dev( ), 2 );
                    cout << "\n";    }    }
          Reduce( hp, hpmult );
          if (REDUCE_VERBOSE)
          {    cout << "after:\n";
               for ( int u = 0; u < hp.isize( ); u++ )
               {    RightPrecisionOut( cout, hp[u].Sep( ), 2 );
                    cout << " +/- "; RightPrecisionOut( cout, hp[u].Dev( ), 2 );
                    cout << "\n";    }    }
          for ( int u = 0; u < hp.isize( ); u++ )
          {    ppp.push_back( pp_pair( pp_read( hp[u].Edges1( ) ),
                    pp_read( hp[u].Edges2( ) ), hp[u].Sep( ), hp[u].Dev( ) ) );
               pppmult.push_back( hpmult[u] );    }
          i = j - 1;    }
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, 
               "in MakePairedPairs, after initial generation of pairs", sub_dir, 
               ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    } 

     // Precompute closure lengths.  If there is only one value <= 2 devs
     // off, and no other value <= 4 devs off, plug it in.

     if (PRECOMPUTE_CLOSURE_LENGTHS)
     {    vecKmerPath paths;
          vec<read_pairing> pairs;
          //DEBUG  // For use with TestFindClosureLengths.
          //DEBUG  Ofstream( pairStrm, "tmp/pairs" );
          //DEBUG  pairStrm << ppp.isize() << endl;
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    const pp_pair& p = ppp[i];
               read_pairing rp;
               rp.t = other;
               rp.weight = 1;
               rp.id1 = paths.size( );
               static KmerPath x;
               x.Clear( );
               for ( int j = 0; j < p.LeftSize( ); j++ ) {
                    x.Append( h.EdgeObject( p.Left(j) ) );
                    //DEBUG  if ( j>0 ) pairStrm << ".";
                    //DEBUG  pairStrm << BaseAlpha( p.Left(j) );
               }
               paths.push_back_reserve(x);
               rp.sep = int( round( p.Gap( ) ) ) - h.K( ) + 1;
               rp.sd = int( ceil( ppp[i].Dev( ) ) );
               //DEBUG  pairStrm << " " << rp.sep << " " << rp.sd << " ";
               rp.id2 = paths.size( );
               x.Clear( );
               for ( int j = 0; j < p.RightSize( ); j++ ) {
                    x.Append( h.EdgeObject( p.Right(j) ) );
                    //DEBUG  if ( j>0 ) pairStrm << ".";
                    //DEBUG  pairStrm << BaseAlpha( p.Right(j) );
               }
               //DEBUG  pairStrm << endl;
               paths.push_back_reserve(x);
               pairs.push_back(rp);    }
          //DEBUG  pairStrm.close();
          //DEBUG  vec<int> unipathLengths;
          //DEBUG  for ( int i = 0; i < h.EdgeObjectCount(); ++i )
          //DEBUG    unipathLengths.push_back( h.EdgeObject(i).KmerCount() );
          //DEBUG  Ofstream( lengthStrm, "tmp/lengths" );
          //DEBUG  lengthStrm << unipathLengths;
          //DEBUG  lengthStrm.close();
          //DEBUG  PRINT( h.K() );
          vec< vec<int> > closure_lengths;
          double fclock = WallClockTime( );
          FindClosureLengths( paths, pairs, 4.0, h.K( ), closure_lengths );
          out << TimeSince(fclock) << " used in FindClosureLengths" << endl;
          out << "\npossible gap sizes:\n";
          for ( int i = 0; i < ppp.isize( ); i++ )
          {    pp_pair& p = ppp[i];
               int len1 = 0, len2 = 0;
               for ( int j = 0; j < p.LeftSize( ); j++ )
                    len1 += h.EdgeLength( p.Left(j) );
               for ( int j = 0; j < p.RightSize( ); j++ )
                    len2 += h.EdgeLength( p.Right(j) );
               static vec<int> seps;
               static vec<double> errs;
               seps.clear( ), errs.clear( );
               for ( int j = 0; j < closure_lengths[i].isize( ); j++ )
               {    int g = closure_lengths[i][j] - len1 - len2 - h.K( ) + 1;
                    double d = ( double(g) - p.Gap( ) ) / p.Dev( );
                    if ( Abs(d) > 4.0 ) continue;
                    seps.push_back(g);
                    errs.push_back(d);    }
               SortSync( errs, seps );
               if ( seps.size( ) >= 2 && errs[0] <= 2.0 && errs[1] > 4.0 )
               {    seps.resize(1);
                    errs.resize(1);    }
               if ( seps.solo( ) && errs[0] <= 2.0 ) 
               {    p.SetGap( seps[0] );
                    p.SetDev( 0.0 );    }
               out << "\n[" << i << "] " << ppp[i] << "\n";
               out << "[" << i << "]";
               for ( int j = 0; j < Min( 6, seps.isize( ) ); j++ )
               {    out << " " << seps[j] << "(" << setprecision(3) 
                         << errs[j] << ")";    }
               if ( seps.size( ) > 6 ) out << " ...";
               out << "\n";    }    }

     // Look for sources and sinks that appear to be hanging ends.

     vec<vrtx_t> to_left_vertex( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.From(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( i, j );
               to_left_vertex[e] = i;    }    } 
     vec<vrtx_t> to_right_vertex( h.EdgeObjectCount( ) );
     for ( int i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.To(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexTo( i, j );
               to_right_vertex[e] = i;    }    }
     cout << TimeSince(clock1) << " in MakePairedPairs 1" << endl;

     // Merge identical edges.

     double clock2 = WallClockTime( );
     MergeIdenticalEdges( ppp, pppmult );

     // Delete pairs conflicting with trusted reads.

     conflicts_tr.resize_and_set( trusteds.size( ), False );
     for ( int i1 = 0; i1 < trusteds.isize( ); i1++ )
     {    const pp_read& t = trusteds[i1];
          for ( int i2 = 0; i2 < trusteds.isize( ); i2++ )
          {    const pp_read& t2 = trusteds[i2];
               static vec<int> offsets;
               GetOverlaps( t, t2, offsets );
               for ( int u1 = 0; u1 < t.isize( ); u1++ )
               {    if ( !unique_edge[ t[u1] ] ) continue;
                    for ( int u2 = 0; u2 < t2.isize( ); u2++ )
                    {    if ( t2[u2] != t[u1] ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) ) 
                         {    if (MERGE_PAIRED_PAIRS_VERBOSE)
                              {    cout << "conflict between trusteds: " << t 
                                        << " vs " << t2 << "\n";    }
                              conflicts_tr[i2] = True;    }    }    }    }    }
     cout << "found " << Sum(conflicts_tr) << " conflicts between trusteds\n";
     vec<Bool> conflicts( ppp.size( ), False );
     for ( int i1 = 0; i1 < trusteds.isize( ); i1++ )
     {    const pp_read& t = trusteds[i1];
          for ( int i2 = 0; i2 < ppp.isize( ); i2++ )
          {    const pp_pair& p2 = ppp[i2];
               static vec<int> offsets;
               GetOverlaps( t, p2.Left( ), offsets );
               for ( int u1 = 0; u1 < t.isize( ); u1++ )
               {    if ( !unique_edge[ t[u1] ] ) continue;
                    for ( int u2 = 0; u2 < p2.LeftSize( ); u2++ )
                    {    if ( p2.Left(u2) != t[u1] ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) ) 
                         {    if (MERGE_PAIRED_PAIRS_VERBOSE)
                              {    cout << "conflict between trusted " << t 
                                        << " and pair " << p2 << "\n";    }
                              conflicts[i2] = True;    }    }    }
               GetOverlaps( t, p2.Right( ), offsets );
               for ( int u1 = 0; u1 < t.isize( ); u1++ )
               {    if ( !unique_edge[ t[u1] ] ) continue;
                    for ( int u2 = 0; u2 < p2.RightSize( ); u2++ )
                    {    if ( p2.Right(u2) != t[u1] ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) )
                         {    if (MERGE_PAIRED_PAIRS_VERBOSE)
                              {    cout << "conflict between trusted " << t 
                                        << " and pair " << p2 << "\n";    }
                              conflicts[i2] = True;    }    }    }    }    }
     cout << "found " << Sum(conflicts) << " conflicts with trusteds\n";
     /*
     if ( Sum(conflicts_tr) == 0 ) 
     {    EraseIf( ppp, conflicts );
          EraseIf( pppmult, conflicts );    }
     */

     // Delete reads that are not trusteds.

     vec<Bool> dangerous( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    Bool OK_left = False, OK_right = False;
          for ( int j = 0; j < trusteds.isize( ); j++ )
          {    if ( ppp[i].Left( ) == trusteds[j] ) OK_left = True;
               if ( ppp[i].Right( ) == trusteds[j] ) OK_right = True;    }
          if ( !OK_left || !OK_right ) 
          {    dangerous[i] = True;
               if (MERGE_PAIRED_PAIRS_VERBOSE)
                    cout << "deleting dangerous " << ppp[i] << "\n";    }    }
     EraseIf( ppp, dangerous );
     EraseIf( pppmult, dangerous );
     cout << TimeSince(clock2) << " in MakePairedPairs 2" << endl;
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, "in MakePairedPairs, after deleting dangerous pairs",
               sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    }    }

void MakePairedPairs2(

     /* input:   */   const HyperKmerPath& h,
                      const String& sub_dir,
                      const vec<HyperPairPlacement>&  hpairs,  // must be sorted!
                      vec<Bool>& unique_edge,  /* Edited! */
                      const vec<int>& edge_copyno,
                      const vec<double>& edge_copyno_p,
                      const vec<Bool>& hstart_known,
                      const vec<int>& hstart,
                      const int NHOOD_RADIUS,
                      const int NHOOD_RADIUS_INTERNAL,
                      const double short_insert_coverage,
                      const double short_insert_readlength,
                      const double short_insertlength,
                      const double short_insertdev,
                      const vec<pp_read>& trusteds,
                      const vec<pp_pair>& trusted_pairs,

     /* outputs: */   ostream& out,                // for reporting
                      vec<pp_pair>& ppp,           // the pp_pairs
                      vec<int>& pppL,              // edge lengths

     /* optional: */  Bool test_valid,             // compare to genome
                      const vecbasevector& genome, // the genome
                      const KmerBaseBroker& kbb,   // to compare to genome
                      const serfvec<placement>& locsv,
                      Bool CHEAT_BY_DELETING_INVALID_EDGES,
                      Bool DELETE_HANGING_ENDS,
                      Bool DELETE_POORLY_COVERED_PAIRS,
                      Bool POORLY_COVERED_PAIRS_VERBOSE,
                      Bool MERGE_PAIRED_PAIRS_VERBOSE,
                      Bool MERGE_COPY_NUMBER_TWO,
                      Bool MERGE_COPY_NUMBER_TWO_VERBOSE,
                      Bool REMOVE_SOLO,
                      int SHOW_PAIR_PLACEMENTS,

     /* other: */     vec<int>& pppmult,
                      vec<Bool>& conflicts_tr

          )

{
     vec<vrtx_t> to_left_vertex( h.EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.From(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( i, j );
               to_left_vertex[e] = i;    }    } 
     vec<vrtx_t> to_right_vertex( h.EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.To(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexTo( i, j );
               to_right_vertex[e] = i;    }    }

     // Delete pairs that contradict trusted pairs.

     double clock3 = WallClockTime( );
     digraphE<int> G( h, pppL );
     vec<Bool> ppp_trusted( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          if ( pppmult[i] == 1 ) continue;
          for ( int j = 0; j < trusted_pairs.isize( ); j++ )
          {    const pp_pair& t = trusted_pairs[j];
               if ( p.Left( ) == t.Left( ) && p.Right( ) == t.Right( ) )
               {    double sep_diff = Abs( p.Gap( ) - t.Gap( ) );
                    if ( sep_diff <= 3.0 * p.Dev( ) + 3.0 * t.Dev( ) )
                         ppp_trusted[i] = True;    }    }    }

     // Look for conflicts between trusted pairs.

     vec<Bool> ppp_untrusted( ppp.size( ), False );
     double dmult = 3.85;
     vec<int> pppleftlen( ppp.size( ), 0 ), ppprightlen( ppp.size( ), 0 );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          for ( int j = 0; j < p.LeftSize( ); j++ )
               pppleftlen[i] += pppL[ p.Left(j) ];
          for ( int j = 0; j < p.RightSize( ); j++ )
               ppprightlen[i] += pppL[ p.Right(j) ];    }
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !ppp_trusted[i] ) continue;
          const pp_pair& p = ppp[i];
          for ( int j = 0; j < ppp.isize( ); j++ )
          {    if ( j == i ) continue;
               if ( !ppp_trusted[j] ) continue;
               const pp_pair& q = ppp[j];
               static vec<int> loffsets, roffsets;
               GetOverlaps( p.Left( ), q.Left( ), loffsets );
               GetOverlaps( p.Right( ), q.Right( ), roffsets );
               Bool have_unique = False;
               for ( int u = 0; u < loffsets.isize( ); u++ )
               {    int lo = loffsets[u];
                    Bool have_unique = False;
                    for ( int v = Max( 0, lo ); 
                         v < Min( p.LeftSize( ), lo + q.LeftSize( ) ); v++ )
                    {    if ( unique_edge[ p.Left(v) ] ) have_unique = True;    }
                    if ( !have_unique ) continue;
                    int Lo = 0;
                    if ( lo >= 0 )
                    {    for ( int z = 0; z < lo; z++ )
                              Lo += pppL[ p.Left(z) ];    }
                    else
                    {    for ( int z = 0; z < -lo; z++ )
                              Lo -= pppL[ q.Left(z) ];    }
                    double Ro = Lo + pppleftlen[j] + q.Gap( )
                         - pppleftlen[i] - p.Gap( );
                    int Rolow 
                         = int( ceil( Ro - dmult * p.Dev( ) - dmult * q.Dev( ) ) );
                    int Rohigh 
                         = int( floor( Ro + dmult * p.Dev( ) + dmult * q.Dev( ) ) );
                    if ( Rolow <= -ppprightlen[j] ) continue;
                    if ( Rohigh >= ppprightlen[i] ) continue;
                    Bool found_right_overlap = False;
                    for ( int v = 0; v < roffsets.isize( ); v++ )
                    {    int ro = roffsets[v];
                         int Ro = 0;
                         if ( ro >= 0 )
                         {    for ( int z = 0; z < ro; z++ )
                                   Ro += pppL[ p.Right(z) ];    }
                         else
                         {    for ( int z = 0; z < -ro; z++ )
                                   Ro -= pppL[ q.Right(z) ];    }
                         if ( Ro >= Rolow && Ro <= Rohigh ) 
                              found_right_overlap = True;    }
                    if ( !found_right_overlap )
                    {    ppp_untrusted[i] = True;
                         ppp_untrusted[j] = True;    
                         if (MERGE_PAIRED_PAIRS_VERBOSE)
                         {    cout << "Found conflict between trusted pairs "
                                   << ppp[i] << " and " << ppp[j] 
                                   << "\n";    }    }    }
               for ( int u = 0; u < roffsets.isize( ); u++ )
               {    int ro = roffsets[u];
                    Bool have_unique = False;
                    for ( int v = Max( 0, ro ); 
                         v < Min( p.RightSize( ), ro + q.RightSize( ) ); v++ )
                    {    if ( unique_edge[ p.Right(v) ] ) have_unique = True;    }
                    if ( !have_unique ) continue;
                    int Ro = 0;
                    if ( ro >= 0 )
                    {    for ( int z = 0; z < ro; z++ )
                              Ro += pppL[ p.Right(z) ];    }
                    else
                    {    for ( int z = 0; z < -ro; z++ )
                              Ro -= pppL[ q.Right(z) ];    }
                    double Lo = Ro + pppleftlen[i] + p.Gap( )
                         - pppleftlen[j] - q.Gap( );
                    int Lolow 
                         = int( ceil( Lo - dmult * p.Dev( ) - dmult * q.Dev( ) ) );
                    int Lohigh 
                         = int( floor( Lo + dmult * p.Dev( ) + dmult * q.Dev( ) ) );
                    if ( Lolow <= -pppleftlen[j] ) continue;
                    if ( Lohigh >= pppleftlen[i] ) continue;
                    Bool found_left_overlap = False;
                    for ( int v = 0; v < loffsets.isize( ); v++ )
                    {    int lo = loffsets[v];
                         int Lo = 0;
                         if ( lo >= 0 )
                         {    for ( int z = 0; z < lo; z++ )
                                   Lo += pppL[ p.Left(z) ];    }
                         else
                         {    for ( int z = 0; z < -lo; z++ )
                                   Lo -= pppL[ q.Left(z) ];    }
                         if ( Lo >= Lolow && Lo <= Lohigh ) 
                              found_left_overlap = True;    }
                    if ( !found_left_overlap )
                    {    ppp_untrusted[i] = True;
                         ppp_untrusted[j] = True;
                         if (MERGE_PAIRED_PAIRS_VERBOSE)
                         {    cout << "Found conflict between trusted pairs "
                                   << ppp[i] << " and " << ppp[j] 
                                   << "\n";    }    }    }    }    }
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !ppp_untrusted[i] ) continue;
          const pp_pair& p = ppp[i];
          const double almost_always_true = 0.99999;
          for ( int pass = 1; pass <= 2; pass++ )
          {    const pp_read& v = ( pass == 1 ? p.Left( ) : p.Right( ) );
               for ( int j = 0; j < v.isize( ); j++ )
               {    int e = v[j];
                    if ( unique_edge[e] && edge_copyno_p[e] < almost_always_true )
                    {    if (MERGE_PAIRED_PAIRS_VERBOSE)
                         {    cout << "Distrusting uniqueness of " << BaseAlpha(e)
                                   << ", p(uniqueness) = ";
                              RightPrecisionOut( cout, edge_copyno_p[e], 6 );
                              cout << "\n";    }
                         unique_edge[e] = False;    }    }    }    }

     // More trusted pair stuff.

     for ( int i = 0; i < ppp.isize( ); i++ )
     {    int r = BinPosition( trusteds, ppp[i].Left( ) );
          if ( r >= 0 && conflicts_tr[r] ) ppp_trusted[i] = False;
          r = BinPosition( trusteds, ppp[i].Right( ) );
          if ( r >= 0 && conflicts_tr[r] ) ppp_trusted[i] = False;    }
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !ppp_trusted[i] ) continue;
          const pp_pair& p = ppp[i];
          if ( !ReadyToClose( p, 4.0 ) ) continue;
          static vec<pp_closure> closures;
          GetClosures( p, pppL, 4.0, closures, False );
          if ( closures.empty( ) ) ppp_trusted[i] = False;    }
     for ( int i = 0; i < ppp.isize( ); i++ )
          if ( ppp_untrusted[i] ) ppp_trusted[i] = False;
     vec<Bool> solo( ppp.size( ) );
     for ( int i = 0; i < ppp.isize( ); i++ )
          solo[i] = ( pppmult[i] == 1 );
     for ( int pass = 1; pass <= 2; pass++ )
     {    vec<Bool> pppvalid;
          if (test_valid)
          {    static vec<Bool> remove;
               PrintPairLocs( False, "", "", ppp, h, kbb, genome, NHOOD_RADIUS,
                    NHOOD_RADIUS_INTERNAL, locsv, remove, &pppvalid );    }
          DeletePairsContraTrustedPairs( ppp, solo, ppp_trusted, pppL, 
               unique_edge, G, to_right_vertex, to_left_vertex, pppvalid,
               test_valid );    }
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, "in MakePairedPairs, after DeletePairsContra",
               sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    } 
     cout << TimeSince(clock3) << " in MakePairedPairs 3" << endl;

     // Merge trusted edges where possible.  Then delete pairs conflicting with
     // trusted pairs.

     vec<Bool> remove( ppp.size( ), False );
     dmult = 3.0;
     MergeAndTighten( ppp, pppL, unique_edge, dmult, to_right_vertex, 
          to_left_vertex, hstart_known, hstart, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
          G, ppp_trusted, edge_copyno, edge_copyno_p, remove, 
          MERGE_PAIRED_PAIRS_VERBOSE, MERGE_COPY_NUMBER_TWO, SHOW_PAIR_PLACEMENTS, 
          sub_dir, h, kbb, genome, locsv, MERGE_COPY_NUMBER_TWO_VERBOSE );
     double clock4b = WallClockTime( );
     solo.resize( ppp.size( ), False );
     EraseIf( ppp, remove );
     EraseIf( solo, remove );
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, "in MakePairedPairs, after MergeAndTighten",
               sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    } 
     vec<Bool> pppvalid;
     if (test_valid)
     {    static vec<Bool> remove;
          PrintPairLocs( False, "", "", ppp, h, kbb, genome, NHOOD_RADIUS,
               NHOOD_RADIUS_INTERNAL, locsv, remove, &pppvalid );    }
     cout << TimeSince(clock4b) << " in MakePairedPairs 4b" << endl;

     // If gap is <= 0 and dev is 0, close the gap.

     double clock5 = WallClockTime( );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    pp_pair& p = ppp[i];
          if ( p.Gap( ) > 0.0 || p.Dev( ) != 0.0 ) continue;
          int over = -int(floor(p.Gap( )));
          static vec<int> offsets;
          GetOverlaps( p.Left( ), p.Right( ), offsets );
          for ( int j = 0; j < offsets.isize( ); j++ )
          {    int o = offsets[j];
               int overlap = 0;
               for ( int k = Max( 0, o ); k < p.LeftSize( ); k++ )
                    overlap += pppL[ p.Left(k) ]; 
               for ( int k = 0; k < -o; k++ )
                    overlap += pppL[ p.Right(k) ]; 
               if ( overlap == over )
               {    static pp_read r;
                    JoinReads( p.Left( ), p.Right( ), o, r );
                    int len = 0;
                    for ( int m = 0; m < r.isize( ); m++ )
                         len += pppL[ r[m] ];
                    p = pp_pair( r, r, -len, 0 );    }    }    }

     // Resuscitate singleton edges that are covered by other edges.

     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( !solo[i] ) continue;
          const pp_pair& p = ppp[i];
          static vec<Bool> covL, covR;
          covL.resize_and_set( p.LeftSize( ), False );
          covR.resize_and_set( p.RightSize( ), False );
          for ( int m = 0; m < ppp.isize( ); m++ )
          {    if ( m == i ) continue;
               const pp_pair& q = ppp[m];
               static vec<int> o1, o2;
               GetOverlaps( p.Left( ), q.Left( ), o1 );
               if ( o1.empty( ) ) continue;
               GetOverlaps( p.Right( ), q.Right( ), o2 );
               if ( o2.empty( ) ) continue;
               for ( int i1 = 0; i1 < o1.isize( ); i1++ )
               {    for ( int i2 = 0; i2 < o2.isize( ); i2++ )
                    {    double g1 = p.Gap( ), g2 = q.Gap( );
                         for ( int j = p.LeftSize( ) - o1[i1]; j < q.LeftSize( ); 
                              j++ )
                         {    g1 -= pppL[ q.Left(j) ];    }
                         for ( int j = 0; j < -o2[i2]; j++ )
                              g1 -= pppL[ q.Right(j) ];
                         for ( int j = o1[i1] + q.LeftSize( ); j < p.LeftSize( ); 
                              j++ )
                         {    g2 -= pppL[ p.Left(j) ];    }
                         for ( int j = 0; j < o2[i2]; j++ )
                              g2 -= pppL[ p.Right(j) ];
                         double d1 = p.Dev( ), d2 = q.Dev( );
                         double g, d;
                         if ( !CombineMeasurements( g1, g2, d1, d2, dmult, g, d ) )
                              continue;
                         if (MERGE_PAIRED_PAIRS_VERBOSE)
                         {    cout << "Might use [" << m << "] to resuscitate solo ["
                                   << i << "].\n";    }
                         int low_L = Max( 0, o1[i1] );
                         int high_L = Min( p.LeftSize( ), o1[i1] + q.LeftSize( ) );
                         if ( !( low_L < high_L ) ) continue;
                         int low_R = Max( 0, o2[i2] );
                         int high_R = Min( p.RightSize( ), o2[i2] + q.RightSize( ) );
                         if ( !( low_R < high_R ) ) continue;
                         if (MERGE_PAIRED_PAIRS_VERBOSE)
                              PRINT4( low_L, high_L, low_R, high_R );
                         if ( low_L > 0 )
                         {    Bool eq_left = True, eq_right = True;
                              int x = low_L;
                              for ( int u = 0; u < x; u++ )
                                   if ( p.Left(u) != p.Left(x) ) eq_left = False;
                              for ( int u = x+1; u < p.LeftSize( ); u++ )
                                   if ( p.Left(u) != p.Left(x) ) eq_right = False;
                              if ( !eq_left && !eq_right )
                              {    while( low_L+1 < high_L )
                                   {    if ( p.Left(low_L) == p.Left(low_L+1) ) 
                                             ++low_L;
                                        else break;   }    }    }
                         if ( high_L < p.LeftSize( ) )
                         {    Bool eq_left = True, eq_right = True;
                              int x = high_L-1;
                              for ( int u = 0; u < x; u++ )
                                   if ( p.Left(u) != p.Left(x) ) eq_left = False;
                              for ( int u = x+1; u < p.LeftSize( ); u++ )
                                   if ( p.Left(u) != p.Left(x) ) eq_right = False;
                              if ( !eq_left && !eq_right )
                              {    while( low_L < high_L-1 )
                                   {    if ( p.Left(high_L-1) == p.Left(high_L-2) )
                                             --high_L;    
                                        else break;    }    }    }
                         if ( low_R > 0 )
                         {    Bool eq_left = True, eq_right = True;
                              int x = low_R;
                              for ( int u = 0; u < x; u++ )
                                   if ( p.Right(u) != p.Right(x) ) eq_left = False;
                              for ( int u = x+1; u < p.RightSize( ); u++ )
                                   if ( p.Right(u) != p.Right(x) ) eq_right = False;
                              if ( !eq_left && !eq_right )
                              {    while( low_R+1 < high_R )
                                   {    if ( p.Right(low_R) == p.Right(low_R+1) ) 
                                             ++low_R;
                                        else break;   }    }    }
                         if ( high_R < p.RightSize( ) )
                         {    Bool eq_left = True, eq_right = True;
                              int x = high_R-1;
                              for ( int u = 0; u < x; u++ )
                                   if ( p.Right(u) != p.Right(x) ) eq_left = False;
                              for ( int u = x+1; u < p.RightSize( ); u++ )
                                   if ( p.Right(u) != p.Right(x) ) eq_right = False;
                              if ( !eq_left && !eq_right )
                              {    while( low_R < high_R-1 )
                                   {    if ( p.Right(high_R-1) == p.Right(high_R-2) )
                                             --high_R;    
                                        else break;    }    }    }
                         if (MERGE_PAIRED_PAIRS_VERBOSE)
                              PRINT4( low_L, high_L, low_R, high_R );
                         for ( int u = low_L; u < high_L; u++ )
                              covL[u] = True;
                         for ( int u = low_R; u < high_R; u++ )
                              covR[u] = True;    }    }    }
          if ( Sum(covL) == p.LeftSize( ) && Sum(covR) == p.RightSize( ) )
          {    solo[i] = False;
               if (MERGE_PAIRED_PAIRS_VERBOSE)
               {    cout << "Resuscitating solo [" << i << "] " 
                         << ppp[i] << "\n";    }    }    }
               

     // Remove singleton edges.

     if (REMOVE_SOLO)
     {    for ( int i = 0; i < ppp.isize( ); i++ )
          {    if ( solo[i] )
               {    if ( IsClosed( ppp[i], pppL ) ) solo[i] = False;
                    else if (MERGE_PAIRED_PAIRS_VERBOSE)
                    {    cout << "Removing solo [" << i << "] " << ppp[i] 
                              << "\n";    }    }    }
          EraseIf( ppp, solo );    }

     // Print graph edges, checking their validity versus truth.

     if (test_valid)
     {    static vec<Bool> remove;
          PrintPairLocs( False, "", "", ppp, h, kbb, genome, NHOOD_RADIUS,
               NHOOD_RADIUS_INTERNAL, locsv, remove, &pppvalid );    }
     if (MERGE_PAIRED_PAIRS_VERBOSE) out << "\nRevised graph edges:\n" << endl;
     vec<Bool> cheat_delete( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    const pp_pair& p = ppp[i];
          if (MERGE_PAIRED_PAIRS_VERBOSE) out << "[" << i << "] " << p;
          if ( test_valid && !pppvalid[i] )
          {    if (MERGE_PAIRED_PAIRS_VERBOSE) out << " [INVALID]";    
               if (CHEAT_BY_DELETING_INVALID_EDGES) cheat_delete[i] = True;    }
          if (MERGE_PAIRED_PAIRS_VERBOSE) out << "\n";    }
     if (CHEAT_BY_DELETING_INVALID_EDGES) EraseIf( ppp, cheat_delete );
     cout << TimeSince(clock5) << " in MakePairedPairs 5" << endl;    }

void MergePairedPairs(

     /* both:     */  vec<pp_pair>& ppp,        // the pp_pairs

     /* input:    */  const HyperKmerPath& h,
                      const vec<Bool>& unique_edge,
                      const vec<int>& edge_copyno,
                      const vec<double>& edge_copyno_p,
                      const vec<int>& hstart,
                      const vec<Bool>& hstart_known,
                      const int NHOOD_RADIUS,
                      const int NHOOD_RADIUS_INTERNAL,
                      const double dmult,
                      vec<int>& pppL,           // edge lengths

     /* output:   */  ostream& out,             // for reporting

     /* control:  */  const Bool verbose,
                      const Bool MERGE_COPY_NUMBER_TWO,
                      const Bool MERGE_COPY_NUMBER_TWO_VERBOSE,

     /* for pair  */  const int SHOW_PAIR_PLACEMENTS,
     /* placement */  const String& sub_dir,
     /* printing: */  const KmerBaseBroker& kbb,
                      const vecbasevector& genome,
                      const serfvec<placement>& locsv

          )

{    vec<vrtx_t> to_right_vertex( h.EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.To(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexTo( i, j );
               to_right_vertex[e] = i;    }    }
     vec<vrtx_t> to_left_vertex( h.EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < h.N( ); i++ )
     {    for ( int j = 0; j < h.From(i).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexFrom( i, j );
               to_left_vertex[e] = i;    }    }
     digraphE<int> G( h, pppL );

     // Find contradictions.  Unique edges party to contradictions are treated as
     // untrusted.  If a given edge e contradicts four or more others, and each of
     // those contradict no edges other than e, delete e.

     vec<Bool> safe_unique = unique_edge;
     if (verbose) cout << "\n" << Date( ) << ": finding contradictions" << endl;
     vec<Bool> printed( ppp.size( ), False );
     vec< vec<int> > contra( ppp.size( ) );
     for ( int i1 = 0; i1 < ppp.isize( ); i1++ )
     {    const pp_pair& p1 = ppp[i1];
          for ( int i2 = 0; i2 < ppp.isize( ); i2++ )
          {    const pp_pair& p2 = ppp[i2];
               static vec<int> offsets;
               GetOverlaps( p1.Left( ), p2.Left( ), offsets );
               for ( int u1 = 0; u1 < p1.LeftSize( ); u1++ )
               {    if ( !unique_edge[ p1.Left(u1) ] ) continue;
                    for ( int u2 = 0; u2 < p2.LeftSize( ); u2++ )
                    {    if ( p2.Left(u2) != p1.Left(u1) ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) )
                         {    if (verbose)
                              {    cout << "[" << i1 << "] vs [" << i2 << "] at "
                                        << "unique " << BaseAlpha( p1.Left(u1) ) 
                                        << "\n";
                                   if ( !printed[i1] )
                                   {    cout << "[" << i1 << "] = " << ppp[i1] 
                                             << "\n";    }
                                   if ( !printed[i2] )
                                   {    cout << "[" << i2 << "] = " << ppp[i2] 
                                             << "\n";    }    }
                              printed[i1] = printed[i2] = True;
                              contra[i1].push_back(i2);
                              contra[i2].push_back(i1);
                              safe_unique[ p1.Left(u1) ] = False;    }    }    }    
               GetOverlaps( p1.Left( ), p2.Right( ), offsets );
               for ( int u1 = 0; u1 < p1.LeftSize( ); u1++ )
               {    if ( !unique_edge[ p1.Left(u1) ] ) continue;
                    for ( int u2 = 0; u2 < p2.RightSize( ); u2++ )
                    {    if ( p2.Right(u2) != p1.Left(u1) ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) )
                         {    if (verbose)
                              {    cout << "[" << i1 << "] vs [" << i2 << "] at "
                                        << "unique " << BaseAlpha( p1.Left(u1) ) 
                                        << "\n";
                                   if ( !printed[i1] )
                                   {    cout << "[" << i1 << "] = " << ppp[i1] 
                                             << "\n";    }
                                   if ( !printed[i2] )
                                   {    cout << "[" << i2 << "] = " << ppp[i2] 
                                             << "\n";    }    }
                              printed[i1] = printed[i2] = True;
                              contra[i1].push_back(i2);
                              contra[i2].push_back(i1);
                              safe_unique[ p1.Left(u1) ] = False;    }    }    }    
               GetOverlaps( p1.Right( ), p2.Right( ), offsets );
               for ( int u1 = 0; u1 < p1.RightSize( ); u1++ )
               {    if ( !unique_edge[ p1.Right(u1) ] ) continue;
                    for ( int u2 = 0; u2 < p2.RightSize( ); u2++ )
                    {    if ( p2.Right(u2) != p1.Right(u1) ) continue;
                         int o = u1 - u2;
                         if ( !Member( offsets, o ) )
                         {    if (verbose)
                              {    cout << "[" << i1 << "] vs [" << i2 << "] at "
                                        << "unique " << BaseAlpha( p1.Right(u1) ) 
                                        << "\n";
                                   if ( !printed[i1] )
                                   {    cout << "[" << i1 << "] = " << ppp[i1] 
                                             << "\n";    }
                                   if ( !printed[i2] )
                                   {    cout << "[" << i2 << "] = " << ppp[i2] 
                                             << "\n";    }    }
                              printed[i1] = printed[i2] = True;
                              contra[i1].push_back(i2);
                              contra[i2].push_back(i1);
                              safe_unique[ p1.Right(u1) ] 
                                   = False;    }    }    }    }    }
     for ( int i = 0; i < ppp.isize( ); i++ )
          UniqueSort( contra[i] );
     vec<Bool> contra_remove( ppp.size( ), False );
     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( contra[i].size( ) >= 4 )
          {    Bool bad = False;
               for ( int j = 0; j < contra[i].isize( ); j++ )
                    if ( !contra[ contra[i][j] ].solo( ) ) bad = True;
               if ( !bad ) 
               {    if (verbose)
                    {    cout << "contradiction - removing [" << i << "] " << ppp[i]
                              << "\n";    }
                    contra_remove[i] = True;    }    }    }
     EraseIf( ppp, contra_remove );
     if (verbose) cout << Date( ) << ": done" << endl;
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, "in MergePairedPairs, after contra_remove", sub_dir, 
               ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    } 
          
     // Merge and tighten up edges where possible.

     vec<Bool> remove( ppp.size( ), False );
     vec<Bool> editable( ppp.size( ), True );
     MergeAndTighten( ppp, pppL, safe_unique, dmult, to_right_vertex, 
          to_left_vertex, hstart_known, hstart, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
          G, editable, edge_copyno, edge_copyno_p, remove, verbose, 
          MERGE_COPY_NUMBER_TWO, SHOW_PAIR_PLACEMENTS, sub_dir, h, kbb, genome, 
          locsv, MERGE_COPY_NUMBER_TWO_VERBOSE );
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, 
               "in MergePairedPairs, after MergeAndTighten", sub_dir, ppp, h, kbb, 
               genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, locsv, remove );    } 

     // Delete edges having a highly negative gap but no closure.

     for ( int i = 0; i < ppp.isize( ); i++ )
     {    if ( remove[i] ) continue;
          if ( ppp[i].Gap( ) < -5.0 * ppp[i].Dev( ) && ppp[i].Dev( ) > 0 ) 
          {    pp_mpair p( ppp[i] );
               vec<pp_closure> closures;
               GetClosures( p, pppL, 4.0, closures );
               if ( closures.empty( ) ) 
               {    cout << "\nDELETING NEG: " << p << "\n";
                    remove[i] = True;    }    }    }
     EraseIf( ppp, remove );
     if ( SHOW_PAIR_PLACEMENTS >= 1 )
     {    PrintPairLocs( True, "in MergePairedPairs, after highly negative gaps", 
               sub_dir, ppp, h, kbb, genome, NHOOD_RADIUS, NHOOD_RADIUS_INTERNAL, 
               locsv );    } 

     // Try again to merge copy number two stuff.

     if (MERGE_COPY_NUMBER_TWO)
     {    vec<Bool> remove( ppp.size( ), False );
          MergeCopyNumberTwo( h, kbb, ppp, pppL, to_left_vertex, to_right_vertex, 
               G, edge_copyno, remove, MERGE_COPY_NUMBER_TWO_VERBOSE );
          EraseIf( ppp, remove );    }

     // Print pairs.

     if (verbose)
     {    cout << "\nPairs after merging:\n";
          for ( int u = 0; u < ppp.isize( ); u++ )
               cout << "[" << u << "] " << ppp[u] << "\n";    }    }

void MakePairedPairsAncillary(

     /* input:   */  const vec<pp_pair>& ppp,  // the pp_pairs
                     const HyperKmerPath& h,   // the HyperKmerPath

     /* outputs: */  vecKmerPath& paths,       // KmerPaths of reads in the pairs
                     vec<read_pairing>& pairs  // the pairs
          )

{    paths.clear( ), pairs.clear( );
     for ( int u = 0; u < ppp.isize( ); u++ )
     {    int id1 = paths.size( );
          static KmerPath p;
          p.Clear( );
          for ( int m = 0; m < ppp[u].LeftSize( ); m++ )
               p.Append( h.EdgeObject( ppp[u].Left(m) ) );
          paths.push_back_reserve(p);
          int id2 = paths.size( );
          p.Clear( );
          for ( int m = 0; m < ppp[u].RightSize( ); m++ )
               p.Append( h.EdgeObject( ppp[u].Right(m) ) );
          p.Reverse( );
          paths.push_back_reserve(p);
          read_pairing rp;
          rp.id1 = id1, rp.id2 = id2;
          rp.sep = int( round( ppp[u].Gap( ) ) ) - ( h.K( ) - 1 );
          rp.sd =  int( ceil( ppp[u].Dev( ) ) );
          rp.t = other, rp.weight = 1;
          pairs.push_back(rp);    }    }

template Bool digraphE<int>::ThisClose(vrtx_t, vrtx_t, int) const;
