// Copyright (c) 2004 Broad Institute/Massachusetts Institute of Technology

#ifndef PATHS_KMERPATHDATABASE_H
#define PATHS_KMERPATHDATABASE_H

#include "paths/KmerPath.h"

// An encapsulated vec<tagged_rpint>, which allows per-database
// caching for lookups.

class KmerPathDatabase
{
 public:
  KmerPathDatabase();

  KmerPathDatabase( const String& filename );
  KmerPathDatabase( const KmerPathDatabase& other );
  KmerPathDatabase( const vec<tagged_rpint>* p_vecOfTaggedRpints );
  KmerPathDatabase( const vecKmerPath& fwdPaths,
                    const vecKmerPath& revPaths );
  KmerPathDatabase( const vec<KmerPath>& fwdPaths,
                    const vec<KmerPath>& revPaths );
  
  KmerPathDatabase& operator= ( const KmerPathDatabase& other );
  
  ~KmerPathDatabase();

  
  void Contains( longlong kmerId, 
                 vec<longlong>& answer, bool append = false ) const;

  void Intersects( const KmerPathInterval& kpi, 
                   vec<longlong>& answer, bool append = false ) const;

  longlong Instance( longlong kmerId ) const;

  void GetHighFrequencyKmers( vec<longlong>& hiFreqKmers, const int minFreq ) const;

  const tagged_rpint& operator[] ( const size_t index ) const
  {
    return (*mp_taggedRpints)[ index ];
  }

  const vec<tagged_rpint>& ConstRef() const { return *mp_taggedRpints; }

  vec<tagged_rpint>::const_iterator Begin() const { return mp_taggedRpints->begin(); }
  vec<tagged_rpint>::const_iterator End()   const { return mp_taggedRpints->end(); }

  longlong size() const { return mp_taggedRpints->size(); }

  // The mp_rpints vector is written out to or read in from the given filename.
  void Write( const String& filename ) const;
  void Read( const String& filename );

  float GetHitRate() const { return (float)m_hits/max(1.0f,(float)(m_hits+m_misses)); }
  float GetMissDistance() const { return (float)m_missDistance/max(1.0f,(float)m_misses); }

  void ResetHitRate() const { m_hits = 0; m_misses = 0; m_missDistance = 0; }

 private:
  template <class KmerPathVector>
  void ConstructFromKmerPaths( const KmerPathVector& fwPaths, const KmerPathVector& rcPaths );
  
  void CopyFrom( const KmerPathDatabase& other );

  const vec<tagged_rpint>* mp_taggedRpints;
  bool m_ownRpints;
  mutable unsigned int m_to;
  mutable longlong m_hits;
  mutable longlong m_misses;
  mutable longlong m_missDistance;
};

#endif
