/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include <queue>

#include "CoreTools.h"
#include "graph/Digraph.h"
#include "Equiv.h"
#include "math/Functions.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "paths/ImproperMerge.h"
#include "paths/InternalMerge.h"
#include "paths/PileReadsOnPath.h"
#include "paths/SuperBaseVector.h"

// Defined below.

void InternalMergeImpl( HyperKmerPath& h, 
     const NegativeGapValidator& ngv, int min_overlap, 
     int min_proper_overlap, Bool SPECIAL,
     Bool seed_unique, Bool seed_unique_weak, const vec<tagged_rpint>& uniqdb );

// These two functions massage their arguments and pass them on to the Impl function.

void InternalMerge( HyperKmerPath& h, 
     const NegativeGapValidator& ngv, int min_overlap, 
     int min_proper_overlap, Bool SPECIAL )
{    static vec<tagged_rpint> uniqdb_null;
     InternalMergeImpl( h, ngv, min_overlap, min_proper_overlap, SPECIAL,
          False, False, uniqdb_null );    }


void InternalMergeOnUnique( HyperKmerPath& h, 
     const NegativeGapValidator& ngv, int min_overlap, 
     int min_proper_overlap, Bool SPECIAL, const vec<tagged_rpint>& uniqdb,
     Bool seed_unique_weak )
{    InternalMergeImpl( h, ngv, min_overlap, min_proper_overlap, SPECIAL,
          True, seed_unique_weak, uniqdb );    }


// This calls InternalMergeOnUnique on groups of components of the input HKPS.

void GroupedInternalMerge( const vec<HyperKmerPath>& hin, HyperKmerPath& hout, 
     const KmerBaseBroker& kbb, const int min_perfect_match_to_group, 
     const int group_steps, const NegativeGapValidator& ngv, int min_overlap, 
     int min_proper_overlap, Bool SPECIAL, const vec<tagged_rpint>& uniqdb,
     const Bool SHORTEST_MERGE )
{    int K = kbb.GetK( );
     vec<int> source;
     PRINT( hin.size( ) ); // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
     vecbasevector bases;
     for ( int i = 0; i < hin.isize( ); i++ )
     {    const HyperKmerPath& h = hin[i];
          for ( int j = 0; j < h.EdgeObjectCount( ); j++ )
          {    static basevector b;
               b = kbb.Seq( h.EdgeObject(j) );
               if ( b.isize( ) < min_perfect_match_to_group ) continue;
               source.push_back(i);
               bases.push_back_reserve(b);    }    }
     PerfectAligner pal( 96, PerfectAligner::findProperOnly );
     vec<alignment_plus> aligns;
     double pclock = WallClockTime( );
     pal.Align( bases, aligns );
     cout << TimeSince(pclock) << " used by PerfectAligner in GroupedInternalMerge"
          << endl;
     vec< vec<int> > connected( hin.size( ) );
     for ( int i = 0; i < aligns.isize( ); i++ )
     {    const alignment_plus& ap = aligns[i];
          if ( ap.a.Pos1( ) - ap.a.pos1( ) < min_perfect_match_to_group ) continue;
          int id1 = source[ ap.Id1( ) ], id2 = source[ ap.Id2( ) ];
          connected[id1].push_back(id2);
          connected[id2].push_back(id1);    }
     for ( int i = 0; i < hin.isize( ); i++ )
          UniqueSort( connected[i] );
     vec<Bool> used( hin.size( ), False );
     int first_unused = 0;
     vec<HyperKmerPath> p;
     while( first_unused < hin.isize( ) )
     {    if ( used[first_unused] )
          {    ++first_unused;
               continue;    }
          static vec<HyperKmerPath> hgroup;
          static vec<int> group, to;
          hgroup.clear( ), group.clear( );
          group.push_back(first_unused);
          used[first_unused] = True;
          hgroup.push_back( hin[first_unused] );
          for ( int i = 0; i < group_steps; i++ )
          {    to.clear( );
               for ( int j = 0; j < group.isize( ); j++ )
               {    const vec<int>& c = connected[ group[j] ];
                    for ( int u = 0; u < c.isize( ); u++ )
                    {    if ( used[ c[u] ] ) continue;
                         to.push_back( c[u] );
                         hgroup.push_back( hin[ c[u] ] );
                         used[ c[u] ] = True;    }    }
               group.append(to);    }
          HyperKmerPath hg( K, hgroup );
          if ( hgroup.size( ) > 1 )
          {    cout << Date( ) << ": merging group of size " << hgroup.size( )
                    << endl;    }
          // hg.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False ); // *****************
          InternalMergeOnUnique( hg, ngv, min_proper_overlap, min_proper_overlap, SPECIAL, uniqdb );
          p.push_back(hg);    }
     hout.SetK(K);
     hout.SetToDisjointUnionOf(p);
     cout << Date( ) << ": starting final merge" << endl; // XXXXXXXXXXXXXXXXXXXXXXX
     PRINT( p.size( ) ); // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
     InternalMergeOnUnique( hout, ngv, min_proper_overlap, min_proper_overlap, SPECIAL, uniqdb );
     InternalMergeOnUnique( hout, ngv, min_overlap, min_overlap, SPECIAL, uniqdb, True );
     //InternalMergeOnUnique( hout, ngv, 5000, 500, SPECIAL, uniqdb );

     //InternalMergeOnUnique( hout, ngv, 2000, 500, SPECIAL, uniqdb ); // ?????????????????????
     if (SHORTEST_MERGE) InternalMergeOnUnique( hout, ngv, 2000, 200, SPECIAL, uniqdb );

     //InternalMergeOnUnique( hout, ngv, 5000, 5000, SPECIAL, uniqdb, True );    
}



// InternalMergeImpl helper functions.

KmerPath Between( const KmerPath& p, const KmerPathSplitPoint& left,
     const KmerPathSplitPoint& right )
{    KmerPath answer;
     ForceAssert( left < right );                               
     ForceAssertLe( right.IntervalsToLeft( ), p.NSegments( ) );
     if ( left.IntervalsToLeft( ) == right.IntervalsToLeft( ) )
     {    const KmerPathInterval& I = p.Segment( left.IntervalsToLeft( ) );
          answer.AddSegment( I.Start( ) + left.KmersInPartialToLeft( ),
               I.Start( ) + right.KmersInPartialToLeft( ) - 1 );    }
     else
     {    const KmerPathInterval& Ileft = p.Segment( left.IntervalsToLeft( ) );
          if ( left.KmersInPartialToLeft( ) == 0 ) answer.AddSegment(Ileft);
          else
          {    answer.AddSegment( Ileft.Start( ) + left.KmersInPartialToLeft( ), 
                    Ileft.Stop( ) );    }
          for ( int u = left.IntervalsToLeft( ) + 1;  
               u < right.IntervalsToLeft( ); u++ )
          {    answer.AddSegment( p.Segment(u) );    }
          if ( right.KmersInPartialToLeft( ) > 0 )
          {    const KmerPathInterval& Iright 
                    = p.Segment( right.IntervalsToLeft( ) );
               answer.AddSegment( Iright.Start( ), Iright.Start( ) 
                    + right.KmersInPartialToLeft( ) - 1 );    }    }
     return answer;    }

void PrintLengths( const KmerPath& p )
{    for ( int i = 0; i < p.NSegments( ); i++ )
     {    if ( i > 0 ) cout << " ";
          cout << p.Segment(i).Length( );    }
     cout << "\n";    }

void PrintLengths( const EmbeddedSubPath<KmerPath> & p )
{    for ( int i = 0; i < p.NEdges( ); i++ )
     {    if ( i > 0 ) cout << " / ";
          for ( int j = 0; j < p.EdgeObject(i).NSegments( ); j++ )
          {    if ( j > 0 ) cout << " ";
               cout << p.EdgeObject(i).Segment(j).Length( );    }    }
     cout << "\n";    }

// SplitEdgeInPointedSubPath: given an EmbeddedSubPath u = (a,e) of a HyperKmerPath 
// 
//    e0       em-1
// a0 --> ...  --> am,
//
// and a kmer at position x on the EmbeddedSubPath (comprising a "PointedSubPath"), 
// an edge ej (either e0 or em-1), and a splitting of that edge in two (given by two 
// KmerPaths s1, s2 whose concatenation is the edge object ej), do the following:
// - edit the HyperKmerPath by splitting the edge;
// - update the data (u,x) accordingly.
// We assume that if the edge split point is to the left of x, then j = 0, and if
// the edge split point is to the right of x, then j = m-1 (and split_to_left_of_x
// must be specified, so giving j is redundant).  Moreover, in the first case, we
// edit u so that it starts at the edge split point, and in the second case, we
// edit u so that it stops at the edge split point.

void SplitEdgeInPointedSubPath( HyperKmerPath& h, EmbeddedSubPath<KmerPath>& u,
     MultiKmerPathLoc& x, int j, const KmerPath& s1, const KmerPath& s2,
     Bool split_to_left_of_x )
{    int segments_in_split_edge = u.EdgeObject(j).NSegments( );
     int next_to_last_vertex = u.Vertex( u.NVertices( ) - 2 );
     int new_vertex = h.N( );
     int new_edge1 = h.EdgeObjectCount( ), new_edge2 = h.EdgeObjectCount( ) + 1;
     h.SplitEdge( u.Vertex(j), u.EdgeObjectFromIndex(j), s1, s2 );
     int new_edge2_from_new_vertex 
          = h.EdgeObjectIndexToFromIndex( new_vertex, new_edge2 );
     int nsegs1 = s1.NSegments( ), nsegs2 = s2.NSegments( );
     Bool split_at_segment_boundary = ( nsegs1 + nsegs2 == segments_in_split_edge );
     if (split_to_left_of_x)
     {    ForceAssertEq( j, 0 );
          u.SetVertex( 0, new_vertex );
          u.SetEdge( 0, new_edge2_from_new_vertex );
          if ( x.Edge( ) == 0 )
          {    if ( !split_at_segment_boundary )
               {    x.SetSegmentOnEdge( x.SegmentOnEdge( ) - ( nsegs1 - 1 ) );
                    if ( x.SegmentOnEdge( ) == 0 )
                    {    x.SetKmerOnSegment( 
                              x.KmerOnSegment( ) 
                                   - s1.Segment( nsegs1 - 1 ).Length( ) );    }    }
               else x.SetSegmentOnEdge( x.SegmentOnEdge( ) - nsegs1 );    }    }
     else
     {    ForceAssertEq( j, u.NEdges( ) - 1 );
          int new_edge1_from_next_to_last_vertex
               = h.EdgeObjectIndexToFromIndex( next_to_last_vertex, new_edge1 );
          u.SetVertex( u.NVertices( ) - 1, new_vertex );
          u.SetEdge( u.NEdges( ) - 1, new_edge1_from_next_to_last_vertex );    }    }

// We define a class which can be stuffed into a priority queue and used to track
// extensions as we iteratively extend them through the graph.

class AlignExtend {
     
     public:

     AlignExtend( ) { }
     AlignExtend( const PointedSubPathPair& p, 
          const vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> >& ends,
          int align_length )
          : p_(p), ends_(ends), align_length_(align_length), terminal_(False) { }

     const PointedSubPathPair& PathPair( ) const { return p_; }

     const EmbeddedSubPath<KmerPath>& Path( int i ) const { return p_.Path(i); }
     const EmbeddedSubPath<KmerPath>& Path1( ) const { return p_.Path1( ); }
     const EmbeddedSubPath<KmerPath>& Path2( ) const { return p_.Path2( ); }

     const MultiKmerPathLoc& Loc1( ) const { return p_.Loc1( ); }
     const MultiKmerPathLoc& Loc2( ) const { return p_.Loc2( ); }

     const vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> >& Ends( ) { return ends_; }
     const pair<MultiKmerPathLoc,MultiKmerPathLoc>& Ends(int i) { return ends_[i]; }

     int AlignLength( ) const { return align_length_; }

     Bool Terminal( ) const { return terminal_; }

     void SetTerminal( const Bool term = True ) { terminal_ = term; }

     // Priority function for extensions: 
     // * alignments that can't be extended are lower priority that ones that can be.
     // * alignments that are longer are lower priority than ones that are shorter.
     // * alignments are then ordered by the vertices in their paths, so that alignments 
     //   of equal termination state, equal length, and equal start and vertices on both 
     //   components are considered equivalent.
     // This results in a breadth-first search where equivalent length paths that
     // terminate at the same set of vertices collapse into a single result (regardless
     // of the content of the path).  This compromise makes searching through diploid
     // components much faster.
     friend bool operator<( const AlignExtend& ae1, const AlignExtend& ae2 )
     {
       // Return true if ae1 is lower priority than ae2.
       if ( ae1.Terminal( ) && !ae2.Terminal( ) ) return true;
       if ( !ae1.Terminal( ) && ae2.Terminal( ) ) return false;
       if ( ae1.AlignLength( ) > ae2.AlignLength( ) ) return true;
       if ( ae1.AlignLength( ) < ae2.AlignLength( ) ) return false;
       for ( int p = 0; p < 2; ++p ) {
         if ( ae1.Path(p).FirstVertex() < ae2.Path(p).FirstVertex() ) return true;
         if ( ae1.Path(p).FirstVertex() > ae2.Path(p).FirstVertex() ) return false;
         if ( ae1.Path(p).LastVertex() < ae2.Path(p).LastVertex() ) return true;
         if ( ae1.Path(p).LastVertex() > ae2.Path(p).LastVertex() ) return false;
       }
       return false;
     }

     private:

     PointedSubPathPair p_;
     vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> > ends_;
     int align_length_;
     Bool terminal_;

};

class AlignExtendQueue 
     : public set<AlignExtend>  {

     public:

     const AlignExtend& top() const { Assert( !empty() ); return *(--end()); }
     void pop() { Assert( !empty() ); erase(--end()); }
     void push( const AlignExtend& ae ) { insert(ae); }

     Bool WorkToDo( ) const
     {    if ( empty( ) ) return False;
          return !top( ).Terminal( ) ;    }
};

// InternalMergeLeftExtend and InternalMergeRightExtend are helpers for 
// InternalMerge.  Given a PointedSubPathPair p and ends of an alignment of it, it 
// tries to enlarge the alignment by extending to the left [resp. right].  

void InternalMergeLeftExtend( const HyperKmerPath& h, AlignExtendQueue& Q,
     const NegativeGapValidator& ngv )
{    KmerPath M;
     int naligns;
     vec<KmerPathSplitPoint> il, jl;
     vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> > endsplus;
     while( Q.WorkToDo( ) )
     {    static AlignExtend E;
          E = Q.top( );
          Q.pop( );

          // If alignment does not go to the left end on either sequence, there is
          // nothing to do.

          if ( !E.Ends(0).first.LeftEnd( ) && !E.Ends(1).first.LeftEnd( ) )
          {    E.SetTerminal( );
               Q.push(E);
               continue;    }
     
          // Handle case where alignment goes to the left end on one sequence
          // but not on the other.
          
          Bool found_extension = False;
          Bool not_to_end = 
               !E.Ends(0).first.LeftEnd( ) || !E.Ends(1).first.LeftEnd( );
          for ( int f = 0; f < 2; f++ )
          {    int g = 1 - f;
               if ( E.Ends(f).first.LeftEnd( ) && !E.Ends(g).first.LeftEnd( ) )
               {    int w = E.Path(f).Vertex(0);
                    for ( int i = 0; i < h.To(w).isize( ); i++ )
                    {    int v = h.To(w)[i];
                         const KmerPath& e = h.EdgeObjectByIndexTo( w, i );
                         int ei = h.InputToOutputFrom( w, i );

                         // Modify path by prepending e to the one sequence.

                         EmbeddedSubPath<KmerPath> 
                              path1 = E.Path1( ), path2 = E.Path2( );
                         MultiKmerPathLoc loc1 = E.Loc1( ), loc2 = E.Loc2( );
                         if ( f == 0 )
                         {    path1.Prepend( v, ei );
                              loc1.SetEdge( loc1.Edge( ) + 1 );    }
                         else
                         {    path2.Prepend( v, ei );
                              loc2.SetEdge( loc2.Edge( ) + 1 );    }
                         PointedSubPathPair pplus( path1, path2, loc1, loc2 );
                         // If either path involves a loop, skip it.
                         if ( pplus.HasDups( ) ) continue;
                         // If the paths cross, skip it.
                         if ( NSharedEdges( path1, path2 ) != 0 ) continue;

                         // Align.

                         h.AlignSubpaths( pplus, naligns, endsplus, M, il, jl, ngv );
                         if ( naligns == 0 ) continue;

                         // If the extension did not use the added edge, forget it.

                         if ( endsplus[f].first.Edge( ) != 0 ) continue;

                         // Save extension.

                         found_extension = True;
                         AlignExtend Enew( pplus, endsplus, M.KmerCount( ) );
                         Q.push(Enew);    }    }    }
          if (not_to_end)
          {    if ( !found_extension )
               {    E.SetTerminal( );
                    Q.push(E);    }
               continue;    }
     
          // Handle case where alignment goes to the left end on both sequences.

          int w1 = E.Path1( ).Vertex(0), w2 = E.Path2( ).Vertex(0);
          for ( int i1 = 0; i1 < h.To(w1).isize( ); i1++ )
          {    int v1 = h.To(w1)[i1];
               const KmerPath& e1 = h.EdgeObjectByIndexTo( w1, i1 );
               int ei1 = h.InputToOutputFrom( w1, i1 );
               for ( int i2 = 0; i2 < h.To(w2).isize( ); i2++ )
               {    int v2 = h.To(w2)[i2];
                    const KmerPath& e2 = h.EdgeObjectByIndexTo( w2, i2 );
                    if ( h.EdgeObjectIndexByIndexTo( w1, i1 )
                         == h.EdgeObjectIndexByIndexTo( w2, i2 ) )
                    {    continue;    }
                    int ei2 = h.InputToOutputFrom( w2, i2 );

                    // Modify p by prepending e1 and e2 to the sequences.

                    EmbeddedSubPath<KmerPath> path1 = E.Path1( ), path2 = E.Path2( );
                    MultiKmerPathLoc loc1 = E.Loc1( ), loc2 = E.Loc2( );
                    path1.Prepend( v1, ei1 ), path2.Prepend( v2, ei2 );
                    loc1.SetEdge( loc1.Edge( ) + 1 );
                    loc2.SetEdge( loc2.Edge( ) + 1 );
                    PointedSubPathPair pplus( path1, path2, loc1, loc2 );
                    // If either path involves a loop, skip it.
                    if ( pplus.HasDups( ) ) continue;
                    // If the paths cross, skip it.
                    if ( NSharedEdges( path1, path2 ) != 0 ) continue;

                    // Align.

                    h.AlignSubpaths( pplus, naligns, endsplus, M, il, jl, ngv );
                    if ( naligns == 0 ) continue;

                    // If the extension did not use the added edges, forget it.

                    if ( endsplus[0].first.Edge( ) != 0 ) continue;

                    // Save extension.

                    found_extension = True;
                    AlignExtend Enew( pplus, endsplus, M.KmerCount( ) );
                    Q.push(Enew);    }    }
          if ( !found_extension )
          {    E.SetTerminal( );
               Q.push(E);    }    }    }

void InternalMergeRightExtend( const HyperKmerPath& h, AlignExtendQueue& Q,
     const NegativeGapValidator& ngv )
{    KmerPath M;
     int naligns;
     vec<KmerPathSplitPoint> il, jl;
     vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> > endsplus;
     while( Q.WorkToDo( ) )
     {    static AlignExtend E;
          E = Q.top( );
          Q.pop( );

          // If alignment does not go to the right end on either sequence, there is
          // nothing to do.

          if ( !E.Ends(0).second.RightEnd( E.Path1( ) ) 
               && !E.Ends(1).second.RightEnd( E.Path2( ) ) )
          {    E.SetTerminal( );
               Q.push(E);
               continue;    }

          // Handle case where alignment goes to the right end on one sequence
          // but not on the other.

          Bool found_extension = False;
          Bool not_to_end = !E.Ends(0).second.RightEnd( E.Path1( ) ) 
               || !E.Ends(1).second.RightEnd( E.Path2( ) );
          for ( int f = 0; f < 2; f++ )
          {    int g = 1 - f;
               if ( E.Ends(f).second.RightEnd( E.Path(f) ) 
                    && !E.Ends(g).second.RightEnd( E.Path(g) ) )
               {    int v = E.Path(f).Vertex( E.Path(f).NVertices( ) - 1 );
                    for ( int i = 0; i < h.From(v).isize( ); i++ )
                    {    int w = h.From(v)[i];
                         const KmerPath& e = h.EdgeObjectByIndexFrom( v, i );
                         int ei = i;

                         // Modify p by appending e to the one sequence.

                         EmbeddedSubPath<KmerPath> 
                              path1 = E.Path1( ), path2 = E.Path2( );
                         MultiKmerPathLoc loc1 = E.Loc1( ), loc2 = E.Loc2( );
                         if ( f == 0 ) path1.Append( w, ei );
                         else path2.Append( w, ei );
                         PointedSubPathPair pplus( path1, path2, loc1, loc2 );
                         // If either path involves a loop, skip it.
                         if ( pplus.HasDups( ) ) continue;
                         // If the paths cross, skip it.
                         if ( NSharedEdges( path1, path2 ) != 0 ) continue;

                         // Align.

                         h.AlignSubpaths( pplus, naligns, endsplus, M, il, jl, ngv );
                         if ( naligns == 0 ) continue;

                         // If the extension did not use the added edge, forget it.

                         int last_edge = E.Path(f).NEdges( );
                         if ( endsplus[f].second.Edge( ) != last_edge ) continue;

                         // Save extension.

                         found_extension = True;
                         AlignExtend Enew( pplus, endsplus, M.KmerCount( ) );
                         Q.push(Enew);    }    }    }
          if (not_to_end)
          {    if ( !found_extension )
               {    E.SetTerminal( );
                    Q.push(E);    }
               continue;    }
     
          // Handle case where alignment goes to the right end on both sequences.

          int v1 = E.Path1( ).Vertex( E.Path1( ).NVertices( ) - 1 ); 
          int v2 = E.Path2( ).Vertex( E.Path2( ).NVertices( ) - 1 );
          for ( int i1 = 0; i1 < h.From(v1).isize( ); i1++ )
          {    int w1 = h.From(v1)[i1];
               const KmerPath& e1 = h.EdgeObjectByIndexFrom( v1, i1 );
               int ei1 = i1;
               for ( int i2 = 0; i2 < h.From(v2).isize( ); i2++ )
               {    int w2 = h.From(v2)[i2];
                    const KmerPath& e2 = h.EdgeObjectByIndexFrom( v2, i2 );
                    if ( h.EdgeObjectIndexByIndexFrom( v1, i1 )
                         == h.EdgeObjectIndexByIndexFrom( v2, i2 ) )
                    {    continue;    }
                    int ei2 = i2;

                    // Modify p by appending e1 and e2 to the sequences.

                    EmbeddedSubPath<KmerPath> path1 = E.Path1( ), path2 = E.Path2( );
                    MultiKmerPathLoc loc1 = E.Loc1( ), loc2 = E.Loc2( );
                    path1.Append( w1, ei1 ), path2.Append( w2, ei2 );
                    PointedSubPathPair pplus( path1, path2, loc1, loc2 );
                    // If either path involves a loop, skip it.
                    if ( pplus.HasDups( ) ) continue;
                    // If the paths cross, skip it.
                    if ( NSharedEdges( path1, path2 ) != 0 ) continue;

                    // Align.

                    h.AlignSubpaths( pplus, naligns, endsplus, M, il, jl, ngv );
                    if ( naligns == 0 ) continue;

                    // If the extension did not use the added edges, forget it.

                    if ( endsplus[0].second.Edge( ) != E.Path1( ).NEdges( ) ) 
                         continue;

                    // Save extension.

                    found_extension = True;
                    AlignExtend Enew( pplus, endsplus, M.KmerCount( ) );
                    Q.push(Enew);    }    }
          if ( !found_extension )
          {    E.SetTerminal( );
               Q.push(E);    }    }    }

void InternalMergeImpl( HyperKmerPath& hkp, 
     const NegativeGapValidator& ngv, int min_overlap, 
     int min_proper_overlap, Bool SPECIAL, 
     Bool seed_unique, Bool seed_unique_weak, const vec<tagged_rpint>& uniqdb )
{    
     hkp.TestValid( );
     
     // If there is more than one component, first process each component 
     // separately.

     equiv_rel ee( hkp.N( ) );
     for ( int i = 0; i < hkp.N( ); i++ )
          for ( int j = 0; j < hkp.From(i).isize( ); j++ )
               ee.Join( i, hkp.From(i)[j] );
     int norbits = ee.OrbitCount( );
     if (SPECIAL)
     {    if ( norbits > 1 )
          {    vec<HyperKmerPath> merged_components;
               merged_components.reserve(norbits);
               for ( vrtx_t v = 0; v < hkp.N( ); v++ )
               {    if ( !ee.Representative(v) ) continue;
                    static vec<vrtx_t> o;
                    ee.Orbit( v, o );
                    HyperKmerPath h( hkp, o );
                    if ( h.EdgeObjectCount( ) > 1 )
                    {
                         /*
                         if ( h.EdgeObjectCount( ) > 10 ) // ZZZ
                              PRINT3( v, N( ), h.EdgeObjectCount( ) ); // ZZZZZZZZZZ
                         */
                         InternalMerge( h,
                              ngv, min_overlap, min_proper_overlap, False );    }
                    merged_components.push_back(h);    }
               hkp.SetToDisjointUnionOf(merged_components);    }    }

     // There are three phases.
     // Phase 1: we accept only anchored joins within a given component.
     // Phase 2: we accept only long proper joins and anchored joins.  
     // Phase 3: we accept anything.
     // However, if there is only one component, we only perform phase 1, and if
     // there is more than one component, we only perform phases 2 and 3.
     // [Only on if SPECIAL=True!]

     int phase = ( norbits == 1 ? 1 : 2 );

     while(1)
     {    Bool found_merge = False;

          // Find connected components of HyperKmerPath.

          equiv_rel e( hkp.N( ) );
          for ( int i = 0; i < hkp.N( ); i++ )
               for ( int j = 0; j < hkp.From(i).isize( ); j++ )
                    e.Join( i, hkp.From(i)[j] );
          vec<int> reps;
          e.OrbitReps(reps);
          int nreps = reps.size( );
          vec< vec<int> > components0(nreps);
          for ( int i = 0; i < nreps; i++ )
               e.Orbit( reps[i], components0[i] );
          vec<int> to_rep( hkp.N( ) );
          for ( int i = 0; i < nreps; i++ )
          {    for ( int j = 0; j < components0[i].isize( ); j++ )
                    to_rep[ components0[i][j] ] = i;    }

          // If HyperKmerPath has more than one component, append reversed copy of 
          // HyperKmerPath.

          int nverts = hkp.N( );
          int nedges = hkp.EdgeObjectCount( );
          longlong nkmers = 0; // ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ
          for ( int i = 0; i < nedges; i++ ) // ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ
               nkmers += hkp.EdgeObject(i).KmerCount( ); // ZZZZZZZZZZZZZZZZZZZZZZZZZZZZ
          if ( !SPECIAL || nreps > 1 )
          {    HyperKmerPath rev_copy(hkp);
               rev_copy.Reverse( );
               for ( int i = 0; i < nedges; i++ )
               {    static KmerPath p;
                    p = hkp.EdgeObject(i);
                    p.Reverse( );
                    rev_copy.SetEdgeObject( i, p );    }
               hkp.Append(rev_copy);    }

          // Set up structure to track connected components and correspondence
          // with reverse complements.

          vec< vec<int> > components = components0;
          if ( !SPECIAL || nreps > 1 )
          {    components.append(components0);
               for ( int i = nreps; i < 2 * nreps; i++ )
               {    for ( int j = 0; j < components[i].isize( ); j++ )
                         components[i][j] += nverts;    }    }
          vec<int> touched( components.size( ), False );
          // PRINT( components.size( ) ); // XXX

          // Find shared kmers, and remove those for which both copies lie in 
          // the reversed copy of the original HyperKmerPath.  Also remove those
          // which are between an edge and its reverse.  Also swap order in shares
          // if needed so that reverse comes second.

          vec<KmerPathCorrespondence> shares;
          hkp.SharedKmers( shares, min_overlap );
          int count = 0;
          for ( int i = 0; i < shares.isize( ); i++ )
          {    if ( ( shares[i].From1( ) < nverts || shares[i].From2( ) < nverts )
                    && shares[i].Id1( ) != shares[i].Id2( ) + nedges
                    && shares[i].Id2( ) != shares[i].Id1( ) + nedges )
               {    if ( shares[i].From1( ) >= nverts ) shares[i].Swap( );
                    if ( count != i ) shares[count] = shares[i];
                    ++count;    }    }
          shares.resize(count);
          // PRINT( shares.size( ) ); // XXX
          /*
          // if ( reps.size( ) > 1 ) // ZZZ
          if ( EdgeObjectCount( ) > 10 ) // ZZZ
          {    cout << Date( ) << ": #" << phase << ", ncomp = " // ZZZ
                    << components.size( ) << ", nshare = " << shares.size( ) // ZZZ
                    << ", nkmers = " << nkmers << endl;    } // ZZZ
          */

          // If seed_unique = True, remove shares whose perfect extension does not 
          // contain unique sequence, except for cases where we can "zipper up".

          if ( seed_unique ) {
               static vec<vrtx_t> to_left, to_right;
	       static vec<int> edgelength;
               hkp.ToLeft(to_left), hkp.ToRight(to_right);
               edgelength.resize( hkp.EdgeObjectCount( ) );
               for ( int i = 0; i < edgelength.isize( ); i++ )
                    edgelength[i] = hkp.EdgeLength(i);
               static vec<Bool> to_remove;
               to_remove.resize_and_set( shares.size( ), False );
               for ( int i = 0; i < shares.isize( ); i++ )
               {    const KmerPathCorrespondence& s = shares[i];
                    int e1 = s.Id1( ), e2 = s.Id2( );
                    vrtx_t v1 = to_left[e1], w1 = to_right[e1];
                    vrtx_t v2 = to_left[e2], w2 = to_right[e2];
                    KmerPathLoc scanStart1( hkp.EdgeObject(e1), s.Pos1( ).Interval( ),
                         s.Pos1( ).PosOnInterval( ) );
                    KmerPathLoc scanStart2( hkp.EdgeObject(e2), s.Pos2( ).Interval( ),
                         s.Pos2( ).PosOnInterval( ) );

                    KmerPathLoc rightScan1( scanStart1 ), rightScan2( scanStart2 );
                    ScanRightPerfectMatch( rightScan1, rightScan2 );
                    int right_ext = KmersInInterval( scanStart1, rightScan1 ) - 1;

                    KmerPathLoc leftScan1( scanStart1 ), leftScan2( scanStart2 );
                    ScanLeftPerfectMatch( leftScan1, leftScan2 );
                    int left_ext = KmersInInterval( leftScan1, scanStart1 ) - 1;

                    int pos1 = s.Pos1( ).PosOnPath( ), pos2 = s.Pos2( ).PosOnPath( );
                    int len1 = edgelength[e1], len2 = edgelength[e2];
                    int left1 = pos1 - left_ext, right1 = pos1 + right_ext + 1;
                    int left2 = pos2 - left_ext, right2 = pos2 + right_ext + 1;
                    Bool zipper_up = False;
                    if ( v1 == v2 && left1 == 0 && left2 == 0 ) zipper_up = True;
                    if ( w1 == w2 && right1 == len1 && right2 == len2 ) zipper_up = True;
                    if ( seed_unique && !zipper_up )
                    {    if ( !seed_unique_weak || ( !hkp.Source(v1) && !hkp.Source(v2)
                              && !hkp.Sink(w1) && !hkp.Sink(w2) ) )
                         {    Bool uniq = False;
                              static KmerPath p;
                              p.Clear( );
                              hkp.EdgeObject(e1).CopySubpath( leftScan1, rightScan1, p );
                              static vec<longlong> places;
                              for ( int j = 0; j < p.NSegments( ); j++ )
                              {    Contains( uniqdb, p.Segment(j), places );
                                   if ( places.nonempty( ) )
                                   {    uniq = True;
                                        break;    }    }
                              if ( !uniq ) to_remove[i] = True;    }    }    }
               EraseIf( shares, to_remove );

          /*
          cout << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
          PRINT( shares.size( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
          for ( int i = 0; i < shares.isize( ); i++ ) // QQQQQQQQQQQQQQQQQQQQQQQQQQQ
          {    const KmerPathCorrespondence& s = shares[i]; // QQQQQQQQQQQQQQQQQQQQQ
               cout << "\n[" << i << "] SHARE\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               int e1 = s.Id1( ), e2 = s.Id2( ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               int v1 = to_left[e1], w1 = to_right[e1]; // QQQQQQQQQQQQQQQQQQQQQQQQQ
               int v2 = to_left[e2], w2 = to_right[e2]; // QQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "v1=" << v1 << " ---(e1=" << e1 << ")---> " // QQQQQQQQQQQQQQ
                    << "w1=" << w1 << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "v2=" << v2 << " ---(e2=" << e2 << ")---> " // QQQQQQQQQQQQQQ
                    << "w2=" << w2 << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               PRINT( EdgeLength(e1) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               PRINT( EdgeLength(e2) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               PRINT( s.Offset( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               ForceAssertEq( // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
                    EdgeObject(e1).Start( s.Pos1( ).Interval( ) ) // QQQQQQQQQQQQQQQ
                         + s.Pos1( ).PosOnInterval( ), // QQQQQQQQQQQQQQQQQQQQQQQQQQ
                    EdgeObject(e2).Start( s.Pos2( ).Interval( ) ) // QQQQQQQQQQQQQQQ
                         + s.Pos2( ).PosOnInterval( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQ
               cout << s.Pos1( ).PosOnPath( ) << " <---> " // QQQQQQQQQQQQQQQQQQQQQQ
                    << s.Pos2( ).PosOnPath( ) << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQ
               KmerPathLoc r1( EdgeObject(e1), s.Pos1( ).Interval( ), // QQQQQQQQQQQ
                    s.Pos1( ).PosOnInterval( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               KmerPathLoc r2( EdgeObject(e2), s.Pos2( ).Interval( ), // QQQQQQQQQQQ
                    s.Pos2( ).PosOnInterval( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               int plus_left = -1, plus_right = -1; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               while(1) // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               {    if ( r1.GetKmer( ) != r2.GetKmer( ) ) break; // QQQQQQQQQQQQQQQQ
                    ++plus_right; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
                    if ( r1.atEnd( ) || r2.atEnd( ) ) break; // QQQQQQQQQQQQQQQQQQQQ
                    r1.Increment( ), r2.Increment( );    } // QQQQQQQQQQQQQQQQQQQQQQ
               KmerPathLoc l1( EdgeObject(e1), s.Pos1( ).Interval( ), // QQQQQQQQQQQ
                    s.Pos1( ).PosOnInterval( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               KmerPathLoc l2( EdgeObject(e2), s.Pos2( ).Interval( ), // QQQQQQQQQQQ
                    s.Pos2( ).PosOnInterval( ) ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               while(1) // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               {    if ( l1.GetKmer( ) != l2.GetKmer( ) ) break; // QQQQQQQQQQQQQQQQ
                    ++plus_left; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
                    if ( l1.atBegin( ) || l2.atBegin( ) ) break; // QQQQQQQQQQQQQQQQ
                    l1.Decrement( ), l2.Decrement( );    } // QQQQQQQQQQQQQQQQQQQQQQ
               PRINT2( plus_left, plus_right ); // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "w1 goes to"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               for ( int j = 0; j < From(w1).isize( ); j++ ) // QQQQQQQQQQQQQQQQQQQQ
                    cout << " " << From(w1)[j]; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "v1 comes from"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               for ( int j = 0; j < To(v1).isize( ); j++ ) // QQQQQQQQQQQQQQQQQQQQQQ
                    cout << " " << To(v1)[j]; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "w2 goes to"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               for ( int j = 0; j < From(w2).isize( ); j++ ) // QQQQQQQQQQQQQQQQQQQQ
                    cout << " " << From(w2)[j]; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "v2 comes from"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               for ( int j = 0; j < To(v2).isize( ); j++ ) // QQQQQQQQQQQQQQQQQQQQQQ
                    cout << " " << To(v2)[j]; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
               cout << "\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
                    } // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
          */
          }

          // Note components corresponding to shares.

          vec<int> comp1( shares.size( ) ), comp2( shares.size( ) );
          vec<int> rep1( shares.size( ) ), rep2( shares.size( ) );
          for ( int i = 0; i < shares.isize( ); i++ )
          {    
               // cout << Date( ) << ": processing share [" << i << "]\n"; // QQQQQQ
               int v1 = shares[i].From1( ), v2 = shares[i].From2( );
               if ( v1 < nverts ) 
               {    comp1[i] = to_rep[v1];
                    rep1[i] = comp1[i];    }
               else 
               {    comp1[i] = nreps + to_rep[ v1 - nverts ];
                    rep1[i] = comp1[i] - nreps;    }
               if ( v2 < nverts ) 
               {    comp2[i] = to_rep[v2];
                    rep2[i] = comp2[i];    }
               else 
               {    comp2[i] = nreps + to_rep[ v2 - nverts ];
                    rep2[i] = comp2[i] - nreps;    }    }
     
          // Go through the shares.

          for ( int u = 0; u < shares.isize( ); u++ )
          {    const KmerPathCorrespondence& c = shares[u];

               // Check share to see if it refers to an edge that is no longer
               // active, as a result of edits to the graph.

               if ( c.FromIndex1( ) >= hkp.From( c.From1( ) ).isize( )
                    || hkp.EdgeObjectIndexByIndexFrom( c.From1( ), c.FromIndex1( ) ) 
                    != c.Id1( ) )
               {    continue;    }
               if ( c.FromIndex2( ) >= hkp.From( c.From2( ) ).isize( )
                    || hkp.EdgeObjectIndexByIndexFrom( c.From2( ), c.FromIndex2( ) ) 
                    != c.Id2( ) )
               {    continue;    }
               if ( hkp.From( c.From1( ) )[ c.FromIndex1( ) ] != c.To1( ) ) continue;
               if ( hkp.From( c.From2( ) )[ c.FromIndex2( ) ] != c.To2( ) ) continue;

               // Check share to see if it refers to the same edge (twice).

               if ( c.Id1( ) == c.Id2( ) ) continue;

               // In phase 1 we only align within a given insert.

               if ( SPECIAL && phase == 1 )
               {    if ( comp1[u] != comp2[u] || comp1[u] >= nreps ) continue;    }

               // Don't consider alignments of a component to its reverse.

               if ( Abs( comp1[u] - comp2[u] ) == nreps ) continue;

               // Once a component has been touched, we can't use its reverse.

               if ( nreps > 1 )
               {    int r1, r2;
                    if ( comp1[u] < reps.isize( ) ) r1 = comp1[u] + reps.isize( );
                    else r1 = comp1[u] - reps.isize( );
                    if ( comp2[u] < reps.isize( ) ) r2 = comp2[u] + reps.isize( );
                    else r2 = comp2[u] - reps.isize( );
                    if ( touched[r1] || touched[r2] ) continue;    }

               // Define PointedSubPairPath p.

               vec<vrtx_t> aa(2), bb(2);
               aa[0] = c.From1( ), aa[1] = c.To1( );
               bb[0] = c.From2( ), bb[1] = c.To2( );
               vec<int> e(1), f(1);
               e[0] = c.FromIndex1( );
               f[0] = c.FromIndex2( );
               //const digraphE<KmerPath>& D(hkp);
               PointedSubPathPair p( EmbeddedSubPath<KmerPath>( hkp, aa, e ),
                    EmbeddedSubPath<KmerPath>( hkp, bb, f ),
                    MultiKmerPathLoc( 0, c.Pos1( ) ), 
                    MultiKmerPathLoc( 0, c.Pos2( ) ) );

               KmerPath M;
               int naligns;
               vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> > ends;
               vec<KmerPathSplitPoint> il, jl;
               hkp.AlignSubpaths( p, naligns, ends, M, il, jl, ngv );
               if ( naligns == 0 ) continue;

               // Try to left extend and right extend.
               
               Bool fail;
               AlignExtendQueue Q;
               Q.push( AlignExtend( p, ends, M.KmerCount( ) ) );
               InternalMergeLeftExtend( hkp, Q, ngv ); 
               {
                 AlignExtendQueue Qreset;
                 while ( !Q.empty() ) {
                   AlignExtend E = Q.top();
                   Q.pop();
                   E.SetTerminal( False );
                   Qreset.push( E );
                 }
                 Q = Qreset;
               }
               InternalMergeRightExtend( hkp, Q, ngv );

               // Go through each extension.

               while( !Q.empty( ) )
               {    static AlignExtend E;
                    E = Q.top( );
                    Q.pop( );
                    p = E.PathPair( );
                    ends = E.Ends( );
     
                    // Rebuild the alignment data.

                    hkp.AlignSubpaths( p, naligns, ends, M, il, jl, ngv );
                    ForceAssertGt( naligns, 0 );

//                     cout << "Considering alignment of " << M.KmerCount() << " kmers:" << endl;
//                     cout << "  "; p.PrintSummary( cout, 0, 0 ); cout << endl;
//                     cout << "  "; p.PrintSummary( cout, 1, 0 ); cout << endl;
             
                    // Check to see if aligning region is too short.  There are 
                    // three criteria:
                    // 1. Regardless of length, we accept any alignment if the 
                    // vertices agree on one end and the nonaligning parts on that 
                    // end have the same length (allowing for gap stretching).
                    // 2. If the alignment is proper (i.e. goes from source to sink)
                    // and at least min_proper_overlap bases, we accept it.
                    // 3. If the alignment extends for at least min_overlap bases, we 
                    // accept it.

                    Bool accept_short = False;
                    if ( p.Path1( ).Vertex(0) == p.Path2( ).Vertex(0) )
                    {    vec<int> before_min(2, 0), before_max(2, 0);
                         for ( int i = 0; i < 2; i++ )
                         {    const MultiKmerPathLoc& l = ends[i].first;
                              const KmerPath& kp = p.Path(i).EdgeObject(0);
                              int seg = l.SegmentOnEdge( ); 
                              int kmer = l.KmerOnSegment( );
                              for ( int j = 0; j < seg; j++ )
                              {    before_min[i] += kp.MinLength(j);
                                   before_max[i] += kp.MaxLength(j);    }
                              before_min[i] += kmer;
                              before_max[i] += kmer;    }
                         if ( IntervalOverlap( before_min[0], before_max[0] + 1,
                              before_min[1], before_max[1] + 1 ) > 0 )
                         {    accept_short = True;    }    }
                    if ( p.Path1( ).Vertex( p.Path1( ).NVertices( ) - 1 )
                         == p.Path2( ).Vertex( p.Path2( ).NVertices( ) - 1 ) )
                    {    vec<int> after_min(2, 0), after_max(2, 0);
                         for ( int i = 0; i < 2; i++ )
                         {    const MultiKmerPathLoc& r = ends[i].second;
                              int last = p.Path(i).NEdges( ) - 1;
                              const KmerPath& kp = p.Path(i).EdgeObject(last);
                              int seg = r.SegmentOnEdge( ); 
                              int kmer = r.KmerOnSegment( );
                              after_min[i] += kp.Length(seg) - kmer - 1;
                              after_max[i] += kp.Length(seg) - kmer - 1;
                              for ( int r = seg + 1; r < kp.NSegments( ); r++ )
                              {    after_min[i] += kp.MinLength(r);
                                   after_max[i] += kp.MaxLength(r);    }    }
                         if ( IntervalOverlap( after_min[0], after_max[0] + 1,
                              after_min[1], after_max[1] + 1 ) > 0 )
                         {    accept_short = True;    }    }

//                     if ( accept_short )
//                       cout << " Accepting short align." << endl;

                    if ( !accept_short )
                    {    if ( M.KmerCount( ) < min_proper_overlap ) continue;
                         if ( SPECIAL && phase == 1 ) continue;
                           
                         Bool left_proper =
                           ( ends[0].first.LeftEnd( ) &&
                             hkp.Source( p.Path(0).FirstVertex() ) ) ||
                           ( ends[1].first.LeftEnd( ) &&
                             hkp.Source( p.Path(1).FirstVertex() ) );
                         Bool right_proper =
                           ( ends[0].second.RightEnd( p.Path(0) ) &&
                             hkp.Sink( p.Path(0).LastVertex( ) ) ) ||
                           ( ends[1].second.RightEnd( p.Path(1) ) &&
                             hkp.Sink( p.Path(1).LastVertex( ) ) );

                         Bool proper = left_proper && right_proper;

                         // Check for pseudoproper: all the vertices and edges of a
                         // given component are covered.

                         Bool pseudoproper = False;
                         if ( !proper )
                         {    for ( int j = 0; j < 2; j++ )
                              {    const EmbeddedSubPath<KmerPath>& P = p.Path(j);
                                   if ( ends[j].first.LeftEnd( ) &&
                                        ends[j].second.RightEnd(P) )
                                   {    static vec<vrtx_t> vertices;
				        static vec<int> edges;
                                        vertices.clear( ), edges.clear( );
                                        for ( int u = 0; u < P.NVertices( ); u++ )
                                             vertices.push_back( P.Vertex(u) );
                                        for ( int u = 0; u < P.NEdges( ); u++ )
                                        {    edges.push_back( 
                                                  P.EdgeObjectIndex(u) );    }
                                        UniqueSort(vertices), UniqueSort(edges);
                                        if ( hkp.IsComplete( vertices, edges ) )
                                        {    pseudoproper = True;
                                             break;    }    }    }    }

//                          if ( proper ) 
//                            cout << " Accepting proper." << endl;
//                          else if ( pseudoproper )
//                            cout << " Accepting pseudoproper." << endl;
                         
                         if ( SPECIAL && phase == 2 && ( (!proper && !pseudoproper) 
                              || M.KmerCount( ) < 3000 ) )
                         {    continue;    }
                         if ( !proper && !pseudoproper ) {
                           if ( M.KmerCount( ) < min_overlap ) 
                             continue;    
//                            else
//                              cout << " Accepting improper." << endl;
                         }
                    }

                    // Add vertices if ends are not at the ends of edges, and adjust
                    // p, il, jl accordingly.

                    for ( int i = 0; i < 2; i++ )
                    {    const MultiKmerPathLoc& l = ends[i].first;
                         if ( !l.LeftEnd( ) )
                         {    const KmerPath& kp = p.Path(i).EdgeObject(0);
                              int seg = l.SegmentOnEdge( ); 
                              int kmer = l.KmerOnSegment( );
                              KmerPathLoc left( kp, seg, kmer );
                              KmerPath before, after;
                              for ( int j = 0; j < seg; j++ )
                                   before.AddSegment( kp.Segment(j) );
                              if ( kmer > 0 )
                              {    before.AddSegment( kp.Segment(seg).Start( ),
                                        kp.Segment(seg).Start( ) + kmer - 1 );    }
                              // kp.CopyTail( left, after ); // CopyTail not safe
                              if ( kmer == 0 ) after.AddSegment( kp.Segment(seg) );
                              else 
                              {    after.AddSegment( kp.Segment(seg).Start( ) + kmer,
                                        kp.Segment(seg).Stop( ) );    }
                              for ( int r = seg + 1; r < kp.NSegments( ); r++ )
                                   after.AddSegment( kp.Segment(r) );
                              SplitEdgeInPointedSubPath( hkp, p.PathMutable(i), 
                                   p.LocMutable(i), 0, before, after, True );    
                              p.PathMutable(0).Repair( ), p.PathMutable(1).Repair( );
                              if ( i == 0 ) il[0] = KmerPathSplitPoint( 0, 0 );
                              if ( i == 1 ) jl[0] = KmerPathSplitPoint( 0, 0 );
                              MultiKmerPathLoc& r = ends[i].second;
                              if ( l.Edge( ) == r.Edge( ) )
                              {    if ( l.SegmentOnEdge( ) == r.SegmentOnEdge( ) )
                                   {    r.SetKmerOnSegment( r.KmerOnSegment( ) 
                                             - l.KmerOnSegment( ) );    }    
                                   r.SetSegmentOnEdge( r.SegmentOnEdge( ) 
                                        - l.SegmentOnEdge( ) );    }    }
                         const MultiKmerPathLoc& r = ends[i].second;
                         ForceAssertGe( r.KmerOnSegment( ), 0 ); // XXX
                         if ( !r.RightEnd( p.Path(i) ) )
                         {    int last = p.Path(i).NEdges( ) - 1;
                              const KmerPath& kp = p.Path(i).EdgeObject(last);
                              int seg = r.SegmentOnEdge( ); 
                              int kmer = r.KmerOnSegment( );
                              KmerPathLoc right( kp, seg, kmer );
                              KmerPath before, after;
                              kp.CopyHead( right, before );
                              // Not safe:
                              // kp.CopySubpathNoFirstKmer(right, kp.End( ), after);
                              if ( kmer < kp.Segment(seg).Length( ) - 1 )
                              {    after.AddSegment( 
                                        kp.Segment(seg).Start( ) + kmer + 1,
                                        kp.Segment(seg).Stop( ) );    }
                              for ( int r = seg + 1; r < kp.NSegments( ); r++ )
                                   after.AddSegment( kp.Segment(r) );
                              SplitEdgeInPointedSubPath( hkp, p.PathMutable(i),
                                   p.LocMutable(i), last, before, after, False );    
                              p.PathMutable(0).Repair( ); 
                              p.PathMutable(1).Repair( );    
                              if ( i == 0 ) 
                              {    il[ p.Path1( ).NVertices( ) - 1 ]
                                        = KmerPathSplitPoint( 
                                             M.NSegments( ), 0 );    }
                              if ( i == 1 ) 
                              {    jl[ p.Path2( ).NVertices( ) - 1 ]
                                        = KmerPathSplitPoint( 
                                             M.NSegments( ), 0 );    }    }    }

                    // Glue.

                    /*
                    for ( int v = 0; v < il.isize( ); v++ ) // XXX
                         PRINT2( v, il[v] ); // XXX
                    for ( int v = 0; v < jl.isize( ); v++ ) // XXX
                         PRINT2( v, jl[v] ); // XXX
                    */
                    for ( int v = 0; v < il.isize( ); v++ )
                         ForceAssert( il[v].Initialized( ) );
                    for ( int v = 0; v < jl.isize( ); v++ )
                         ForceAssert( jl[v].Initialized( ) );
                    vec<KmerPathSplitPoint> splits = il;
                    splits.append(jl);
                    UniqueSort(splits);
                    vec<int> EE( il.size( ) ), FF( jl.size( ) );
                    for ( int v = 0; v < il.isize( ); v++ )
                         EE[v] = BinPosition( splits, il[v] );
                    for ( int v = 0; v < jl.isize( ); v++ )
                         FF[v] = BinPosition( splits, jl[v] );
                    vec<KmerPath> Msplit;
                    for ( int v = 0; v < splits.isize( ) - 1; v++ )
                         Msplit.push_back( Between( M, splits[v], splits[v+1] ) );
                    digraphE<KmerPath> Mgraph( 
                         Msplit, digraphE<KmerPath>::EDGES_IN_LINE );

                    p.Path1( ).TestValid( ); // XXX
                    p.Path2( ).TestValid( ); // XXX

                    // cout << "gluing\n"; // QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ
                    hkp.Glue( p.Path1( ), p.Path2( ), EE, FF, Mgraph );
                    found_merge = True;

                    // Note which components have been touched.

                    touched[ comp1[u] ] = touched[ comp2[u] ] = True;
                    break;    }    }

          // Remove superfluous components from original introduction of reverse
          // complements.

          // cout << "at cleaning step\n"; // XXX
          // if ( nreps > 1 )
          {    for ( int i = 0; i < nreps; i++ )
               {    int del;
                    ForceAssert( !touched[i] || !touched[i+nreps] );
                    if ( !touched[i] && !touched[i+nreps] ) del = i + nreps;
                    else if ( touched[i] ) del = i + nreps;
                    else del = i;
                    for ( int j = 0; j < components[del].isize( ); j++ )
                    {    int v = components[del][j];
                         hkp.DeleteEdgesAtVertex(v);    }    }    }
          hkp.RemoveUnneededVertices( );    

          // If there are two edges between the same two vertices, one of which is
          // just a gap, and the other is not, and the lengths are compatible,
          // delete the first edge.

          for ( int v = 0; v < hkp.N( ); v++ )
          {    for ( int i1 = 0; i1 < hkp.From(v).isize( ); i1++ )
               {    int w = hkp.From(v)[i1];
                    if ( v == w ) continue;
                    const KmerPath& p1 = hkp.EdgeObjectByIndexFrom( v, i1 );
                    if ( p1.NSegments( ) != 1 ) continue;
                    if ( !p1.Segment(0).isGap( ) ) continue;
                    for ( int i2 = 0; i2 < hkp.From(v).isize( ); i2++ )
                    {    if ( hkp.From(v)[i2] != w ) continue;
                         const KmerPath& p2 = hkp.EdgeObjectByIndexFrom( v, i2 );
                         if ( p2.NSegments( ) == 1 && p2.Segment(0).isGap( ) )
                              continue;
                         if ( IntervalOverlap( (int) p1.Segment(0).Minimum( ), 
                              (int) p1.Segment(0).Maximum( ) + 1,
                              p2.MinLength( 0, p2.NSegments( ) - 1 ),
                              p2.MaxLength( 0, p2.NSegments( ) - 1 ) + 1 ) == 0 )
                         {    continue;    }
                         hkp.DeleteEdgeFrom( v, i1 );
                         found_merge = True;
                         goto next_v;    }    }
               next_v: continue;    }

          // If there are two edges between the same two vertices, both of which
          // are just gaps, and the gap intervals overlap, then merge them.

          for ( int v = 0; v < hkp.N( ); v++ )
          {    vec<int> gap_edges;
               for ( int i = 0; i < hkp.From(v).isize( ); i++ )
               {    int w = hkp.From(v)[i];
                    if ( v == w ) continue;
                    const KmerPath& p = hkp.EdgeObjectByIndexFrom( v, i );
                    if ( p.NSegments( ) != 1 ) continue;
                    if ( !p.Segment(0).isGap( ) ) continue;
                    gap_edges.push_back(i);    }
               for ( int j1 = 0; j1 < gap_edges.isize( ); j1++ )
               {    for ( int j2 = j1 + 1; j2 < gap_edges.isize( ); j2++ )
                    {    int i1 = gap_edges[j1], i2 = gap_edges[j2];
                         int w = hkp.From(v)[i1];
                         if ( hkp.From(v)[i2] != w ) continue;
                         const KmerPath& p1 = hkp.EdgeObjectByIndexFrom( v, i1 );
                         const KmerPath& p2 = hkp.EdgeObjectByIndexFrom( v, i2 );
                         const KmerPathInterval& I1 = p1.Segment(0);
                         const KmerPathInterval& I2 = p2.Segment(0);
                         if ( IntervalOverlap( I1.Minimum( ), I1.Maximum( ) + 1,
                              I2.Minimum( ), I2.Maximum( ) + 1 ) == 0 )
                         {    continue;    }
                         int new_min = Min( I1.Minimum( ), I2.Minimum( ) );
                         int new_max = Max( I1.Maximum( ), I2.Maximum( ) );
                         KmerPath p;
                         p.AddSegment( new_min, new_max, True );
                         hkp.ChangeEdgeObjectFrom( v, i1, p );
                         hkp.DeleteEdgeFrom( v, i2 );
                         found_merge = True;
                         goto next_vertex;    }    }
               next_vertex: continue;    }

          // Look for cases where an edge consists of a gap only, and can be 
          // merged into another edge.

          for ( int v = 0; v < hkp.N( ); v++ )
          {    for ( int i = 0; i < hkp.From(v).isize( ); i++ )
               {    int w = hkp.From(v)[i];
                    if ( v == w ) continue;
                    const KmerPath& p = hkp.EdgeObjectByIndexFrom( v, i );
                    if ( p.NSegments( ) != 1 ) continue;
                    if ( !p.Segment(0).isGap( ) ) continue;
                    Bool placed = False;
                    for ( int j = 0; j < hkp.To(w).isize( ); j++ )
                    {    const KmerPath& q = hkp.EdgeObjectByIndexTo( w, j );
                         if ( q.LastSegment( ) != p.Segment(0) ) continue;
                         if ( q.NSegments( ) == 1 ) continue;
                         int x = hkp.To(w)[j];
                         if ( x == v || x == w ) continue;
                         KmerPath before;
                         for ( int u = 0; u < q.NSegments( ) - 1; u++ )
                              before.AddSegment( q.Segment(u) );

                         // Check to see if we're creating any illegal adjacencies.

                         Bool illegal = False;
                         for ( int r = 0; r < hkp.From(v).isize( ); r++ )
                         {    const KmerPath& p = hkp.EdgeObjectByIndexFrom( v, r );
                              if ( !ngv.LegalConcat( before, p ) )
                              {    illegal = True;
                                   break;    }    }
                         if (illegal) continue;
          
                         // Have:                  Want:
                         //        v -----> w      x ----> v ----> w
                         // x ------------> w

                         hkp.DeleteEdgeTo( w, j );
                         hkp.AddEdge( x, v, before );
                         placed = True;
                         break;    }
                    if (placed) break;    }    }
          for ( int v = 0; v < hkp.N( ); v++ )
          {    for ( int i = 0; i < hkp.From(v).isize( ); i++ )
               {    int w = hkp.From(v)[i];
                    if ( v == w ) continue;
                    const KmerPath& p = hkp.EdgeObjectByIndexFrom( v, i );
                    if ( p.NSegments( ) != 1 ) continue;
                    if ( !p.Segment(0).isGap( ) ) continue;
                    Bool placed = False;
                    for ( int j = 0; j < hkp.From(v).isize( ); j++ )
                    {    const KmerPath& q = hkp.EdgeObjectByIndexFrom( v, j );
                         if ( q.Segment(0) != p.Segment(0) ) continue;
                         if ( q.NSegments( ) == 1 ) continue;
                         int x = hkp.From(v)[j];
                         if ( x == v || x == w ) continue;
                         KmerPath after;
                         for ( int u = 1; u < q.NSegments( ); u++ )
                              after.AddSegment( q.Segment(u) );

                         Bool illegal = False;
                         for ( int r = 0; r < hkp.To(w).isize( ); r++ )
                         {    const KmerPath& p = hkp.EdgeObjectByIndexTo( w, r );
                              if ( !ngv.LegalConcat( p, after ) )
                              {    illegal = True;
                                   break;    }    }
                         if (illegal) continue;
          
                         // Have:                  Want:
                         // v ----> w              v ----> w ----> x
                         // v ------------> x

                         hkp.DeleteEdgeFrom( v, j );
                         hkp.AddEdge( w, x, after );
                         placed = True;
                         break;    }
                    if (placed) break;    }    }

          // Look for cases where gap edge A from v to w is superfluous because 
          // there are consecutive edges B, C going from v to w and having the
          // same total length (allowing for gap stretching).

          for ( int v = 0; v < hkp.N( ); v++ )
          {    static vec<Bool> remove;
               remove.resize_and_set( hkp.From(v).size( ), False );
               for ( int i = 0; i < hkp.From(v).isize( ); i++ )
               {    int w = hkp.From(v)[i];
                    if ( v == w ) continue;
                    const KmerPath& p = hkp.EdgeObjectByIndexFrom( v, i );
                    if ( p.NSegments( ) != 1 ) continue;
                    if ( !p.Segment(0).isGap( ) ) continue;
                    for ( int j = 0; j < hkp.From(v).isize( ); j++ )
                    {    if ( j == i ) continue;
                         int x = hkp.From(v)[j];
                         if ( x == v ) continue;
                         const KmerPath& p1 = hkp.EdgeObjectByIndexFrom( v, j );
                         for ( int k = 0; k < hkp.From(x).isize( ); k++ )
                         {    if ( hkp.From(x)[k] != w ) continue;
                              const KmerPath& p2 = hkp.EdgeObjectByIndexFrom( x, k );
                              if ( IntervalOverlap( 
                                   p.MinLength( ), p.MaxLength( ) + 1,
                                   p1.MinLength( ) + p2.MinLength( ),
                                   p1.MaxLength( ) + p2.MaxLength( ) + 1 ) > 0 )
                              {    remove[i] = True;    
                                   found_merge = True;    }    }    }    }
               for ( int i = hkp.From(v).isize( ) - 1; i >= 0; i-- )
               {    if ( !remove[i] ) continue;
                    int w = hkp.From(v)[i];
                    hkp.DeleteEdgeFrom( v, i );    }    }

          // Remove identical paths between two given vertices.  Then remove dead
          // edge objects.

          hkp.CompressEdgeObjects( );
          hkp.RemoveDuplicateEdges( );
          hkp.RemoveDeadEdgeObjects( );

          if (SPECIAL)
          {    if ( !found_merge )
               {    ++phase;
                    if ( phase == 2 && norbits == 1 ) break;
                    if ( phase == 4 ) break;    }    }
          else if ( !found_merge ) break;    }

     hkp.TestValid( );    }

