/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include <strstream>

#include "CoreTools.h"
#include "Equiv.h"
#include "PrintAlignment.h"
#include "ReadPairing.h"
#include "assembly/Interval.h"
#include "assembly/display/Channeler.h"
#include "graph/DigraphTemplate.h"
#include "graphics/BasicGraphics.h"
#include "math/Arith.h"
#include "math/Functions.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "pairwise_aligners/SmithWatBandedA.h"
#include "paths/AlignAndMerge.h"
#include "paths/HyperKmerPath.h"
#include "paths/ImproperMerge.h"
#include "paths/KmerBaseBroker.h"
#include "paths/NegativeGapValidator.h"
#include "random/Random.h"

template digraphE<KmerPath>::digraphE( const digraphE& g, int n );
template digraphE<KmerPath>::digraphE( const digraphE& g, const vec< vec<int> >& C );
template digraphE<KmerPath>::digraphE(vec<KmerPath> const&, 
     const ConstructorBehavior );
template digraphE<basevector>::digraphE(vec<basevector> const&, 
     const ConstructorBehavior );
template digraphE<KmerPath>::digraphE(vec<KmerPath> const&, const equiv_rel&);
template 
     void digraphE<KmerPath>::SplitEdge(vrtx_t, int, KmerPath const&, KmerPath const&);
template void digraphE<KmerPath>::Reverse();
template void digraphE<int>::Reverse();
template void digraphE<KmerPath>::ReverseComponent( vrtx_t v );
template void digraphE<KmerPath>::Glue(EmbeddedSubPath<KmerPath> const&, 
     EmbeddedSubPath<KmerPath> const&, vec<int> const&, vec<int> const&, 
     digraphE<KmerPath> const&);
template digraphE<KmerPath>::digraphE(digraphE<KmerPath> const&, 
     vec<vrtx_t> const&);
template void digraphE<KmerPath>::DeleteEdgesAtVertex(vrtx_t);
template void digraphE<KmerPath>::RemoveDuplicateEdges();
template void digraphE<KmerPath>::RemoveDeadEdgeObjects();
template void BinaryWrite(int, const digraphE<KmerPath>&);
template void BinaryRead(int, digraphE<KmerPath>&);
template Bool digraphE<KmerPath>::IsComplete( const vec<vrtx_t>& vertices, 
     const vec<int>& edges ) const;
template void digraphE<KmerPath>::DualComponentRelation(
     equiv_rel& e, const vec<Bool>& exclude ) const;
template digraphE<KmerPath>::digraphE(digraphE<KmerPath> const&, equiv_rel const&);
template void digraphE<KmerPath>::Initialize(vec<digraphE<KmerPath> > const&, 
     vec<pair<pair<int, vrtx_t>, pair<int, vrtx_t> > > const&);
template void digraphE<KmerPath>::ToLeft(vec<vrtx_t>&) const;
template void digraphE<KmerPath>::ToRight(vec<vrtx_t>&) const;
template void digraphE<basevector>::ToLeft(vec<vrtx_t>&) const;
template void digraphE<basevector>::ToRight(vec<vrtx_t>&) const;

void HyperKmerPath::ReduceLoops( )
{    for ( int v = 0; v < N( ); v++ )
     {    if ( To(v).size( ) != 2 || From(v).size( ) != 1 ) continue;
          int w = From(v)[0];
          if ( v == w ) continue;
          int u;
          if ( To(v)[0] == w ) u = To(v)[1];
          else if ( To(v)[1] == w ) u = To(v)[0];
          else continue;
          if ( u == v || u == w ) continue;
          if ( From(w).size( ) != 2 || To(w).size( ) != 1 ) continue;
          int x;
          if ( From(w)[0] == v ) x = From(w)[1];
          else x = From(w)[0];
          if ( x == u || x == v || x == w ) continue;
          static KmerPath vv, vx;
          vv = EdgeObjectByIndexFrom( v, 0 );
          vv.Append( EdgeObjectByIndexTo( v, ( To(v)[0] == w ? 0 : 1 ) ) );
          vx = EdgeObjectByIndexFrom( v, 0 );
          vx.Append( EdgeObjectByIndexFrom( w, ( From(w)[0] == x ? 0 : 1 ) ) );
          int vx_id = EdgeObjectIndexByIndexTo( v, ( To(v)[0] == w ? 0 : 1 ) );
          EdgeObjectMutable(vx_id) = vx;
          FromMutable(w).clear( ), ToMutable(w).clear( );
          FromEdgeObjMutable(w).clear( ), ToEdgeObjMutable(w).clear( );
          int vv_id = EdgeObjectIndexByIndexFrom( v, 0 );
          EdgeObjectMutable(vv_id) = vv;
          FromMutable(v)[0] = v;
          if ( To(v)[0] == w )
          {    ToMutable(v)[0] = u;
               ToEdgeObjMutable(v)[0] = ToEdgeObj(v)[1];    }
          FromEdgeObjMutable(v)[0] = vv_id;
          ToMutable(v).resize(1), ToEdgeObjMutable( )[v].resize(1);
          ToMutable(v).push_back(v);
          ToEdgeObjMutable( )[v].push_back(vv_id);
          FromMutable(v).push_back(x);
          FromEdgeObjMutable( )[v].push_back(vx_id);
          for ( int j = 0; j < To(x).isize( ); j++ )
          {    if ( To(x)[j] == w )
               {    ToMutable(x)[j] = v;
                    ToEdgeObjMutable(x)[j] = vx_id;
                    break;    }    }    
          SortSync( FromMutable(v), FromEdgeObjMutable(v) );
          SortSync( ToMutable(v), ToEdgeObjMutable(v) );
          SortSync( FromMutable(w), FromEdgeObjMutable(w) );
          SortSync( ToMutable(w), ToEdgeObjMutable(w) );
          SortSync( ToMutable(x), ToEdgeObjMutable(x) );    }    }

void HyperKmerPath::DisplayEdge( vec<graphics_primitive>& picture, int e,           
     int K, const vec<HyperSloppyReadPlacement>& locs, const KmerBaseBroker& kbb,
     int start, int stop ) const
{    vec<Interval> ilocs;
     const KmerPath& p = EdgeObject(e);
     for ( int i = 0; i < locs.isize( ); i++ )
     {    const HyperSloppyReadPlacement& l = locs[i];
          int v = l.Vertex( ), ev = l.EdgeFrom( );
          if ( e == EdgeObjectIndexByIndexFrom( v, ev ) )
          {    KmerPathLoc leftp, rightp;
               if ( l.LeftEndMapped( ) )
               {    KmerPathLocAlt left = l.LeftLoc( ).LocOnEdge( );
                    leftp.Set( p, left.Interval( ), left.PosOnInterval( ) );    }
               else leftp = p.Begin( );
               if ( l.RightEndMapped( ) )
               {    KmerPathLocAlt right = l.RightLoc( ).LocOnEdge( );
                    rightp.Set( p, right.Interval( ), right.PosOnInterval( ) );    }
               else rightp = p.End( );
               int xstart = DistMin( p.Begin( ), leftp );
               int xstop = DistMin( p.Begin( ), rightp ) + K;
               if ( start >= 0 )
               {    if ( xstop < start ) continue;
                    if ( xstart < start ) xstart = start;    }
               if ( stop >= 0 )
               {    if ( xstart > stop ) continue;
                    if ( xstop > stop ) xstop = stop;    }
               ilocs.push_back( Interval( xstart, xstop ) );    }    }
     Sort(ilocs);
     Channeler<Interval> ch( ilocs, 5 );
     picture.clear( );
     for ( int i = 0; i < ilocs.isize( ); i++ )
     {    int x1 = ilocs[i].Begin( ), x2 = ilocs[i].End( );
          int y = ch.GetChannelFor( ilocs[i] );
          if ( ilocs[i].Length( ) < 200 ) 
               picture.push_back( SetColor( color( 0, 0.8, 0 ) ) );
          else picture.push_back( SetColor(red) );
          picture.push_back( Segment( x1, y, x2, y ) );    }
     SuperBaseVector v = kbb.ToSequence(p);
     vec<char> bases;
     for ( int i = 0; i < v.size( ); i++ )
     {    const basevector& b = v.Seq(i);
          for ( int j = 0; j < (int) b.size( ); j++ )
               bases.push_back( as_base( b[j] ) );
          if ( i < v.size( ) - 1 )
          for ( int j = 0; j < v.MinGap(i); j++ )
               bases.push_back( 'N' );    }
     vec< pair<double,double> > points;
     for ( int i = 0; i < bases.isize( ); i+= 100 )
     {    if ( i < start || i > stop ) continue;
          int GC = 0, ACGT = 0;
          for ( int j = -50; j < 50; j++ )
          {    int s = i + j;
               if ( s >= 0 && s < bases.isize( ) )
               {    if ( bases[s] == 'G' || bases[s] == 'C' ) ++GC;
                    if ( bases[s] != 'N' ) ++ACGT;    }    }
          if ( ACGT > 0 )
          {    points.push_back( make_pair(
                    double(i), double(45 * GC)/double(ACGT) ) );    }    }
     picture.push_back( SetColor(black) );
     picture.push_back( SetLineWidth(1.2) );
     for ( int i = 0; i < points.isize( ) - 1; i++ )
     {    picture.push_back( Segment( points[i].first, points[i].second,
               points[i+1].first, points[i+1].second ) );    }
     picture.push_back( SetTimesRoman(12.0) );
     picture.push_back( TextToLeft( 
          ToString( double(100)/double(45) * points.front( ).second, 2 ),
               points.front( ).first, points.front( ).second, 5 ) );
     picture.push_back( TextToRight( 
          ToString( double(100)/double(45) * points.back( ).second, 2 ),
               points.back( ).first, points.back( ).second, 5 ) );    }

int HyperSloppyReadPlacement::KmersCovered( const KmerPath& p ) const
{    if ( LeftEndMapped( ) && RightEndMapped( ) )
     {    KmerPathLocAlt left = LeftLoc( ).LocOnEdge( );
          KmerPathLocAlt right = RightLoc( ).LocOnEdge( );
          int left_int = left.Interval( ), right_int = right.Interval( );
          int left_pos = left.PosOnInterval( ), right_pos = right.PosOnInterval( );
          if ( left_int == right_int ) return right_pos - left_pos + 1;
          else
          {    int cov = p.Length(left_int) - left_pos + right_pos + 1;
               for ( int j = left_int + 1; j <= right_int - 1; j++ )
                    if ( !p.isGap(j) ) cov += p.Length(j);
               return cov;    }    }
     else if ( RightEndMapped( ) )
     {    KmerPathLocAlt right = RightLoc( ).LocOnEdge( );
          int right_int = right.Interval( ), right_pos = right.PosOnInterval( );
          int cov = right_pos + 1;
          for ( int j = 0; j <= right_int - 1; j++ )
               if ( !p.isGap(j) ) cov += p.Length(j);
          return cov;    }
     else if ( LeftEndMapped( ) )
     {    KmerPathLocAlt left = LeftLoc( ).LocOnEdge( );
          int left_int = left.Interval( ), left_pos = left.PosOnInterval( );
          int cov = p.Length(left_int) - left_pos;
          for ( int j = left_int + 1; j < p.NSegments( ); j++ )
               if ( !p.isGap(j) ) cov += p.Length(j);
          return cov;    }
     else return p.KmerCount( );    }

void HyperKmerPath::DumpFasta( const String& fn, const KmerBaseBroker& kbb ) const
{    Ofstream( out, fn );
     int count = 0;
     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    int w = From(v)[j];
	  out << ">edge_" << BaseAlpha( EdgeObjectIndexByIndexFrom( v, j ) )
                    << " " << v << ":" << w << "\n";
               const KmerPath& p = EdgeObjectByIndexFrom( v, j );
               kbb.ToSequence(p).PrintN(out);    }    }    }

// HappyPlacement: return true if placement of read pair can be validated.
// A false return value is harder to interpret.  This code was cannibalized
// from ShowLinks.

Bool HyperKmerPath::HappyPlacement( int i1, int i2, 
     const vec<HyperSloppyReadPlacement>& locs, const read_pairing& P, 
     const vec<int>& trims, const vec<int>& comp )
{    HyperSloppyReadPlacement l1 = locs[i1], l2 = locs[i2];
     int v1 = l1.Vertex( ), v2 = l2.Vertex( );
     if ( comp[v1] != comp[v2] || l1.Fw( ) == l2.Fw( ) ) return False;
     if ( !l1.Fw( ) ) 
     {    swap(l1, l2);
          swap(v1, v2);    }
     if ( !l1.RightEndMapped( ) || !l2.LeftEndMapped( ) ) return False;
     int id1 = l1.ReadId( ), id2 = l2.ReadId( );
     int fv1 = l1.EdgeFrom( ), fv2 = l2.EdgeFrom( );
     int w1 = From(v1)[fv1], w2 = From(v2)[fv2];
     static vec< pair<int,int> > allpathlengths;
     allpathlengths.clear( );
     if ( v1 == v2 && fv1 == fv2 )
     {    const KmerPath& p = EdgeObjectByIndexFrom( v1, fv1 );
          const KmerPathLocAlt& a = l1.RightLoc( ).LocOnEdge( );
          KmerPathLoc left( p, a.Interval( ), a.PosOnInterval( ) );
          const KmerPathLocAlt& b = l2.LeftLoc( ).LocOnEdge( );
          KmerPathLoc right( p, b.Interval( ), b.PosOnInterval( ) );
          int sep_low = DistMin( left, right ), sep_high = DistMax( left, right );
          allpathlengths.push_back( make_pair( sep_low, sep_high ) );    }
     else
     {    static vec< vec<vrtx_t> > paths;
          paths.clear( );
          vrtx_t x = w1, y = v2;
          if ( !AllPaths( x, y, paths, 100 ) ) return False;
          if ( paths.empty( ) ) return False;
          if ( !PathLengths( x, y, allpathlengths, 100 ) ) return False;
          int addlow = 0, addhigh = 0;
          const KmerPath& p = EdgeObjectByIndexFrom( v1, fv1 );
          const KmerPathLocAlt& a = l1.RightLoc( ).LocOnEdge( );
          KmerPathLoc left( p, a.Interval( ), a.PosOnInterval( ) );
          addlow += DistMin( left, p.End( ) ) + 1;
          addhigh += DistMax( left, p.End( ) ) + 1;
          const KmerPath& q = EdgeObjectByIndexFrom( v2, fv2 );
          const KmerPathLocAlt& b = l2.LeftLoc( ).LocOnEdge( );
          KmerPathLoc right( q, b.Interval( ), b.PosOnInterval( ) );
          addlow += DistMin( q.Begin( ), right ) + 1;
          addhigh += DistMax( q.Begin( ), right ) + 1;
          for ( int u = 0; u < allpathlengths.isize( ); u++ )
          {    allpathlengths[u].first += addlow;
               allpathlengths[u].second += addhigh;    }    }
     int sep_exp = P.sep + K( ) - 1 + trims[id1] + trims[id2];
     Float sep_dev_limit = Float(3.0);
     for ( int u = 0; u < allpathlengths.isize( ); u++ )
     {    int sep_low = allpathlengths[u].first, sep_high = allpathlengths[u].second;
          int diff;
          if ( sep_exp < sep_low ) diff = sep_low - sep_exp;
          else if ( sep_exp > sep_high ) 
          {    diff = sep_exp - sep_high;    }
          else diff = 0;
          Float sep_dev = Float(diff)/Float(P.sd);
          if ( sep_dev <= sep_dev_limit ) return True;    }
     return False;    }

void HyperKmerPath::SetToDisjointUnionOf( const vec<HyperKmerPath>& v )
{    for ( int i = 0; i < v.isize( ); i++ )
          ForceAssertEq( K( ), v[i].K( ) );
     Clear( );
     int nvert = 0, nedge = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    nvert += v[i].N( );
          nedge += v[i].EdgeObjectCount( );    }
     FromMutable( ).reserve(nvert), ToMutable( ).reserve(nvert);
     FromEdgeObjMutable( ).reserve(nvert), ToEdgeObjMutable( ).reserve(nvert);
     EdgesMutable( ).reserve(nedge);
     int vcount = 0, ecount = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    const HyperKmerPath& h = v[i];
          EdgesMutable( ).append( h.Edges( ) );
          vec< vec<vrtx_t> > from = h.From( ), to = h.To( );
          vec< vec<int> > frome = h.FromEdgeObj( ), toe = h.ToEdgeObj( );
          for ( int j = 0; j < h.N( ); j++ )
          {    for ( int u = 0; u < from[j].isize( ); u++ )
               {    from[j][u] += vcount;
                    frome[j][u] += ecount;    }
               for ( int u = 0; u < to[j].isize( ); u++ )
               {    to[j][u] += vcount;
                    toe[j][u] += ecount;    }    }
          FromMutable( ).append(from), ToMutable( ).append(to);
          FromEdgeObjMutable( ).append(frome), ToEdgeObjMutable( ).append(toe);
          vcount += h.N( );
          ecount += h.EdgeObjectCount( );    }    }

void HyperBasevector::SetToDisjointUnionOf( const vec<HyperBasevector>& v )
{    for ( int i = 0; i < v.isize( ); i++ )
          ForceAssertEq( K( ), v[i].K( ) );
     Clear( );
     int nvert = 0, nedge = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    nvert += v[i].N( );
          nedge += v[i].EdgeObjectCount( );    }
     FromMutable( ).reserve(nvert), ToMutable( ).reserve(nvert);
     FromEdgeObjMutable( ).reserve(nvert), ToEdgeObjMutable( ).reserve(nvert);
     EdgesMutable( ).reserve(nedge);
     int vcount = 0, ecount = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    const HyperBasevector& h = v[i];
          EdgesMutable( ).append( h.Edges( ) );
          vec< vec<vrtx_t> > from = h.From( ), to = h.To( );
          vec< vec<int> > frome = h.FromEdgeObj( ), toe = h.ToEdgeObj( );
          for ( int j = 0; j < h.N( ); j++ )
          {    for ( int u = 0; u < from[j].isize( ); u++ )
               {    from[j][u] += vcount;
                    frome[j][u] += ecount;    }
               for ( int u = 0; u < to[j].isize( ); u++ )
               {    to[j][u] += vcount;
                    toe[j][u] += ecount;    }    }
          FromMutable( ).append(from), ToMutable( ).append(to);
          FromEdgeObjMutable( ).append(frome), ToEdgeObjMutable( ).append(toe);
          vcount += h.N( );
          ecount += h.EdgeObjectCount( );    }    }

void HyperKmerPath::TestKmersGood( const KmerBaseBroker& kbb ) const
{    for ( int z = 0; z < EdgeObjectCount( ); z++ )
     {    const KmerPath& p = EdgeObject(z);
          KmerPath q;
          for ( int w = 0; w < p.NSegments( ); w++ )
          {    if ( w == 0 && p.Segment(w).isGap( ) ) continue;
               if ( w == p.NSegments( ) - 1 && p.Segment(w).isGap( ) )
                    continue;
               q.AddSegment( p.Segment(w) );    }
          SuperBaseVector b = kbb.ToSequence(q);    }
     for ( int v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < From(v).isize( ); i++ )
          {    int w = From(v)[i];
               const KmerPath& p = EdgeObjectByIndexFrom( v, i );
               for ( int j = 0; j < From(w).isize( ); j++ )
               {    const KmerPath& q = EdgeObjectByIndexFrom( w, j );
                    KmerPath pq;
                    for ( int u = 0; u < p.NSegments( ); u++ )
                    {    if ( u == 0 && p.Segment(u).isGap( ) ) continue;
                         pq.AddSegment( p.Segment(u) );    }
                    for ( int u = 0; u < q.NSegments( ); u++ )
                    {    if ( u == q.NSegments( ) - 1 && q.Segment(u).isGap( ) )
                              continue;
                         pq.AddSegment( q.Segment(u) );    }
                    SuperBaseVector bb = kbb.ToSequence(pq);    }    }    }    }

void BinaryWrite( int fd, const HyperKmerPath& h )
{    WriteBytes( fd, &h.K_, sizeof(int) );
     BinaryWrite( fd, (const digraphE<KmerPath>&) h );    }

void BinaryRead( int fd, HyperKmerPath& h )
{    ReadBytes( fd, &h.K_, sizeof(int) );
     BinaryRead( fd, (digraphE<KmerPath>&) h );    }

void HyperKmerPath::ReverseComponent( vrtx_t x )
{    equiv_rel e( N( ) );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < From(v).isize( ); i++ )
          {    int w = From(v)[i];
               e.Join( v, w );    }    }
     vec<int> o; 
     e.Orbit( x, o ); 
     for ( int j = 0; j < o.isize( ); j++ ) 
     {    int i = o[j];
          for ( int u = 0; u < From(i).isize( ); u++ )
          {    int e = EdgeObjectIndexByIndexFrom( i, u );
               KmerPath& p = EdgeObjectMutable(e);
               p.Reverse( );    }    }
     digraphE<KmerPath>::ReverseComponent(x);    }

void HyperKmerPath::Reverse( )
{    for ( int i = 0; i < EdgeObjectCount( ); i++ )
          EdgeObjectMutable(i).Reverse( );
     digraphE<KmerPath>::Reverse( );    }

void HyperBasevector::Reverse( )
{    for ( int i = 0; i < EdgeObjectCount( ); i++ )
          EdgeObjectMutable(i).ReverseComplement( );
     digraphE<basevector>::Reverse( );    }

String EdgeLabel( vrtx_t v, vrtx_t w, const KmerPath& p, int K )
{    float len = 0, dev = 0;
     for ( int u = 0; u < p.NSegments( ); u++ )
     {    const KmerPathInterval& x = p.Segment(u);
          if ( x.isSeq( ) ) len += x.Length( );
          else
          {    len += float( x.Maximum( ) + x.Minimum( ) ) / 2.0;
               dev += float( x.Maximum( ) - x.Minimum( ) ) / 2.0;    }    }
     ostrstream out;
     out << v << " --- " << setprecision(9) << len << " +/- " << dev 
          << " --> " << w << ends;
     return out.str( );    }

void KmerPathCorrespondence::Swap( )
{    swap( id1_, id2_ );
     swap( from1_, from2_ );
     swap( from_index1_, from_index2_ );
     swap( to1_, to2_ );
     swap( l1_, l2_ );    }

Bool operator<( const KmerPathCorrespondence& c1, const KmerPathCorrespondence& c2 )
{    if ( c1.Id1( ) < c2.Id1( ) ) return True;
     if ( c1.Id1( ) > c2.Id1( ) ) return False;
     if ( c1.Id2( ) < c2.Id2( ) ) return True;
     if ( c1.Id2( ) > c2.Id2( ) ) return False;
     if ( c1.Offset( ) < c2.Offset( ) ) return True;
     return False;    }

KmerPath MultiKmerPath::Merger( ) const
{    KmerPath answer;
     const vec<KmerPath>& p = *this;
     for ( int i = 0; i < isize( ); i++ )
     {    for ( int j = 0; j < p[i].NSegments( ); j++ )
               answer.AddSegmentNoConcatenate( p[i].Segment(j) );    }
     return answer;    }

MultiKmerPathLoc::MultiKmerPathLoc( const MultiKmerPath& m, const KmerPathLoc& l )
{    edge_ = 0;
     segment_on_edge_ = l.GetIndex( );
     kmer_on_segment_ = l.GetLoc( );
     for ( int u = 0; u < m.isize( ); u++ )
     {    int segs_in_edge = m[u].NSegments( );
          if ( segs_in_edge > segment_on_edge_ ) break;
          segment_on_edge_ -= segs_in_edge;
          ++edge_;    }    }

void PointedSubPathPair::TestValid( ) const
{    const KmerPath& p1 = Path1( ).EdgeObject( Loc1( ).Edge( ) );
     const KmerPath& p2 = Path2( ).EdgeObject( Loc2( ).Edge( ) );
     int seg1 = Loc1( ).SegmentOnEdge( ), seg2 = Loc2( ).SegmentOnEdge( );      
     ForceAssertEq(
          p1.Segment(seg1).Start( ) + Loc1( ).KmerOnSegment( ),
          p2.Segment(seg2).Start( ) + Loc2( ).KmerOnSegment( ) );    }

KmerPathLoc MultiKmerPathLoc::PosOnMerger( 
     const MultiKmerPath& m, const KmerPath& merger ) const
{    int segment = SegmentOnEdge( );
     for ( int i = 0; i < Edge( ); i++ )
          segment += m[i].NSegments( );
     return KmerPathLoc( merger, segment, KmerOnSegment( ) );    }

Bool operator<=( const MultiKmerPathLoc& l1, const MultiKmerPathLoc& l2 )
{    if ( l1.Edge( ) < l2.Edge( ) ) return True;
     if ( l1.Edge( ) > l2.Edge( ) ) return False;
     if ( l1.SegmentOnEdge( ) < l2.SegmentOnEdge( ) ) return True;
     if ( l1.SegmentOnEdge( ) > l2.SegmentOnEdge( ) ) return False;
     if ( l1.KmerOnSegment( ) < l2.KmerOnSegment( ) ) return True;
     if ( l1.KmerOnSegment( ) > l2.KmerOnSegment( ) ) return False;
     return True;    }

void HyperKmerPath::TestValid( ) const
{    digraphE<KmerPath>::TestValid( );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    int w = From(v)[j];
               const KmerPath& p = EdgeObjectByIndexFrom( v, j );
               ForceAssertGt( p.NSegments( ), 0 );    }    }    }

void HyperKmerPath::PrintSummary( ostream& out ) const
{    for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    int w = From(v)[j];
               const KmerPath& p = EdgeObjectByIndexFrom( v, j );
               out << EdgeLabel( v, w, p, K( ) ) << "\n";    }    }    }

void HyperKmerPath::PrintSummaryDOT( ostream& out ) const
{    vec< vec<String> > edge_labels;
     edge_labels.resize( N( ) );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    const vec<vrtx_t>& from = From(v);
          edge_labels[v].resize( from.size( ) );
          for ( int j = 0; j < from.isize( ); j++ )
          {    const KmerPath& p = EdgeObjectByIndexFrom( v, j );
               vrtx_t w = from[j];
               edge_labels[v][j] = EdgeLabel( v, w, p, K( ) );    }    }
     DOT( out, edge_labels );    }

void HyperKmerPath::PrintSummaryDOT0( ostream& out ) const
{    DOT( out );    }

void HyperKmerPath::PrintSummaryDOT0w( ostream& out, Bool label_contigs,
     Bool label_vertices, Bool label_edges, const vec<vrtx_t>* componentsToPrint ) const
{    
     // Set up output.

     out << "digraph G {\n\n";
     // out << "orientation=landscape;\n";
     if (label_vertices)
          out << "node [width=0.1,height=0.1,fontsize=12,shape=plaintext];\n";
     else out << "node [width=0.1,height=0.1,fontsize=10,shape=point];\n";
     out << "edge [fontsize=12];\n";
     if (label_contigs) out << "margin=1.0;\n";
     out << "rankdir=LR;\n";
     out << "labeljust=l;\n";

     // Define components.

     equiv_rel e;
     ComponentRelation(e);
     vec<vrtx_t> reps;
     e.OrbitRepsAlt(reps);

     // Print the contigs.  We put each contig in its own cluster (the 
     // subgraph's name MUST start with "cluster" for this to have any effect).

     for ( int i = reps.isize( ) - 1; i >= 0; i-- )
     {
       if ( componentsToPrint && !binary_search(componentsToPrint->begin(), componentsToPrint->end(), reps[i] ) )
	    continue;

       out << "\nsubgraph cluster" << i << " {\n";
          out << "color=white;\n";
          // out << "label=\"contig " << i << "\";\n";
          static vec<vrtx_t> o;
          e.Orbit( reps[i], o );

          // Find "leftmost" vertex.

          Sort(o);
          vec<float> pos( o.size( ) );
          vec<Bool> placed( o.size( ), False );
          pos[0] = 0.0, placed[0] = True;
          while( Sum(placed) < o.isize( ) )
          {    for ( int i1 = 0; i1 < o.isize( ); i1++ )
               {    int v = o[i1];
                    for ( int j = 0; j < From(v).isize( ); j++ )
                    {    int w = From(v)[j];
                         int i2 = BinPosition( o, w );
                         if ( !( placed[i1] ^ placed[i2] ) ) continue;
                         const KmerPath& p = EdgeObjectByIndexFrom( v, j );
                         if ( placed[i1] ) pos[i2] = pos[i1] + p.MidLength( );
                         else pos[i1] = pos[i2] - p.MidLength( );    
                         placed[i1] = placed[i2] = True;    }    }    }
          float left = Min(pos);
          int leftj = 0;
          for ( leftj = 0; leftj < pos.isize( ); leftj++ )
               if ( pos[leftj] == left ) break;
          int leftv = o[leftj];

          // Print component.

          for ( int vi = 0; vi < o.isize( ); vi++ )
          {    int v = o[vi];
               if (label_vertices)
               {    out << v << " [label=" << "\"" << v << "\"" 
                         << ",fontcolor=black];\n";    }
               for ( int j = 0; j < From(v).isize( ); j++ )
               {    const KmerPath& e = EdgeObjectByIndexFrom( v, j );
                    int ei = EdgeObjectIndexByIndexFrom( v, j );
                    int w = From(v)[j];
                    float wd = 0.1; // this value not used
                    String color, label;
                    Bool bold = False;
                    double len = e.MidLength( );
                    if ( e.MidLength( ) < 100.0 ) 
                    {    color = "gray";
                         if ( v == w ) label = ToString( int(round(len)) );
                         wd = 1.0;    }
                    else if ( len >= 100.0 && len < 1000.0 )
                    {    color = "black";
                         wd = 2.0;    }
                    else if ( len >= 1000.0 && len < 10000.0 )
                    {    color = "red";
                         wd = 4.0;
                         label = ToString( len/1000.0, 1 ) + " kb";    }
                    else if ( len >= 10000.0 )
                    {    color = "magenta";
                         bold = True;
                         wd = 8.0;
                         label = ToString( int(round(len/1000)) ) + " kb";    }
                    out << v << " -> " << w
                         << " [minlen=" << wd << ",color=" << color;
                    if (bold) out << ",style=bold";
                    if (label_edges)
                    {    if ( label == "" ) label = BaseAlpha(ei);
                         else label = BaseAlpha(ei) + " (" + label + ")";    }
                    if ( label != "" ) out << ",label=\"" << label << "\"";
                    if ( label_contigs && v == leftv && j == 0 )
                    {    out << ",taillabel=\"contig " << i << "\",labelangle=180,"
                              << "labeldistance=4.75,labelfontsize=18,"
                              << "labelfontname=\"Times-Bold\"";    }
                    out << "];\n";    }    }
          out << "}\n";    }
     out << "\n}" << endl;    }

void HyperKmerPath::PrintSummaryPlus( ostream& out, 
     const vec<HyperSloppyReadPlacement>* locs, KmerBaseBroker* kbb,
     vec<read_pairing>* pairs, vec<int>* pairs_index,
     int max_reads_to_show_per, Bool print_kmer_ranges,
     const vec<String>* edge_remarks, Bool print_component_id_line,
     const vec<String>* component_remarks,
     const vec<Bool>* component_remarks_only ) const
{    
     // Find read locations assigned to each edge.

     vec< vec<HyperSloppyReadPlacement> > locs_by_edge;
     if ( max_reads_to_show_per > 0 )
     {    locs_by_edge.resize( EdgeObjectCount( ) );
          const vec<HyperSloppyReadPlacement>& locsx = *locs;
          for ( int i = 0; i < locsx.isize( ); i++ )
          {    const HyperSloppyReadPlacement& l = locsx[i];
               int e = EdgeObjectIndexByIndexFrom( l.Vertex( ), l.EdgeFrom( ) );
               locs_by_edge[e].push_back(l);    }
          for ( int i = 0; i < EdgeObjectCount( ); i++ )
          {    stable_sort( locs_by_edge[i].begin( ), 
                    locs_by_edge[i].end( ), cmp_pos );    }    }

     // Define equivalence relation whose orbits are the components.

     equiv_rel e( N( ) );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
               e.Join( v, From(v)[j] );    }

     // Set up indices.

     int nr = 0;
     if ( max_reads_to_show_per > 0 )
     {    const vec<HyperSloppyReadPlacement>& locsx = *locs;
          for ( int i = 0; i < locsx.isize( ); i++ )
               nr = Max( nr, locsx[i].ReadId( ) + 1 );    }
     vec< vec<int> > to_comp(nr), to_comp_pos(nr);

     // Go through the components.  There are two passes.  In the first pass, we
     // just fill in the indices.

     for ( int pass = 1; pass <= 2; pass++ )
     {
     int count = 0;
     for ( vrtx_t x = 0; x < N( ); x++ )
     {    if ( e.Representative(x) )
          {    if ( pass == 2 && print_component_id_line ) 
                    out << "\ncomponent " << count << "\n";
               if ( pass == 2 && component_remarks != 0 )
                    out << (*component_remarks)[count];
               if ( component_remarks_only != 0 && (*component_remarks_only)[count] )
               {    ++count;
                    continue;    }
               ++count;
               vec<vrtx_t> o;
               e.Orbit( x, o );
               Sort(o);
               vec<float> pos( o.size( ) );
               vec<Bool> placed( o.size( ), False );
               pos[0] = 0.0, placed[0] = True;
               while( Sum(placed) < o.isize( ) )
               {    for ( int i1 = 0; i1 < o.isize( ); i1++ )
                    {    int v = o[i1];
                         for ( int j = 0; j < From(v).isize( ); j++ )
                         {    int w = From(v)[j];
                              int i2 = BinPosition( o, w );
                              if ( !( placed[i1] ^ placed[i2] ) ) continue;
                              const KmerPath& p = EdgeObjectByIndexFrom( v, j );
                              if ( placed[i1] ) pos[i2] = pos[i1] + p.MidLength( );
                              else pos[i1] = pos[i2] - p.MidLength( );    
                              placed[i1] = placed[i2] = True;    }    }    }
               vec<float> opos( o.size( ) );
               for ( int i = 0; i < o.isize( ); i++ )
                    opos[i] = pos[i];
               SortSync( opos, o );
               int edgeid = 0;
               for ( int i = 0; i < o.isize( ); i++ )
               {    vrtx_t v = o[i];
                    vec<vrtx_t> f = From(v);
                    vec<int> ind( f.size( ) );
                    for ( int j = 0; j < ind.isize( ); j++ )
                         ind[j] = j;
                    vec<float> flen( f.size( ) );
                    for ( int j = 0; j < f.isize( ); j++ )
                         flen[j] = EdgeObjectByIndexFrom( v, j ).MidLength( );
                    SortSync( flen, ind );
                    for ( int j = 0; j < ind.isize( ); j++ )
                    {    ++edgeid;
                         vrtx_t w = f[ ind[j] ];
                         const KmerPath& p = EdgeObjectByIndexFrom( v, ind[j] );

                         // Record component and edge indices.

                         int e = EdgeObjectIndexByIndexFrom( v, ind[j] );
                         if ( pass == 1 && max_reads_to_show_per > 0 )
                         {    const vec<HyperSloppyReadPlacement>& 
                                   h = locs_by_edge[e];
                              for ( int l = 0; l < h.isize( ); l++ )
                              {    const HyperSloppyReadPlacement& r = h[l];
                                   to_comp[ r.ReadId( ) ].push_back(count-1);
                                   to_comp_pos[ r.ReadId( ) ].
                                        push_back(edgeid-1);    }    }
                         if ( pass == 1 ) continue;

                         // Print edge.

                         out << "\n" << "[";
                         if (print_component_id_line) 
                              out << count-1 << "." << edgeid-1 
                                  << " = " << BaseAlpha(e) << "] ";
                         else out << BaseAlpha(e) << "] ";
                         out << EdgeLabel( v, w, p, K( ) ) << "\n";
                         if (print_kmer_ranges) PrintFolded( out, p );
                         if ( To(v).empty( ) ) out << "[" << v << " is source]\n";
                         if ( From(w).empty( ) ) out << "[" << w << " is sink]\n";

                         // Check for bubble: v != w and there are exactly two edges 
                         // from v to w, and both are gap-free.  Then print the
                         // alignment of the two edges.

                         vec<int> js;
                         for ( int j2 = 0; j2 < ind.isize( ); j2++ )
                         {    int w2 = f[ ind[j2] ];
                              if ( w2 == w ) js.push_back(j2);    }
                         if ( v != w && js.size( ) == 2 && js[0] == j && kbb != 0 )
                         {    int j2 = js[1];
                              const KmerPath& p2 
                                   = EdgeObjectByIndexFrom( v, ind[j2] );
                              if ( p.GapFree( ) && p2.GapFree( ) )
                              {    basevector b = kbb->ToSequence(p).Seq(0);
                                   basevector b2 = kbb->ToSequence(p2).Seq(0);
                                   align a;
                                   int offset = 0, bandwidth = 20, errors;
                                   SmithWatBandedA( b, b2, offset, 
                                        bandwidth, a, errors );
                                   out << "\nalignment of the two paths from "
                                        << v << " to " << w << "\n";
                                   PrintVisualAlignment( 
                                        True, out, b, b2, a );    }    }

                         if ( edge_remarks != 0 ) out << (*edge_remarks)[e];

                         if ( max_reads_to_show_per > 0 )
                         {    const vec<HyperSloppyReadPlacement>& 
                                   h = locs_by_edge[e];
                              for ( int l = 0; l < h.isize( ); l++ )
                              {    if ( l >= max_reads_to_show_per )
                                   {    out << "...\n";
                                        break;    }
                                   const HyperSloppyReadPlacement& r = h[l];
                                   out << "rd " << r.ReadId( ) << " " 
                                        << ( r.Rc( ) ? "rc" : "fw" ) << " @ ";
                                   if ( r.LeftEndMapped( ) )
                                   {    out << r.LeftLoc( ).Interval( ) << "."
                                             << r.LeftLoc( ).PosOnInterval( );    }
                                   else out << "...";
                                   out << " - ";
                                   if ( r.RightEndMapped( ) )
                                   {    out << r.RightLoc( ).Interval( ) << "."
                                             << r.RightLoc( ).PosOnInterval( );    }
                                   else out << "...";
                                   int id1 = r.ReadId( );
                                   const vec<int>& is_pairs_index = *pairs_index;
                                   int pi = is_pairs_index[id1];
                                   if ( pi >= 0 )
                                   {    const vec<read_pairing>& is_pairs = *pairs;
                                        int id2 = is_pairs[pi].Partner(id1);
                                        out << " [p - " << id2 
                                             << ", sep = " << is_pairs[pi].sep
                                             << " +/- " << is_pairs[pi].sd << "]";
                                        if ( to_comp[id2].size( ) == 0 )
                                             out << " - unplaced";
                                        if ( to_comp[id2].size( ) == 1 )
                                        {    int comp = to_comp[id2][0];
                                             int comp_pos = to_comp_pos[id2][0];
                                             if ( comp != count-1 ||
                                                  comp_pos != edgeid-1 )
                                             {    out << " - comp " << comp << "."
                                                       << comp_pos;    }    }
                                        Bool shown = False;
                                        if ( to_comp[id2].size( ) == 2 )
                                        {    int comp1 = to_comp[id2][0];
                                             int comp2 = to_comp[id2][1];
                                             int comp_pos1 = to_comp_pos[id2][0];
                                             int comp_pos2 = to_comp_pos[id2][1];
                                             if ( comp_pos2 < comp_pos1 )
                                                  swap( comp_pos1, comp_pos2 );
                                             if ( comp1 == comp2 )
                                             {    shown = True;
                                                  out << " - comp " << comp1 << ".{"
                                                       << comp_pos1
                                                       << "," << comp_pos2
                                                       << "}";    }    }
                                        if ( !shown && to_comp[id2].size( ) > 1 )
                                        {    out << " - " << to_comp[id2].size( )
                                                  << " comps";    }    }
                                   else out << " [unpaired]";
                                   out << "\n";    }    }    }    }    }    }    }   
     flush(out);    }

void PointedSubPathPair::PrintSummary( ostream& out, int i, int K ) const
{    const EmbeddedSubPath<KmerPath>& e = Path(i);
     for ( int j = 0; j < e.NVertices( ); j++ )
     {    out << e.Vertex(j);
          if ( j < e.NVertices( ) - 1 )
          {    const KmerPath& p = e.EdgeObject(j);
               float len = 0, dev = 0;
               for ( int u = 0; u < p.NSegments( ); u++ )
               {    const KmerPathInterval& x = p.Segment(u);
                    if ( x.isSeq( ) ) len += x.Length( );
                    else
                    {    len += float( x.Maximum( ) + x.Minimum( ) ) / 2.0;
                         dev += float( x.Maximum( ) - x.Minimum( ) ) / 2.0;    }    }
               ostrstream strout;
               strout << " --- " << len << " +/- " << dev << " --> " << ends;
               out << strout.str( );    }    }    }

// INCOMPLETE:

void AlignFullTo( const vec<KmerPath>& p, const HyperKmerPath& h,
     vec< vec<FullMapToHyperKmerPath> >& m )
{
     // Build path database.

     vec<Bool> used;
     // h.Used(used); // ???????????????????????????????????????????????????????????
     int nused = Sum(used);
     vec<int> from( used.size( ) ), from_index( used.size( ) ), to( used.size( ) );
     for ( int v = 0; v < h.N( ); v++ )
     {    for ( int j = 0; j < h.From(v).isize( ); j++ )
          {    int w = h.From(v)[j];
               int e = h.EdgeObjectIndexByIndexFrom( v, j );
               from[e] = v;
               from_index[e] = j;
               to[e] = w;    }    }
     vec<tagged_rpint> segs;
     segs.reserve(nused);
     for ( int i = 0; i < used.isize( ); i++ )
          if ( used[i] ) h.EdgeObject(i).AppendToDatabase( segs, i );
     Prepare(segs);

     // Go through the paths p.

     for ( int i = 0; i < p.isize( ); i++ )
     {    const KmerPath& pi = p[i];
          ForceAssert( pi.Proper( ) );
          longlong firstkmer = pi.Segment(0).Start( );
          static vec<longlong> places;
          Contains( segs, firstkmer, places );
          for ( int j = 0; j < places.isize( ); j++ )
          {    const tagged_rpint& t = segs[ places[j] ];

               // ...
                          }

          // ...
                    }

     // ...
               }

void HyperKmerPath::SharedKmers( vec<KmerPathCorrespondence>& shares,
     int min_improper )
{    shares.clear( );

     // Build path database.

     vec<Bool> used;
     Used(used);
     int nused = Sum(used);
     vec<int> from( used.size( ) ), from_index( used.size( ) ), to( used.size( ) );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    int w = From(v)[j];
               int e = EdgeObjectIndexByIndexFrom( v, j );
               from[e] = v;
               from_index[e] = j;
               to[e] = w;    }    }
     vec<tagged_rpint> segs;
     segs.reserve(nused);
     for ( int i = 0; i < used.isize( ); i++ )
          if ( used[i] ) EdgeObject(i).AppendToDatabase( segs, i );
     Prepare(segs);
     static vec<vrtx_t> to_left, to_right;
     static vec<int> edgelength;
     ToLeft(to_left), ToRight(to_right);
     edgelength.resize( EdgeObjectCount( ) );
     for ( int i = 0; i < edgelength.isize( ); i++ )
          edgelength[i] = EdgeLength(i);

     // Search path database for shared kmers.  

     vec< vec<int> > starts( EdgeObjectCount( ) );
     for ( int i = 0; i < EdgeObjectCount( ); i++ )
     {    int start = 0;
          for ( int j = 0; j < EdgeObject(i).NSegments( ); j++ )
          {    starts[i].push_back(start);
               start += EdgeObject(i).Length(j);    }    }
     for ( int i = 0; i < segs.isize( ); i++ )
     {    int id1 = segs[i].PathId( ), pos1 = segs[i].PathPos( );
          static vec<longlong> matches;
          const KmerPathInterval& rpi1 = EdgeObject(id1).Segment(pos1);
          Contains( segs, rpi1, matches );
          for ( int j = 0; j < matches.isize( ); j++ )
          {    const tagged_rpint& t = segs[ matches[j] ];
               int id2 = t.PathId( ), ppos2 = t.PathPos( );
               if ( id2 < id1 ) continue;
               if ( id1 == id2 && pos1 == ppos2 ) continue;
               const KmerPathInterval& rpi2 = EdgeObject(id2).Segment(ppos2);

               // Now we have two overlapping path intervals.  Exhibit positions
               // on each which correspond to the same kmer.

               int ind1, ind2;
               longlong start1 = rpi1.Start( ), stop1 = rpi1.Stop( );
               longlong start2 = rpi2.Start( ), stop2 = rpi2.Stop( );
               if ( start1 >= start2 )
               {    ind1 = 0, ind2 = int(start1 - start2);    }
               else
               {    ind2 = 0, ind1 = int(start2 - start1);    }

               // Reject if the kmers to the left match.

               if ( ind1 == 0 && ind2 == 0 && pos1 > 0 && ppos2 > 0 )
               {    if ( EdgeObject(id1).Stop(pos1-1) 
                         == EdgeObject(id2).Stop(ppos2-1) )
                    {    continue;    }    }

               // Generate the share.

               int pathpos1 = starts[id1][pos1] + ind1;
               int pathpos2 = starts[id2][ppos2] + ind2;
               KmerPathLocAlt l1( EdgeObject(id1), pos1, ind1, pathpos1 );
               KmerPathLocAlt l2( EdgeObject(id2), ppos2, ind2, pathpos2 );
               KmerPathCorrespondence s( id1, from[id1], from_index[id1], 
                    to[id1], l1, id2, from[id2], from_index[id2], to[id2], l2 );

               // Remove shares whose perfect extensions P are improper and such
               // that length(P) is less than min_improper.

               int e1 = s.Id1( ), e2 = s.Id2( );
               int v1 = to_left[e1], w1 = to_right[e1];
               int v2 = to_left[e2], w2 = to_right[e2];

               KmerPathLoc scanStart1( EdgeObject(e1), s.Pos1( ).Interval( ),
                    s.Pos1( ).PosOnInterval( ) );
               KmerPathLoc scanStart2( EdgeObject(e2), s.Pos2( ).Interval( ),
                    s.Pos2( ).PosOnInterval( ) );

               KmerPathLoc rightScan1( scanStart1 ), rightScan2( scanStart2 );
               ScanRightPerfectMatch( rightScan1, rightScan2 );
               int right_ext = KmersInInterval( scanStart1, rightScan1 ) - 1;

               KmerPathLoc leftScan1( scanStart1 ), leftScan2( scanStart2 );
               ScanLeftPerfectMatch( leftScan1, leftScan2 );
               int left_ext = KmersInInterval( leftScan1, scanStart1 ) - 1;

               int pos1 = s.Pos1( ).PosOnPath( ), pos2 = s.Pos2( ).PosOnPath( );
               int len1 = edgelength[e1], len2 = edgelength[e2];
               int left1 = pos1 - left_ext, right1 = pos1 + right_ext + 1;
               int left2 = pos2 - left_ext, right2 = pos2 + right_ext + 1;
               Bool left_proper = ( left1 == 0 || left2 == 0 );
               Bool right_proper = ( right1 == len1 || right2 == len2 );
               Bool proper = ( left_proper && right_proper );
               if ( !proper && left_ext + right_ext + 1 < min_improper ) continue;

               // Save the share.

               shares.push_back(s);    }    }

     // Sort shares.

     Sort(shares);    }

void HyperKmerPath::ContractEmptyEdges( ) {
  bool contracted = false;
  for( int v = 0; v < N(); v += (!contracted) ) {
    contracted = false;
    for( int j = From(v).size()-1; j>=0; j-- )
      if( EdgeObjectByIndexFrom(v, j).IsEmpty() ) {
	ContractEdgeFrom(v,j);
	contracted = true;
	break;
      }
  }
}



// BAD BAD BAD:

int MinLen( const KmerPathInterval& I )
{    if ( I.isSeq( ) ) return I.Length( );
     else return 1000000;    }  // VERY SLOPPY.

// Advance: move a KmerPathLoc to the next kmer.  The starting location must not be
// in a gap.

void Advance( KmerPathLoc& l )
{    const KmerPath& p = l.GetPath( );
     ForceAssert( !p.Segment( l.GetIndex( ) ).isGap( ) );
     if ( l.GetLoc( ) + 1 < p.Segment( l.GetIndex( ) ).Length( ) )
          l.SetLoc( l.GetLoc( ) + 1 );
     else
     {    ForceAssertLt( l.GetIndex( ) + 1, p.NSegments( ) );
          l.SetIndex( l.GetIndex( ) + 1 );
          l.SetLoc(0);    }    }

Bool InRange( const KmerPathLoc& l )
{    if ( l.GetIndex( ) < 0 ) return False;
     if ( l.GetIndex( ) >= l.GetPath( ).NSegments( ) ) return False;
     if ( l.GetLoc( ) < 0 ) return False;
     if ( l.GetLoc( ) >= l.GetPath( ).Segment( l.GetIndex( ) ).Length( ) ) 
          return False;
     return True;    }

// Note that if AlignSubpaths finds more than one path, it returns the first one.
// this can't be optimal.

void HyperKmerPath::AlignSubpaths( const PointedSubPathPair& p, int& naligns, 
     vec< pair<MultiKmerPathLoc,MultiKmerPathLoc> >& ends,
     KmerPath& M, vec<KmerPathSplitPoint>& i, vec<KmerPathSplitPoint>& j,
     const NegativeGapValidator& ngv ) const
{
     // Define KmerPaths to be aligned.

     MultiKmerPath m1( p.Path1( ) ), m2( p.Path2( ) );

     // Check inputs.

     ForceAssert( m1.size( ) > 0 && m2.size( ) > 0 );
     ForceAssert( m1[0].NSegments( ) > 0 && m2[0].NSegments( ) > 0 );

     // Create merged paths.

     KmerPath p1 = m1.Merger( ), p2 = m2.Merger( );

     // Test for equality.  (Don't deal with end gap case.)

     if ( m1 == m2 && p.Loc1( ) == p.Loc2( ) && !m1.back( ).LastSegment( ).isGap( ) )
     {    naligns = 1;
          M = p1;
          ends.resize(2);
          ends[0].first = ends[1].first = MultiKmerPathLoc( 0, 0, 0 );
          ends[0].second = ends[1].second
               = MultiKmerPathLoc( m1.size( ) - 1, m1.back( ).NSegments( ) - 1,
                    m1.back( ).LastSegment( ).Length( ) - 1 );
          i.resize( p.Path1( ).NVertices( ) ); j.resize( p.Path2( ).NVertices( ) );
          int sum = 0;
          for ( int u = 0; u < p.Path1( ).NVertices( ); u++ )
          {    i[u] = j[u] = KmerPathSplitPoint( sum, 0 );
               if ( u < p.Path1( ).NVertices( ) - 1 )
                    sum += m1[u].NSegments( );    }
          return;    }

     // Define positions of matching kmers on p1 and p2.

     KmerPathLoc w1 = p.Loc1( ).PosOnMerger( m1, p1 );
     KmerPathLoc w2 = p.Loc2( ).PosOnMerger( m2, p2 );
     ForceAssertEq( w1.GetKmer( ), w2.GetKmer( ) );

     // Check for two gaps in a row.  For now, reject such cases.  I don't know how
     // often this happens or even if it does happen.

     Bool have_two_gaps = False;
     for ( int u = 0; u < p1.NSegments( ) - 1; u++ )
     {    if ( p1.Segment(u).isGap( ) && p1.Segment(u+1).isGap( ) )
               have_two_gaps = True;    }
     for ( int u = 0; u < p2.NSegments( ) - 1; u++ )
     {    if ( p2.Segment(u).isGap( ) && p2.Segment(u+1).isGap( ) )
               have_two_gaps = True;    }
     if (have_two_gaps)
     {    naligns = 0;
          return;    }

     // If there is a gap at the beginning or end of either m1 or m2, remove the
     // gaps now, and put them back at the end of AlignSubpaths.  We do this 
     // because ImproperMerge doesn't handle nonstandard paths.

     KmerPathInterval front_gap1(0,0), front_gap2(0,0);
     KmerPathInterval back_gap1(0,0), back_gap2(0,0);
     Bool have_front_gap1 = m1[0].Segment(0).isGap( );
     Bool have_front_gap2 = m2[0].Segment(0).isGap( );
     Bool have_back_gap1 = m1.back( ).LastSegment( ).isGap( );
     Bool have_back_gap2 = m2.back( ).LastSegment( ).isGap( );
     if (have_front_gap1)
     {    front_gap1 = m1[0].Segment(0);
          KmerPath m1tail;
          for ( int u = 1; u < m1[0].NSegments( ); u++ )
               m1tail.AddSegmentNoConcatenate( m1[0].Segment(u) );
          m1[0] = m1tail;
          p1 = m1.Merger( );
          w1 = KmerPathLoc( p1, w1.GetIndex( ) - 1, w1.GetLoc( ) );    }
     if (have_front_gap2)
     {    front_gap2 = m2[0].Segment(0);
          KmerPath m2tail;
          for ( int u = 1; u < m2[0].NSegments( ); u++ )
               m2tail.AddSegmentNoConcatenate( m2[0].Segment(u) );
          m2[0] = m2tail;
          p2 = m2.Merger( );
          w2 = KmerPathLoc( p2, w2.GetIndex( ) - 1, w2.GetLoc( ) );    }
     if (have_back_gap1)
     {    back_gap1 = m1.back( ).LastSegment( );
          KmerPath m1head;
          for ( int u = 0; u < m1.back( ).NSegments( ) - 1; u++ )
               m1head.AddSegmentNoConcatenate( m1.back( ).Segment(u) );
          m1.back( ) = m1head;
          p1 = m1.Merger( );    }
     if (have_back_gap2)
     {    back_gap2 = m2.back( ).LastSegment( );
          KmerPath m2head;
          for ( int u = 0; u < m2.back( ).NSegments( ) - 1; u++ )
               m2head.AddSegmentNoConcatenate( m2.back( ).Segment(u) );
          m2.back( ) = m2head;
          p2 = m2.Merger( );    }

     // If p1 and/or p2 have adjacent and mergeable KmerPathIntervals, they will not
     // be handled correctly by ImproperMerge.  Therefore we contract mergeable 
     // KmerPathIntervals in p1 and p2, and translate locations to there, reversing
     // the process later.

     KmerPath p1c, p2c;
     int w1c_index = w1.GetIndex( ), w2c_index = w2.GetIndex( );
     int w1c_loc = w1.GetLoc( ), w2c_loc = w2.GetLoc( );
     for ( int u = 0; u < p1.NSegments( ); u++ )
     {    int segs_before = p1c.NSegments( );
          int kmers_before = 0;
          if ( segs_before > 0 )
               kmers_before = p1c.Segment( segs_before - 1 ).Length( );
          p1c.AddSegment( p1.Segment(u) );
          int segs_after = p1c.NSegments( );
          if ( segs_before == segs_after )
          {    if ( u <= w1.GetIndex( ) ) --w1c_index;
               if ( u == w1.GetIndex( ) ) w1c_loc += kmers_before;    }    }
     for ( int u = 0; u < p2.NSegments( ); u++ )
     {    int segs_before = p2c.NSegments( );
          int kmers_before = 0;
          if ( segs_before > 0 )
               kmers_before = p2c.Segment( segs_before - 1 ).Length( );
          p2c.AddSegment( p2.Segment(u) );
          int segs_after = p2c.NSegments( );
          if ( segs_before == segs_after )
          {    if ( u <= w2.GetIndex( ) ) --w2c_index;
               if ( u == w2.GetIndex( ) ) w2c_loc += kmers_before;    }    }
     KmerPathLoc w1c( p1c, w1c_index, w1c_loc );
     ForceAssert( InRange(w1c) ); // XXX
     KmerPathLoc w2c( p2c, w2c_index, w2c_loc );
     ForceAssert( InRange(w2c) ); // XXX
     ForceAssertEq( w1c.GetKmer( ), w2c.GetKmer( ) );

     /*
     vecKmerPath psave; // ZZZ
     psave.push_back(p1c); // ZZZ
     psave.push_back(p2c); // ZZZ
     cout << "\np1c:\n" << p1c << "\n\n"; // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
     cout << "\np2c:\n" << p2c << "\n\n"; // XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
     psave.WriteAll( "paths.problemc" ); // ZZZ
     PRINT2( w1c.GetIndex( ), w2c.GetIndex( ) ); // ZZZ
     */

     // Try to merge.

     vec<ImproperMerger> merges0, merges;
     ImproperMergePaths( p1c, p2c, w1c.GetIndex( ), w2c.GetIndex( ), 
          merges0, 1, &ngv );
     for ( int u = 0; u < merges0.isize( ); u++ )
          if ( ngv.MakeValid( merges0[u].merged ) ) merges.push_back( merges0[u] );
     naligns = merges.size( );
     if ( naligns == 0 ) return;
     M = merges[0].merged;

     // Translate positions on p1c and p2c back to p1 and p2, by counting kmers.

     KmerPathLoc left1c = merges[0].left_end1, right1c = merges[0].right_end1;
     KmerPathLoc left2c = merges[0].left_end2, right2c = merges[0].right_end2;
     int left1_count = 0, left2_count = 0, right1_count = 0, right2_count = 0;
     for ( int u = 0; u < left1c.GetIndex( ); u++ )
          left1_count += MinLen( p1c.Segment(u) );
     left1_count += left1c.GetLoc( );
     for ( int u = 0; u < left2c.GetIndex( ); u++ )
          left2_count += MinLen( p2c.Segment(u) );
     left2_count += left2c.GetLoc( );
     for ( int u = 0; u < right1c.GetIndex( ); u++ )
          right1_count += MinLen( p1c.Segment(u) );
     right1_count += right1c.GetLoc( );
     for ( int u = 0; u < right2c.GetIndex( ); u++ )
          right2_count += MinLen( p2c.Segment(u) );
     right2_count += right2c.GetLoc( );
     int left1_index = 0, left2_index = 0, right1_index = 0, right2_index = 0;
     int left1_loc = 0, left2_loc = 0, right1_loc = 0, right2_loc = 0;
     for ( int u = 0; u < p1.NSegments( ); u++ )
     {    if ( left1_count == MinLen( p1.Segment(u) ) )
          {    left1_loc = 0;
               left1_index = u+1;
               break;     }
          if ( left1_count < MinLen( p1.Segment(u) ) )
          {    left1_loc = left1_count;
               left1_index = u;
               break;    }
          left1_count -= MinLen( p1.Segment(u) );    }
     for ( int u = 0; u < p2.NSegments( ); u++ )
     {    if ( left2_count == MinLen( p2.Segment(u) ) )
          {    left2_loc = 0;
               left2_index = u+1;
               break;     }
          if ( left2_count < MinLen( p2.Segment(u) ) )
          {    left2_loc = left2_count;
               left2_index = u;
               break;    }
          left2_count -= MinLen( p2.Segment(u) );    }
     for ( int u = 0; u < p1.NSegments( ); u++ )
     {    if ( right1_count == MinLen( p1.Segment(u) ) )
          {    right1_loc = 0;
               right1_index = u+1;
               break;     }
          if ( right1_count < MinLen( p1.Segment(u) ) )
          {    right1_loc = right1_count;
               right1_index = u;
               break;    }
          right1_count -= MinLen( p1.Segment(u) );    }
     for ( int u = 0; u < p2.NSegments( ); u++ )
     {    if ( right2_count == MinLen( p2.Segment(u) ) )
          {    right2_loc = 0;
               right2_index = u+1;
               break;    }
          if ( right2_count < MinLen( p2.Segment(u) ) )
          {    right2_loc = right2_count;
               right2_index = u;
               break;    }
          right2_count -= MinLen( p2.Segment(u) );    }

     // Compute positions.

     KmerPathLoc left1( p1, left1_index, left1_loc );
     KmerPathLoc right1( p1, right1_index, right1_loc );
     KmerPathLoc left2( p2, left2_index, left2_loc );
     KmerPathLoc right2( p2, right2_index, right2_loc );
     ends.resize(2);
     ends[0].first = MultiKmerPathLoc( m1, left1 );
     ends[0].second = MultiKmerPathLoc( m1, right1 );
     ends[1].first = MultiKmerPathLoc( m2, left2 );
     ends[1].second = MultiKmerPathLoc( m2, right2 );
     ForceAssertEq( ends[0].first.Kmer(m1), ends[1].first.Kmer(m2) );
     ForceAssertEq( ends[0].second.Kmer(m1), ends[1].second.Kmer(m2) );
     i.clear( ), j.clear( );
     i.resize( p.Path1( ).NVertices( ) ); j.resize( p.Path2( ).NVertices( ) );

     for ( int x = 0; x < 2; x++ )
     {    vec<KmerPathSplitPoint>& ij = ( x == 0 ? i : j );
          for ( int u = 0; u < p.Path(x).NVertices( ); u++ )
          {    MultiKmerPathLoc uloc( u, 0, 0 );
               if ( uloc >= ends[x].first && uloc <= ends[x].second )
               {    KmerPathLoc l, onm;
                    if ( x == 0 ) onm = uloc.PosOnMerger(m1, p1);
                    else onm = uloc.PosOnMerger(m2, p2);
                    ForceAssert( InRange(onm) ); // XXX
                    KmerPath mc;
                    int index = onm.GetIndex( ), loc = onm.GetLoc( );
                    const KmerPath& p12 = ( x == 0 ? p1 : p2 );
                    for ( int v = 0; v < p12.NSegments( ); v++ )
                    {    int segs_before = mc.NSegments( );
                         int kmers_before = 0;
                         if ( segs_before > 0 )
                              kmers_before = mc.Segment( segs_before - 1 ).Length( );
                         mc.AddSegment( p12.Segment(v) );
                         int segs_after = mc.NSegments( );
                         if ( segs_before == segs_after )
                         {    if ( v <= onm.GetIndex( ) ) --index;
                              if ( v == onm.GetIndex( ) ) 
                                   loc += kmers_before;    }    }
                    const KmerPath& p12c = ( x == 0 ? p1c : p2c );
                    KmerPathLoc onmc( p12c, index, loc );
                    ForceAssert( InRange(onmc) ); // XXX
                    if ( p12c.Segment(index).isGap( ) )
                    {    ForceAssertGe( index, 1 );
                         ForceAssertEq( loc, 0 );
                         ForceAssert( !p12c.Segment(index-1).isGap( ) );
                         KmerPathLoc onmcleft( p12c, index-1, 
                              p12c.Segment(index-1).Length( ) - 1 );
                         KmerPathLoc lleft;
                         if ( x == 0 ) lleft = merges[0].PushLoc1(onmcleft);
                         else lleft = merges[0].PushLoc2(onmcleft);
                         ForceAssert( InRange(lleft) ); // XXX
                         Advance(lleft);
                         l = lleft;    }
                    else
                    {    if ( x == 0 ) l = merges[0].PushLoc1(onmc);
                         else l = merges[0].PushLoc2(onmc);
                         ForceAssert( InRange(l) );    } // XXX
                    ForceAssertGe( l.GetIndex( ), 0 );
                    ij[u] = KmerPathSplitPoint( l.GetIndex( ), l.GetLoc( ) );    }
               const MultiKmerPath& m12 = ( x == 0 ? m1 : m2 );
               if ( ends[x].second.AtRightEndOfEdge(m12) 
                    && u == ends[x].second.Edge( ) + 1 )
               {    ij[u] = KmerPathSplitPoint( 
                         M.NSegments( ), 0 );    }    }    }

     // Restore front and back gaps if we had to remove them.  We try to extend the
     // alignment over the gaps, but we could try harder.

     MultiKmerPathLoc &e0a = ends[0].first, &e1a = ends[1].first;
     MultiKmerPathLoc &e0b = ends[0].second, &e1b = ends[1].second;
     if ( have_front_gap1 || have_front_gap2 )
     {    Bool equal_front
               = have_front_gap1 && have_front_gap2 && front_gap1 == front_gap2;
          if (equal_front)
          {
               // Check to see if the alignment goes right up to the gap on both 
               // paths.

               if ( e0a.LeftEnd( ) && e1a.LeftEnd( ) )
               {    
                    for ( int u = 0; u < 2; u++ )
                    {    if ( ends[u].second.Edge( ) == 0 )
                              ends[u].second.AddToSegmentOnEdge(1);    }
                    KmerPath Mnew;
                    Mnew.AddSegment(front_gap1);
                    for ( int u = 0; u < M.NSegments( ); u++ )
                         Mnew.AddSegmentNoConcatenate( M.Segment(u) );
                    M = Mnew;
                    for ( int u = 0; u < i.isize( ); u++ )
                    {    if ( i[u].IntervalsToLeft( ) > 0 
                              || i[u].KmersInPartialToLeft( ) > 0 )
                         {    i[u].SetIntervalsToLeft( 
                                   i[u].IntervalsToLeft( ) + 1 );    }    }
                    for ( int u = 0; u < j.isize( ); u++ )
                    {    if ( j[u].IntervalsToLeft( ) > 0 
                              || j[u].KmersInPartialToLeft( ) > 0 )
                         {    j[u].SetIntervalsToLeft( 
                                   j[u].IntervalsToLeft( ) + 1 );    }    }    }
               else
               {    if ( e0a.Edge( ) == 0 ) e0a.AddToSegmentOnEdge(1);
                    if ( e1a.Edge( ) == 0 ) e1a.AddToSegmentOnEdge(1);
                    if ( e0b.Edge( ) == 0 ) e0b.AddToSegmentOnEdge(1);
                    if ( e1b.Edge( ) == 0 ) e1b.AddToSegmentOnEdge(1);    }    }
          if ( !equal_front && have_front_gap1 )
          {    if ( e0a.Edge( ) == 0 ) e0a.AddToSegmentOnEdge(1);
               if ( e0b.Edge( ) == 0 ) e0b.AddToSegmentOnEdge(1);    }
          if ( !equal_front && have_front_gap2 )
          {    if ( e1a.Edge( ) == 0 ) e1a.AddToSegmentOnEdge(1);
               if ( e1b.Edge( ) == 0 ) e1b.AddToSegmentOnEdge(1);    }    }
     if ( have_back_gap1 || have_back_gap2 )
     {    Bool equal_back
               = have_back_gap1 && have_back_gap2 && back_gap1 == back_gap2;
          if (equal_back)
          {
               // Check to see if the alignment goes right up to the gap on both 
               // paths.

               if ( e0b.Edge( ) == m1.isize( ) - 1 
                    && e0b.SegmentOnEdge( ) == m1.back( ).NSegments( ) - 1
                    && e0b.KmerOnSegment( ) 
                         == m1.back( ).LastSegment( ).Length( ) - 1
                    && e1b.Edge( ) == m2.isize( ) - 1 
                    && e1b.SegmentOnEdge( ) == m2.back( ).NSegments( ) - 1
                    && e1b.KmerOnSegment( ) 
                         == m2.back( ).LastSegment( ).Length( ) - 1 )
               {    M.AddSegment(back_gap1);
                    int ne1 = p.Path1( ).NEdges( ) - 1;
                    int ne2 = p.Path2( ).NEdges( ) - 1;
                    e0b.SetEdge(ne1);
                    e0b.SetSegmentOnEdge( 
                         p.Path1( ).EdgeObject(ne1).NSegments( ) - 1 );
                    e0b.SetKmerOnSegment( p.Path1( ).EdgeObject(ne1).LastSegment( )
                         .Length( ) - 1 );
                    e1b.SetEdge(ne2);
                    e1b.SetSegmentOnEdge( 
                         p.Path2( ).EdgeObject(ne2).NSegments( ) - 1 );
                    e1b.SetKmerOnSegment( p.Path2( ).EdgeObject(ne2).LastSegment( )
                         .Length( ) - 1 );
                    i.back( ) = KmerPathSplitPoint( M.NSegments( ), 0 );
                    j.back( ) = KmerPathSplitPoint( 
                         M.NSegments( ), 0 );    }    }    }    }

void HyperKmerPath::CompressEdgeObjects( )
{    for ( int i = 0; i < EdgeObjectCount( ); i++ )
     {    KmerPath& p = EdgeObjectMutable(i);
          static KmerPath pnew;
          pnew.Clear( );
          for ( int j = 0; j < p.NSegments( ); j++ )
               pnew.AddSegment( p.Segment(j) );
          p = pnew;    }    }

void HyperKmerPath::RemoveReadlessEdges( const vec<HyperSloppyReadPlacement>& locs )
{    vec<Bool> covered( EdgeObjectCount( ), False );
     for ( int i = 0; i < locs.isize( ); i++ )
     {    int e = EdgeObjectIndexByIndexFrom( 
               locs[i].Vertex( ), locs[i].EdgeFrom( ) );
          covered[e] = True;    }
     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = From(v).isize( ) - 1; j >= 0; j-- )
          {    int e = EdgeObjectIndexByIndexFrom( v, j );
               if ( !covered[e] ) DeleteEdgeFrom( v, j );    }    }
     RemoveUnneededVertices( );
     CompressEdgeObjects( );
     RemoveDeadEdgeObjects( );    }

// RemoveWeakBits: Suppose that one of the following two situations is encountered:
// (a) there are edges X ---> Y, Y ---> Z of length >= 4kb, and an edge
//     w: B ---> Y or w: Y ---> B of length <= 100, supported by only one read,
//     and B is a source or sink, and there are no other edges involving Y;
// (b) there are edges X ---> Y, Z ---> W of length >= 4kb, and edges
//     w: Y ---> Z, k: Y ---> Z of length <= 100, such that h is supported by only
//     one read, k is supported by at least three reads, and k has no gaps, and
//     there are no other edges involving Y and Z.
// Then edge w is deleted.

void HyperKmerPath::RemoveWeakBits( const vec<HyperSloppyReadPlacement>& locs )
{    
     // Compute coverage.

     vec<int> coverage( EdgeObjectCount( ), 0 );
     for ( int i = 0; i < locs.isize( ); i++ )
     {    int e = EdgeObjectIndexByIndexFrom( 
               locs[i].Vertex( ), locs[i].EdgeFrom( ) );
          ++coverage[e];    }

     // Case (a).

     for ( int Y = 0; Y < N( ); Y++ )
     {    if ( From(Y).size( ) == 1 && To(Y).size( ) == 2 )
          {    if ( EdgeObjectByIndexFrom( Y, 0 ).MinLength( ) < 4000 ) continue;
               for ( int i = 0; i < 2; i++ )
               {    if ( EdgeObjectByIndexTo( Y, 1-i ).MinLength( ) >= 4000
                         && EdgeObjectByIndexTo( Y, i ).MinLength( ) <= 100
                         && coverage[ EdgeObjectIndexByIndexTo( Y, i ) ] <= 1
                         && Source( To(Y)[i] ) )
                    {    DeleteEdgeTo( Y, i );
                         break;    }    }    }
          else if ( From(Y).size( ) == 2 && To(Y).size( ) == 1 )
          {    if ( EdgeObjectByIndexTo( Y, 0 ).MinLength( ) < 4000 ) continue;
               for ( int i = 0; i < 2; i++ )
               {    if ( EdgeObjectByIndexFrom( Y, 1-i ).MinLength( ) >= 4000
                         && EdgeObjectByIndexFrom( Y, i ).MinLength( ) <= 100
                         && coverage[ EdgeObjectIndexByIndexFrom( Y, i ) ] <= 1
                         && Sink( From(Y)[i] ) )
                    {    DeleteEdgeFrom( Y, i );
                         break;    }    }    }    }

     // Case (b).

     for ( int Y = 0; Y < N( ); Y++ )
     {    if ( From(Y).size( ) != 2 ) continue;
          int Z = From(Y)[0];
          if ( From(Y)[1] != Z ) continue;
          if ( To(Y).size( ) != 1 || From(Z).size( ) != 1 ) continue;
          if ( EdgeObjectByIndexTo( Y, 0 ).MinLength( ) < 4000 ) continue;
          if ( EdgeObjectByIndexFrom( Z, 0 ).MinLength( ) < 4000 ) continue;
          int i1 = 0, i2 = 1;
          if ( coverage[ EdgeObjectIndexByIndexFrom( Y, i1 ) ] > 1 ) swap( i1, i2 );
          if ( coverage[ EdgeObjectIndexByIndexFrom( Y, i1 ) ] > 1 ) continue;
          if ( coverage[ EdgeObjectIndexByIndexFrom( Y, i2 ) ] < 3 ) continue;
          if ( !EdgeObjectByIndexFrom( Y, i2 ).GapFree( ) ) continue;
          DeleteEdgeFrom( Y, i1 );    }

     // Clean up.

     RemoveUnneededVertices( );
     CompressEdgeObjects( );
     RemoveDeadEdgeObjects( );    }

void HyperKmerPath::GroupReadPlacements( const vec<HyperSloppyReadPlacement>& locs,
     int nreads, vec< vec< vec<int> > >& locgrps ) const
{    vec< vec<int> > locs_index(nreads); 
     for ( int i = 0; i < locs.isize( ); i++ )
          locs_index[ locs[i].ReadId( ) ].push_back(i);
     locgrps.clear( );
     locgrps.resize(nreads);
     for ( int id = 0; id < nreads; id++ )
     {    static vec<int> all, grp;
          static vec<HyperSloppyReadPlacement> h;
          all.clear( ), h.clear( );
          for ( int j = 0; j < locs_index[id].isize( ); j++ )
          {    h.push_back( locs[ locs_index[id][j] ] );
               all.push_back( locs_index[id][j] );    }
          static vec<Bool> used;
          used.resize_and_set( h.size( ), False );
          for ( int i = 0; i < h.isize( ); i++ )
          {    Bool source = True;
               for ( int j = 0; j < h.isize( ); j++ )
               {    if ( From( h[j].Vertex( ) )[ h[j].EdgeFrom( ) ] == h[i].Vertex( )
                         && h[j].Rc( ) == h[i].Rc( ) && j != i )
                    {    source = False;
                         break;    }    }
               if ( !source ) continue;
               grp.clear( );
               grp.push_back( all[i] );
               used[i] = True;
               int u = i;
               while(1)
               {    Bool extended = False;
                    for ( int j = 0; j < h.isize( ); j++ )
                    {    if ( From( h[u].Vertex( ) )[ h[u].EdgeFrom( ) ] 
                              == h[j].Vertex( )
                              && h[j].Rc( ) == h[u].Rc( ) && !used[j] )
                         {    grp.push_back( all[j] );
                              used[j] = True;
                              u = j;
                              extended = True;    }    }
                    if ( !extended ) break;    }
               locgrps[id].push_back(grp);    }
          for ( int i = 0; i < h.isize( ); i++ )
          {    if ( used[i] ) continue;
               grp.clear( );
               grp.push_back( all[i] );
               locgrps[id].push_back(grp);    }    }    }

void HyperKmerPath::PileReadsOnHyperPath( vec<HyperSloppyReadPlacement>& locs, 
     const ReadsOnPathPiler& piler, const vec<read_pairing>& pairs,
     const vec<int>& pairs_index, const vec<int>& trims,
     const vec<CompletedInsert>& inserts )
{    locs.clear( );

     // Phase 1: pile reads on paths.

     for ( int v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    const KmerPath& p = EdgeObjectByIndexFrom( v, j );
               vec<ReadOnPath> path_pile;
               list<KmerPath> subpath_storage;

               // Deal with cases where gaps are at the beginning or end of the
               // path, then call the piler.

               Bool gap_first = p.FirstSegment( ).isGap( );
               Bool gap_last = p.LastSegment( ).isGap( );
               if ( gap_first && p.NSegments( ) == 1 ) continue;
               static KmerPath p1;
               if ( !gap_first && !gap_last ) 
                    piler.PileReadsOnPath( p, path_pile, subpath_storage );
               else
               {    p1.Clear( );
                    int start = ( gap_first ? 1 : 0 );
                    int stop = ( gap_last ? p.NSegments( ) - 1 : p.NSegments( ) );
                    for ( int i = start; i < stop; i++ )
                         p1.AddSegment( p.Segment(i) );
                    piler.PileReadsOnPath( p1, path_pile, subpath_storage );    }

               // Process the pile.

               for ( int i = 0; i < path_pile.isize( ); i++ )
               {    const ReadOnPath& r = path_pile[i];
                    const PathEmbedding& into_read = r.intoRead;
                    const PathEmbedding& into_path = r.intoPath;
                    KmerPathLocAlt left_end_to, right_end_to;
                    HyperKmerPathLoc left_end_toh, right_end_toh;
                    Bool left_end_mapped = into_read.SubStartLoc( ).atBegin( );
                    Bool right_end_mapped = into_read.SubStopLoc( ).atEnd( );
                    if (left_end_mapped)
                    {    left_end_to = KmerPathLocAlt( into_path.SubStartLoc( ) );
                         if (gap_first)
                         {    left_end_to = KmerPathLocAlt( p,
                                   left_end_to.Interval( ) + 1,
                                   left_end_to.PosOnInterval( ) );    }
                         left_end_toh = HyperKmerPathLoc( v, j, left_end_to );    }
                    if (right_end_mapped)
                    {    right_end_to = KmerPathLocAlt( into_path.SubStopLoc( ) );
                         if (gap_first)
                         {    right_end_to = KmerPathLocAlt( p,
                                   right_end_to.Interval( ) + 1,
                                   right_end_to.PosOnInterval( ) );    }
                         right_end_toh = HyperKmerPathLoc( v, j, right_end_to );    }
                    HyperSloppyPathEmbedding e( left_end_mapped, right_end_mapped,
                         left_end_toh, right_end_toh );
                    int read_id;
                    if ( r.readId >= 0 ) read_id = r.readId;
                    else read_id = -r.readId - 1;
                    HyperSloppyReadPlacement s( v, j, read_id, pairs_index[read_id], 
                         r.readId < 0, e );
                    locs.push_back(s);    }    }    }
     Sort(locs);    

     // Phase 2: in a higgledly-piggledy way, use inserts to impose conditions on
     // valid read placements.  This should be redone rigorously, by actually
     // mapping the insert paths back to the hyperpath.
     //
     // If there are any fake inserts, we don't perform this test at present.

     Bool fake = False;
     for ( int i = 0; i < inserts.isize( ); i++ )
     {    const CompletedInsert& I = inserts[i];
          int id1 = I.Id1( ), id2 = I.Id2( );
          if ( id1 < 0 || id2 < 0 ) fake = True;    }
     int nreads = pairs_index.size( );
     vec< vec<int> > locs_index(nreads);
     for ( int i = 0; i < locs.isize( ); i++ )
          locs_index[ locs[i].ReadId( ) ].push_back(i);
     vec<Bool> invalid;
     if ( !fake )
     {    invalid.resize_and_set( locs.size( ), True );
          for ( int i = 0; i < inserts.isize( ); i++ )
          {    const CompletedInsert& I = inserts[i];
               int id1 = I.Id1( ), id2 = I.Id2( );
               const vec<int> &v1 = locs_index[id1], &v2 = locs_index[id2];
               for ( int u1 = 0; u1 < v1.isize( ); u1++ )
               for ( int u2 = 0; u2 < v2.isize( ); u2++ )
               {    HyperSloppyReadPlacement h1 = locs[ v1[u1] ], 
                         h2 = locs[ v2[u2] ];
                    if ( h1.Rc( ) ) swap( h1, h2 );
                    if ( !( h1.Fw( ) && h2.Rc( ) ) ) continue;
                    vrtx_t v1 = h1.Vertex( );
		    int e1 = h1.EdgeFrom( );
                    vrtx_t v2 = h2.Vertex( );
		    int e2 = h2.EdgeFrom( );
     
                    // Now we have a placement of I's left read, fw on the edge
                    // defined by (v1,e1), and a placement of its right read, rc on
                    // an edge defined by (v2,e2).

                    vrtx_t w1 = From(v1)[e1], w2 = From(v2)[e2];

                    // We find all paths from w1 to v2.  Then we find all the edges
                    // which I might conceivably map to.

                    static vec< vec<vrtx_t> > paths;
                    AllPaths( w1, v2, paths );
                    static vec< pair<vrtx_t,int> > edgelocs;
                    edgelocs.clear( );
                    if ( paths.nonempty( ) || ( v1 == v2 && e1 == e2 ) )
                         edgelocs.push_back( make_pair(v1,e1), make_pair(v2,e2) );
                    for ( int j = 0; j < paths.isize( ); j++ )
                    {    const vec<vrtx_t>& p = paths[j];
                         for ( int k = 0; k < p.isize( ); k++ )
                         {    vrtx_t v = p[k];
                              for ( int r = 0; r < From(v).isize( ); r++ )
                              {    vrtx_t w = From(v)[r];
                                   Bool accept = False;
                                   if ( k < p.isize( ) - 1 && w == p[k+1] ) 
                                        accept = True;
                                   else
                                   {    
                                        // Check for possible cycles in the path.
     
                                        for ( int s = 0; s <= k; s++ )
                                        {    if ( w == p[s] )
                                             {    accept = True;
                                                  break;    }    }    }
                                   if ( !accept ) continue;
                                   edgelocs.push_back( 
                                        make_pair(v,r) );    }    }    }
                    UniqueSort(edgelocs);

                    // Now validate any placement which lies on one of these edges.

                    static vec<int> ids;
                    ids.clear( );
                    for ( int z = 0; z < I.NPaths( ); z++ )
                    {    for ( int m = 0; m < I.Locs(z).size( ); m++ )
                              ids.push_back( I.Locs(z)[m].ReadId( ) );    }
                    UniqueSort(ids);
                    for ( int z = 0; z < ids.isize( ); z++ )
                    {    int id = ids[z];
                         for ( int m = 0; m < locs_index[id].isize( ); m++ )
                         {    int g = locs_index[id][m];
                              pair<vrtx_t,int> 
                                   eloc( locs[g].Vertex( ), locs[g].EdgeFrom( ) );
                              if ( BinMember( edgelocs, eloc ) )
                              // {    cout << "validating placement of read " << id
                              //           << " after vertex " << locs[g].Vertex( )
                              //           << " using completed insert for "
                              //           << id1 << "/" << id2 << "\n";
                                   invalid[g] = False;    
                              //           }
                                   }    }    }    }
          EraseIf( locs, invalid );    }

     // Find HyperKmerPath components.

     equiv_rel e( N( ) );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < From(v).isize( ); i++ )
          {    int w = From(v)[i];
               e.Join( v, w );    }    }
     vec<int> comp( N( ) );
     int ncomp = 1;
     for ( int x = 0; x < N( ); x++ )
     {    if ( e.Representative(x) )
          {    vec<int> o;
               e.Orbit( x, o );
               for ( int i = 0; i < o.isize( ); i++ )
                    comp[ o[i] ] = ncomp;
               ++ncomp;    }    }
     --ncomp;

     // Index read placements.

     locs_index.clear( ), locs_index.resize(nreads);
     for ( int i = 0; i < locs.isize( ); i++ )
          locs_index[ locs[i].ReadId( ) ].push_back(i);

     // Phase 3: if a read pair is placed happily, but it's ends are placed
     // elsewhere as well, delete those extra placements.  Only partially
     // implemented.

     vec< vec< vec<int> > > locgrps;
     GroupReadPlacements( locs, nreads, locgrps );
     invalid.resize_and_set( locs.size( ), False );
     for ( int pi = 0; pi < pairs.isize( ); pi++ )
     {    int id1 = pairs[pi].id1, id2 = pairs[pi].id2;

          // First find the placements which are obviously good.

          static vec< pair<int,int> > goods;
          goods.clear( );
          int nplace1 = locgrps[id1].size( ), nplace2 = locgrps[id2].size( );
          for ( int i1 = 0; i1 < nplace1; i1++ )
          {    for ( int i2 = 0; i2 < nplace2; i2++ )
               {    int u1, u2;
                    if ( locs[ locgrps[id1][i1][0] ].Fw( ) ) 
                         u1 = locgrps[id1][i1].back( );
                    else u1 = locgrps[id1][i1].front( );
                    if ( locs[ locgrps[id2][i2][0] ].Fw( ) ) 
                         u2 = locgrps[id2][i2].back( );
                    else u2 = locgrps[id2][i2].front( );
                    if ( HappyPlacement( u1, u2, locs, pairs[pi], trims, comp ) )
                    {    goods.push_back( make_pair( i1, i2 ) );    }    }    }

          // Now exclude the others, in very limited circumstances.

          if ( goods.size( ) == 1 && ( nplace1 == 1 || nplace2 == 1 ) )
          {    for ( int i1 = 0; i1 < nplace1; i1++ )
               {    if ( i1 == goods[0].first ) continue;
                    for ( int u = 0; u < locgrps[id1][i1].isize( ); u++ )
                         invalid[ locgrps[id1][i1][u] ] = True;    }
               for ( int i2 = 0; i2 < nplace2; i2++ )
               {    if ( i2 == goods[0].second ) continue;
                    for ( int u = 0; u < locgrps[id2][i2].isize( ); u++ )
                         invalid[ locgrps[id2][i2][u] ] = True;    }    }    }
     EraseIf( locs, invalid );    }

Bool HyperKmerPath::PathLengths( vrtx_t x, vrtx_t y, vec< pair<int,int> >& lengths,
     int maxpaths, Bool include_gaps ) const
{    vec< vec<vrtx_t> > paths;
     if ( !AllPaths( x, y, paths, maxpaths ) ) return False;
     lengths.clear( );
     for ( int u = 0; u < paths.isize( ); u++ )
     {    vec< pair<int,int> > pathlengths;
          pathlengths.push_back( make_pair(0, 0) );
          for ( int v = 1; v < paths[u].isize( ); v++ )
          {    int s = paths[u][v-1], t = paths[u][v];
               vec< pair<int,int> > edgelengths;
               for ( int z = 0; z < From(s).isize( ); z++ )
               {    if ( From(s)[z] != t ) continue;
                    const KmerPath& p = EdgeObjectByIndexFrom( s, z );
                    if (include_gaps)
                    {    edgelengths.push_back( 
                              make_pair( p.MinLength( ), p.MaxLength( ) ) );    }
                    else 
                    {    edgelengths.push_back( 
                              make_pair( p.KmerCount( ), p.KmerCount( ) ) );   }   }
               UniqueSort(edgelengths);
               vec< pair<int,int> > newpathlengths;
               for ( int a = 0; a < pathlengths.isize( ); a++ )
               {    for ( int b = 0; b < edgelengths.isize( ); b++ )
                    {    int low = pathlengths[a].first + edgelengths[b].first;
                         int high = pathlengths[a].second + edgelengths[b].second;
                         newpathlengths.push_back( make_pair(low, high) );    }    }
               pathlengths = newpathlengths;    }
          lengths.append(pathlengths);    }
     UniqueSort(lengths);
     return True;    }

// A hyperlink represents a gap edge that might be added to a HyperKmerPath,
// from vertex v1 to vertex v2, in different components, possibly after reversing 
// one of the components.

class hyperlink {

     public:

     int c1, c2;
     int v1, v2;
     Bool fw1, fw2;
     int gap_min, gap_max;

     Bool rc1( ) const { return !fw1; }
     Bool rc2( ) const { return !fw2; }

     hyperlink( ) { }
     hyperlink( int c1, int c2, int v1, int v2, Bool fw1, Bool fw2, int gap_min, 
          int gap_max ) : c1(c1), c2(c2), v1(v1), v2(v2), fw1(fw1), fw2(fw2), 
          gap_min(gap_min), gap_max(gap_max) { }

     int Spread( ) const { return gap_max - gap_min; }

     friend Bool operator<( const hyperlink& l1, const hyperlink& l2 )
     {    if ( l1.c1 < l2.c1 ) return True;
          if ( l1.c1 > l2.c1 ) return False;
          if ( l1.c2 < l2.c2 ) return True;
          if ( l1.c2 > l2.c2 ) return False;
          if ( l1.fw1 < l2.fw1 ) return True;
          if ( l1.fw1 > l2.fw1 ) return False;
          if ( l1.fw2 < l2.fw2 ) return True;
          if ( l1.fw2 > l2.fw2 ) return False;
          if ( l1.v1 < l2.v1 ) return True;
          if ( l1.v1 > l2.v1 ) return False;
          if ( l1.v2 < l2.v2 ) return True;
          if ( l1.v2 > l2.v2 ) return False;
          if ( l1.Spread( ) < l2.Spread( ) ) return True;
          return False;    }

};

void HyperKmerPath::ShowLinks( const vec<HyperSloppyReadPlacement>& locs,
     const vec<read_pairing>& pairs, const vec<int>& trims, Bool silent,
     Bool allow_short_edges )
{    
     // Do sanity checks.

     for ( int i = 0; i < locs.isize( ); i++ )
     {    if ( locs[i].EndsMapped( ) )
          {    ForceAssertEq( locs[i].LeftLoc( ).Vertex( ),
                    locs[i].RightLoc( ).Vertex( ) );
               ForceAssertEq( locs[i].LeftLoc( ).EdgeFromVertex( ),
                    locs[i].RightLoc( ).EdgeFromVertex( ) );    }    }

     // Define connectedness equivalence relation on vertices.

     equiv_rel e( N( ) );
     for ( int v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < From(v).isize( ); i++ )
          {    int w = From(v)[i];
               e.Join( v, w );    }    }

     // Define component numbering.

     vec<int> comp( N( ) );
     int ncomp = 1;
     for ( int x = 0; x < N( ); x++ )
     {    if ( e.Representative(x) )
          {    vec<int> o;
               e.Orbit( x, o );
               for ( int i = 0; i < o.isize( ); i++ )
                    comp[ o[i] ] = ncomp;
               ++ncomp;    }    }
     --ncomp;

     // Go through the read locations.

     vec<hyperlink> links;
     for ( int i = 0; i < locs.isize( ); i++ )
     {    int pid = locs[i].PairId( ), j;
          for ( j = i + 1; j < locs.isize( ); j++ )
               if ( locs[j].PairId( ) != pid ) break;
          int id1 = locs[i].ReadId( ), id2 = locs[j-1].ReadId( );
          if ( pid >= 0 && id1 != id2 )
          {    
               // Now we know that both ends of a read pair have been placed.

               int sep = pairs[pid].sep, dev = pairs[pid].sd;

               // If the ends are uniquely and fully placed on the same component 
               // of the graph, find the lengths in kmers of the intervening paths.
               // If they are consistent with the pairing, print nothing.

               Bool ufp = False;
               ostrstream uout;
               if ( j - i == 2 && locs[i].EndsMapped( ) && locs[i+1].EndsMapped( ) )
               {    int v1 = locs[i].LeftLoc( ).Vertex( );
                    int v2 = locs[i+1].LeftLoc( ).Vertex( );
                    if ( comp[v1] == comp[v2] && locs[i].Rc( ) != locs[i+1].Rc( ) )
                    {    ufp = True;
                         HyperSloppyReadPlacement m1 = locs[i], m2 = locs[i+1];
                         if ( !locs[i].Fw( ) ) 
                         {    swap(m1, m2);
                              swap(v1, v2);    }
                         int fv1 = m1.LeftLoc( ).EdgeFromVertex( );
                         int fv2 = m2.LeftLoc( ).EdgeFromVertex( );
                         int w1 = From(v1)[fv1], w2 = From(v2)[fv2];
                         vec< pair<int,int> > allpathlengths;
                         if ( v1 == v2 && fv1 == fv2 )
                         {    const KmerPath& p = EdgeObjectByIndexFrom( v1, fv1 );
                              const KmerPathLocAlt& a = m1.RightLoc( ).LocOnEdge( );
                              KmerPathLoc left(p, a.Interval(), a.PosOnInterval());
                              const KmerPathLocAlt& b = m2.LeftLoc( ).LocOnEdge( );
                              KmerPathLoc right(p, b.Interval(), b.PosOnInterval());
                              int sep_low = DistMin( left, right );
                              int sep_high = DistMax( left, right );
                              allpathlengths.push_back( 
                                   make_pair( sep_low, sep_high ) );    }
                         else
                         {    vec< vec<vrtx_t> > paths;
                              vrtx_t x = w1, y = v2;
                              if ( !AllPaths( x, y, paths, 100 ) )
                              {    uout << "too many paths\n";
                                   continue;    }
                              if ( paths.empty( ) ) uout << "NO PATHS FOUND\n";
                              for ( int u = 0; u < paths.isize( ); u++ )
                              {    uout << "path " << u+1 << ": ";
                                   for ( int v = 0; v < paths[u].isize( ); v++ )
                                   {    if ( v > 0 ) uout << " --> ";
                                        uout << paths[u][v];    }
                                   uout << "\n";    }
                              if ( !PathLengths( x, y, allpathlengths, 100 ) )
                              {    uout << "too many paths\n";
                                   continue;    }
                              int addlow = 0, addhigh = 0;
                              const KmerPath& p = EdgeObjectByIndexFrom( v1, fv1 );
                              const KmerPathLocAlt& a = m1.RightLoc( ).LocOnEdge( );
                              KmerPathLoc left( p, a.Interval(), a.PosOnInterval() );
                              addlow += DistMin( left, p.End( ) ) + 1;
                              addhigh += DistMax( left, p.End( ) ) + 1;
                              const KmerPath& q = EdgeObjectByIndexFrom( v2, fv2 );
                              const KmerPathLocAlt& b = m2.LeftLoc( ).LocOnEdge( );
                              KmerPathLoc right(q, b.Interval(), b.PosOnInterval());
                              addlow += DistMin( q.Begin( ), right ) + 1;
                              addhigh += DistMax( q.Begin( ), right ) + 1;
                              for ( int u = 0; u < allpathlengths.isize( ); u++ )
                              {    allpathlengths[u].first += addlow;
                                   allpathlengths[u].second += addhigh;    }    }
                         int sep_exp = sep + K( ) - 1 + trims[id1] + trims[id2];
                         Float sep_dev_limit = Float(3.0);
                         Bool good = False;
                         if ( allpathlengths.nonempty( ) )
                         {    uout << "possible path lengths:\n";
                              for ( int u = 0; u < allpathlengths.isize( ); u++ )
                              {    int sep_low = allpathlengths[u].first;
                                   int sep_high = allpathlengths[u].second;
                                   uout << "[" << sep_low << "," << sep_high << "]";
                                   int diff;
                                   if ( sep_exp < sep_low ) diff = sep_low - sep_exp;
                                   else if ( sep_exp > sep_high ) 
                                   {    diff = sep_exp - sep_high;    }
                                   else diff = 0;
                                   Float sep_dev = Float(diff)/Float(dev);
                                   if ( sep_dev <= sep_dev_limit ) good = True;
                                   uout << ", off by " << sep_dev 
                                        << " deviations\n";    }    }
                         if (good) 
                         {    i = j - 1;
                              continue;    }
                         uout << ends;    }    }

               // Analyze the case where ends are uniquely and fully placed on 
               // different components of the graph.

               Bool diffcomp = False;
               ostrstream dout;
               if ( j - i == 2 && locs[i].EndsMapped( ) && locs[i+1].EndsMapped( ) )
               {    int v1 = locs[i].LeftLoc( ).Vertex( );
                    int v2 = locs[i+1].LeftLoc( ).Vertex( );
                    if ( comp[v1] != comp[v2] )
                    {    diffcomp = True;
                         vec< vec<vrtx_t> > sss(2);
                         vec<int> dlow(2), dhigh(2);
                         for ( int u = 0; u < 2; u++ )
                         {    HyperSloppyReadPlacement m = locs[i+u];
                              vrtx_t v = m.LeftLoc( ).Vertex( );
                              int fv = m.LeftLoc( ).EdgeFromVertex( );
                              vrtx_t w = From(v)[fv];
                              vec< vec<vrtx_t> > paths;
                              vec< pair<int,int> > allpathlengths;
                              Bool fw = m.Fw( );
                              vrtx_t x = ( fw ? w : vrtx_t(-1) ), y = ( fw ? vrtx_t(-1) : v );
                              if ( !AllPaths( x, y, paths, 100 ) )
                              {    dout << "too many paths\n";
                                   continue;    }
                              vec<vrtx_t> ss;
                              for ( int z = 0; z < paths.isize( ); z++ )
                                   ss.push_back( fw ? paths[z].back( ) 
                                        : paths[z].front( ) );
                              UniqueSort(ss);
                              sss[u] = ss;
                              if ( !PathLengths( x, y, allpathlengths, 100 ) )
                              {    dout << "too many paths\n";
                                   continue;    }
                              const KmerPath& p = EdgeObjectByIndexFrom(v, fv);
                              const KmerPathLocAlt& 
                                   a = ( fw ? m.RightLoc( ).LocOnEdge( )
                                        : m.LeftLoc( ).LocOnEdge( ) );
                              KmerPathLoc l( p, a.Interval(), a.PosOnInterval() );
                              int addlow, addhigh;
                              if (fw)
                              {    addlow = DistMin( l, p.End( ) ) + 1;
                                   addhigh = DistMax( l, p.End( ) ) + 1;    }
                              else
                              {    addlow = DistMin( p.Begin( ), l ) + 1;
                                   addhigh = DistMax( p.Begin( ), l ) + 1;    }
                              for (int v = 0; v < allpathlengths.isize( ); v++)
                              {    allpathlengths[v].first += addlow;
                                   allpathlengths[v].second += addhigh;    }
                              dout << ( fw ? "sinks" : "sources" )
                                   << " reachable from placement " << u+1 << ":";
                              for ( int z = 0; z < ss.isize( ); z++ )
                                   dout << " " << ss[z];
                              dout << "\ndistance to "
                                   << ( fw ? "sinks" : "sources" ) << ":";
                              if ( allpathlengths.nonempty( ) )
                              {    dlow[u] = allpathlengths[0].first;
                                   dhigh[u] = allpathlengths[0].second;
                                   for (int z = 1; z < allpathlengths.isize( ); z++)
                                   {    dlow[u] = Min( dlow[u],
                                             allpathlengths[z].first );
                                        dhigh[u] = Max( dhigh[u],
                                             allpathlengths[z].second );    }    }
                              for (int z = 0; z < allpathlengths.isize( ); z++)
                              {    int sep_low = allpathlengths[z].first;
                                   int sep_high = allpathlengths[z].second;
                                   dout << " [" << sep_low << "," << sep_high 
                                        << "]";    }
                              dout << "\n";    }
                         int sep_exp = sep + K( ) - 1 + trims[id1] + trims[id2];
                         if ( sss[0].size( ) == 1 && sss[1].size( ) == 1 )
                         {    int v1 = sss[0][0], v2 = sss[1][0];
                              Bool fw1 = locs[i].Fw( ), fw2 = locs[i+1].Fw( );
                              if (fw2)
                              {    swap( v1, v2 );
                                   swap( fw1, fw2 );    }
                              int gap_low = sep_exp - Sum(dlow) - 4 * dev;
                              int gap_high = sep_exp - Sum(dhigh) + 4 * dev;
                              dout << "could add gap from " << ( fw1 ? "fw" : "rc" )
                                   << " " << v1 << " to " << ( fw2 ? "rc" : "fw" ) 
                                   << " " << v2 << " of size ["
                                   << gap_low << "," << gap_high << "]\n";
                              if ( v1 <= v2 )
                              {    links.push_back( hyperlink( comp[v1], comp[v2], 
                                        v1, v2, fw1, fw2, gap_low, 
                                        gap_high ) );    }
                              else
                              {    links.push_back( hyperlink( comp[v2], comp[v1], 
                                        v2, v1, fw2, fw1, gap_low, 
                                        gap_high ) );    }    }
                         dout << ends;    }    }

               // Print info.

               if ( !silent )
               {    cout << "\npairing " << pid 
                         << ", sep = " << sep << " +/- " << dev << "\n";
                    PRINT2( id1, id2 );
                    for ( int k = i; k < j; k++ )
                    {    if ( locs[k].ReadId( ) == id1 ) cout << "from ";
                         else cout << "to ";
                         cout << ( locs[k].Rc( ) ? "rc " : "fw " );
                         if ( locs[k].LeftEndMapped( ) && locs[k].RightEndMapped( ) )
                         {    ForceAssertEq( comp[ locs[k].LeftLoc( ).Vertex( ) ],
                                   comp[ locs[k].RightLoc( ).Vertex( ) ] );    }
                         if ( locs[k].LeftEndMapped( ) || locs[k].RightEndMapped( ) )
                         {    cout << "<component ";
                              if ( locs[k].LeftEndMapped( ) )
                                   cout << comp[ locs[k].LeftLoc( ).Vertex( ) ];
                              else cout << comp[ locs[k].RightLoc( ).Vertex( ) ];
                              cout << "> ";    }
                         if ( !locs[k].LeftEndMapped( ) ) cout << "[unmapped]";
                         else
                         {    HyperKmerPathLoc l = locs[k].LeftLoc( );
                              cout << "[vert " << l.Vertex( )
                                   << ", edge " << l.EdgeFromVertex( )
                                   << ", pos " << l.LocOnEdge( ).Interval( ) << "."
                                   << l.LocOnEdge( ).PosOnInterval( ) << "]";    }
                         cout << " .. ";
                         if ( !locs[k].RightEndMapped( ) ) cout << "[unmapped]";
                         else
                         {    HyperKmerPathLoc l = locs[k].RightLoc( );
                              cout << "[vert " << l.Vertex( )
                                   << ", edge " << l.EdgeFromVertex( )
                                   << ", pos " << l.LocOnEdge( ).Interval( ) << "."
                                   << l.LocOnEdge( ).PosOnInterval( ) << "]";    }
                         cout << "\n";    }    
                    if (ufp) cout << uout.str( );
                    if (diffcomp) cout << dout.str( );    }    }

          i = j - 1;    }

     // Summarize hyperlinks.

     Sort(links);
     if ( !silent )
     {    cout << "\nLINKS:\n";
          for ( int i = 0; i < links.isize( ); i++ )
          {    int j;
               for ( j = i + 1; j < links.isize( ); j++ )
               {    if ( links[j].c1 != links[i].c1 ) break;
                    if ( links[j].c2 != links[i].c2 ) break;
                    if ( links[j].fw1 != links[i].fw1 ) break;
                    if ( links[j].fw2 != links[i].fw2 ) break;    }
               if ( j - i >= 2 )
               {    const hyperlink &l1 = links[i], &l2 = links[i+1];
                    cout << "\n" << l1.c1 << "." << l1.v1 << ( !l1.fw1 ? "rc" : "" ) 
                         << " --[" << Min( l1.gap_min, l2.gap_min ) << "," 
                         << Max( l1.gap_max, l2.gap_max ) << "]--> "
                         << l1.c2 << "." << l1.v2 << ( !l1.fw2 ? "" : "rc" )  
                         << "\n";    }    
               i = j - 1;    }    }

     // Catalog linking.

     vec< set< pair<int,Bool> > > fw_links(ncomp+1), rc_links(ncomp+1);
     for ( int i = 0; i < links.isize( ); i++ )
     {    int j;
          for ( j = i + 1; j < links.isize( ); j++ )
          {    if ( links[j].c1 != links[i].c1 ) break;
               if ( links[j].c2 != links[i].c2 ) break;
               if ( links[j].fw1 != links[i].fw1 ) break;
               if ( links[j].fw2 != links[i].fw2 ) break;     
               if ( links[j].v1 != links[i].v1 ) break;
               if ( links[j].v2 != links[i].v2 ) break;    }
          if ( j - i >= 2 && links[i].gap_max >= 0 )
          {    const hyperlink& l = links[i];
               if (l.fw1) fw_links[l.c1].insert( make_pair( l.c2, l.fw2 ) );
               else rc_links[l.c1].insert( make_pair( l.c2, l.fw2 ) );
               if (l.fw2) fw_links[l.c2].insert( make_pair( l.c1, l.fw1 ) );
               else rc_links[l.c2].insert( make_pair( l.c1, l.fw1 ) );    }
          i = j - 1;    }
     
     // Propose links.

     vec<Bool> rc( ncomp + 1, False );
     equiv_rel ec( ncomp + 1 );
     vec<hyperlink> gap_edges;
     for ( int i = 0; i < links.isize( ); i++ )
     {    int j;
          for ( j = i + 1; j < links.isize( ); j++ )
          {    if ( links[j].c1 != links[i].c1 ) break;
               if ( links[j].c2 != links[i].c2 ) break;
               if ( links[j].fw1 != links[i].fw1 ) break;
               if ( links[j].fw2 != links[i].fw2 ) break;     
               if ( links[j].v1 != links[i].v1 ) break;
               if ( links[j].v2 != links[i].v2 ) break;    }
          if ( j - i >= 2 )
          {    const hyperlink &l1 = links[i], &l2 = links[i+1];
               int min_gap = Min( l1.gap_min, l2.gap_min );
               int max_gap = Max( l1.gap_max, l2.gap_max );
               Bool unique = True;
               if ( l1.fw1 && fw_links[l1.c1].size( ) > 1 ) unique = False;
               if ( l1.rc1( ) && rc_links[l1.c1].size( ) > 1 ) unique = False;
               if ( l1.fw2 && fw_links[l1.c2].size( ) > 1 ) unique = False;
               if ( l1.rc2( ) && rc_links[l1.c2].size( ) > 1 ) unique = False;
               if ( ( min_gap >= -2000 && max_gap <= 4000 )
                    || ( unique && max_gap >= 0 ) )
               {    if ( ec.ClassId(l1.c1) != ec.ClassId(l1.c2) || 
                         ( rc[l1.c1] ^ l1.rc1( ) ) != ( rc[l1.c2] ^ l1.rc2( ) ) )
                    {    if ( !silent )
                         {    cout << "\njoining: " << l1.c1 << "." << l1.v1 
                                   << ( !l1.fw1 ? "rc" : "" ) << " --[" 
                                   << min_gap << "," << max_gap << "]--> " << l1.c2 
                                   << "." << l1.v2 << ( !l1.fw2 ? "" : "rc" )  
                                   << "\n";    }
                         gap_edges.push_back( hyperlink( l1.c1, l1.c2, l1.v1, 
                              l1.v2, l1.fw1, l1.fw2, min_gap, max_gap ) );
                         if ( ec.ClassId(l1.c1) != ec.ClassId(l1.c2) )
                         {    vec<int> o1, o2;
                              ec.Orbit( l1.c1, o1 );
                              ec.Orbit( l1.c2, o2 );
                              if ( l1.rc1( ) != rc[l1.c1] )
                              {    for ( int u = 0; u < o1.isize( ); u++ )
                                        rc[ o1[u] ] = !rc[ o1[u] ];    }
                              if ( l1.rc2( ) == rc[l1.c2] )
                              {    for ( int u = 0; u < o2.isize( ); u++ )
                                        rc[ o2[u] ] = !rc[ o2[u] ];    }
                              ec.Join( l1.c1, l1.c2 );    }    }    }    }
          i = j - 1;    }
     vec<int> rep( ncomp + 1 );
     for ( int v = 0; v < N( ); v++ )
          rep[ comp[v] ] = v;
     for ( int i = 1; i < rc.isize( ); i++ )
          if ( rc[i] ) ReverseComponent( rep[i] );
     for ( int i = 0; i < gap_edges.isize( ); i++ )
     {    const hyperlink& e = gap_edges[i];
          Bool join12 = ( e.fw1 == !rc[e.c1] && e.fw2 != !rc[e.c2] );
          Bool join21 = ( e.fw1 != !rc[e.c1] && e.fw2 == !rc[e.c2] );
          if ( !join12 && !join21 )
          {    PRINT2( e.c1, e.c2 );
               PRINT2( int(rc[e.c1]), int(rc[e.c2]) );
               FatalErr( "Orientation inconsistency." );    }

          // If there is a possible negative gap, we see if it could be real.
          // For now we reject many valid joins.  We do not check for interweaving
          // without shared kmers, which could cause errors in rare cases.

          if ( e.gap_min < 0 )
          {    int v = ( join12 ? e.v1 : e.v2 );
               int w = ( join12 ? e.v2 : e.v1 );

               // We require that there's a unique edge entering v, a unique
               // edge exiting w, that the're both long, and share no kmers.

               if ( To(v).size( ) != 1 || From(w).size( ) != 1 )
               {    if ( !silent )
                    {    cout << "Refusing to join " << v << " to " << w 
                              << " because of nonuniqueness of nearby edges.\n";    }
                    continue;    }
               const KmerPath& p = EdgeObjectByIndexTo( v, 0 );
               const KmerPath& q = EdgeObjectByIndexFrom( w, 0 );
               if ( !allow_short_edges )
               {    if ( p.MinLength( ) < -e.gap_min || q.MinLength( ) < -e.gap_min )
                    {    if ( !silent )
                         {    cout << "Refusing to join " << v << " to " << w 
                                   << " because nearby edges "
                                   << "are too short.\n";    }    }
                    continue;    }
               Bool have_overlap = False;
               for ( int i = 0; i < p.NSegments( ); i++ )
               {    const KmerPathInterval& I = p.Segment(i);
                    for ( int j = 0; j < q.NSegments( ); j++ )
                    {    const KmerPathInterval& J = q.Segment(j);
                         if ( I.Overlaps(J) )
                         {    vec<MergedKmerPath> mergers;
                              MergePaths( p, q, i, j, mergers );
                              if ( mergers.nonempty( ) )
                              {    have_overlap = True;
                                   break;    }    }    }
                    if (have_overlap) break;    }
               if (have_overlap)
               {    if ( !silent )
                    {    cout << "Refusing to join " << v << " to " << w 
                              << " because of possible overlap.\n";    }
                    continue;    }    }

          // Do the join.

          KmerPath p;
          p.AddGap( Max( 0, e.gap_min ), e.gap_max );
          if (join12) AddEdge( e.v1, e.v2, p );
          else AddEdge( e.v2, e.v1, p );    }
     RemoveUnneededVertices( );    }

HyperKmerPath::HyperKmerPath( int K, const vec<CompletedInsert>& inserts )
{    K_ = K;
     vecKmerPath p;
     int npaths = 0, nsegments = 0;
     for ( int i = 0; i < inserts.isize( ); i++ )
     {    npaths += inserts[i].NPaths( );
          for ( int j = 0; j < inserts[i].NPaths( ); j++ )
               nsegments += inserts[i].Path(j).NSegments( );    }
     p.Reserve( nsegments, npaths );
     equiv_rel e(npaths);
     npaths = 0;
     for ( int i = 0; i < inserts.isize( ); i++ )
     {    for ( int j = 0; j < inserts[i].NPaths( ); j++ )
          {    p.push_back( inserts[i].Path(j) );
               if ( j > 0 ) e.Join( npaths + j - 1, npaths + j );    }
          npaths += inserts[i].NPaths( );    }
     EdgeEquivConstructor( VecOfKmerPath( p ), e );    }

HyperKmerPath::HyperKmerPath( const String& filename )
{    int fd = OpenForRead(filename);
     BinaryRead( fd, *this );
     close(fd);    }

vec<basevector> Convert( const vec<KmerPath>& in, const KmerBaseBroker& kbb )
{    vec<basevector> out;
     for ( int j = 0; j < in.isize( ); j++ )
     {    SuperBaseVector s = kbb.ToSequence( in[j] );
          ForceAssertEq( s.size( ), 1 );
          out.push_back( s.Seq(0) );    }
     return out;    }

HyperBasevector::HyperBasevector( const HyperKmerPath& h, const KmerBaseBroker& kbb )
     : digraphE<basevector>( h.From( ), h.To( ), Convert( h.Edges( ), kbb ), 
          h.ToEdgeObj( ), h.FromEdgeObj( ) )
{    K_ = h.K( );    }

void BinaryWrite( int fd, const HyperBasevector& h )
{    WriteBytes( fd, &h.K_, sizeof(int) );
     BinaryWrite( fd, (const digraphE<basevector>&) h );    }

void BinaryRead( int fd, HyperBasevector& h )
{    ReadBytes( fd, &h.K_, sizeof(int) );
     BinaryRead( fd, (digraphE<basevector>&) h );    }

void HyperKmerPath::RemoveSmallComponents( int min_kmers )
{    equiv_rel e;
     ComponentRelation(e);
     vec<vrtx_t> reps, o, keep;
     e.OrbitRepsAlt(reps);
     for ( int i = 0; i < reps.isize( ); i++ )
     {    e.Orbit( reps[i], o );
          int nkmers = 0;
          for ( int j = 0; j < o.isize( ); j++ )
          {    vrtx_t v = o[j];
               for ( int t = 0; t < From(v).isize( ); t++ )
                    nkmers += EdgeObjectByIndexFrom( v, t ).KmerCount( );    }
          if ( nkmers < min_kmers ) continue;
          keep.append(o);    }
     HyperKmerPath h( *this, keep );
     *this = h;    }

void HyperBasevector::LowerK( int newK )
{    ForceAssertLe( newK, K( ) );
     for ( int i = 0; i < EdgeObjectCount( ); i++ )
     {    ForceAssertGt( EdgeObject(i).size( ), 0 );
          EdgeObjectMutable(i).resize( EdgeObject(i).size( ) + newK - K( ) );    }
     K_ = newK;    }

void HyperKmerPath::MakeEdgeDatabase( vec<tagged_rpint>& edgedb ) const
{    edgedb.clear( );
     edgedb.reserve( EdgeObjectCount( ) );
     for ( int i = 0; i < EdgeObjectCount( ); i++ )
          EdgeObject(i).AppendToDatabase( edgedb, i );
     Prepare(edgedb);    }



bool ComponentsAreIsomorphic( const HyperKmerPath& hkp1, vrtx_t seed_vx_1,
			      const HyperKmerPath& hkp2, vrtx_t seed_vx_2,
			      vec<vrtx_t>* p_iso ) {
  vec<vrtx_t> local_iso;
  if( p_iso == NULL ) p_iso = &local_iso;

  p_iso->resize( hkp1.N(), -1 );
  // (*p_iso)[i]=-1 means hkp1 vx i unused; j>=0 means matches hkp2 vx j.
  vec<Bool> used2( hkp2.N(), false );
  
  // The hypothesis <i,j> means vertices i and j ought to match
  vec< pair<vrtx_t,vrtx_t> > hypotheses;
  // The total number of hypotheses will be two per edge in the
  // connected component (but generally not all on the stack at once).
  hypotheses.reserve( 2 * hkp1.EdgeObjectCount() );
  // Initial hypothesis: the given seed vertices match.
  hypotheses.push_back( make_pair(seed_vx_1,seed_vx_2) );

  while( ! hypotheses.empty() ) {

    vrtx_t v1 = hypotheses.back().first, v2=hypotheses.back().second;
    hypotheses.pop_back();

    if( (*p_iso)[v1]==v2 )                     // already knew this hypothesis
      continue;
    else if( used2[v2] || (*p_iso)[v1]!=-1 )   // knew a contradictory hypothesis
      return false;

    // Otherwise:
    // (1) Assume this hypothesis is true
    (*p_iso)[v1] = v2;
    used2[v2] = true;

    // (2) check that the edges in/out of these vertices match up
    vec<vrtx_t> from_v1 = hkp1.From(v1), to_v1 = hkp1.To(v1);
    vec<vrtx_t> from_v2 = hkp2.From(v2), to_v2 = hkp2.To(v2);

    if( from_v1.size() != from_v2.size() || to_v1.size() != to_v2.size() )
      return false;

    vec<KmerPath> from_v1_edges( from_v1.size() );
    vec<KmerPath> from_v2_edges( from_v2.size() );
    vec<KmerPath> to_v1_edges( to_v1.size() );
    vec<KmerPath> to_v2_edges( to_v2.size() );

    for(int i=0; i<from_v1.isize(); i++)
      from_v1_edges[i] = hkp1.EdgeObjectByIndexFrom(v1,i);
    for(int i=0; i<from_v2.isize(); i++)
      from_v2_edges[i] = hkp2.EdgeObjectByIndexFrom(v2,i);

    for(int j=0; j<to_v1.isize(); j++)
      to_v1_edges[j] = hkp1.EdgeObjectByIndexTo(v1,j);
    for(int j=0; j<to_v2.isize(); j++)
      to_v2_edges[j] = hkp2.EdgeObjectByIndexTo(v2,j);

    SortSync( from_v1_edges, from_v1 );
    SortSync( from_v2_edges, from_v2 );
    SortSync( to_v1_edges, to_v1 );
    SortSync( to_v2_edges, to_v2 );

    if( from_v1_edges != from_v2_edges || to_v2_edges != to_v2_edges )
      return false;

    ForceAssert( from_v1_edges.UniqueOrdered() );
    ForceAssert( from_v2_edges.UniqueOrdered() );
    ForceAssert( to_v1_edges.UniqueOrdered() );
    ForceAssert( to_v2_edges.UniqueOrdered() );

    for( int i=0; i < from_v1.isize(); i++ )
      hypotheses.push_back( make_pair(from_v1[i],from_v2[i]) );
    for( int j=0; j < to_v1.isize(); j++ )
      hypotheses.push_back( make_pair(to_v1[j],to_v2[j]) );


    // (3) formulate all the corollary hypotheses imposed by edge matching
    for( int i=0; i < from_v1.isize(); i++ )
      hypotheses.push_back( make_pair(from_v1[i],from_v2[i]) );
    for( int j=0; j < to_v1.isize(); j++ )
      hypotheses.push_back( make_pair(to_v1[j],to_v2[j]) );

  } // end of processing of hypotheses.back()

  // If no hypotheses gave rise to a contradiction,
  // then we have an isomorphism.  Hooray!
  return true;
}

Bool operator==( const HyperBasevector& h1, const HyperBasevector& h2 )
{    if ( h1.K( ) != h2.K( ) ) return False;
     return (const digraphE<basevector>&) h1 
          == (const digraphE<basevector>&) h2;    }

void HyperKmerPath::Zipper( )
{    for ( int pass = 1; pass <= 2; pass++ )
     {    for ( int v = 0; v < N( ); v++ )
          {    restart:
               for ( int j1 = 0; j1 < From(v).isize( ); j1++ )
               {    int w1 = From(v)[j1];
                    if ( w1 == v ) continue;
                    const KmerPath& e1 = EdgeObjectByIndexFrom( v, j1 );
                    for ( int j2 = j1+1; j2 < From(v).isize( ); j2++ )
                    {    int w2 = From(v)[j2];
                         if ( w2 == v ) continue;
                         const KmerPath& e2 = EdgeObjectByIndexFrom( v, j2 );
                         if ( e1.Start(0) != e2.Start(0) ) continue;
                         KmerPathLoc loc1(e1, 0), loc2(e2, 0);
                         ScanRightPerfectMatch( loc1, loc2 );
                         if ( loc1.atEnd( ) || loc2.atEnd( ) ) 
                              continue; // SUCH CASES NOT YET IMPLEMENTED!
                         KmerPath before, after1, after2;
                         e1.CopySubpath( e1.Begin( ), loc1, before );
                         e1.CopySubpathNoFirstKmer( loc1, e1.End( ), after1 );
                         e2.CopySubpathNoFirstKmer( loc2, e2.End( ), after2 );
                         DeleteEdgeFrom( v, j2 );
                         DeleteEdgeFrom( v, j1 );
                         int x = N( );
                         AddVertices(1);
                         AddEdge( v, x, before );
                         AddEdge( x, w1, after1 );
                         AddEdge( x, w2, after2 );
                         goto restart;    }    }    }
          Reverse( );    }    }


// MethodDecl: CanonicalizeEdges
// Canonicalize the KmerPaths on all the edges.
void HyperKmerPath::CanonicalizeEdges() {
  vec<KmerPath>& edges = EdgesMutable();
  for ( int i = 0; i < edges.isize(); i++ )
    edges[i].Canonicalize();
}


/**
   Method: FindIsomorphicComponents

   Test whether the HyperKmerPath has two isomorphic connected components.

   Return the equivalence relation of the components, and the list of ids of those components
   that have an isomorphic partner in the graph.
   
*/
Bool HyperKmerPath::FindIsomorphicComponents( equiv_rel& componentRelation,
					      vec<vrtx_t>& isomorphicComponentReps,
					      Bool stopIfFoundOne ) const {
  // Determine the connected components
  componentRelation.Initialize( N() );
  ComponentRelation( componentRelation );

  // For each connected component compute a "signature" -- a vector of
  // integers such that isomorphic components will have equal signatures.
  // We'll then test components with equal signatures for isomorphism.

  // A component's signature is a vector of integers that looks like this:
  // #vertices #edges sorted-list-of-vertex-out-degrees sorted-list-of-vertex-in-degrees sorted-list-of-edge-lengths 

  typedef vec<int> component_signature_t;

  vec< component_signature_t > componentSignatures;

  vec<vrtx_t> componentReps;
  componentRelation.OrbitRepsAlt( componentReps );
  for ( int i = 0 ; i < componentReps.isize(); i++ ) {
    vrtx_t thisComponentRep = componentReps[i];
    vec<vrtx_t> thisComponentVerts;
    componentRelation.Orbit( thisComponentRep, thisComponentVerts );
    component_signature_t thisComponentSig;
    thisComponentSig.push_back( thisComponentVerts.isize() );

    int thisComponentNumEdges = 0;
    for ( int j = 0; j < thisComponentVerts.isize(); j++ )
      thisComponentNumEdges += From(thisComponentVerts[j]).isize();

    thisComponentSig.push_back( thisComponentNumEdges );

    vec<int> thisComponentOutDegrees;
    for ( int j = 0; j < thisComponentVerts.isize(); j++ )
      thisComponentOutDegrees.push_back( From(thisComponentVerts[j]).isize() );
    Sort( thisComponentOutDegrees );
    thisComponentSig.append( thisComponentOutDegrees );
    
    vec<int> thisComponentInDegrees;
    for ( int j = 0; j < thisComponentVerts.isize(); j++ )
      thisComponentInDegrees.push_back( To(thisComponentVerts[j]).isize() );
    Sort( thisComponentInDegrees );
    thisComponentSig.append( thisComponentInDegrees );

    vec<int> thisComponentEdgeLens;
    for ( int j = 0; j < thisComponentVerts.isize(); j++ ) {
      const vec<int>& fromThisVrtx = FromEdgeObj(thisComponentVerts[j]);
      for ( int k = 0; k < fromThisVrtx.isize(); k++ )
	thisComponentEdgeLens.push_back( EdgeObject( fromThisVrtx[k] ).KmerCount() );
    }
      
    Sort( thisComponentEdgeLens );
    thisComponentSig.append( thisComponentEdgeLens );

    componentSignatures.push_back( thisComponentSig );
  }

  SortSync( componentSignatures, componentReps );

  Bool foundIsomorphicComponents = False;
  isomorphicComponentReps.clear();
  for ( int i = 0; i < componentSignatures.isize()-1; i++ ) {
    if ( componentSignatures[i] == componentSignatures[i+1] &&
	 ComponentsAreIsomorphic( *this, componentReps[i], *this, componentReps[i+1] ) ) {
      foundIsomorphicComponents = True;
      isomorphicComponentReps.push_back( componentReps[i] );
      isomorphicComponentReps.push_back( componentReps[i+1] );
      if ( stopIfFoundOne )
	return True;
    }
  }
  UniqueSort( isomorphicComponentReps );

  return foundIsomorphicComponents;
}

// Method: DumpGraphML
// Output the structure HyperKmerPath in a textual format that can be easily read without
// reference to our code base.
void HyperKmerPath::DumpGraphML( const String& graphMLFileName ) const {

  vec< vec< String > > edgeLabels( N() );
  for ( int v = 0; v < N( ); v++ )
    {
      for ( int j = 0; j < From(v).isize( ); j++ )
	{
	  int w = From(v)[j];
	  edgeLabels[ v ].push_back( BaseAlpha( EdgeObjectIndexByIndexFrom( v, j ) ) );
	}
    }
  Ofstream( grml, graphMLFileName );
  WriteGraphML( grml, edgeLabels );
}
