/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Alignment.h"
#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "math/Functions.h"
#include "math/HoInterval.h"
#include "pairwise_aligners/PerfectAligner.h"
#include "paths/Hospital.h"
#include "paths/HyperKmerPath.h"
#include "paths/KmerBaseBroker.h"
#include "paths/KmerPath.h"

// RecomputeAligns: given two HyperKmerPaths h1 and h2, such that every edge object
// in h2 is equal to an edge object in h1, and given all perfect alignments of
// sequences S to the edge sequences of h1, find all perfect alignments of S to the
// edge sequences of h2.

void RecomputeAligns( const HyperKmerPath& h1, const HyperKmerPath& h2,
     const vecbasevector& S, const vec<alignment_plus>& aligns1,
     vec<alignment_plus>& aligns2 )
{    vec<KmerPath> edges1;
     vec<int> edge_id1;
     for ( int e = 0; e < h1.EdgeObjectCount( ); e++ )
     {    edges1.push_back( h1.EdgeObject(e) );
          edge_id1.push_back(e);    }
     SortSync( edges1, edge_id1 );
     vec< vec<int> > ind1( h1.EdgeObjectCount( ) );
     for ( int i = 0; i < aligns1.isize( ); i++ )
          ind1[ aligns1[i].Id2( ) ].push_back(i);
     aligns2.clear( );
     for ( int e2 = 0; e2 < h2.EdgeObjectCount( ); e2++ )
     {    int m = BinPosition( edges1, h2.EdgeObject(e2) );
          ForceAssertGe( m, 0 );
          int e1 = edge_id1[m];
          for ( int j = 0; j < ind1[e1].isize( ); j++ )
          {    alignment_plus ap = aligns1[ ind1[e1][j] ];
               ap.SafeSetId2(e2);
               aligns2.push_back(ap);    }    }    }

// Given a directed graph G, a "handled subgraph" S of G is a complete subgraph of
// G, such that 
// - if v is a vertex of G - S, w is a vertex of S, and there is an edge between
//   v and w, then w is a source or a sink of S;
// - if v is a source of S that is not a source of G, then there is a unique edge 
//   in S exiting v;
// - if v is a sink of S that is not a sink of G, then there is a unique edge 
//   in S entering v.
// The edges exiting sources and entering sinks (of S, not G) are called the 
// "handles" of S.

// Handles.  Return the handles of a handled subgraph S of a HyperKmerPath h.

void Handles( const HyperKmerPath& h, const vec<vrtx_t>& S, vec<int>& handles )
{    vec<vrtx_t> sources, sinks;
     h.SubgraphSources( S, sources ), h.SubgraphSinks( S, sinks );
     handles.clear( );
     for ( int i = 0; i < sources.isize( ); i++ )
     {    vrtx_t v = sources[i];
          if ( h.To(v).empty( ) ) continue;
          vec<int> S_From_v;
          for ( int j = 0; j < h.From(v).isize( ); j++ )
               if ( BinMember( S, h.From(v)[j] ) ) S_From_v.push_back(j);
          ForceAssertEq( S_From_v.size( ), 1 );
          handles.push_back( h.EdgeObjectIndexByIndexFrom( v, S_From_v[0] ) );    }
     for ( int i = 0; i < sinks.isize( ); i++ )
     {    vrtx_t v = sinks[i];
          if ( h.From(v).empty( ) ) continue;
          vec<int> S_To_v;
          for ( int j = 0; j < h.To(v).isize( ); j++ )
               if ( BinMember( S, h.To(v)[j] ) ) S_To_v.push_back(j);
          ForceAssertEq( S_To_v.size( ), 1 );
          handles.push_back( h.EdgeObjectIndexByIndexTo( v, S_To_v[0] ) );    }
     UniqueSort(handles);    }

// GrowHandledNhood.  Given an edge in a HyperKmerPath, find a handled subgraph S
// containing it, with the property that its handles have length at least min_length
// and coverage between min_cov and max_cov (these are called qualified handles), 
// and which is relatively small, by gradually growing outward from the given edge, 
// iteratively generating S.  This is done by repeating the following steps:
// - If S is a handled subgraph with qualified handles, stop.
// - If S has an unqualified handle that is a source, pick one, and add all the 
//   vertices that enter it.
// - If S has an unqualified handle that is a sink, pick one, and add all the 
//   vertices that exit it.
// - If S has a source x that is not a source of G and that has more than one edge 
//   in S exiting it, pick such an x, and add all the vertices that enter it.
// - If S has a sink x that is not a sink of G and that has more than one edge 
//   in S entering it, pick such an x, and add all the vertices that exit it.
// - If v is a vertex of G - S, w is a vertex of S, and there is an edge between
//   v and w, and w is not a source or a sink of S, pick such a vertex v and add
//   it to S.
//
// Initially, S should be set to the vertices in the edge.

void GrowHandledNhood( vec<vrtx_t>& S, const HyperKmerPath& h, const vec<double>& cov,
     int min_length, double min_cov, double max_cov )
{    restart:
     UniqueSort(S);
     static vec<vrtx_t> sources, sinks;
     h.SubgraphSources( S, sources ), h.SubgraphSinks( S, sinks );
     for ( int i = 0; i < sources.isize( ); i++ )
     {    vrtx_t v = sources[i];
          vec<int> S_From_v;
          for ( int j = 0; j < h.From(v).isize( ); j++ )
               if ( BinMember( S, h.From(v)[j] ) ) S_From_v.push_back(j);
          if ( h.To(v).nonempty( ) && S_From_v.solo( ) )
          {    int e = h.EdgeObjectIndexByIndexFrom( v, S_From_v[0] );
               if ( h.EdgeLength(e) < min_length || cov[e] < min_cov 
                    || cov[e] > max_cov )
               {    S.append( h.To(v) );
                    goto restart;    }    }    }
     for ( int i = 0; i < sinks.isize( ); i++ )
     {    vrtx_t v = sinks[i];
          vec<int> S_To_v;
          for ( int j = 0; j < h.To(v).isize( ); j++ )
               if ( BinMember( S, h.To(v)[j] ) ) S_To_v.push_back(j);
          if ( h.From(v).nonempty( ) && S_To_v.solo( ) )
          {    int e = h.EdgeObjectIndexByIndexTo( v, S_To_v[0] );
               if ( h.EdgeLength(e) < min_length || cov[e] < min_cov 
                    || cov[e] > max_cov )
               {    S.append( h.From(v) );
                    goto restart;    }    }    }
     for ( int i = 0; i < sources.isize( ); i++ )
     {    vrtx_t v = sources[i];
          vec<int> S_From_v;
          for ( int j = 0; j < h.From(v).isize( ); j++ )
               if ( BinMember( S, h.From(v)[j] ) ) S_From_v.push_back(j);
          if ( h.To(v).nonempty( ) && !S_From_v.solo( ) )
          {    S.append( h.To(v) );
               goto restart;    }    }
     for ( int i = 0; i < sinks.isize( ); i++ )
     {    vrtx_t v = sinks[i];
          vec<int> S_To_v;
          for ( int j = 0; j < h.To(v).isize( ); j++ )
               if ( BinMember( S, h.To(v)[j] ) ) S_To_v.push_back(j);
          if ( h.From(v).nonempty( ) && !S_To_v.solo( ) )
          {    S.append( h.From(v) );
               goto restart;    }    }
     for ( int i = 0; i < S.isize( ); i++ )
     {    vrtx_t w = S[i];
          for ( int j = 0; j < h.To(w).isize( ); j++ )
          {    vrtx_t v = h.To(w)[j];
               if ( BinMember( S, v ) || BinMember( sources, w ) 
                    || BinMember( sinks, w ) ) 
               {    continue;    }
               S.push_back(v);
               goto restart;    }    }
     for ( int i = 0; i < S.isize( ); i++ )
     {    vrtx_t w = S[i];
          for ( int j = 0; j < h.From(w).isize( ); j++ )
          {    vrtx_t v = h.From(w)[j];
               if ( BinMember( S, v ) || BinMember( sources, w ) 
                    || BinMember( sinks, w ) ) 
               {    continue;    }
               S.push_back(v);
               goto restart;    }    }    }

void Hospital( HyperKmerPath& h, const vecbasevector& reads,
     vec<alignment_plus>& Aligns, const KmerBaseBroker* kbb, 
     const vec<int>& partner, Bool verbose, const vec<tagged_rpint>& uniqdb )
{
     int nreads = reads.size( ), K = h.K( );
     HyperBasevector hb( h, *kbb );

     if (verbose)
     {    cout << "\n------------------------------------------------------"
               << "--------------------------\n";
          cout << "\nEntering hospital.  Here is the HyperKmerPath:\n";
          h.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );    }

     beginning:

     // Generate indices to alignments.

     int naligns = Aligns.size( );
     vec< vec<int> > Aligns_index( reads.size( ) );
     for ( int i = 0; i < naligns; i++ )
          Aligns_index[ Aligns[i].Id1( ) ].push_back(i);
     vec< vec<int> > Aligns_index2( h.EdgeObjectCount( ) );
     for ( int i = 0; i < naligns; i++ )
          Aligns_index2[ Aligns[i].Id2( ) ].push_back(i);
     vec<int> pos1(naligns), Pos1(naligns);
     for ( int i = 0; i < naligns; i++ )
     {    pos1[i] = Aligns[i].a.pos1( );
          Pos1[i] = Aligns[i].a.Pos1( );    }

     // Determine which read alignments hit unique sequence.

     int nedges = h.EdgeObjectCount( );
     vec<int> L(nedges);
     for ( int i = 0; i < nedges; i++ )
          L[i] = h.EdgeLength(i);
     vec< vec<ho_interval> > huniq(nedges);
     for ( int i = 0; i < nedges; i++ )
     {    static vec<ho_interval> huniq0;
          huniq0.clear( );
          const KmerPath& e = h.EdgeObject(i);
          int pos = 0;
          for ( int j = 0; j < e.NSegments( ); j++ )
          {    static vec<longlong> places;
               const KmerPathInterval& J = e.Segment(j);
               Contains( uniqdb, J, places );
               for ( int u = 0; u < places.isize( ); u++ )
               {    const tagged_rpint& t = uniqdb[ places[u] ];
                    int start = Max( 0, int( t.Start( ) - J.Start( ) ) );
                    int stop = Min( J.Length( ), int( t.Stop( ) - J.Start( ) ) );
                    huniq0.push_back( ho_interval( pos+start, pos+stop ) );    }
               pos += J.Length( );    }
          ExtractGivenCoverage( L[i], 1, huniq0, huniq[i] );    }
     vec<Bool> hits_unique( Aligns.size( ), False );
     for ( int i = 0; i < Aligns.isize( ); i++ )
     {    const alignment_plus& ap = Aligns[i];
          int id1 = ap.Id1( ), id2 = ap.Id2( );
          int nkmers = reads[id1].isize( ) - K + 1;
          int pos2 = ( !ap.Rc2( ) ? ap.a.pos2( ) : L[id2] + K - 1 - ap.a.Pos2( ) );
          int Pos2 = Min( L[id2] - 1, pos2 + nkmers - 1 );
          pos2 = Max( 0, pos2 );
          ho_interval h( pos2, Pos2+1 );
          if ( Overlap( h, huniq[id2] ) > 0 ) hits_unique[i] = True;    }

     // Compute coverage of each edge.

     vec<double> cov( nedges, 0 );
     for ( int i = 0; i < reads.size( ); i++ )
     {    int np = Aligns_index[i].size( );
          for ( int j = 0; j < np; j++ )
          {    const alignment_plus& ap = Aligns[ Aligns_index[i][j] ];
               int id2 = ap.Id2( );
               cov[id2] += double( ap.a.extent1( ) ) 
                    / ( double(np) * double( hb.EdgeLength(id2) ) );    }    }
     vec< pair<int,double> > ecov;
     for ( int e = 0; e < nedges; e++ )
          ecov.push_back( make_pair( h.EdgeLength(e), cov[e] ) );
     ReverseSort(ecov);
     double avecov = 0.0;
     longlong totalEdgeLength = Sum(L);
     int numSamples = 0;
     int maxToSample = 10;
     longlong sumSampledLength = 0;
     for ( numSamples = 0; numSamples < maxToSample; numSamples++ ) {
          if ( sumSampledLength > totalEdgeLength/10 ) break;
          avecov += ecov[numSamples].second;
          sumSampledLength += ecov[numSamples].first;
     }
     avecov /= double(numSamples);
     for ( int e = 0; e < nedges; e++ )
          cov[e] /= avecov;

     // Compute coverage of each base in each edge.  For each read alignment,
     // compute the coverage by all reads of the edge bases in the alignment.
     // (NOTE: handling of rc case is wrong!)

     /*
     vec< vec<double> > bcov(nedges);
     for ( int e = 0; e < nedges; e++ )
          bcov[e].resize( hb.EdgeLength(e), 0 );
     for ( int i = 0; i < reads.size( ); i++ )
     {    int np = Aligns_index[i].size( );
          for ( int j = 0; j < np; j++ )
          {    const alignment_plus& ap = Aligns[ Aligns_index[i][j] ];
               int id2 = ap.Id2( );
               for ( int u = ap.a.pos2( ); u < ap.a.Pos2( ); u++ )
                    bcov[id2][u] += 1.0 / double(np);    }    }
     vec<double> aligns_cov( Aligns.size( ) );
     double aligns_cov_mean = 0.0;
     for ( int i = 0; i < Aligns.isize( ); i++ )
     {    const alignment_plus& ap = Aligns[i];
          double x = 0.0;
          for ( int u = ap.a.pos2( ); u < ap.a.Pos2( ); u++ )
               x += bcov[ ap.Id2( ) ][u];
          x /= double( ap.a.extent2( ) );
          aligns_cov_mean += x;
          aligns_cov[i] = x;    }
     aligns_cov_mean /= double( Aligns.size( ) );
     for ( int i = 0; i < Aligns.isize( ); i++ )
     {    aligns_cov[i] /= aligns_cov_mean;
          const alignment_plus& ap = Aligns[i];
          PRINT4( i, ap.Id1( ), ap.Id2( ), aligns_cov[i] );    }
     */

     // Set up edge indices.

     vec<vrtx_t> to_left_vertex(nedges), to_right_vertex(nedges);
     for ( vrtx_t w = 0; w < h.N( ); w++ )
     {    for ( int j = 0; j < h.To(w).isize( ); j++ )
          {    int e = h.EdgeObjectIndexByIndexTo( w, j );
               vrtx_t v = h.To(w)[j];
               to_left_vertex[e] = v, to_right_vertex[e] = w;    }    }

     // Find edges that are substantially duplicated.  We don't check edges of
     // length < 100.

     vec<Bool> duplicated( nedges, False );
     vecKmerPath edges;
     for ( int i = 0; i < nedges; i++ )
          edges.push_back_reserve( h.EdgeObject(i) );
     vecKmerPath edgesrc(edges);
     for ( int i = 0; i < nedges; i++ )
          edgesrc[i].Reverse( );
     vec<tagged_rpint> edgedb;
     CreateDatabase( edges, edgesrc, edgedb );
     for ( int id1 = 0; id1 < nedges; id1++ )
     {    static vec<longlong> places;
          longlong x = edges[id1].Start(0);
          Contains( edgedb, x, places );
          for ( int j = 0; j < places.isize( ); j++ )
          {    const tagged_rpint& t = edgedb[ places[j] ];
               int id2 = t.ReadId( );
               Bool rc = ( t.PathId( ) < 0 );
               if ( id1 == id2 ) continue;
               if ( duplicated[id1] && duplicated[id2] ) continue;
               if ( L[id1] < 100 && L[id2] < 100 ) continue;
               if ( L[id1] < 100 && L[id2] >= 200 ) continue;
               if ( L[id2] < 100 && L[id1] >= 200 ) continue;
               if ( duplicated[id1] && L[id2] > 2 * L[id1] ) continue;
               if ( duplicated[id2] && L[id1] > 2 * L[id2] ) continue;
               const KmerPath& e2 = ( !rc ? edges[id2] : edgesrc[id2] );
               if ( ProperOverlapExt( edges[id1], e2, 0, t.PathPos( ) ) )
               {    int ext = x - t.Start( );
                    for ( int j = 0; j < t.PathPos( ); j++ )
                         ext += e2.Length(j);
                    int overlap = IntervalOverlap( 0, L[id2], ext, ext + L[id1] );
                    if ( 2 * overlap >= L[id1] ) duplicated[id1] = True;
                    if ( 2 * overlap >= L[id2] ) 
                         duplicated[id2] = True;    }    }    }

     /*
     vecbasevector edges;
     for ( int i = 0; i < nedges; i++ )
          edges.push_back_reserve( hb.EdgeObject(i) );
     vec<alignment_plus> ealigns;
     PerfectAligner pal( 96, PerfectAligner::findProperOnly );
     pal.Align( edges, ealigns );
     for ( int i = 0; i < ealigns.isize( ); i++ )
     {    const alignment_plus& ap = ealigns[i];
          int id1 = ap.Id1( ), id2 = ap.Id2( );
          if ( id1 == id2 ) continue;
          int len1 = edges[id1].size( ), len2 = edges[id2].size( );
          double cov1 = double( ap.a.extent1( ) ) / double(len1);
          double cov2 = double( ap.a.extent2( ) ) / double(len2);
          if ( cov1 >= 0.5 ) duplicated[id1] = True;
          if ( cov2 >= 0.5 ) duplicated[id2] = True;    }
     */

     // Find long overcollapsed edges and neighborhoods of them.

     for ( int e = 0; e < nedges; e++ )
     {    if ( h.EdgeLength(e) >= 400 && cov[e] >= 1.3 )
          {    vrtx_t v = to_left_vertex[e], w = to_right_vertex[e];
               cout << "\n------------------------------------------------------"
                    << "--------------------------\n"
                    << "\ntrying to fix edge " << BaseAlpha(e) << " from " << v << " to " << w 
                    << " having length " << h.EdgeLength(e) << " and coverage " 
                    << cov[e] << "\n";
               vec<vrtx_t> S;
               S.push_back( v, w );
               UniqueSort(S);
               int min_length = 5000;                // for handles
               double min_cov = 0.9, max_cov = 1.1;  // for handles
               GrowHandledNhood( S, h, cov, min_length, min_cov, max_cov );
               cout << "vertices in neighborhood: ";
               CompactPrint( cout, S );
               cout << "\n";    
               vec<int> handles;
               Handles( h, S, handles );
               if ( handles.empty() ) continue;
               cout << "found handles: ";
               for ( unsigned int i = 0; i < handles.size(); ++i )
                 cout << BaseAlpha(handles[i]) << " ";
               cout << endl;

               // Check for some pathologies, which for now we don't deal with:
               // - two handles are equal as KmerPaths;
               // - a handle goes from a source to a sink.

               Bool equal_handles = False, bihandle = False;
               vec<vrtx_t> sources, sinks;
               h.SubgraphSources( S, sources ), h.SubgraphSinks( S, sinks );
               vec<int> lsources, lsinks;
               for ( int i = 0; i < sources.isize( ); i++ )
                    lsources.push_back( BinPosition( S, sources[i] ) );
               for ( int i = 0; i < sinks.isize( ); i++ )
                    lsinks.push_back( BinPosition( S, sinks[i] ) );
               for ( int i1 = 0; i1 < handles.isize( ); i1++ )
               {    vrtx_t v = to_left_vertex[ handles[i1] ];
                    vrtx_t w = to_right_vertex[ handles[i1] ];
                    if ( BinMember( sources, v ) && BinMember( sinks, w ) )
                         bihandle = True;
                    for ( int i2 = i1 + 1; i2 < handles.isize( ); i2++ )
                         if ( handles[i2] == handles[i1] ) equal_handles = True;    }
               if ( equal_handles || bihandle )
               {    cout << "Bad handles!\n";
                    continue;    }

               // Separate out the neighborhood and localize data structures.

               HyperKmerPath hs( h, S );
               int nsedges = hs.EdgeObjectCount( );
               vec<int> global_edge_ids;
               for ( int i = 0; i < S.isize( ); i++ )
               {    vrtx_t v = S[i];
                    for ( int j = 0; j < h.From(v).isize( ); j++ )
                    {    vrtx_t w = h.From(v)[j];
                         if ( !BinMember( S, w ) ) continue;
                         int e = h.EdgeObjectIndexByIndexFrom( v, j );
                         global_edge_ids.push_back(e);    }    }
               vec<int> to_local(nedges, -1);
               for ( int i = 0; i < global_edge_ids.isize( ); i++ )
                    to_local[ global_edge_ids[i] ] = i;
               vec<double> scov;
               for ( int i = 0; i < global_edge_ids.isize( ); i++ )
                    scov.push_back( cov[ global_edge_ids[i] ] );
               vec<int> sto_left_vertex(nsedges), sto_right_vertex(nsedges);
               for ( vrtx_t w = 0; w < hs.N( ); w++ )
               {    for ( int j = 0; j < hs.To(w).isize( ); j++ )
                    {    int e = hs.EdgeObjectIndexByIndexTo( w, j );
                         vrtx_t v = hs.To(w)[j];
                         sto_left_vertex[e] = v; 
                         sto_right_vertex[e] = w;    }    }
               if (verbose)
               {    cout << "\nLocal HyperKmerPath:\n";
                    hs.PrintSummaryPlus( cout, 0, 0, 0, 0, 0, False );
                    cout << "\n";    }

               // Define untrusted edges.

               vec<Bool> untrusted( nsedges, False );
               for ( int e = 0; e < nsedges; e++ )
               { 
                 if ( scov[e] > 1.2 ||
                      scov[e] < 0.8 ||
                      duplicated[ global_edge_ids[e] ] ||
                      hs.EdgeLength(e) < 100 ) {
                   untrusted[e] = True;    
                 }
               }
               if (verbose)
               {    for ( int e = 0; e < nsedges; e++ )
                    {    if ( untrusted[e] )
                         {    cout << "untrusted(" << scov[e];
                              if ( duplicated[ global_edge_ids[e] ] )
                                cout << ",dup";
                              if ( hs.EdgeLength(e) < 100 )
                                cout << ",short";
                              cout << "): "
                                   << sto_left_vertex[e] << " -" << BaseAlpha(e) << "-> "
                                   << sto_right_vertex[e] << "\n";     }
                         else
                         {    cout << "TRUSTED(" << scov[e] << "): "
                                   << sto_left_vertex[e] << " -" << BaseAlpha(e) << "-> "
                                   << sto_right_vertex[e] << "\n";     }    }    }

               // Define equivalence relation on edges.

               equiv_rel e;
               hs.DualComponentRelation( e, untrusted );
               equiv_rel ev;
               hs.ComponentRelation(ev);
               vec< pair<int,int> > join_prints;
               if (verbose) cout << "\n";
               for ( int id1 = 0; id1 < nreads; id1++ )
               {    if ( !Aligns_index[id1].solo( ) ) continue;
                    int ai1 = Aligns_index[id1][0];
                    if ( !hits_unique[ai1] ) continue;
                    const alignment_plus& ap1 = Aligns[ai1];
                    int id1 = ap1.Id1( );
                    int id2 = partner[id1];
                    if ( pos1[ai1] > 0 || Pos1[ai1] < reads[id1].isize( ) ) continue;
                    if ( id2 < 0 ) continue;
                    if ( !Aligns_index[id2].solo( ) ) continue;
                    int ai2 = Aligns_index[id2][0];
                    if ( !hits_unique[ai2] ) continue;
                    const alignment_plus& ap2 = Aligns[ai2];
                    if ( ap1.Rc2( ) == ap2.Rc2( ) ) continue;
                    if ( pos1[ai2] > 0 || Pos1[ai2] < reads[id2].isize( ) ) continue;
                    int e1 = to_local[ ap1.Id2( ) ], e2 = to_local[ ap2.Id2( ) ];
                    if ( e1 < 0 || e2 < 0 ) continue;
                    if ( untrusted[e1] || untrusted[e2] ) continue;
                    if ( !ev.Equiv( sto_left_vertex[e1], sto_left_vertex[e2] ) )
                         continue;

                    // We require that in the local graph, there is a path from
                    // e1 to e2 (or the other way in the rc case).

                    int xe1 = e1, xe2 = e2;
                    if ( ap1.Rc2( ) ) swap( xe1, xe2 );
                    vrtx_t v = sto_right_vertex[xe1], w = sto_left_vertex[xe2];
                    static vec<vrtx_t> v_only(1), from_v;
                    v_only[0] = v;
                    hs.digraph::GetSuccessors( v_only, from_v );
                    if ( !Member( from_v, w ) ) continue;

                    // Report.

                    if (verbose && e1 != e2
                         && !Member( join_prints, make_pair(e1,e2) )
                         && !Member( join_prints, make_pair(e2,e1) ) )
                    {    join_prints.push_back( make_pair(e1,e2) );
                         cout << "joining edge " << sto_left_vertex[e1]
                              << " -" << BaseAlpha(e1) << "-> " << sto_right_vertex[e1]
                              << " to " << sto_left_vertex[e2] << " -"
                              << BaseAlpha(e2) << "-> " << sto_right_vertex[e2]
                              /*
                              << ", coverage = " << aligns_cov[ai1] << ", "
                              << aligns_cov[ai2];
                              */
                              << "\n";
                              /*
                              cout << readlocs[ readlocs_index[id1] ];
                              cout << readlocs[ readlocs_index[id2] ];
                              */
                                          }

                    // Join.

                    e.Join( e1, e2 );    }

               // Go through the components. ????????

               vec<int> reps; 
               e.OrbitRepsAlt(reps);
               vec< vec<vrtx_t> > C;
	       vec< vec<vrtx_t> > D;
               vec<Bool> used( hs.N( ), False );
               int count = 1;
               for ( int pass = 1; pass <= 2; pass++ )
               {    for ( int i = 0; i < reps.isize( ); i++ )
                    {    static vec<vrtx_t> o, verts;
                         e.Orbit( reps[i], o );
                         Sort(o);
                         if ( o.solo( ) && untrusted[ o[0] ] ) continue;
                         verts.clear( );
                         for ( int j = 0; j < o.isize( ); j++ )
                         {    verts.push_back( sto_left_vertex[ o[j] ] );
                              verts.push_back( sto_right_vertex[ o[j] ] );    }
                         UniqueSort(verts);
                         /*
                         vec<int> linked_to;
                         for ( int j = 0; j < o.isize( ); j++ )
                         {    int m1 = o[j];
                              if ( hs.EdgeLength(m1) < 1000 ) continue;
                              int gm1 = global_edge_ids[m1];
                              for ( int r = 0; r < Aligns_index2[gm1].isize( ); 
                                   r++ )
                              {    const alignment_plus& ap1 
                                        = Aligns[ Aligns_index2[gm1][r] ];
                                   int id1 = ap1.Id1( );
                                   int id2 = partner[id1];
                                   if ( id2 < 0 ) continue;
                                   for ( int s = 0; 
                                        s < Aligns_index[id2].isize( ); s++ )
                                   {    const alignment_plus& ap2
                                             = Aligns[ Aligns_index[id2][s] ];
                                        int m2 = to_local[ ap2.Id2( ) ];
                                        if ( m2 < 0 ) continue;
                                        if ( !ev.Equiv( sto_left_vertex[m1], 
                                             sto_left_vertex[m2] ) )
                                        {    continue;    }
                                        if ( BinMember( o, m2 ) ) continue;
                                        linked_to.push_back(m2);    }    }    }
                         UniqueSort(linked_to);
                         */
                         vec<vrtx_t> to, from;
			 vec<vrtx_t> between;
                         hs.GetPredecessors( verts, to );
                         hs.digraph::GetSuccessors( verts, from );
                         Intersection( to, from, between );
                         /*
                         for ( int j = 0; j < linked_to.isize( ); j++ )
                         {    between.push_back( sto_left_vertex[ linked_to[j] ] );
                              between.push_back( 
                                   sto_right_vertex[ linked_to[j] ] );    }
                         UniqueSort(between);
                         */
                         if ( pass == 1 ) C.push_back(between);
                         if ( pass == 2 )
                         {    Bool subsumed = False;
                              for ( int j = 0; j < C.isize( ); j++ )
                              {    if ( Subset( between, C[j] )
                                        && between.size( ) < C[j].size( ) )
                                   {    subsumed = True;    }    }
                              if (subsumed) continue;
                              D.push_back(between);    }
                         for ( int u = 0; u < between.isize( ); u++ )
                              used[ between[u] ] = True;
                         if ( pass == 2 && verbose )
                         {    cout << "\n[" << i+1 << "] verts ";
                              CompactPrint( cout, verts );
                              cout << "\no = ";
                              CompactPrint( cout, o );
                              cout << "\n[" << count++ << "] ";
                              CompactPrint( cout, between );
                              cout << "\n";    }    }    }
               if ( D.size( ) >= 2 )
               {    cout << "\nConsidering surgery, inserting " 
                         << D.size( ) << " components.\n";

                    // Check that each handle of S appears exactly once in D.

                    Bool good = True;
                    cout << "lsources = ";
                    CompactPrint( cout, lsources );
                    cout << "\nlsinks = ";
                    CompactPrint( cout, lsinks );
                    cout << "\n";
                    for ( int i = 0; i < handles.isize( ); i++ )
                    {    int v = BinPosition( S, to_left_vertex[ handles[i] ] );
                         int w = BinPosition( S, to_right_vertex[ handles[i] ] );
                         int touch_end = 0, contain_handle = 0;
                         for ( int j = 0; j < D.isize( ); j++ )
                         {    Bool cv = BinMember( D[j], v );
                              Bool cw = BinMember( D[j], w );
                              if ( cv && cw ) ++contain_handle;
                              if ( BinMember( lsources, v ) && cv ) ++touch_end;
                              if ( BinMember( lsinks, w ) && cw ) ++touch_end;    }
                         if ( touch_end != 1 || contain_handle != 1 )
                         {    good = False;
                              PRINT4( v, w, touch_end, contain_handle );    }    }
                    if ( !good ) cout << "Problem with handles.\n";
                    else
                    {    HyperKmerPath hm(h);
                         vec<int> del;
                         for ( int i = 0; i < S.isize( ); i++ )
                         {    vrtx_t v = S[i];
                              for ( int j = 0; j < h.From(v).isize( ); j++ )
                              {    vrtx_t w = h.From(v)[j];
                                   if ( BinMember( S, w ) )
                                   {    del.push_back( h.EdgeObjectIndexByIndexFrom(
                                             v, j ) );    }    }    }
                         Sort(del);
                         hm.DeleteEdges(del);
                         vec<HyperKmerPath> hypers;
                         hypers.push_back(hm);
                         for ( int i = 0; i < D.isize( ); i++ )
                               hypers.push_back( HyperKmerPath( hs, D[i] ) );
                         vec< pair< pair<int,vrtx_t>, pair<int,vrtx_t> > > joins;
                         for ( int i = 0; i < handles.isize( ); i++ )
                         {    vrtx_t gv = to_left_vertex[ handles[i] ];
                              int v = BinPosition( S, gv );
                              vrtx_t gw = to_right_vertex[ handles[i] ];
                              int w = BinPosition( S, gw );
                              for ( int j = 0; j < D.isize( ); j++ )
                              {    Bool cv = BinMember( D[j], v );
                                   Bool cw = BinMember( D[j], w );
                                   if ( !( cv && cw ) ) continue;
                                   if ( BinMember( lsources, v ) )
                                   {    int iv = BinPosition( D[j], v );
                                        joins.push_back( make_pair(
                                             make_pair( 0, gv ),
                                             make_pair( 1 + j, iv ) ) );    }
                                   if ( BinMember( lsinks, w ) )
                                   {    int iw = BinPosition( D[j], w );
                                        joins.push_back( make_pair(
                                             make_pair( 0, gw ),
                                             make_pair( 1 + j, 
                                                  iw ) ) );    }    }    }
                         HyperKmerPath hnew( K, hypers, joins );
                         hnew.RemoveEdgelessVertices( );
                         hnew.RemoveDeadEdgeObjects( );
                         vec<alignment_plus> Aligns_new;
                         RecomputeAligns( h, hnew, reads, Aligns, Aligns_new );
                         h = hnew;
                         hb = HyperBasevector( h, *kbb );
                         Aligns = Aligns_new;
                         goto beginning;    }    }    }    }

     if (verbose)
     {    cout << "\n------------------------------------------------------"
               << "--------------------------\n\n";    }    }
