/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// This code builds the unipath link graphs used by LocalizeReads.

#include "MainTools.h"

#include "paths/Unipath.h"
#include "paths/UnipathNhood.h"
#include "paths/simulation/Placement.h"

int main( int argc, char** argv ) {
  
  RunTime();

  BeginCommandArguments;
  CommandArgument_String(PRE);
  CommandArgument_String(DATA);
  CommandArgument_String(RUN);
  CommandArgument_Int(K);
  CommandArgument_Int_OrDefault(MAX_COPY_NUMBER, 2);
  CommandArgument_Int_OrDefault(MIN_KMERS, 25);
  CommandArgument_Bool_OrDefault(USE_TRUTH, False);
  CommandArgument_Int_OrDefault(MIN_EDGE_MULTIPLICITY, 3);
  CommandArgument_Bool_OrDefault(TRANSITIVE_FILL_IN, True);
  CommandArgument_Int_OrDefault(TRANSITIVE_FILL_IN_VERBOSITY, 0);
  CommandArgument_Bool_OrDefault(BUILD_GRAPH_VERBOSE, False);
  EndCommandArguments;

  String data_dir = PRE + "/" + DATA;
  String run_dir = PRE + "/" + DATA + "/" + RUN;

  String KS = ToString(K);

  // Read in read paths, unipaths, reads, and pairing data.
  
  MakeMappedMastervec( run_dir + "/reads.paths.k" + KS, vecKmerPath, paths );
  MakeMappedMastervec( run_dir + "/reads.unipaths.k" + KS, vecKmerPath, unipaths );
  int nreads = paths.size( ), nuni = unipaths.size( );

  vec<read_pairing> pairs;
  ReadPairsFile( run_dir + "/reads.pairto", pairs );

  // Find involution which maps a unipath to its reverse complement.  Compute
  // length of each unipath.  Read in output of UnipathCoverage.
  
  vec<int> to_rc, ulen(nuni);
  {
    MakeMappedVec( run_dir + "/reads.unipathsdb.k" + KS, tagged_rpint, unipathsdb );
    UnipathInvolution( unipaths, unipathsdb, to_rc );
  }
  for ( int i = 0; i < nuni; i++ )
    ulen[i] = unipaths[i].KmerCount( );

  vecvec<pdf_entry> cp;
  cp.ReadAll( run_dir + "/reads.unipaths.predicted_count.k" + KS );

  // Define the normal unipaths.
  
  vec<Bool> normal(nuni, False), normalplus(nuni, False);
  vec<int> predicted_copyno(nuni, -1);
  vec<double> predicted_copyno_p(nuni, -1);
  for ( int i = 0; i < nuni; i++ ) {
    int copyno = -1;
    double maxp = 0;
    for ( int j = 0; j < cp[i].size( ); j++ ) {
      if ( cp[i][j].second > maxp ) {
        copyno = cp[i][j].first;
        maxp = cp[i][j].second;
      }
    }
    predicted_copyno[i] = copyno;
    predicted_copyno_p[i] = maxp;
    if ( copyno > MAX_COPY_NUMBER ) continue;
    normalplus[i] = True;
    if ( ulen[i] < MIN_KMERS ) continue;
    normal[i] = True;    
  }

  // Load the read placements on unipaths.
  
  String unilocs_file = run_dir + "/reads.unilocs." + KS + ".10.1";
  BREAD2( unilocs_file, vec<read_location_short>, ulocs );
  vecvec<int> ulocs_indexr;
  ulocs_indexr.ReadAll( unilocs_file + ".indexr" );

  // Load in unipath locs on reference if needed.
  vecvec<placement> locs;
  if ( TRANSITIVE_FILL_IN_VERBOSITY && USE_TRUTH )
    locs.ReadAll( run_dir + "/reads.unipaths.k" + KS + ".locs" );    

  // Build the unipath graphs.
  
  digraphE<sepdev> G, Gplus;
  cout << Date() << ": building G" << endl;
  BuildUnipathLinkGraph( K, pairs, ulocs, ulocs_indexr, normal, ulen, 
                         paths, to_rc, MIN_EDGE_MULTIPLICITY, G );
  cout << Date() << ": building Gplus" << endl;
  BuildUnipathLinkGraph( K, pairs, ulocs, ulocs_indexr, normalplus, ulen, 
                         paths, to_rc, MIN_EDGE_MULTIPLICITY, Gplus );
  digraphE<fsepdev> FG;
  cout << Date() << ": building FG" << endl;
  BuildUnipathLinkGraph( K, pairs, ulocs, ulocs_indexr, normal, ulen, 
                         paths, to_rc, MIN_EDGE_MULTIPLICITY, FG, BUILD_GRAPH_VERBOSE );
  if (TRANSITIVE_FILL_IN) {
    cout << Date() << ": filling in FG" << endl;
    FillInTransitiveEdges( FG, 6000, 1000.0, 10.0, predicted_copyno, ulen,
                           TRANSITIVE_FILL_IN_VERBOSITY,
                           (TRANSITIVE_FILL_IN_VERBOSITY && USE_TRUTH ? &locs : NULL) );    
  }
  
  // Save the graphs.
  cout << Date() << ": saving graphs" << endl;
  BinaryWrite( run_dir + "/reads.unipathgraph.k" + KS, G );
  BinaryWrite( run_dir + "/reads.unipathgraphplus.k" + KS, Gplus );
  BinaryWrite( run_dir + "/reads.unipathgraph_fp.k" + KS, FG );
}
