/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#include "paths/BigPicture.h"

#include <set>

int GetNodeIdFromPathId( const OrientedKmerPathId& pathId )
{
  return pathId.GetId()*2 + pathId.IsRc(); 
}

void PrintOneNode( ostream& out, const OrientedKmerPathId& okpid, 
                   set<OrientedKmerPathId>& seen, const int partition ) {
  if ( ! seen.count( okpid ) ) {
    out << "  " << GetNodeIdFromPathId( okpid ) 
        << " ["
        << "label=\"" << okpid << "\"";
    if ( okpid.GetId() < partition )
      out << ",style=filled,fillcolor=grey";
    out << "];" << "\n";
    seen.insert( okpid );
  }
}
  

void
PrintBigPictureDot( const String& filename, 
                    const MuxGraph& muxGraph,
                    const SubsumptionList& subList,
                    const int partition ) 
{
  ofstream out( filename.c_str() );

  out << "digraph BigPicture {" << "\n";
  out << "  concentrate=true;" << "\n";
  out << "  rankdir=LR;" << "\n";
  
  set<OrientedKmerPathId> seen;

  vec<Mux> muxes;
  for ( int i = 0; i < muxGraph.size(); ++i ) {
    for ( int rc = 0; rc < 2; ++rc ) {
      OrientedKmerPathId thisOkpid( i, rc );
      muxGraph.GetMuxesOf( thisOkpid, muxes );
      if ( ! muxes.empty() )
        PrintOneNode( out, thisOkpid, seen, partition );
      for ( int m = 0; m < muxes.isize(); ++m ) {
        OrientedKmerPathId muxOkpid( muxes[m].GetPathId() );
        PrintOneNode( out, muxOkpid, seen, partition );
        out << "  " << GetNodeIdFromPathId( thisOkpid ) 
            << " -> " << GetNodeIdFromPathId( muxOkpid )
            << " [label=\"" << muxes[m].GetNumKmers() << "\"];" << "\n";
      }
    }
  }

  vec<SubsumptionRecord> subs;
  for ( int i = 0; i < muxGraph.size(); ++i ) {
    for ( int rc = 0; rc < 2; ++rc ) {
      OrientedKmerPathId thisOkpid( i, rc );
      subList.GetFullRecordsFor( thisOkpid, subs );
      if ( ! subs.empty() )
        PrintOneNode( out, thisOkpid, seen, partition );
      for ( int s = 0; s < subs.isize(); ++s ) {
        OrientedKmerPathId superOkpid( subs[s].GetSuperPathId() );
        PrintOneNode( out, superOkpid, seen, partition );
        out << "  " << GetNodeIdFromPathId( superOkpid ) 
            << " -> " << GetNodeIdFromPathId( thisOkpid )
            << " [color=red,constraint=false];" << "\n";
      }
    }
  }
    
  out << "}" << "\n";
}
