/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef ALIGN_PAIRS_TO_HYPER_H
#define ALIGN_PAIRS_TO_HYPER_H

#include "Basevector.h"
#include "Bitvector.h"
#include "CoreTools.h"
#include "ReadPairing.h"
#include "graph/Digraph.h"
#include "paths/HyperKmerPath.h"
#include "paths/SeqOnHyper.h"


// Computes alignments of reads to the hyperkmer path directly using Perfectlookup

typedef pair<int,int> IndexPair;

void AlignPairsToHyper( const HyperKmerPath& h, const HyperBasevector& hb,
			// note that hb must be HyperBasevector(h)
			const vecbasevector& reads, const vec<read_pairing>& pairs,
			const String& sub_dir,
			vec<CompressedSeqOnHyper>& csaligns,
			vec<vec<int> >& csaligns_index,
			vec< IndexPair >& pair_places, 
			Bool verbose,  const int max_insert_size = -1);

// EvaluatePair: given alignments ap1, ap2 of the reads of a read pair P to a
// HyperKmerPath, determine if the insert could be placed contiguously on the
// HyperKmerPath, without stretching too much.  Return True in that case.

Bool EvaluatePair( 
     const SeqOnHyper& ap1,            // alignment of first read
     const SeqOnHyper& ap2,            // alignment of second read
     const read_pairing& P,            // the read pair
     double dev_mult,                  // how much stretch is allowed in devs
     const HyperBasevector& hb,        // from the HyperKmerPath
     const vec<vrtx_t>& to_right_vertex,  // index from edge to right vertex
     const vec<vrtx_t>& to_left_vertex,   // index from edge to left vertex
     const digraphE<int>& G,           // mirrors hb but edges are lengths in kmers
     const vec<IndexPair>& paired,
     const vec<Bool>& have_path,
     const vec<int>& min_path,
     const vec<int>& max_path,
     const vec<int>& a,
     const vec<int>& b );

// EvaluatePair: given alignments cs1, cs2 of the reads of a read pair P to a
// HyperKmerPath, determine if the insert could be placed contiguously on the
// on a single edge of the HyperKmerPath, without stretching too much.
// Return True in that case.
// Special case of the more general EvalutePair function. Optimized for single
// edge condition using CompressedSeqOnHyper information only.

Bool EvaluatePairOnSingleEdge( 
     const CompressedSeqOnHyper& cs1,     // alignment of first read
     const CompressedSeqOnHyper& cs2,     // alignment of second read
     const read_pairing& P,               // the read pair
     double dev_mult );                   // how much stretch is allowed in devs

// Find placements of read pairs that are not too stretched.  The returned data is
// in pair_places, and we guarantee that csaligns[ pair_places[i].first ] is
// forward aligned, whereas csaligns[ pair_places[i].second ] is reverse aligned.
// Note that pairs of alignments going between components in the HyperKmerPath are 
// not included.

void FindPairPlacements( const HyperKmerPath& h, const HyperBasevector& hb,
			 const vec<CompressedSeqOnHyper>& csaligns,
			 const vec<vec<int> >& csaligns_index,
			 const int nreads, const vec<read_pairing>& pairs,
			 vec< IndexPair >& pair_places, Bool verbose,
			 const String& sub_dir, const int max_insert_size = -1);


// Find the bases on the HyperBasevector edges which are covered by alignments of 
// read pairs that are not too stretched.  Note that read pairs going between 
// components are not counted (which may be a good thing or a bad thing).  One can
// filter to only use pairs within a given separation range and to only use 
// forward- or reverse-aligned reads.

void FindWellCovered( int min_sep, int max_sep, Bool use_fw, Bool use_rc,
     const HyperKmerPath& h, const HyperBasevector& hb,
     const vec<CompressedSeqOnHyper>& csaligns, 
     const vec< IndexPair >& pair_places,
     int nreads, const vec<read_pairing>& pairs, const vec<int>& pairs_index,
     vecbitvector& cov, Bool verbose );


// Deletes edges of HyperKmerPath that are not covered by bases in a good pair.

void FindPoorlyCovered( HyperKmerPath& h, HyperBasevector& hb,
     const KmerBaseBroker& kbb, const vec<CompressedSeqOnHyper>& csaligns, 
     const vec< IndexPair >& pair_places, const vecbasevector& reads, 
     const vec<read_pairing>& pairs, Bool remove, int verbosity,
     const Bool NEW_POORLY_COVERED = False );

#endif
