// Copyright (c) 2004 Broad Institute/Massachusetts Institute of Technology

// SmithWatFree( S, T, gap_penalty )
//
// Let S and T be basevectors, with S very small relative to T.  Return the best 
// (lowest) score of an alignment of S with T, relative to the following rules:
//
// (a) a mismatch scores +1
// (b) each blank in a gap scores +2
// (c) each base of S must align with T (so S must be found as a subset of T)
// (d) there is optionally a penalty for gaps on either side of S

// Compile with -funroll-all-loops (perhaps 5% faster).

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Basevector.h"
#include "math/Functions.h"

#include "Alignment.h"
#include "ShortVector.h"
#include "pairwise_aligners/SmithWatFree.h"

unsigned int SmithWatFree( const basevector& S, const basevector& T, 
			   int& best_loc, alignment& a,
			   bool penalize_left_gap,
			   bool penalize_right_gap, 
                           unsigned int mismatch_penalty, unsigned int gap_penalty )
{
     const unsigned int gap_penalty1 = gap_penalty;
     const unsigned int gap_penalty2 = gap_penalty;
     const unsigned int outer_gap_penalty = 3;

     ForceAssertGt( S.size(), 0 );
     ForceAssertGt( T.size(), 0 );

     unsigned int n = S.size( ), N = T.size( );

     ForceAssertLe( n, N );

     static avector<char> s, t;
     s.resize(n);
     for ( unsigned int i = 0; i < n; i++ )
          s(i) = S[i];
     t.resize(N);
     for ( unsigned int i = 0; i < N; i++ )
          t(i) = T[i];

//      if ( penalize_left_gap )
//        cout << "penalizing left gap" << endl;
//      if ( penalize_right_gap )
//        cout << "penalizing right gap" << endl;
//      PRINT2( n, N );

     const unsigned int Infinity = 1000000000;
     unsigned int best_score = Infinity;
     avector<unsigned int> x(n+1);
     for ( unsigned int i = 0; i <= n; i++ )
          x(i) = Infinity;
     best_loc = 0;

     for ( unsigned int j = 0; j < N; ++j )
     {    
	  unsigned int* xp = x.x;
          unsigned int lastx = 0;

          if ( penalize_left_gap )
	    lastx += outer_gap_penalty * j;

	  for ( unsigned int i = 0; i < n; ++i )
	  {    unsigned int a = lastx + mismatch_penalty * ( s(i) != t(j) );
	       unsigned int b = *xp + gap_penalty2;
	       ++xp;
	       unsigned int c = *xp + gap_penalty1;
	       lastx = *xp;
	       *xp = Min( Min( a, b ), c );    }

//  	  for ( unsigned int i = 0; i <= n; ++i )
//  	    if ( x(i) >= Infinity )
//  	      cout << "I\t";
//  	    else
//  	      cout << x(i) << "\t";
//  	  cout << endl;

	  unsigned int this_score2 = x(n);
	  if ( penalize_right_gap )
	    this_score2 += outer_gap_penalty * ( N - j - 1 );
	  
//  	  PRINT2( j, this_score2 );

          if ( this_score2 <= best_score ) best_loc = j;
          best_score = Min( best_score, this_score2 );    }

//      S.Print( cout, "S" );
//      T.Print( cout, "T" );
     
//       PRINT( best_loc );
//       PRINT( best_score );

     // Redo to find the alignment.

     unsigned int subN = Min( best_loc, 
			      (int) (S.size( ) + 
				     best_score/(Min(gap_penalty1, gap_penalty2)) + 1) ) + 1;

//      PRINT( subN );

     t.resize(subN);
     for ( unsigned int i = 0; i < subN; ++i )
          t(i) = T[ best_loc - subN + i + 1 ];

//      cout << "t: ";
//      for ( unsigned int i = 0; i < subN; ++i )
//        cout << as_base(t(i));
//      cout << endl;

     vec< vec<unsigned char> > from( subN, vec<unsigned char>(n) );

     unsigned int best_score2 = Infinity;
     for ( unsigned int i = 0; i <= n; ++i )
          x(i) = Infinity;
     int best_loc2 = 0;

     for ( unsigned int j = 0; j < subN; ++j )
     {    unsigned int* xp = x.x;
          unsigned int lastx = 0;

	  if ( penalize_left_gap )
	    lastx += outer_gap_penalty * j;
          
          for ( unsigned int i = 0; i < n; ++i )
          {    unsigned int a = lastx + mismatch_penalty * ( s(i) != t(j) );
               unsigned int b = *xp + gap_penalty2;
               ++xp;
               unsigned int c = *xp + gap_penalty1;
               lastx = *xp;

               if ( a <= b )
               {    if ( a <= c ) from[j][i] = 'a';
                    else from[j][i] = 'c';    }
               else
               {    if ( b <= c ) from[j][i] = 'b';
                    else from[j][i] = 'c';    }
//   	       cout << from[j][i];
               *xp = Min( Min( a, b ), c );    }
//  	  cout << endl;

// 	  for ( unsigned int i = 0; i <= n; ++i )
// 	    if ( x(i) >= Infinity )
// 	      cout << "I\t";
// 	    else
// 	      cout << x(i) << "\t";
//  	  cout << endl;

	  unsigned int this_score2 = x(n);
	  if ( penalize_right_gap )
	    this_score2 += outer_gap_penalty * ((N - best_loc - 1) + (subN - j - 1));

// 	  PRINT2( j, this_score2 );

          if ( this_score2 <= best_score2 ) best_loc2 = j;
          best_score2 = Min( best_score2, this_score2 );    }

//       PRINT( N );
//      PRINT( best_loc2 );
//      PRINT( best_score2 );

     ForceAssert( best_loc2 == (int) subN-1 );

     int j = best_loc2;
     int i = ((int) n) - 1;
     int lcount = 0, g1count = 0, g2count = 0;
     int last_length = 0;
     avector<int> gaps(0), lengths(0);
     while(1)
     {    // cout << from[j][i];
          if ( from[j][i] == 'a' )
          {    if ( g1count > 0 )
               {    if ( last_length > 0 )
		    {    gaps.Prepend( g1count );
		         lengths.Prepend( last_length );    }
	            g1count = 0;    }
               if ( g2count > 0 )
               {    if ( last_length > 0 )
	            {    gaps.Prepend( -g2count );
                         lengths.Prepend( last_length );    }
                    g2count = 0;    }
               ++lcount;
               --i;
               --j;    }
          else if ( from[j][i] == 'b' )  // gap on long sequence
          {    if ( lcount > 0 )
               {    last_length = lcount;
                    lcount = 0;    }
               ForceAssert( g1count == 0 );
               ++g2count;
               --i;    }
          else                           // gap on short sequence
          {    if ( lcount > 0 )
               {    last_length = lcount;
                    lcount = 0;    }
               ForceAssert( g2count == 0 );
               ++g1count;
               --j;    }
          if ( i < 0 ) break;
          ForceAssert( j >= 0 );    }

     //     cout << endl;

//      PRINT2( i, j );

     if ( g1count != 0 ) gaps.Prepend( g1count );
     else if ( g2count != 0 ) gaps.Prepend( -g2count );
     else gaps.Prepend(0);

     lengths.Prepend( lcount );

     int pos1 = 0;
     int pos2 = best_loc - subN + 1 + j + 1;

//      S.Print( cout, "S" );
//      T.Print( cout, "T" );
//      cout << "pos1: " << pos1 << " pos2: " << pos2 << endl;
//      cout << "(gaps) lengths: ";
//      for ( int g = 0; g < (int) gaps.length; ++g )
//        cout << "(" << gaps(g) << ") "
// 	    << lengths(g) << " ";
//      cout << endl;
//      PRINT( best_score );

     ForceAssertLe( gaps(0), 0 );

     int first_gap = gaps(0);
     if ( first_gap < 0 )
     {
       pos2 -= first_gap;
       gaps(0) = 0;
     }

     int errors = best_score;
     a = alignment( pos1, pos2, errors, gaps, lengths );

     ForceAssertEq( best_score, best_score2 );
     return best_score;    }
