/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

//// ImperfectLookup:  Find good unique alignments with no indels quickly.
///
/// \fn ImperfectLookup
///
/// Given a vector "query" of sequences Q,
/// (a) find all gap-free alignments A of Q to target (defined by "lookup_file", 
/// from MakeLookupTable with the given "K"), that subsume a perfect match of length 
/// K and that are unique in the sense that there is no other such alignment B for 
/// which errors(B) <= errors(A) + best_prox;
/// (b) find the minimum number of mismatches in a gap-free alignment of Q to target 
/// that subsumes a perfect match of length K.
///
/// If no alignment of Q is found, the value for (b) is infinitely_many.
///
/// The variable AlignDir controls whether we look for forward alignments, or for 
/// both forward and reverse alignments.
///
/// Note that this code will not work for finding indel polymorphisms!
///
/// Note that for certain applications, the code could be speeded up by keeping the
/// lookup table in memory, rather than reading it from disk on each call.
///
/// I'm not sure how this handles Ns in the target.
///
/// If unique_aligned is specified, do not modify aligns, and instead return a
/// boolean value as to whether each read aligns uniquely.
///
/// If quality scores are passed in, they are used to weight the errors. 
/// Details of the algorithm are in the doc for CountMismatches in .cc file.
///
/// \sa PerfectLookup()
/// \ingroup grp_eval

#ifndef IMPERFECT_LOOKUP_H
#define IMPERFECT_LOOKUP_H

#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "lookup/LookAlign.h"
#include "lookup/PerfectLookup.h"

class TaskTimer;
class lookup_table;

#define infinitely_many 1000000000

/// Version that reads in the lookup file from disk
/// The last argument is a dirty trick! if it is set to \c true,
/// the sequence on the second pass (lookig up rc alignments)
/// will be reverted rather than reverse complemented: this is 
/// used for the alignments in color space.
void ImperfectLookup( 
     // inputs:
     const unsigned int K, const vecbasevector& query, const String& lookup_file, 
     // outputs:
     vec<look_align>& aligns, vec<int>& min_errors,
     // parameters:
     const AlignDir direction, const int best_prox, const int max_errors = -1,
     // other:
     vec<Bool>* unique_aligned = 0,
     vec<TaskTimer *> * timers = 0,
     const vecqualvector * quals = 0,
     bool useReverse = false);

/// This version accepts lookup table from outside.  When multiple
/// lookups are to be done this can signficantly reduce I/O load.
/// The last argument is a dirty trick! if it is set to \c true,
/// the sequence on the second pass (lookig up rc alignments)
/// will be reverted rather than reverse complemented: this is 
/// used for the alignments in color space.
void ImperfectLookup( 
     // inputs:
     lookup_table &look, const vecbasevector& query, 
     // outputs:
     vec<look_align>& aligns, vec<int>& min_errors,
     // parameters:
     const AlignDir direction, const int best_prox, const int max_errors = -1,
     // other:
     vec<Bool>* unique_aligned = 0,
     vec<TaskTimer *> * timers = 0,
     const vecqualvector * quals = 0,
     bool useReverse = false);

#endif
