/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
#define NDEBUG
#endif

#include "Basevector.h"
#include "CoreTools.h"
#include "Feudal.h"
#include "BasevectorTools.h"
#include "lookup/LookAlign.h"
#include "lookup/LookupTable.h"
#include "lookup/ImperfectLookup.h"
#include "TaskTimer.h"

/** Add up mismatches, considering quality scores if given.
The return value of this function is:
- if there are no quals, the total number of mismatches.
- if there are quals, each mismatch is multiplied by max(quality at that
base, minQual), and the total error score is then divided by expectedQual
and rounded.
.
We return max_mismatches + 1 as soon as we go beyond our upper error limit
in order to save time.
*/
/*
int CountMismatches(unsigned int length, const basevector & S,
		    unsigned int q_start, lookup_table &look, 
		    unsigned int startx,  int max_mismatches = 1000 * 1000,
		    const qualvector * quals = 0, 
		    double expectedQual = 30, 
		    unsigned char minQual = 10){
  int mismatches = 0;
  if (!quals) {
    int mismatches = 0;
    for ( unsigned int y = 0; y < length; y++ )  {    
      if ( S[q_start + y] != look.Base( startx + y ) ) {    
	++mismatches;
	if ( mismatches > max_mismatches ) 
	  return mismatches;    
      }    
    }
    return mismatches;
  } else {
    float maxBad = expectedQual * max_mismatches;
    float bad = 0;
    for ( unsigned int y = 0; y < length; y++ )  {    
      if ( S[q_start + y] != look.Base( startx + y ) ) {    
	bad += Max(minQual,(*quals)[q_start + y]);
	if ( bad > maxBad ) {
	  return max_mismatches + 1;
	}
      }    
    }
    return int( round(bad/ expectedQual));
  }
}

*/

void ImperfectLookup( const unsigned int K, const vecbasevector& query, 
		      const String& lookup_file, vec<look_align>& aligns, 
		      vec<int>& best_errors,
		      const AlignDir direction, const int best_prox, const int max_errors,
		      vec<Bool>* unique_aligned,
		      vec<TaskTimer *> * timers,
		      const vecqualvector * quals,
		      bool useReverse)
{
  // Read header information from lookup table.

  lookup_table look(lookup_file);
  ForceAssertEq( look.K( ), K );
  ImperfectLookup(look, query, aligns, best_errors, direction, best_prox, max_errors,
		  unique_aligned, timers, quals,useReverse);
}

void ImperfectLookup( lookup_table &look, const vecbasevector& query, 
		      vec<look_align>& aligns, vec<int>& best_errors,
		      const AlignDir direction, const int best_prox, const int max_errors,
		      vec<Bool>* unique_aligned,
		      vec<TaskTimer *> * timers,
		      const vecqualvector * quals,
		      bool useReverse)
{
  const unsigned int K = look.K();
  // Deal with quality scores: verify that the quals match the bases
  // and calculate the mean quality
  double meanQual=0;
  if (quals) {
    vec<int> qs, bs;
    quals->ElementSizes(qs);
    query.ElementSizes(bs);
    ForceAssert(bs == qs);
    longlong qsum=0, qcount=0;
    for (int i=0; i != quals->size(); ++i) {
      qsum = accumulate((*quals)[i].begin(), (*quals)[i].end(), qsum);
    }
    qcount=accumulate(qs.begin(), qs.end(), qcount);
    meanQual = double(qsum)/qcount;
    PRINT3(meanQual, qsum, qcount);//exit(0);
  }

  // Initialize unique_aligned.

  if ( unique_aligned != 0 ) 
    unique_aligned->resize_and_set( query.size( ), False );

  // For each query (or its rc), find the indices of each of its kmers.

  const int npasses = ( direction == FW ? 1 : 2 );
  const int nqueries = query.size( );
  vec< vec<int> > all_pos( nqueries * 2 );
  vec< vec<unsigned int> > all_index( nqueries * 2 );
  for ( int id = 0; id < nqueries; id++ ) {    
    const basevector& s = query[id];
    if ( s.size( ) < K ) continue;
    for ( int pass = 0; pass < npasses; pass++ ) {    
      static basevector src;
      if ( pass == 1 ) {
	if ( useReverse ) src.Reverse(s) ;
	else src.ReverseComplement(s);
      }
      const basevector& S = ( pass == 0 ? s : src );
      int length = S.isize( ) - (int) K;
      all_pos[ 2*id + pass ].reserve( 
				     all_pos[ 2*id + pass ].isize( ) + length + 1 );
      all_index[ 2*id + pass ].reserve( 
				       all_index[ 2*id + pass ].isize( ) + length + 1 );
      for ( int j = 0; j <= length; j++ ) {    
	unsigned int index = Index( S, j, K );
	unsigned int freq = look.Freq(index);
	if ( freq == 0 ) continue;
	all_pos[ 2*id + pass ].push_back(j);    
	all_index[ 2*id + pass ].push_back(index);    
      }    
    }    
  }

  // Set up for alignment generation.

  static look_align la;
  la.nhits = la.indels = 0;
  la.a.SetNblocks(1);
  la.a.SetGap( 0, 0 );

  // Go through the lookup table chunks.

  const unsigned int max_query_size = 200 * 1000 * 1000;
  best_errors.resize_and_set( nqueries, (int) infinitely_many );
  vec<int> second_best_errors( nqueries, infinitely_many );
  vec<look_align> best_align(nqueries);
  for ( unsigned int i = 0; i < look.NChunks( ); i++ ) {  
    if (timers) (*timers)[0]->Start();
    look.ReadChunk(i);
    if (timers) (*timers)[0]->Stop();
    if (timers) (*timers)[1]->Start();

    // Go through the query sequences.

    for ( int id = 0; id < query.size( ); id++ ) {    
      if ( best_errors[id] <= best_prox && 
		 second_best_errors[id] - best_errors[id] <= best_prox ) 
	continue;
      const basevector& s = query[id];
      unsigned int query_length = s.size();
      if ( query_length < K ) continue;

      // Go through the orientations.

      for ( int pass = 0; pass < npasses; pass++ ) {
	int n = 2*id + pass;
	static basevector src;
	if ( pass == 1 ) {
	  if ( useReverse ) src.Reverse(s);
	  else src.ReverseComplement(s);
	}
	const basevector& S = ( pass == 0 ? s : src );
	const qualvector * q = 0;
	if (quals) {
	  if (0 == pass) q = &((*quals)[id]);
	  else {
	    qualvector * rq = new qualvector((*quals)[id]);
	    rq->ReverseMe();
	    q = rq;
	  }
	}
	static vec<unsigned int> offsets;
	offsets.clear( );
	for ( int u = 0; u < all_pos[n].isize( ); u++ ) {    
	  int r = all_pos[n][u];
	  unsigned int index = all_index[n][u];
	  unsigned int start = look.StartLocs(index);
	  unsigned int stop = look.StopLocs(index);
	  for ( unsigned int l = start; l < stop; l++ ) {    
	    if ( look.Locs(l) >= (unsigned int ) r )
	      {    
		offsets.push_back( look.Locs(l) 
				   + ( max_query_size - r ) );    
	      }    
	  }    
	}
	UniqueSort(offsets);
	for ( int u = 0; u < offsets.isize( ); u++ ) {    
	  unsigned int offset = offsets[u];
	  unsigned int tig, rpos; 
	  look.GetContigPos( offset - max_query_size, tig, rpos );
	  unsigned int startx = offset - max_query_size, q_start = 0, t_start = rpos;
	  unsigned int length = query_length;
	  // The valid alignment region is bounded by this contig or
	  // the chunk, whichever ends first.
	  unsigned int first = max(look.ContigStart(tig), look.BasesStart());
	  unsigned int last = min(look.ContigStop(tig), look.BasesStop());
	  // We want subsumed alignments only.
	  if ( offset < first + max_query_size || startx + length > last ) 
	    continue;

	  // Validate alignment.
	  int max_mismatches  = Min( best_errors[id] + best_prox,
				     second_best_errors[id] - 1 );
	  if ( max_errors >= 0 )  {    
	    max_mismatches = Min(max_mismatches, max_errors + best_prox );    
	  }
	  //	  int mismatches = CountMismatches(length, S, q_start, look, startx,
	  //					   max_mismatches, q, meanQual);
	  int mismatches = MismatchCount<BaseExactMatch>(S, 
							 look.Bases(),
							 q_start, 
							 startx - look.BasesStart(),
							 length,
							 max_mismatches, 
							 q, 
							 meanQual);
	  if ( mismatches > max_mismatches ) continue;

	  // See if alignment need not be generated.
                         
	  if ( mismatches > best_errors[id] ) {    
	    second_best_errors[id] = Min( second_best_errors[id], mismatches );
	  if ( best_errors[id] <= best_prox && 
	       second_best_errors[id] - best_errors[id] <= best_prox ) {    
	      goto done_with_this_read;    
	    }
	    continue;    
	  }
	  if ( mismatches == best_errors[id]
	       && best_errors[id] == second_best_errors[id] )
	    {    continue;    }

	  // Create alignment.

	  la.query_id = id;
	  la.target_id = tig;
	  la.a.Setpos1(q_start);
	  la.a.Setpos2(t_start);
	  la.query_length = query_length;
	  la.target_length = look.ContigSize(tig);
	  la.rc1 = ( pass == 1 );
	  la.a.SetLength( 0, length );
	  la.mutations = (0 == quals) 
	    ? mismatches
	    //	    : CountMismatches(length, S, q_start, look, startx);
	    : MismatchCount<BaseExactMatch>(S,look.Bases(),q_start,
					    startx-look.BasesStart(),length);

	  // Update.

	  if ( mismatches < best_errors[id] ) {    
	    second_best_errors[id] = best_errors[id];
	    best_errors[id] = mismatches;
	    best_align[id] = la;    
	  }
	  else if ( mismatches == best_errors[id] ) {    
	    if ( la != best_align[id] )
	      second_best_errors[id] = best_errors[id];    
	  }
	  if ( best_errors[id] <= best_prox && 
	       second_best_errors[id] - best_errors[id] <= best_prox ) {    
	    goto done_with_this_read;    
	  }    
	}    
      }
    
    done_with_this_read: continue;    
    }    
    if (timers) (*timers)[1]->Stop();
  }

  // Finish up.

  if ( unique_aligned == 0 ) aligns.clear( );
  for ( int id = 0; id < nqueries; id++ ) {    
    if ( ( max_errors<0 || best_errors[id] <= max_errors ) &&
	 second_best_errors[id] - best_errors[id] > best_prox )
      if ( unique_aligned == 0 ) aligns.push_back( best_align[id] );
      else (*unique_aligned)[id] = True;    
  }    
}
