/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/* 
   Program: ProvisionalEdits

   Given a <.fastb> file of reads, and the table T of nonunique k-mers in
   them (together with their multiplicities, as produced by
   <FindKmerFrequencies>), provisionally edit the reads as follows.
   
   Scan each read R starting at the left.  Take the k-mer x starting
   at the given position, and binary search T.  If x is in T and, advance
   to the next k-mer in R.  Otherwise, look up all 3k of the point mutations
   of x in T, and if exactly one is present in T, edit R to incorporate
   the corresponding correction, and advance to the kmer which starts
   after the changed position.  Then repeat the process, going
   backwards.

   The name "provisional" is outdated -- right now (04-06-07) this is the main
   error correction method.

   Most of the actual implementation is in <MakeProvisionalChanges>.

   In group: Edits
*/

#ifndef FORCE_DEBUG
  #define NDEBUG
#endif

#include "MainTools.h"

#include "math/Functions.h"
#include "ParseSet.h"
#include "Qualvector.h"
#include "KmerShape.h"
#include "kmer_freq/KmerShortMap.h"
#include "kmer_freq/MakeProvisionalChanges.h"
#include "paths/BaseErrorProb.h"

int main( int argc, char *argv[] )
{
  RunTime( );
  
  BeginCommandArguments;
  CommandArgument_String(PRE);
  CommandArgument_String(DATA); 
  CommandArgument_String(RUN); 
  CommandArgument_String(K);
  CommandArgument_String(READS_IN);
  CommandArgument_String_OrDefault(QUALS_IN, "");
  CommandArgument_String_OrDefault(READS_OUT, "");
  CommandArgument_String_OrDefault(READ_ID,"");
  CommandArgument_String_OrDefault(STRONG_IN, "")
  CommandArgument_String_OrDefault(ERROR_TABLE_NAME, "");
  CommandArgument_Int_OrDefault(MAX_ERRORS, 1);
  CommandArgument_Int_OrDefault(MAX_ERROR_ENTRIES, 0);
  CommandArgument_Int_OrDefault(ERROR_PROB_CUTOFF, 10);
  CommandArgument_Bool_OrDefault(ERROR_PEEK_AHEAD, True);
  
  EndCommandArguments; 

  vec<KmerShapeId> Ks;
  ParseKmerShapeIdSet( K, Ks );

  String run_dir = PRE + "/" + DATA + "/" + RUN;

  // Load protected read file.
  vec<Bool> Protected;
  if ( IsRegularFile(run_dir + "/reads.protected.orig") )
    BinaryRead3(run_dir + "/reads.protected.orig" , Protected );

  // Load Probability Distribution Table
  BaseErrorProbProfile probProfile;
  if (ERROR_TABLE_NAME != "")
    probProfile = BaseErrorProbProfile(PRE + "/" + ERROR_TABLE_NAME);

  // Load reads
  vecbasevector allReads( run_dir + "/" + READS_IN );

  // Load Quality Score (optional)
  vecqualvector quals;
  if (QUALS_IN != "")
    quals =  vecqualvector( run_dir + "/" + QUALS_IN );
  
  vec<int> readIds;
  vecbasevector* pReadsToCorrect = 0;
  if ( READ_ID.empty() )
    pReadsToCorrect = &allReads;
  else {
    ParseIntSet( READ_ID, readIds );
    pReadsToCorrect = new vecbasevector;
    pReadsToCorrect->SparseRead( run_dir + "/" + READS_IN, readIds );
  }

  
  // Load strong kmer table
  String strongKmersFileBase = run_dir + "/" + READS_IN + "." + "strong" + ".k";
  if (STRONG_IN != "")
    strongKmersFileBase =  run_dir + "/" + STRONG_IN + ".k";
  vec<KmerShortMap*> strongTablePtrs;
  for ( unsigned int i = 0; i < Ks.size(); ++i ) {
    strongTablePtrs.push_back( new KmerShortMap(Ks[i],
						strongKmersFileBase+ToString(Ks[i]) ) );
  }
  
  cout << "Data loaded." << endl;

  bool verbose = ( readIds.size() == 1 );
 
  // Perform Error Correction
  MakeProvisionalChanges( *pReadsToCorrect, quals, strongTablePtrs, verbose,
			  probProfile, MAX_ERRORS, MAX_ERROR_ENTRIES,
			  ERROR_PROB_CUTOFF, ERROR_PEEK_AHEAD);
    
  for ( unsigned int i = 0; i < strongTablePtrs.size(); ++i )
    delete strongTablePtrs[i];

  // Revert protected reads.

  if ( READ_ID.empty() && !Protected.empty() )
  {    vecbasevector orig_reads( run_dir + "/" + READS_IN );
       for ( int id = 0; id < Protected.isize( ); id++ )
       {    if ( Protected[id] ) (*pReadsToCorrect)[id] = orig_reads[id];    }    }

  // Write corrected reads.
  
  if ( READS_OUT != "" ) pReadsToCorrect->WriteAll( run_dir + "/" + READS_OUT );

  return 0;
}
