/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef DIGRAPH_TEMPLATE_H
#define DIGRAPH_TEMPLATE_H

#include "CoreTools.h"
#include "Equiv.h"
#include "Set.h"
#include "graph/Digraph.h"

template<class E> void digraphE<E>::Initialize( 
     const digraphE& g, const vec<vrtx_t>& v )
{    from_.resize( v.size( ) ), to_.resize( v.size( ) );
     from_edge_obj_.resize( v.size( ) ), to_edge_obj_.resize( v.size( ) );
     int edgecount = 0;
     for ( int i = 0; i < v.isize( ); i++ )
     {    vrtx_t x = v[i];
          for ( int j = 0; j < g.From(x).isize( ); j++ )
          {    vrtx_t y = g.From(x)[j];
               int i2 = Position( v, y );
               if ( i2 < 0 ) continue;
               from_[i].push_back(i2);    
               to_[i2].push_back(i);
               from_edge_obj_[i].push_back(edgecount);
               to_edge_obj_[i2].push_back(edgecount);
               ++edgecount;    }    }
     edges_.reserve(edgecount);
     for ( int i = 0; i < v.isize( ); i++ )
     {    vrtx_t x = v[i];
          for ( int j = 0; j < g.From(x).isize( ); j++ )
          {    vrtx_t y = g.From(x)[j];
               int i2 = Position( v, y );
               if ( i2 < 0 ) continue;
               edges_.push_back( g.EdgeObjectByIndexFrom( x, j ) );    }    }
     for ( int i = 0; i < v.isize( ); i++ )
     {    SortSync( from_[i], from_edge_obj_[i] );
          SortSync( to_[i], to_edge_obj_[i] );    }    }

template<class E> digraphE<E>::digraphE( const digraphE& g, const vec<vrtx_t>& v )
{    Initialize( g, v );    }

template<class E> digraphE<E>::digraphE( 
     const digraphE& g, const vec< vec<int> >& C )
{    int nedges = 0;
     for ( int i = 0; i < C.isize( ); i++ )
          nedges += C[i].size( );
     edges_.reserve(nedges);
     vec<vrtx_t> to_left, to_right;
     g.ToLeft(to_left), g.ToRight(to_right);
     for ( int i = 0; i < C.isize( ); i++ )
     {    for ( int j = 0; j < C[i].isize( ); j++ )
               edges_.push_back( g.EdgeObject( C[i][j] ) );    }
     for ( int pass = 1; pass <= 2; pass++ )
     {    int nverts = 0, nedges = 0;
          for ( int i = 0; i < C.isize( ); i++ )
          {    static vec<vrtx_t> verts;
               verts.clear( );
               for ( int j = 0; j < C[i].isize( ); j++ )
                    verts.push_back( to_left[ C[i][j] ], to_right[ C[i][j] ] );
               UniqueSort(verts);
               if ( pass == 2 )
               {    for ( int j = 0; j < C[i].isize( ); j++ )
                    {    vrtx_t v = BinPosition( verts, to_left[ C[i][j] ] );
                         vrtx_t w = BinPosition( verts, to_right[ C[i][j] ] );
                         from_[ nverts + v ].push_back( nverts + w );
                         to_[ nverts + w ].push_back( nverts + v );
                         from_edge_obj_[ nverts + v ].push_back(nedges + j);
                         to_edge_obj_[ nverts + w ].push_back(nedges + j);    }    }
               nverts += verts.size( );
               nedges += C[i].size( );    }
          if ( pass == 1 )
          {    from_.resize(nverts), to_.resize(nverts);
               from_edge_obj_.resize(nverts), to_edge_obj_.resize(nverts);    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    SortSync( from_[v], from_edge_obj_[v] );
          SortSync( to_[v], to_edge_obj_[v] );    }    }

template<class E> digraphE<E>::digraphE( const digraphE& g, int n )
{    equiv_rel e;
     g.ComponentRelation(e);
     vec<vrtx_t> reps, o;
     e.OrbitRepsAlt(reps);
     ForceAssertLt( n, reps.isize( ) );
     e.Orbit( reps[n], o );
     Initialize( g, o );    }

template<class E> void BinaryWrite( int fd, const digraphE<E>& g )
{    BinaryWrite( fd, (const digraph&) g );
     int n = g.N( );
     for ( int v = 0; v < n; v++ )
     {    BinaryWrite( fd, g.from_edge_obj_[v] );
          BinaryWrite( fd, g.to_edge_obj_[v] );    }
     int e = g.edges_.size( );
     WriteBytes( fd, &e, sizeof(int) );
     for ( int i = 0; i < e; i++ )
          BinaryWrite( fd, g.EdgeObject(i) );    }

template<class E> void BinaryRead( int fd, digraphE<E>& g )
{    BinaryRead( fd, (digraph&) g );
     int n = g.N( );
     g.from_edge_obj_.resize(n), g.to_edge_obj_.resize(n);
     for ( int v = 0; v < n; v++ )
     {    BinaryRead( fd, g.from_edge_obj_[v] );
          BinaryRead( fd, g.to_edge_obj_[v] );    }
     int e;
     ReadBytes( fd, &e, sizeof(int) );
     g.edges_.resize(e);
     for ( int i = 0; i < e; i++ )
          BinaryRead( fd, g.EdgeObjectMutable(i) );    }

template<class E> void digraphE<E>::EdgeEquivConstructor( 
     const vec<E>& edges, const equiv_rel& e )
{    edges_ = edges;
     int ne = edges.size( );
     vec<int> reps;
     e.OrbitReps(reps);
     int nv = 2 * reps.isize( );
     to_edge_obj_.resize(nv);
     from_edge_obj_.resize(nv);
     to_.resize(nv);
     from_.resize(nv);
     for ( int i = 0; i < reps.isize( ); i++ )
     {    static vec<int> o;
          e.Orbit( reps[i], o );
          for ( int j = 0; j < o.isize( ); j++ )
          {    from_[ 2*i ].push_back( 2*i + 1 );
               from_edge_obj_[ 2*i ].push_back( o[j] );
               to_[ 2*i + 1 ].push_back( 2*i );
               to_edge_obj_[ 2*i + 1 ].push_back( o[j] );    }    }    }

template<class E> digraphE<E>::digraphE( const vec<E>& edges, const equiv_rel& e )
{    EdgeEquivConstructor( edges, e );    }

template<class E> digraphE<E>::digraphE( const vec<E>& edges, 
     const ConstructorBehavior constructor_type )
{    edges_ = edges;
     int ne = edges.size( );
     int nv = ( constructor_type == EDGES_SEPARATE ? ne * 2 : ne + 1 );
     to_edge_obj_.resize(nv);
     from_edge_obj_.resize(nv);
     to_.resize(nv);
     from_.resize(nv);
     if ( constructor_type == EDGES_SEPARATE )
     {    for ( int i = 0; i < ne; i++ )
          {    from_[ 2*i ].push_back( 2*i + 1 );
               from_edge_obj_[ 2*i ].push_back(i);
               to_[ 2*i + 1 ].push_back( 2*i );
               to_edge_obj_[ 2*i + 1 ].push_back(i);    }    }
     else if ( constructor_type == EDGES_IN_LINE )
     {    for ( int i = 0; i <= ne; i++ )
          {    if ( i < ne )
               {    from_[i].push_back(i+1);
                    from_edge_obj_[i].push_back(i);    }
               if ( i > 0 ) 
               {    to_[i].push_back(i-1);
                    to_edge_obj_[i].push_back(i-1);    }    }    }
     else ForceAssert( 0 == 1 );    }

template<class E> void digraphE<E>::Used( vec<Bool>& used )
{    used.resize_and_set( edges_.size( ), False );
     for ( int i = 0; i < N( ); i++ )
     {    for ( int j = 0; j < to_edge_obj_[i].isize( ); j++ )
               used[ to_edge_obj_[i][j] ] = True;    }    }

template<class E> void digraphE<E>::JoinEdges( vrtx_t x, const E& e )
{    ForceAssert( from_[x].size( ) == 1 && to_[x].size( ) == 1 );
     vrtx_t v = to_[x][0], w = from_[x][0];
     ForceAssert( x != v || x != w );
     from_[x].clear( ), from_edge_obj_[x].clear( );
     to_[x].clear( ), to_edge_obj_[x].clear( );
     for ( int i = 0; i < from_[v].isize( ); i++ )
     {    if ( from_[v][i] == x )
          {    from_[v].erase( from_[v].begin( ) + i );
               from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + i );
               break;    }    }
     for ( int i = 0; i < to_[w].isize( ); i++ )
     {    if ( to_[w][i] == x )
          {    to_[w].erase( to_[w].begin( ) + i );
               to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + i );
               break;    }    }
     AddEdge( v, w, e );    }

template<class E> void digraphE<E>::RemoveUnneededVertices( )
{    for ( int i = 0; i < N( ); i++ )
     {    if ( From(i).size( ) == 1 && To(i).size( ) == 1 && From(i)[0] != i )
          {    E p = EdgeObjectByIndexTo( i, 0 );
               p.Append( EdgeObjectByIndexFrom( i, 0 ) );
               JoinEdges( i, p );    }    }
     RemoveEdgelessVertices( );    }

template<class E> void digraphE<E>::RemoveEdgelessVertices( )
{    vec<Bool> remove( N( ), False );
     vec<vrtx_t> new_vertex_id( N( ), -1 );
     vrtx_t id = 0;
     for ( int i = 0; i < N( ); i++ )
     {    if ( from_[i].empty( ) && to_[i].empty( ) ) remove[i] = True;
          else 
          {    new_vertex_id[i] = id;
               ++id;    }    }
     EraseIf( from_, remove ), EraseIf( from_edge_obj_, remove );
     EraseIf( to_, remove ), EraseIf( to_edge_obj_, remove );
     for ( int i = 0; i < N( ); i++ )
     {    for ( int j = 0; j < from_[i].isize( ); j++ )
               from_[i][j] = new_vertex_id[ from_[i][j] ];
          for ( int j = 0; j < to_[i].isize( ); j++ )
               to_[i][j] = new_vertex_id[ to_[i][j] ];    }    }

template<class E> void digraphE<E>::Reverse( )
{    for ( vrtx_t i = 0; i < N( ); i++ )
     {    swap( from_[i], to_[i] );
          swap( from_edge_obj_[i], to_edge_obj_[i] );    }    }

template<class E> void digraphE<E>::ReverseComponent( vrtx_t x )
{    equiv_rel e( N( ) );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < from_[v].isize( ); i++ )
          {    vrtx_t w = from_[v][i];
               e.Join( v, w );    }    }
     vec<vrtx_t> o;
     e.Orbit( x, o );
     for ( int j = 0; j < o.isize( ); j++ )
     {    vrtx_t i = o[j];
          swap( from_[i], to_[i] );
          swap( from_edge_obj_[i], to_edge_obj_[i] );    }    }

template<class E> void digraphE<E>::ReorderVertices( const vec<vrtx_t>& new_order )
{    ForceAssertEq( new_order.isize( ), N( ) );
     vec<vrtx_t> order_new( N( ) );
     for ( vrtx_t i = 0; i < N( ); i++ )
          order_new[ new_order[i] ] = i;
     PermuteVec( from_, order_new );
     PermuteVec( from_edge_obj_, order_new );
     PermuteVec( to_, order_new );
     PermuteVec( to_edge_obj_, order_new );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
               from_[v][j] = order_new[ from_[v][j] ];
          for ( int j = 0; j < to_[v].isize( ); j++ )
               to_[v][j] = order_new[ to_[v][j] ];
          SortSync( from_[v], from_edge_obj_[v] );
          SortSync( to_[v], to_edge_obj_[v] );    }    }

template<class E> void digraphE<E>::ReorderComponents( const vec<component_id_t>& new_order )
{    equiv_rel e( N( ) );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < from_[v].isize( ); i++ )
          {    vrtx_t w = from_[v][i];
               e.Join( v, w );    }    }
     vec<vrtx_t> reps;
     for ( vrtx_t v = 0; v < N( ); v++ )
          if ( e.Representative(v) ) reps.push_back(v);
     ForceAssertEq( new_order.size( ), reps.size( ) );
     vec<vrtx_t> new_vertex_order;
     for ( int i = 0; i < reps.isize( ); i++ )
     {    vrtx_t v = reps[ new_order[i] ];
          static vec<vrtx_t> o;
          e.Orbit( v, o );
          new_vertex_order.append(o);    }
     ReorderVertices(new_vertex_order);    }

template<class E> void digraphE<E>::Append( const digraphE<E>& D )
{    int nedges = edges_.size( );
     edges_.append( D.edges_ );
     int nvertices = from_.size( );
     from_.append( D.from_ );
     to_.append( D.to_ );
     from_edge_obj_.append( D.from_edge_obj_ );
     to_edge_obj_.append( D.to_edge_obj_ );
     for ( vrtx_t i = nvertices; i < N( ); i++ )
     {    for ( int j = 0; j < from_[i].isize( ); j++ )
          {    from_[i][j] += nvertices;
               from_edge_obj_[i][j] += nedges;    }
          for ( int j = 0; j < to_[i].isize( ); j++ )
          {    to_[i][j] += nvertices;
               to_edge_obj_[i][j] += nedges;    }    }    }

template<class E>
void digraphE<E>::SplitEdge( vrtx_t v, int j, const E& e1, const E& e2 )
{    int n = N( );
     int ne = edges_.size( );
     edges_.push_back( e1, e2 );
     vrtx_t w = from_[v][j];
     int we = from_edge_obj_[v][j];
     int i = InputFromOutputTo( v, j );
     from_[v].erase( from_[v].begin( ) + j );
     from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + j );
     to_[w].erase( to_[w].begin( ) + i );
     to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + i );
     from_[v].push_back(n), from_edge_obj_[v].push_back(ne);
     vec<vrtx_t> nfrom, nto;
     vec<int> nfrom_edge_obj, nto_edge_obj;
     nfrom.push_back(w), nfrom_edge_obj.push_back(ne+1);
     nto.push_back(v), nto_edge_obj.push_back(ne);
     from_.push_back(nfrom), to_.push_back(nto);
     from_edge_obj_.push_back(nfrom_edge_obj);
     to_edge_obj_.push_back(nto_edge_obj);
     for ( int u = 0; u < to_[w].isize( ); u++ )
     {    if ( to_[w][u] == v && we == to_edge_obj_[w][u] )
          {    to_.erase( to_.begin( ) + u );
               to_edge_obj_.erase( to_edge_obj_.begin( ) + u );
               break;    }    }
     to_[w].push_back(n), to_edge_obj_[w].push_back(ne+1);    }

template<class E> void digraphE<E>::Glue( const EmbeddedSubPath<E>& a,
     const EmbeddedSubPath<E>& b, const vec<int>& EE, const vec<int>& FF, 
     const digraphE<E>& c )
{    
     // Sanity check.

     ForceAssertGe( a.NVertices( ), 2 ); ForceAssertGe( b.NVertices( ), 2 );
     ForceAssertEq( NSharedEdges(a, b), 0 );
     ForceAssertEq( EE.isize( ), a.NVertices( ) ); 
     ForceAssertEq( FF.isize( ), b.NVertices( ) );
     vec<int> Esort = EE, Fsort = FF;
     Sort(Esort), Sort(Fsort);
     ForceAssert( Esort.UniqueOrdered( ) );
     ForceAssert( Fsort.UniqueOrdered( ) );
     ForceAssertEq( EE.front( ), 0 ); ForceAssertEq( EE.back( ), c.N( ) - 1 );
     ForceAssertEq( FF.front( ), 0 ); ForceAssertEq( FF.back( ), c.N( ) - 1 );

     // Delete edges appearing in a and b.

     for ( int i = 0; i < a.NVertices( ) - 1; i++ )
     {    vrtx_t v = a.Vertex(i), w = a.Vertex(i+1);
          int e = a.EdgeObjectIndexAbs(i);
          int ef = EdgeObjectIndexToFromIndex( v, e );
          int et = InputFromOutputTo( v, ef );
          from_[v].erase( from_[v].begin( ) + ef );
          from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + ef );
          to_[w].erase( to_[w].begin( ) + et );
          to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + et );    }
     for ( int i = 0; i < b.NVertices( ) - 1; i++ )
     {    vrtx_t v = b.Vertex(i), w = b.Vertex(i+1);
          int e = b.EdgeObjectIndexAbs(i);
          int ef = EdgeObjectIndexToFromIndex( v, e );
          int et = InputFromOutputTo( v, ef );
          from_[v].erase( from_[v].begin( ) + ef );
          from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + ef );
          to_[w].erase( to_[w].begin( ) + et );
          to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + et );    }

     // Attach c.

     int nvertices = N( );
     Append(c);
     for ( int i = 0; i < a.NVertices( ); i++ )
          TransferEdges( a.Vertex(i), EE[i] + nvertices );
     for ( int i = 0; i < b.NVertices( ); i++ )
          TransferEdges( b.Vertex(i), FF[i] + nvertices );    

     // If data implies that some vertices in c should be identified, do so.

     vec< vec<vrtx_t> > sources( c.N( ) );
     for ( int i = 0; i < a.NVertices( ); i++ )
          sources[ EE[i] ].push_back( a.Vertex(i) );
     for ( int i = 0; i < b.NVertices( ); i++ )
          sources[ FF[i] ].push_back( b.Vertex(i) );
     for ( int i = 0; i < c.N( ); i++ )
          Sort( sources[i] );
     for ( int i1 = 0; i1 < c.N( ); i1++ )
     {    for ( int i2 = i1 + 1; i2 < c.N( ); i2++ )
          {    if ( Meet( sources[i1], sources[i2] ) )
                    TransferEdges( i1 + nvertices, i2 + nvertices );    }    }    }

template<class E> void digraphE<E>::TransferEdges( vrtx_t v, vrtx_t w )
{    ForceAssert( v != w );

     // Change edges v --> v to edges w --> w.

     static vec<Bool> remove_from_v;
     remove_from_v.resize_and_set( from_[v].size( ), False );
     for ( int i = 0; i < from_[v].isize( ); i++ )
     {    if ( from_[v][i] == v )
          {    from_[w].push_back(w);
               from_edge_obj_[w].push_back( from_edge_obj_[v][i] );
               to_[w].push_back(w);
               to_edge_obj_[w].push_back( from_edge_obj_[v][i] );
               remove_from_v[i] = True;
               int j = InputFromOutputTo( v, i );
               to_[v].erase( to_[v].begin( ) + j );
               to_edge_obj_[v].erase( to_edge_obj_[v].begin( ) + j );    }    }
     EraseIf( from_[v], remove_from_v );
     EraseIf( from_edge_obj_[v], remove_from_v );
     SortSync( from_[w], from_edge_obj_[w] );
     SortSync( to_[w], to_edge_obj_[w] );

     // Change edges u --> v to edges u --> w.
     
     for ( int i = 0; i < to_[v].isize( ); i++ )
     {    vrtx_t u = to_[v][i];
          int j = InputToOutputFrom( v, i );
          from_[u][j] = w;
          SortSync( from_[u], from_edge_obj_[u] );    }

     // Change edges v --> x to edges w --> x.

     for ( int i = 0; i < from_[v].isize( ); i++ )
     {    vrtx_t x = from_[v][i];
          int j = InputFromOutputTo( v, i );
          to_[x][j] = w;
          SortSync( to_[x], to_edge_obj_[x] );    }

     // Do the rest.

     from_[w].append( from_[v] );
     from_edge_obj_[w].append( from_edge_obj_[v] );
     SortSync( from_[w], from_edge_obj_[w] );
     to_[w].append( to_[v] );
     to_edge_obj_[w].append( to_edge_obj_[v] );
     SortSync( to_[w], to_edge_obj_[w] );
     from_[v].clear( ), to_[v].clear( );
     from_edge_obj_[v].clear( ), to_edge_obj_[v].clear( );    }

#define VALID_FAILS(REASON)                \
{    cout << "\nDigraph is invalid.\n";    \
     cout << REASON << "\n";               \
     cout << "Abort.\n";                   \
     TracebackThisProcess( );    }

template<class E> void digraphE<E>::RemoveDuplicateEdges( )
{    for ( vrtx_t v = 0; v < N( ); v++ )
     {    static vec<Bool> remove;
          remove.resize_and_set( from_[v].size( ), False );
          for ( int j = 0; j < from_[v].isize( ); j++ )
          {    int k;
               for ( k = j + 1; k < from_[v].isize( ); k++ )
                    if ( from_[v][k] != from_[v][j] ) break;
               for ( int u1 = j; u1 < k; u1++ )
               {    if ( remove[u1] ) continue;
                    for ( int u2 = u1 + 1; u2 < k; u2++ )
                    {    if ( edges_[ from_edge_obj_[v][u1] ]
                                   == edges_[ from_edge_obj_[v][u2] ] )
                         {    remove[u2] = True;    }    }    }
               j = k - 1;    }
          for ( int i = 0; i < remove.isize( ); i++ )
          {    if ( remove[i] )
               {    vrtx_t w = from_[v][i];
	            int j = InputFromOutputTo( v, i );
                    to_[w].erase( to_[w].begin( ) + j );
                    to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + j );    }    }
          EraseIf( from_[v], remove );
          EraseIf( from_edge_obj_[v], remove );    }    }

template<class E> void digraphE<E>::DeleteEdgesAtVertex( vrtx_t v )
{    for ( int i = 0; i < from_[v].isize( ); i++ )
     {    vrtx_t w = from_[v][i];
          int j = InputFromOutputTo( v, i );
          if ( v == w ) continue;
          to_[w].erase( to_[w].begin( ) + j );
          to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + j );    }
     for ( int i = 0; i < to_[v].isize( ); i++ )
     {    vrtx_t w = to_[v][i];
          int j = InputToOutputFrom( v, i );
          if ( v == w ) continue;
          from_[w].erase( from_[w].begin( ) + j );
          from_edge_obj_[w].erase( from_edge_obj_[w].begin( ) + j );    }
     from_[v].clear( ), from_edge_obj_[v].clear( );
     to_[v].clear( ), to_edge_obj_[v].clear( );    }
                    
template<class E> void digraphE<E>::RemoveDeadEdgeObjects( )
{    vec<Bool> used;
     Used(used);
     int count = 0;
     vec<int> to_new_id( edges_.size( ) );
     for ( int i = 0; i < edges_.isize( ); i++ )
     {    if ( used[i] )
          {    if ( count != i ) edges_[count] = edges_[i];
               to_new_id[i] = count;
               ++count;    }    }
     edges_.resize(count);
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int i = 0; i < from_[v].isize( ); i++ )
               from_edge_obj_[v][i] = to_new_id[ from_edge_obj_[v][i] ];
          for ( int i = 0; i < to_[v].isize( ); i++ )
               to_edge_obj_[v][i] = to_new_id[ to_edge_obj_[v][i] ];    }    }

template<class E> void digraphE<E>::TestValid( ) const
{    if ( from_.size( ) != to_.size( ) )
          VALID_FAILS( "sizes of from_ and to_ are different" );
     if ( from_edge_obj_.size( ) != to_edge_obj_.size( ) )
          VALID_FAILS( "sizes of from_edge_obj_ and to_edge_obj_ are different" );
     if ( from_.size( ) != from_edge_obj_.size( ) )
          VALID_FAILS( "sizes of from_ and from_edge_obj_ are different" );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    if ( !from_[v].Ordered( ) )
          {    VALID_FAILS( "from_[" << v << "] is not ordered" );    }
          if ( !to_[v].Ordered( ) )
          {    VALID_FAILS( "to_[" << v << "] is not ordered" );    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    if ( from_[v].size( ) != from_edge_obj_[v].size( ) )
          {    VALID_FAILS( "sizes of from_[" << v << "] and "
                    << "from_edge_obj_[" << v << "] are different" );    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    if ( to_[v].size( ) != to_edge_obj_[v].size( ) )
          {    VALID_FAILS( "sizes of to_[" << v << "] and "
                    << "to_edge_obj_[" << v << "] are different" );    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    if ( !to_[v].Ordered( ) )
               VALID_FAILS( "to_[" << v << "] is not sorted" );    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
          {    vrtx_t w = from_[v][j];
               if ( w < 0 )
               {    VALID_FAILS( "There is an edge from " << v << " to \"vertex\" "
                         << w << "." );    }
               int ei = from_edge_obj_[v][j];
               Bool found = False;
               for ( int r = 0; r < to_[w].isize( ); r++ )
                    if ( to_[w][r] == v && to_edge_obj_[w][r] == ei ) found = True;
               if ( !found )
               {    VALID_FAILS( "There is an edge from " << v << " to " << w
                         << " in from_[" << v 
                         << "], but not in to_[" << w << "]." );    }    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < to_[v].isize( ); j++ )
          {    vrtx_t w = to_[v][j];
               if ( w < 0 )
               {    VALID_FAILS( "There is an edge from \"vertex\" " << w 
                         << " to " << v << "." );    }
               int ei = to_edge_obj_[v][j];
               Bool found = False;
               for ( int r = 0; r < from_[w].isize( ); r++ )
               {    if ( from_[w][r] == v && from_edge_obj_[w][r] == ei ) 
                         found = True;    }
               if ( !found )
               {    VALID_FAILS( "There is an edge from " << v << " to " << w
                         << " in to_[" << v 
                         << "], but not in from_[" << w 
                         << "]." );    }    }    }    }

template<class E>
void digraphE<E>::Initialize( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to,
          const vec<E>& edges, const vec< vec<int> >& to_edge_obj,
          const vec< vec<int> >& from_edge_obj )
{    digraph::Initialize( from, to );
     edges_ = edges; 
     to_edge_obj_ = to_edge_obj;
     from_edge_obj_ = from_edge_obj;
     int N = from.size( );
     ForceAssertEq( N, to_edge_obj.isize( ) );
     ForceAssertEq( N, from_edge_obj.isize( ) );
     vec<int> used( edges.size( ), 0 );
     for ( vrtx_t i = 0; i < N; i++ ) 
     {    ForceAssertEq( to_edge_obj[i].size( ), to[i].size( ) );
          ForceAssertEq( from_edge_obj[i].size( ), from[i].size( ) );
          for ( int j = 0; j < to_edge_obj[i].isize( ); j++ )
          {    int o = to_edge_obj[i][j];
               ForceAssertGe( o, 0 );
               ForceAssertLt( o, edges.isize( ) );
               ++used[o];
               vrtx_t w = i, v = to_[i][j];
               int wf = BinPosition( from[v], w );
               // The following assert won't do what we want if there are multiple
               // edges between two given vertices (in which case wf doesn't
               // make sense).
               // ForceAssertEq( o, from_edge_obj[v][wf] );    
                    }    }
     for ( int i = 0; i < used.isize( ); i++ )
          ForceAssertEq( used[i], 1 );    }

template<class E>
digraphE<E>::digraphE( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to,
          const vec<E>& edges, const vec< vec<int> >& to_edge_obj,
          const vec< vec<int> >& from_edge_obj ) : digraph(from, to)
{    Initialize( from, to, edges, to_edge_obj, from_edge_obj );    }

template<class E> Bool digraphE<E>::IsComplete( 
     const vec<vrtx_t>& vertices, const vec<int>& edges ) const
{    ForceAssert( vertices.UniqueOrdered( ) );
     ForceAssert( edges.UniqueOrdered( ) );
     for ( int u = 0; u < vertices.isize( ); u++ )
     {    vrtx_t v = vertices[u];
          for ( int j = 0; j < From(v).isize( ); j++ )
          {    vrtx_t w = From(v)[j];
               if ( !BinMember( vertices, w ) ) return False;
               int e = EdgeObjectIndexByIndexFrom( v, j );
               if ( !BinMember( edges, e ) ) return False;    }
          for ( int j = 0; j < To(v).isize( ); j++ )
          {    vrtx_t w = To(v)[j];
               if ( !BinMember( vertices, w ) ) return False;
               int e = EdgeObjectIndexByIndexTo( v, j );
               if ( !BinMember( edges, e ) ) return False;    }    }
     return True;    }

template<class E> void digraphE<E>::DualComponentRelation( 
     equiv_rel& e, const vec<Bool>& exclude ) const
{    e.Initialize( EdgeObjectCount( ) );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j1 = 0; j1 < To(v).isize( ); j1++ )
          {    int e1 = EdgeObjectIndexByIndexTo( v, j1 );
               if ( exclude.nonempty( ) && exclude[e1] ) continue;
               for ( int j2 = 0; j2 < From(v).isize( ); j2++ )
               {    int e2 = EdgeObjectIndexByIndexFrom( v, j2 );
                    if ( exclude.nonempty( ) && exclude[e2] ) continue;
                    e.Join(e1, e2);    }     }    }    }

template<class E> void digraphE<E>::Initialize( 
     const digraphE& g, const equiv_rel& e )
{    edges_ = g.edges_;
     vec<vrtx_t> reps;
     e.OrbitRepsAlt(reps);
     int nreps = reps.size( );
     vec<vrtx_t> to_reps( g.N( ) );
     for ( int i = 0; i < nreps; i++ )
     {    vec<vrtx_t> o;
          e.Orbit( reps[i], o );
          for ( int j = 0; j < o.isize( ); j++ )
               to_reps[ o[j] ] = i;    }
     from_.resize(nreps), to_.resize(nreps);
     from_edge_obj_.resize(nreps), to_edge_obj_.resize(nreps);
     int nedges = g.EdgeObjectCount( );
     vec<int> to_left_vertex(nedges, -1), to_right_vertex(nedges, -1);
     for ( vrtx_t w = 0; w < g.N( ); w++ )
     {    for ( int j = 0; j < g.To(w).isize( ); j++ )
          {    int m = g.EdgeObjectIndexByIndexTo( w, j );
               vrtx_t v = g.To(w)[j];
               to_left_vertex[m] = v, to_right_vertex[m] = w;    }    }
     for ( int m = 0; m < nedges; m++ )
     {    if ( to_left_vertex[m] < 0 || to_right_vertex[m] < 0 ) continue;
          vrtx_t v = to_reps[ to_left_vertex[m] ];
          vrtx_t w = to_reps[ to_right_vertex[m] ];
          from_[v].push_back(w), to_[w].push_back(v);
          from_edge_obj_[v].push_back(m), to_edge_obj_[w].push_back(m);    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    SortSync( from_[v], from_edge_obj_[v] );
          SortSync( to_[v], to_edge_obj_[v] );    }    }

template<class E> digraphE<E>::digraphE( 
     const digraphE& g, const equiv_rel& e ) : edges_( g.edges_ )
{    Initialize( g, e );    }

template<class E> digraphE<E>::digraphE( const vec<digraphE>& g )
{    Initialize(g);    }

template<class E> void digraphE<E>::Initialize( const vec<digraphE>& g )
{    for ( int i = 0; i < g.isize( ); i++ )
          Append( g[i] );    }

template<class E> void digraphE<E>::Initialize( const vec<digraphE>& g,
     const vec< pair< pair<int,vrtx_t>, pair<int,vrtx_t> > >& joins )
{    digraphE<E> G(g);
     equiv_rel e( G.N( ) );
     vec<int> start( g.isize( ) );
     start[0] = 0;
     for ( int i = 1; i < g.isize( ); i++ )
          start[i] = start[i-1] + g[i-1].N( );
     for ( int i = 0; i < joins.isize( ); i++ )
     {    vrtx_t v = start[ joins[i].first.first ] + joins[i].first.second;
          vrtx_t w = start[ joins[i].second.first ] + joins[i].second.second;
          e.Join( v, w );    }
     Initialize( G, e );    }

template<class E> digraphE<E>::digraphE( const vec<digraphE>& g,
     const vec< pair< pair<int,vrtx_t>, pair<int,vrtx_t> > >& joins )
{    Initialize( g, joins );    }

template<class E> Bool digraphE<E>::ThisClose( vrtx_t v, vrtx_t w, E d ) const
{    if ( d < 0 ) return False;
     if ( v == w ) return True;
     static set< pair<vrtx_t,E> > unprocessed, processed;
     unprocessed.clear( ), processed.clear( );
     unprocessed.insert( make_pair( v, 0 ) );
     while( !unprocessed.empty( ) )
     {    vrtx_t x = unprocessed.begin( )->first;
          E dx = unprocessed.begin( )->second;
          typename set< pair<vrtx_t,E> >::iterator u 
               = processed.lower_bound( make_pair( x, 0 ) );
          unprocessed.erase( unprocessed.begin( ) );
          if ( u != processed.end( ) && u->first == x )
          {    if ( u->second <= dx ) continue;
               processed.erase(u);    }
          processed.insert( make_pair( x, dx ) );
          for ( int j = 0; j < From(x).isize( ); j++ )
          {    vrtx_t y = From(x)[j];
               E dy = dx + EdgeObjectByIndexFrom( x, j );
               if ( dy > d ) continue;
               if ( y == w ) return True;
               typename set< pair<vrtx_t,E> >::iterator p 
                    = processed.lower_bound( make_pair( y, 0 ) );
               if ( p != processed.end( ) && p->first == y )
               {    if ( p->second <= dy ) continue;
                    processed.erase(p);    }
               typename set< pair<vrtx_t,E> >::iterator u 
                    = unprocessed.lower_bound( make_pair( y, 0 ) );
               if ( u != unprocessed.end( ) && u->first == y )
               {    if ( u->second <= dy ) continue;
                    unprocessed.erase(u);    }
               unprocessed.insert( make_pair( y, dy ) );    }    }
     return False;    }

template<class E> void digraphE<E>::ToLeft( vec<vrtx_t>& to_left ) const
{    to_left.resize( EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < N( ); i++ )
     {    for ( int j = 0; j < From(i).isize( ); j++ )
          {    int e = EdgeObjectIndexByIndexFrom( i, j );
               to_left[e] = i;    }    }    }

template<class E> void digraphE<E>::ToRight( vec<vrtx_t>& to_right ) const
{    to_right.resize( EdgeObjectCount( ) );
     for ( vrtx_t i = 0; i < N( ); i++ )
     {    for ( int j = 0; j < To(i).isize( ); j++ )
          {    int e = EdgeObjectIndexByIndexTo( i, j );
               to_right[e] = i;    }    }    }

template<class E>
void digraphE<E>::GetSuccessors( const vec<vrtx_t>& v, vec< pair<vrtx_t,E> >& from_v )
{    static set< pair<vrtx_t,E> > check, fromv;
     check.clear( ), fromv.clear( );
     for ( int i = 0; i < v.isize( ); i++ )
          check.insert( make_pair( v[i], 0 ) );
     while( !check.empty( ) )
     {    vrtx_t x = check.begin( )->first;
          E dx = check.begin( )->second;
          typename set< pair<vrtx_t,E> >::iterator u 
               = fromv.lower_bound( make_pair( x, 0 ) );
          check.erase( check.begin( ) );
          if ( u != fromv.end( ) && u->first == x )
          {    if ( u->second <= x ) continue;
               fromv.erase(u);    }
          fromv.insert( make_pair( x, dx ) );
          for ( int i = 0; i < From(x).isize( ); i++ )
          {    vrtx_t y = From(x)[i]; 
               E dy = dx + EdgeObjectByIndexFrom( x, i );
               typename set< pair<vrtx_t,E> >::iterator a 
                    = check.lower_bound( make_pair( y, 0 ) );
               if ( a != check.end( ) && a->first == y )
               {    if ( a->second <= dy ) continue;
                    check.erase(a);    }
               typename set< pair<vrtx_t,E> >::iterator b 
                    = fromv.lower_bound( make_pair( y, 0 ) );
               if ( b != fromv.end( ) && b->first == y )
               {    if ( b->second <= dy ) continue;
                    fromv.erase(b);    }
               check.insert( make_pair( y, dy ) );    }    }
     from_v.clear( );
     for ( typename set< pair<vrtx_t,E> >::iterator i = fromv.begin( ); 
          i != fromv.end( ); ++i )
     {    from_v.push_back(*i);    }    }

#endif
