/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2006) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/**
   File: Digraph.h

   This file contains the definition of class digraph, which represents a 
   finite directed graph, and digraphE<E>, which represents a finite directed
   graph whose edges are objects from class E.
*/

#ifndef DIGRAPH_H
#define DIGRAPH_H

#include "CoreTools.h"
#include "Bitvector.h"
#include "Equiv.h"
#include <map.h>
#include "math/Functions.h"
#include "system/TraceVal.h"

// Semantic Type: vrtx_t
// The id of a digraph vertex.  Should probably be called vrtx_id_t for consistency with other *id_t types,
// but vrtx_t is so ubiquitous that we use the shorter name.
typedef int vrtx_t;

//typedef TraceInt vrtx_t;

typedef pair<vrtx_t, vrtx_t> VertexPair;

// Semantic Type: edge_t
// The id of a digraphE edge object: an index into <digraphE::edges_>.
// Should probably be called edge_id_t but is called edge_t for consistency
// with vrtx_t.
typedef int edge_t;

// Semantic Type: component_id_t
// The id of a connected component of a digraph; typically, the position of this
// component's representative in the list of orbit represenatives returned by
// equiv_rel::OrbitRepsAlt() for the equiv_rel returned by
// digraph::ComponentRelation().  Note that this is not the vrtx_t component
// representative itself, but its index in the canonical list of representatives.
typedef int component_id_t;

/**
   Class: digraph

   A digraph may have multiple edges between two given vertices, but no edges
   from a vertex to itself.

   A digraph is stored as two vec< vec<vrtx_t> >'s, "to" and "from".  For each vertex
   v, to[v] is the sorted list of vertices which have an edge to v, and from[v]
   is the sorted list of vertices which have an edge from v.

*/
class digraph {

     public:

     digraph( ) { }
     digraph( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to );

     void Initialize( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to );

     void Clear( )
     {    from_.clear( ), to_.clear( );    }

     int N( ) const { return from_.size( ); } // number of vertices

     void CheckGoodVertex( vrtx_t v ) const
     {    AssertGe( v, 0 );
          AssertLt( v, N( ) );    }

     const vec< vec<vrtx_t> >& From( ) const
     {    return from_;    }
     const vec< vec<vrtx_t> >& To( ) const
     {    return to_;    }
     vec< vec<vrtx_t> >& FromMutable( )
     {    return from_;    }
     vec< vec<vrtx_t> >& ToMutable( )
     {    return to_;    }

     const vec<vrtx_t>& From( vrtx_t v ) const
     {    CheckGoodVertex(v);
          return from_[v];    }
     const vec<vrtx_t>& To( vrtx_t v ) const
     {    CheckGoodVertex(v);
          return to_[v];    }
     int FromSize( vrtx_t v ) const
     {    CheckGoodVertex(v);
          return from_[v].size( );    }
     int ToSize( vrtx_t v ) const
     {    CheckGoodVertex(v);
          return to_[v].size( );    }

     vec<vrtx_t>& FromMutable( vrtx_t v )
     {    CheckGoodVertex(v);
          return from_[v];    }
     vec<vrtx_t>& ToMutable( vrtx_t v )
     {    CheckGoodVertex(v);
          return to_[v];    }

     Bool Source( vrtx_t v ) const { return To(v).empty( ); }
     Bool Sink( vrtx_t v ) const { return From(v).empty( ); }

     void Sources( vec<vrtx_t>& v ) const;
     void Sinks( vec<vrtx_t>& v ) const;

     // Given the complete subgraph defined by a set of vertices S, find its sources
     // and sinks.  The set S must be sorted.

     void SubgraphSources( const vec<vrtx_t>& S, vec<vrtx_t>& v ) const;
     void SubgraphSinks( const vec<vrtx_t>& S, vec<vrtx_t>& v ) const;

     int SelfEdges( ) const
     {    int self = 0;
          for ( vrtx_t v = 0; v < N( ); v++ )
          {    for ( int i = 0; i < From(v).isize( ); i++ )
                    if ( From(v)[i] == v ) ++self;    }
          return self;    }

     // GetPredecessors: find all vertices which have a directed path to a vertex
     // in v.  Return a sorted list to_v.  GetSuccessors: go the other way.
     // See also another version of GetSuccessors for class digraphE.

     void GetPredecessors( const vec<vrtx_t>& v, vec<vrtx_t>& to_v );
     void GetSuccessors( const vec<vrtx_t>& v, vec<vrtx_t>& from_v );

     // Determine if the connected component defined by vertices sub has a directed
     // cycle.  Don't call this with anything other than a connected component.

     Bool HasCycle( const vec<vrtx_t>& sub ) const;

     // Return equivalence relation corresponding to connected components.

     void ComponentRelation( equiv_rel& e ) const;

     // Return number of connected components in a graph.

     int ConnectedComponents( ) const;

     // Find all vertices whose removal would disconnect the graph.  This code
     // assumes (but does not check) that the graph is connected.

     void CutPoints( vec<vrtx_t>& cuts ) const;


     // Remove all edges entering or exiting a given vertex.

     void DeleteEdgesAtVertex( vrtx_t v );

     // AllPaths: find all paths between two vertices which contain no duplicated
     // vertices.  Zero length paths (i.e. paths from a vertex to the same vertex)
     // are included.  Return answer as a list of lists, with each inner list a 
     // sequence of vertices.  This last can then be expanded out to reflect
     // alternative edges between adjacent vertices.
     //
     // If w = -1, instead find all paths from v to a sink.
     // If v = -1, instead find all paths from a source to w.
     //
     // If maxpaths >= 0, fail, returning False if greater than that many paths or 
     // partial paths found.
     //
     // If allow_self_loop and v = w (and >= 0), then loops from v to v are
     // allowed (but v cannot appear more than twice).

     Bool AllPaths( vrtx_t v, vrtx_t w, vec< vec<vrtx_t> >& paths, int maxpaths = -1,
          const Bool allow_self_loop = False,
          const int maxpushes = -1 ) const;

     friend void BinaryWrite( int fd, const digraph& g );
     friend void BinaryRead( int fd, digraph& g );

     // Create a representation of a given graph in DOT:
     // http://www.research.att.com/~erg/graphviz/info/lang.html.
     // The edge_colors and edge_labels arguments should be in bijective 
     // correspondence with from_.

     void DOT( ostream& out ) const;
     void DOT( ostream& out, const vec<String>& vertex_colors ) const;
     void DOT( ostream& out, const vec<String>& vertex_colors,
          const vec< vec<String> >& edge_colors ) const;
     void DOT( ostream& out, const vec< vec<String> >& edge_labels ) const;

     // Create a representation of a given graph in GraphML:
     // http://graphml.graphdrawing.org
     // The edge_labels arguments should be in bijective 
     // correspondence with from_.
     void WriteGraphML( ostream& out, const vec< vec<String> >& edge_labels ) const;
     
     protected:

     vec< vec<vrtx_t> > from_, to_;

};

template<class E> class EmbeddedSubPath; // forward declaration

template<class E> class digraphE;

template<class E> void BinaryWrite( int fd, const digraphE<E>& g );
template<class E> void BinaryRead( int fd, digraphE<E>& g );

/**
   Class Template: digraphE

   A digraphE<E> is stored as a digraph, plus a vec<E> edges_, plus indexing vectors
   (vec< vec<int> >'s to_edge_obj_ and from_edge_obj_) such that 
   if to_[w][i] = v, then to_edge_obj_[w][i] is the index in edges_ of the edge
   object for v --> w, and similarly for from_edge_obj_.

   The vec<E> edges_ is allowed to have unused entries (although not initially) and 
   there is a RemoveDeadEdgeObjects( ) function to eliminate them.

   Note that there are a bunch of member functions given here for digraphE<E>,
   that could also be implemented for class digraph.
*/
template<class E> class digraphE : public digraph {

     public:

     void TestValid( ) const; // incomplete test

     // Constructor 1: build the empty digraph.

     digraphE( ) { }

     // Constructor 2: build digraph from arbitrary digraph data.
     // (Note the insane order of the arguments: first from before to, then
     // to_edge_obj before from_edge_obj.)

     digraphE( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to,
          const vec<E>& edges, const vec< vec<edge_t> >& to_edge_obj,
          const vec< vec<edge_t> >& from_edge_obj );
     void Initialize( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to,
          const vec<E>& edges, const vec< vec<edge_t> >& to_edge_obj,
          const vec< vec<edge_t> >& from_edge_obj );

     // Constructor 3a: given a collection of edge objects, create a graph having
     // one edge and two vertices for each of the edge objects.
     // [ invoked by: digraphE( const vec<E>& edges, EDGES_SEPARATE ); ]
     // Constructor 3b: given a collection of edge objects, create a line graph out
     // of the edges.
     // [ invoked by: digraphE( const vec<E>& edges, EDGES_IN_LINE ); ]

     enum ConstructorBehavior {
       EDGES_SEPARATE,
       EDGES_IN_LINE 
     };

     digraphE( const vec<E>& edges, const ConstructorBehavior constructor_type  );

     // Constructor 4: given a collection of edge objects, and an equivalence 
     // relation on them, build a graph having two vertices per equivalence class,
     // with one edge between those two vertices for each member of the equivalence
     // class.

     digraphE( const vec<E>& edges, const equiv_rel& e );

     // Same as above, but not a constructor:

     void EdgeEquivConstructor( const vec<E>& edges, const equiv_rel& e );

     // Constructor 5: given a digraph, and given a list of vertex indices,
     // create the digraph having those vertices (with indices starting at 0,
     // but in the given order), and having all the edges that were between those
     // vertices.  Thus this is a "complete subgraph" constructor.  The order of the
     // edges is first by vertex v, and then by the order within From(v).

     digraphE( const digraphE& g, const vec<vrtx_t>& v );
     void Initialize( const digraphE& g, const vec<vrtx_t>& v );

     // Constructor 6: extract the nth connected component from another digraph.

     digraphE( const digraphE& g, int n );

     // Constructor 7: from another digraph and replacement edge objects.

     template<class F> digraphE( const digraphE<F>& g, const vec<E>& edges )
     {    from_ = g.From( );
          to_ = g.To( );
          from_edge_obj_ = g.FromEdgeObj( );
          to_edge_obj_ = g.ToEdgeObj( );
          edges_ = edges;    }

     // Constructor 8: from a given digraph and an equivalence relation on the
     // vertices.  The vertices of the constructed digraph are the equivalence
     // classes of vertices of the given graph.
     
     digraphE( const digraphE& g, const equiv_rel& e );
     void Initialize( const digraphE& g, const equiv_rel& e );

     // Constructor 9: form the disjoint union of a collection of digraphs.

     digraphE( const vec<digraphE>& g );
     void Initialize( const vec<digraphE>& g );

     // Constructor 10: from a collection of digraphs and a set of identifications
     // between vertices in their disjoint union, each of which is specified as
     // ( (g1,v1), (g2,v2) ) where g1, g2 refer to graphs and v1, v2 refer to
     // vertices on those graphs.

     digraphE( const vec<digraphE>& g, 
          const vec< pair< pair<int,vrtx_t>, pair<int,vrtx_t> > >& joins );
     void Initialize( const vec<digraphE>& g, 
          const vec< pair< pair<int,vrtx_t>, pair<int,vrtx_t> > >& joins );

     // Constructor 11: from a given digraph and a collection of subsets of its
     // edges, each of which is given the induced subgraph structure, when are then
     // merged into a disjoin union.

     digraphE( const digraphE& g, const vec< vec<int> >& C );

     // Delete the entire graph.

     void Clear( )
     {    from_.clear( ), to_.clear( );
          from_edge_obj_.clear( ), to_edge_obj_.clear( );
          edges_.clear( );    }

     // Return number of edge objects.

     int EdgeObjectCount( ) const { return edges_.size( ); }

     // A bunch of ways to access edge objects:

     const E& EdgeObject( int i ) const
     {    AssertGe( i, 0 );
          AssertLt( i, edges_.isize( ) );
          return edges_[i];    }

     E& EdgeObjectMutable( int i )
     {    AssertGe( i, 0 );
          AssertLt( i, edges_.isize( ) );
          return edges_[i];    }

     const E& EdgeObjectByIndexFrom( vrtx_t v, int j ) const
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, from_edge_obj_[v].isize( ) );
          return edges_[ from_edge_obj_[v][j] ];    }

     E& EdgeObjectByIndexFromMutable( vrtx_t v, int j )
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, from_edge_obj_[v].isize( ) );
          return edges_[ from_edge_obj_[v][j] ];    }

     edge_t EdgeObjectIndexByIndexFrom( vrtx_t v, int j ) const
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, from_edge_obj_[v].isize( ) );
          return from_edge_obj_[v][j];    }

     const E& EdgeObjectByIndexTo( vrtx_t v, int j ) const
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, to_edge_obj_[v].isize( ) );
          return edges_[ to_edge_obj_[v][j] ];    }

     E& EdgeObjectByIndexToMutable( vrtx_t v, int j )
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, to_edge_obj_[v].isize( ) );
          return edges_[ to_edge_obj_[v][j] ];    }

     edge_t EdgeObjectIndexByIndexTo( vrtx_t v, int j ) const
     {    CheckGoodVertex(v);
          AssertGe( j, 0 );
          AssertLt( j, to_edge_obj_[v].isize( ) );
          return to_edge_obj_[v][j];    }
   
     int EdgeObjectIndexToFromIndex( vrtx_t v, edge_t e ) const
     {    AssertGe( v, 0 );
          AssertLt( v, from_edge_obj_.isize( ) );
          for ( int i = 0; i < from_edge_obj_[v].isize( ); i++ )
               if ( from_edge_obj_[v][i] == e ) return i;
          Assert( 0 == 1 );
          return -1;    }

     int EdgeObjectIndexToToIndex( vrtx_t v, edge_t e ) const
     {    AssertGe( v, 0 );
          AssertLt( v, to_edge_obj_.isize( ) );
          for ( int i = 0; i < to_edge_obj_[v].isize( ); i++ )
               if ( to_edge_obj_[v][i] == e ) return i;
          Assert( 0 == 1 );
          return -1;    }

     const vec<edge_t>& FromEdgeObj( vrtx_t v ) const { return from_edge_obj_[v]; }
     const vec<edge_t>& ToEdgeObj( vrtx_t v ) const { return to_edge_obj_[v]; }
     vec<edge_t>& FromEdgeObjMutable( vrtx_t v ) { return from_edge_obj_[v]; }
     vec<edge_t>& ToEdgeObjMutable( vrtx_t v ) { return to_edge_obj_[v]; }

     const vec< vec<edge_t> >& FromEdgeObj( ) const { return from_edge_obj_; }
     const vec< vec<edge_t> >& ToEdgeObj( ) const { return to_edge_obj_; }
     vec< vec<edge_t> >& FromEdgeObjMutable( ) { return from_edge_obj_; }
     vec< vec<edge_t> >& ToEdgeObjMutable( ) { return to_edge_obj_; }

     const vec<E>& Edges( ) const { return edges_; }
     vec<E>& EdgesMutable( ) { return edges_; }

     // Given an edge referred to by To, find the corresponding edge in From,
     // and the other way.

     int InputToOutputFrom( vrtx_t w, int i ) const
     {    vrtx_t v = to_[w][i];
          edge_t ei = to_edge_obj_[w][i];
          for ( int j = 0; j < from_[v].isize( ); j++ )
               if ( from_edge_obj_[v][j] == ei ) return j;
          ForceAssert( 0 == 1 );
          return -1;    }
     int InputFromOutputTo( vrtx_t w, int i ) const
     {    vrtx_t v = from_[w][i];
          edge_t ei = from_edge_obj_[w][i];
          for ( int j = 0; j < to_[v].isize( ); j++ )
               if ( to_edge_obj_[v][j] == ei ) return j;
          ForceAssert( 0 == 1 );
          return -1;    }

     // Change an edge object.

     void SetEdgeObject( int i, const E& e )
     {    edges_[i] = e;    }

     void ChangeEdgeObjectFrom( vrtx_t v, int i, const E& e )
     {    int ne = edges_.size( );
          edges_.push_back(e);
          vrtx_t w = From(v)[i];
          int j = InputFromOutputTo( v, i );
          from_edge_obj_[v][i] = ne;
          to_edge_obj_[w][j] = ne;    }

     // Move an edge's endpoint, without changing the edge object itself.

     void GiveEdgeNewFromVx( edge_t edge_id, vrtx_t old_from_v, vrtx_t new_from_v ) {
       int i = Position( from_edge_obj_[old_from_v], edge_id );
       ForceAssert( i != -1 );
       vrtx_t w = from_[old_from_v][i];
       int j = Position( to_edge_obj_[w],edge_id );
       ForceAssert( j != -1 );
       to_[w][j] = new_from_v;
       from_[old_from_v].erase( from_[old_from_v].begin() + i );
       from_edge_obj_[old_from_v].erase( from_edge_obj_[old_from_v].begin() + i );
       from_[new_from_v].push_back(w);
       from_edge_obj_[new_from_v].push_back(edge_id);
       SortSync( to_[w], to_edge_obj_[w] );
       SortSync( from_[new_from_v], from_edge_obj_[new_from_v] );
     }

     void GiveEdgeNewToVx( edge_t edge_id, vrtx_t old_to_w, vrtx_t new_to_w ) {
       int j = Position( to_edge_obj_[old_to_w], edge_id );
       ForceAssert( j != -1 );
       vrtx_t v = to_[old_to_w][j];
       int i = Position( from_edge_obj_[v],edge_id );
       ForceAssert( i != -1 );
       from_[v][i] = new_to_w;
       to_[old_to_w].erase( to_[old_to_w].begin() + j );
       to_edge_obj_[old_to_w].erase( to_edge_obj_[old_to_w].begin() + j );
       to_[new_to_w].push_back(v);
       to_edge_obj_[new_to_w].push_back(edge_id);
       SortSync( from_[v], from_edge_obj_[v] );
       SortSync( to_[new_to_w], to_edge_obj_[new_to_w] );
     }

     // MinEdge: find the minimum length of an edge from v to w.  Assert if there is
     // no edge.  This only makes sense if Min(E,E) is defined.

     E MinEdge( vrtx_t v, vrtx_t w )
     {    E m = 0;
          Bool first = True;
          for ( int j = 0; j < From(v).isize( ); j++ )
          {    if ( From(v)[j] != w ) continue;
               if (first) m = EdgeObjectByIndexFrom( v, j );
               else m = Min( m, EdgeObjectByIndexFrom( v, j ) );
               first = False;    }
          ForceAssert( !first );
          return m;    }

     // MaxEdge: find the maximum length of an edge from v to w.  Assert if there is
     // no edge.  This only makes sense if Max(E,E) is defined.

     E MaxEdge( vrtx_t v, vrtx_t w )
     {    E M = 0;
          Bool first = True;
          for ( int j = 0; j < From(v).isize( ); j++ )
          {    if ( From(v)[j] != w ) continue;
               if (first) M = EdgeObjectByIndexFrom( v, j );
               else M = Max( M, EdgeObjectByIndexFrom( v, j ) );
               first = False;    }
          ForceAssert( !first );
          return M;    }

     // GetSuccessors: find all vertices which have a directed path from a vertex
     // in v, and return the minimum distance to each, as a sorted list.

     void GetSuccessors( const vec<vrtx_t>& v, vec< pair<vrtx_t,E> >& from_v );

     // Return equivalence relation e on the edges generated by the rule that if 
     // edge A follows edge B, then they are equivalent.  The classes of e are thus
     // the components of the directed line graph associated to the given graph.
     // If "exclude" is specified, it should have one entry per edge.  The excluded
     // edges are not used in creating the equivalence relation.

     void DualComponentRelation( equiv_rel& e, const vec<Bool>& exclude ) const;

     // ToLeft, ToRight: create vectors that map edge indices to vertex indices.

     void ToLeft( vec<vrtx_t>& to_left ) const;
     void ToRight( vec<vrtx_t>& to_right ) const;

     // Add vertices.

     void AddVertices( int nadd )
     {    int nvert = N( );
          from_.resize( nvert + nadd );
          to_.resize( nvert + nadd );
          from_edge_obj_.resize( nvert + nadd );
          to_edge_obj_.resize( nvert + nadd );    }

     // Add an edge from v to w.
     // Indices of edges from v to any x<=w are untouched, while
     // ones from v to x>w are incremented by 1; likewise for w's edges

     void AddEdge( vrtx_t v, vrtx_t w, const E& e )
     {    int n = EdgeObjectCount( );
          edges_.push_back(e);
	  int i = upper_bound( from_[v].begin(), from_[v].end(), w ) 
	    - from_[v].begin();
	  from_[v].insert( from_[v].begin()+i, w );
	  from_edge_obj_[v].insert( from_edge_obj_[v].begin()+i, n );
	  int j = upper_bound( to_[w].begin(), to_[w].end(), v )
	    - to_[w].begin();
	  to_[w].insert( to_[w].begin()+j, v );
	  to_edge_obj_[w].insert( to_edge_obj_[w].begin()+j, n );    }

     // Delete an edge.
     // Indices of edges <j are untouched, while those >j decrease by 1.
     // In particular, adding an edge and then deleting it is
     // guaranteed to leave the indices of other edges unchanged.

     void DeleteEdgeTo( vrtx_t w, int j )
     {    vrtx_t v = to_[w][j];
          int i = InputToOutputFrom( w, j );
          to_[w].erase( to_[w].begin( ) + j );
          to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + j );
          from_[v].erase( from_[v].begin( ) + i );
          from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + i );    }
     void DeleteEdgeFrom( vrtx_t v, int j )
     {    int w = from_[v][j];
          int i = InputFromOutputTo( v, j );
          from_[v].erase( from_[v].begin( ) + j );
          from_edge_obj_[v].erase( from_edge_obj_[v].begin( ) + j );
          to_[w].erase( to_[w].begin( ) + i );
          to_edge_obj_[w].erase( to_edge_obj_[w].begin( ) + i );    }

     // Delete a bunch of edges.  Input list must be sorted.

     void DeleteEdges( const vec<int>& to_delete )
     {    ForceAssert( to_delete.UniqueOrdered( ) );
          for ( vrtx_t v = 0; v < N( ); v++ )
          {    for ( int j = From(v).isize( ) - 1; j >= 0; j-- )
               {    int e = EdgeObjectIndexByIndexFrom( v, j );
                    if ( BinMember( to_delete, e ) )
                         DeleteEdgeFrom( v, j );    }    }    }

     // ThisClose: determine if there is a path from v to w whose sum of edge
     // objects is <= d.  This assumes that edge objects can be added and that
     // <= makes sense on them.  In the special case v = w, return True (so long
     // as d >= 0).

     Bool ThisClose( vrtx_t v, vrtx_t w, E d ) const;

     // Determine if the given vertices and edges (sorted lists) comprise the 
     // union of zero or more connected components.

     Bool IsComplete( const vec<vrtx_t>& vertices, const vec<edge_t>& edges ) const;

     // Remove all edges entering or exiting a given vertex.

     void DeleteEdgesAtVertex( vrtx_t v );

     // If two edges have the same stop and start, and are same as E-objects,
     // delete one.  This does not actually delete the objects.

     void RemoveDuplicateEdges( );

     // Determine which edge objects are used.

     void Used( vec<Bool>& used );

     // Eliminate unused edge objects.

     void RemoveDeadEdgeObjects( );

     // Remove vertices having no edges coming in or going out.

     void RemoveEdgelessVertices( );

     // Reverse the graph or the connected component containing a given vertex of 
     // it.  The current version does nothing to the edge objects.

     void Reverse( );
     void ReverseComponent( vrtx_t v );

     // Change order of vertices.

     void ReorderVertices( const vec<vrtx_t>& new_order );

     // Change order of components.

     void ReorderComponents( const vec<component_id_t>& new_order );

     // Distance: determine the lengths of all directed paths from vertex v to
     // vertex w, which are no longer than max_dist.  Return answer as D.
     // This is only implemented for the case where E = int.

     void Distance( vrtx_t v, vrtx_t w, int max_dist, vec<int>& D ) const;


     // AllPathsFixedLength: find all paths from v to w having length L.  This
     // is only implemented for the case where E = int.

     void AllPathsFixedLength( vrtx_t v, vrtx_t w, int L, vec< vec<vrtx_t> >& paths ) const;

     // AllPathsLengthRange: find all paths from v to w having length between L1
     // and L2.  This is only implemented for the case where E = int.  If max_paths
     // is specified and more than that many paths are found, return False.  If
     // max_loops is specified and the code loops more than that number of times,
     // return False.

     Bool AllPathsLengthRange( vrtx_t v, vrtx_t w, int L1, int L2, 
          const vec<vrtx_t>& to_right, vec< vec<vrtx_t> >& paths, 
          int max_paths = 0, int max_loops = 0 ) const;

     // SplitEdge( v, j, e1, e2 ): Replace edge from_[v][j] from v to w by a pair 
     // of edges 
     //
     //   e1    e2
     // v --> n --> w,
     //
     // where n is a new vertex [numbered N( ), if N( ) is called before SplitEdge].
     // This pushes e1 and then e2 onto edges_, and leaves an unused entry in 
     // edges_ where the original edge was.

     void SplitEdge( vrtx_t v, int j, const E& e1, const E& e2 );

     // JoinEdges( x, e ): Suppose vertex x has exactly one edge entering it (from
     // a vertex v) and exactly one edge exiting it (to a vertex w, v != w).  
     // Remove both edges and substitute a single edge from v to w, leaving x as a 
     // vertex having no edges coming in or going out, and also leaving unused 
     // entries in edges_.

     void JoinEdges( vrtx_t x, const E& e );

     // RemoveUnneededVertices: Apply JoinEdges to all applicable vertices x.  This 
     // assumes that class E has a member function Append( const E& d ), used here 
     // to compute the edge e in JoinEdges( x, e ).  This function removes edgeless 
     // vertices, but will leave unused entries in edges_.  If you want to remove
     // them, call RemoveDeadEdgeObjects.

     void RemoveUnneededVertices( );

     // Append(D): append another digraph, forming the disjoint union.  The
     // new vertices from D are numbered after the existing vertices.

     void Append( const digraphE<E>& D );

     // TransferEdges( v, w ): move all edges entering and exiting v, so that they
     // enter and exit w instead, leaving v without any edges entering and exiting 
     // it.  

     void TransferEdges( vrtx_t v, vrtx_t w );

     // ContractEdgeTo, ContractEdgeFrom: contract the specified edge.
     // Afterwards that edge no longer exists, and all other edges of
     // the far-end vertex are transfered to v.

     void ContractEdgeTo( vrtx_t w, int j ) 
     {    vrtx_t v = to_[w][j];
          DeleteEdgeTo( w, j );
	  if ( v != w ) TransferEdges( v, w );    }
     void ContractEdgeFrom( vrtx_t v, int j ) 
     {    vrtx_t w = from_[v][j];
          DeleteEdgeFrom( v, j );
	  if ( v != w ) TransferEdges( w, v );    }

     // Glue.  Given two sequences of edges in the graph, given by subpaths a and b,
     //
     //    e0       em-1            f0      fn-1
     // a0 --> ...  --> am,      b0 --> ... --> bn
     //
     // such that {e0,...,em-1} does not meet {f0,...,fn-1},
     //
     // plus a third sequence of edges not in the graph
     //
     //    f0       fp-1  
     // c0 --> ...  --> cp,  
     // 
     // and two injective maps:
     //
     // EE: {0,...m} --> {0,...p},     FF: {0,...,n} --> {0,...,p},
     //
     // such that EE(0) = 0, EE(m) = p, FF(0) = 0, and FF(n) = p,
     //
     // modify the graph by replacing the "a" and "b" sequences by the "c"
     // sequence, using the maps EE and FF to transfer edges involving any of
     // a0,...,am,b0,...,bn.  However, these vertices are left in the graph as
     // vertices having no edges entering or exiting them.
     //
     // Notes on inputs:
     // The digraph c gives c1,...,cp and the intervening edges.

     void Glue( const EmbeddedSubPath<E>& a, const EmbeddedSubPath<E>& b,
          const vec<int>& EE, const vec<int>& FF, const digraphE<E>& c );

     friend void BinaryWrite<E>( int fd, const digraphE<E>& g );
     friend void BinaryRead<E>( int fd, digraphE<E>& g );

     // Routines to compare two digraphs.

     friend Bool operator!=( const digraphE<E>& g1, const digraphE<E>& g2 )
     {    if ( g1.From( ) != g2.From( ) ) return True;
          if ( g1.To( ) != g2.To( ) ) return True;
          if ( g1.edges_ != g2.edges_ ) return True;
          if ( g1.to_edge_obj_ != g2.to_edge_obj_ ) return True;
          if ( g1.from_edge_obj_ != g2.from_edge_obj_ ) return True;
          return False;    }
     friend Bool operator==( const digraphE<E>& g1, const digraphE<E>& g2 )
     {    return !( g1 != g2 );    }
     friend void Compare( 
          ostream& out, const digraphE<E>& g1, const digraphE<E>& g2 )
     {    if ( g1.N( ) != g2.N( ) )
               cout << "first graph has " << g1.N( ) << " vertices but "
                    << "second graph has " << g2.N( ) << "\n";
          if ( g1.From( ) != g2.From( ) ) cout << "from_ not the same\n";
          if ( g1.To( ) != g2.To( ) ) cout << "to_ not the same\n";
          if ( g1.edges_ != g2.edges_ ) cout << "edges_ not the same\n";
          if ( g1.to_edge_obj_ != g2.to_edge_obj_ )
               cout << "to_edge_obj_ not the same\n";
          if ( g1.from_edge_obj_ != g2.from_edge_obj_ )
               cout << "from_edge_obj_ not the same\n";
          if ( g1 != g2 ) cout << "DIGRAPHS ARE NOT EQUAL\n";
          return;    }

     private:

     vec<E> edges_;
     vec< vec<edge_t> > to_edge_obj_, from_edge_obj_;

};

// Class Template: EmbeddedSubPath
//
// An EmbeddedSubPath is a directed path within a given <digraphE>.  It keeps the
// address of the digraph, so should not be used in situations where that address
// might change.
//
// Constructor: takes as input a sequence of consecutive edges
//
//    e0       em-1
// a0 --> ...  --> am,
//
// where the edge from ai to ai+1 is given by from_[ a[i] ][ e[i] ].  

template<class E> class EmbeddedSubPath {

     public:

     void TestValid( ) const
     {    ForceAssertEq( e_.isize( ), a_.isize( ) - 1 );
          for ( int u = 0; u < a_.isize( ) - 1; u++ )
          {    const vec<vrtx_t>& fr = D_->From( a_[u] );
               ForceAssertGe( e_[u], 0 );
               ForceAssertLt( e_[u], fr.isize( ) );
               ForceAssertEq( fr[ e_[u] ], a_[u+1] );    
               ForceAssertEq( D_->EdgeObjectIndexByIndexFrom( a_[u], e_[u] ),
                    esafe_[u] );    }    }

     // Repair e_ entries which are wrong because digraph has been edited.

     void Repair( )
     {    for ( int u = 0; u < e_.isize( ); u++ )
          {    if ( D_->EdgeObjectIndexByIndexFrom( a_[u], e_[u] ) != esafe_[u] )
                    e_[u] = D_->EdgeObjectIndexToFromIndex( 
                         a_[u], esafe_[u] );    }    }

     EmbeddedSubPath( ) { D_ = 0; }

     EmbeddedSubPath( const digraphE<E>& D, const vec<vrtx_t>& a, const vec<int>& e )
          : D_(&D), a_(a), e_(e)
     {    esafe_.resize( e.size( ) );
          for ( int i = 0; i < e.isize( ); i++ )
               esafe_[i] = D.EdgeObjectIndexByIndexFrom( a[i], e[i] );
          TestValid( );    }

     // The following constructor takes a vector of vertices.  There must be a
     // unique edge from each vertex to the next.

     EmbeddedSubPath( const digraphE<E>& D, const vec<vrtx_t>& a ) : D_(&D), a_(a)
     {    ForceAssertGt( a.size( ), 0 );
          e_.resize( a.isize( ) - 1 ), esafe_.resize( a.isize( ) - 1 );
          for ( int i = 0; i < e_.isize( ); i++ )
          {    int v = a[i], w = a[i+1];
               Bool found = False;
               for ( int j = 0; j < D.From(v).isize( ); j++ )
               {    if ( D.From(v)[j] == w )
                    {    ForceAssert( !found );
                         found = True;
                         e_[i] = j;
                         esafe_[i] = D.EdgeObjectIndexByIndexFrom( v, j );    }    }
               if ( !found )
               {    cout << "No edge found from vertex " << v << " to vertex "
                         << w << ".\n";
                    ForceAssert( 0 == 1 );    }    }    }

     int NVertices( ) const { return a_.size( ); }
     int NEdges( ) const { return e_.size( ); }

     vrtx_t Vertex( int i ) const { return a_[i]; }
     vrtx_t FirstVertex( ) const { return a_.front( ); }
     vrtx_t LastVertex( ) const { return a_.back( ); }

     const E& EdgeObject( int i ) const
     {    AssertGe( i, 0 );
          AssertLt( i, e_.isize( ) );
          return D_->EdgeObjectByIndexFrom( a_[i], e_[i] );    }

     int EdgeObjectIndexAbs( int i ) const
     {    return esafe_[i];    }

     int EdgeObjectIndex( int i ) const
     {    AssertGe( i, 0 );
          AssertLt( i, e_.isize( ) );
          return D_->EdgeObjectIndexByIndexFrom( a_[i], e_[i] );    }

     int EdgeObjectFromIndex( int i ) const
     {    AssertGe( i, 0 );
          AssertLt( i, e_.isize( ) );
          return e_[i];    }

     void SetVertex( int i, vrtx_t v ) { a_[i] = v; }
     void SetEdge( int i, int e ) 
     {    e_[i] = e; 
          esafe_[i] = D_->EdgeObjectIndexByIndexFrom( a_[i], e );    }

     // Add vertex and edge to left or right.

     void Prepend( vrtx_t a, int e )
     {    a_.push_front(a);
          e_.push_front(e);
          esafe_.push_front( D_->EdgeObjectIndexByIndexFrom(a,e) );    }
     void Append( vrtx_t a, int e )
     {    esafe_.push_back( D_->EdgeObjectIndexByIndexFrom(a_.back(),e) );
          a_.push_back(a);
          e_.push_back(e);    }

     friend int NSharedEdges( const EmbeddedSubPath& p1, const EmbeddedSubPath& p2 )
     {    vec<int> edges1, edges2;
          for ( int i = 0; i < p1.NEdges( ); i++ )     
               edges1.push_back( p1.D_->
                    EdgeObjectIndexByIndexFrom( p1.a_[i], p1.e_[i] ) );
          for ( int i = 0; i < p2.NEdges( ); i++ )     
               edges2.push_back( p2.D_->
                    EdgeObjectIndexByIndexFrom( p2.a_[i], p2.e_[i] ) );
          Sort(edges1), Sort(edges2);
          return Intersection( edges1, edges2 ).size( );    }

     friend int DuplicatedEdges( const EmbeddedSubPath& p )
     {    vec<int> edges;
          for ( int i = 0; i < p.NEdges( ); i++ )     
               edges.push_back( p.D_->
                    EdgeObjectIndexByIndexFrom( p.a_[i], p.e_[i] ) );
          UniqueSort(edges);
          return p.NEdges( ) - edges.isize( );    }

     private:

     const digraphE<E>* D_;
     vec<vrtx_t> a_;
     vec<int> e_;
     vec<int> esafe_;

};

#endif
