/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include <map.h>
#include "CoreTools.h"
#include "Bitvector.h"
#include "Equiv.h"
#include "Set.h"
#include "graph/Digraph.h"
#include "graph/DigraphTemplate.h"

template void digraphE<int>::ToLeft(vec<vrtx_t>&) const;
template void digraphE<int>::ToRight(vec<vrtx_t>&) const;

void digraph::Sources( vec<vrtx_t>& v ) const
{    v.clear( );
     for ( vrtx_t i = 0; i < N( ); i++ )
          if ( Source(i) ) v.push_back(i);    }

void digraph::Sinks( vec<vrtx_t>& v ) const
{    v.clear( );
     for ( vrtx_t i = 0; i < N( ); i++ )
          if ( Sink(i) ) v.push_back(i);    }

Bool digraph::HasCycle( const vec<vrtx_t>& sub ) const
{    ForceAssert( sub.nonempty( ) );
     static vec<vrtx_t> s;
     s = sub;
     Sort(s);
     const int white = 0, red = 1, black = 2;
     for ( int i = 0; i < s.isize( ); i++ )
     {    static vec<int> color;
     static vec<vrtx_t> reds;
          color.resize_and_set( s.size( ), white );
          reds.clear( );
          reds.push_back(i);
          color[i] = red;
          while( reds.nonempty( ) )
          {    vrtx_t j = reds.back( );
               color[j] = black;
               reds.resize( reds.isize( ) - 1 );
               vrtx_t v = s[j];
               for ( int t = 0; t < From(v).isize( ); t++ )
               {    vrtx_t w = From(v)[t];
                    int p = BinPosition( s, w );
                    ForceAssertGe( p, 0 );
                    if ( p == i ) return True;
                    if ( color[p] == white ) color[p] = red;    }    }    }
     return False;    }

void BinaryWrite( int fd, const digraph& g )
{    int n = g.from_.size( );
     WriteBytes( fd, &n, sizeof(int) );
     for ( int v = 0; v < n; v++ )
     {    BinaryWrite( fd, g.from_[v] );
          BinaryWrite( fd, g.to_[v] );    }    }

void BinaryRead( int fd, digraph& g )
{    int n;
     ReadBytes( fd, &n, sizeof(int) );
     g.from_.resize(n), g.to_.resize(n);
     for ( int v = 0; v < n; v++ )
     {    BinaryRead( fd, g.from_[v] );
          BinaryRead( fd, g.to_[v] );    }    }

void digraph::Initialize( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to )
{    ForceAssertEq( from.size( ), to.size( ) );
     int N = from.size( );
     from_ = from, to_ = to;
     for ( vrtx_t i = 0; i < N; i++ )
     {    for ( int j = 0; j < from[i].isize( ); j++ )
               CheckGoodVertex( from[i][j] );
          for ( int j = 0; j < to[i].isize( ); j++ )
               CheckGoodVertex( to[i][j] );
          ForceAssert( from[i].Ordered( ) );
          ForceAssert( to[i].Ordered( ) );    }
     for ( vrtx_t i = 0; i < N; i++ )
     {    for ( int j = 0; j < from[i].isize( ); j++ )
               ForceAssert( BinMember( to[ from[i][j] ], i ) );
          for ( int j = 0; j < to[i].isize( ); j++ )
               ForceAssert( BinMember( from[ to[i][j] ], i ) );    }    }

digraph::digraph( const vec< vec<vrtx_t> >& from, const vec< vec<vrtx_t> >& to )
{    Initialize( from, to );    }

void PrintStandardDOTHeader( ostream& out )
{    out << "digraph G {\n\n";
     out << "orientation=landscape;\n";
     out << "node [width=0.1,height=0.1,fontsize=10,shape=point];\n";
     out << "edge [fontsize=12];\n";    }

void digraph::DOT( ostream& out ) const
{    PrintStandardDOTHeader(out);
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
               out << v << " -> " << from_[v][j] << ";\n";    }
     out << "\n}\n";    }

void digraph::DOT( ostream& out, const vec<String>& vertex_colors ) const
{    PrintStandardDOTHeader(out);
     for ( int v = 0; v < vertex_colors.isize( ); v++ )
     {    if ( vertex_colors[v] != "" )
          {    out << v << " [color=" << vertex_colors[v] << "];\n";    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
               out << v << " -> " << from_[v][j] << ";\n";    }
     out << "\n}\n";    }

void digraph::DOT( ostream& out, const vec<String>& vertex_colors,
     const vec< vec<String> >& edge_colors ) const
{    PrintStandardDOTHeader(out);
     for ( int v = 0; v < vertex_colors.isize( ); v++ )
     {    if ( vertex_colors[v] != "" )
          {    out << v << " [color=" << vertex_colors[v] << "];\n";    }    }
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
          {    const String& color = edge_colors[v][j];
               out << v << " -> " << from_[v][j];
               if ( color != "" ) out << " [color=" << color << "]";
               out << ";\n";    }    }
     out << "\n}\n";    }

void digraph::DOT( ostream& out, const vec< vec<String> >& edge_labels ) const
{    PrintStandardDOTHeader(out);
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < from_[v].isize( ); j++ )
          {    vrtx_t w = from_[v][j];
               const String& label = edge_labels[v][j];
               String ename = "E_" + ToString(v) + "_" + ToString(w);
               out << "\n" << ename << " [label=\"" << label << "\",shape=box];\n";
               out << v << " -> " << ename << " [arrowhead=none];\n";
               out << ename << " -> " << w << ";\n";    }    }
     out << "\n}\n";    }


// Create a representation of a given graph in XML.
// http://graphml.graphdrawing.org
// The edge_labels arguments should be in bijective 
// correspondence with from_.
void digraph::WriteGraphML( ostream& out, const vec< vec<String> >& edge_labels ) const {
  out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    "<graphml xmlns=\"http://graphml.graphdrawing.org/xmlns\"  \n"
   "  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"  \n"
   "  xsi:schemaLocation=\"http://graphml.graphdrawing.org/xmlns \n"
   "  http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd\">  \n";
  out << "<graph id=\"G\" edgedefault=\"directed\">\n";

  out << "<key id=\"d0\" for=\"edge\"  attr.name=\"edgeName\" attr.type=\"string\" />\n";
  
  for ( vrtx_t v = 0; v < N(); v++ )
    out << "  <node id=\"n" << v << "\" />\n";

  for ( vrtx_t v = 0; v < N(); v++ ) {
    for ( int j = 0; j < from_[v].isize(); j++ ) {
      vrtx_t w = from_[v][j];
      out << "<edge source=\"n" << v << "\" target=\"n" << w << "\" >\n";
      out << "  <data key=\"d0\">" << edge_labels[v][j] << "</data>\n";
      out << "</edge>\n";
    }
  }
  out << "</graph>\n";
  
  out << "</graphml>\n";
}


void digraph::ComponentRelation( equiv_rel& e ) const
{    e.Initialize( N( ) );
     for ( vrtx_t v = 0; v < N( ); v++ )
     {    for ( int j = 0; j < From(v).isize( ); j++ )
          {    vrtx_t w = From(v)[j];
               e.Join(v, w);    }    }    }

int digraph::ConnectedComponents( ) const
{    equiv_rel e;
     ComponentRelation(e);
     return e.OrbitCount( );    }

void digraph::CutPoints( vec<vrtx_t>& cuts ) const
{    cuts.clear( );
     int components = ConnectedComponents();
     for ( vrtx_t i = 0; i < N( ); i++ )
     {    equiv_rel e( N( ) );
          for ( vrtx_t v = 0; v < N( ); v++ )
          {    if ( v == i ) continue;
               for ( int j = 0; j < From(v).isize( ); j++ )
               {    vrtx_t w = From(v)[j];
                    if ( w == i ) continue;
                    e.Join(v, w);    }    }
          if ( e.OrbitCount( ) > components + 1 ) cuts.push_back(i);    }    }

void digraph::GetPredecessors( const vec<vrtx_t>& v, vec<vrtx_t>& to_v )
{    set<vrtx_t> check, tov;
     for ( int i = 0; i < v.isize( ); i++ )
     {    check.insert( v[i] );
          tov.insert( v[i] );    }
     while( !check.empty( ) )
     {    int x = *check.begin( );
          check.erase( check.begin( ) );
          for ( int i = 0; i < To(x).isize( ); i++ )
          {    vrtx_t y = To(x)[i];
               if ( Member( tov, y ) ) continue;
               check.insert(y);
               tov.insert(y);    }    }
     to_v.clear( );
     for ( set<vrtx_t>::iterator i = tov.begin( ); i != tov.end( ); ++i )
          to_v.push_back(*i);    }

void digraph::GetSuccessors( const vec<vrtx_t>& v, vec<vrtx_t>& from_v )
{    set<vrtx_t> check, fromv;
     for ( int i = 0; i < v.isize( ); i++ )
     {    check.insert( v[i] );
          fromv.insert( v[i] );    }
     while( !check.empty( ) )
     {    vrtx_t x = *check.begin( );
          check.erase( check.begin( ) );
          for ( vrtx_t i = 0; i < From(x).isize( ); i++ )
          {    vrtx_t y = From(x)[i];
               if ( Member( fromv, y ) ) continue;
               check.insert(y);
               fromv.insert(y);    }    }
     from_v.clear( );
     for ( set<vrtx_t>::iterator i = fromv.begin( ); i != fromv.end( ); ++i )
          from_v.push_back(*i);    }

void digraph::SubgraphSources( const vec<vrtx_t>& S, vec<vrtx_t>& v ) const
{    ForceAssert( S.UniqueOrdered( ) );
     v.clear( );
     for ( int i = 0; i < S.isize( ); i++ )
     {    vrtx_t x = S[i];
          Bool source = True;
          for ( int j = 0; j < To(x).isize( ); j++ )
          {    vrtx_t y = To(x)[j];
               if ( BinMember( S, y ) )
               {    source = False;
                    break;    }    }
          if (source) v.push_back(x);    }    }

void digraph::SubgraphSinks( const vec<vrtx_t>& S, vec<vrtx_t>& v ) const
{    ForceAssert( S.UniqueOrdered( ) );
     v.clear( );
     for ( int i = 0; i < S.isize( ); i++ )
     {    vrtx_t x = S[i];
          Bool sink = True;
          for ( int j = 0; j < From(x).isize( ); j++ )
          {    vrtx_t y = From(x)[j];
               if ( BinMember( S, y ) )
               {    sink = False;
                    break;    }    }
          if (sink) v.push_back(x);    }    }

template<class E>
     void digraphE<E>::Distance( vrtx_t v, vrtx_t w, int max_dist, vec<int>& D ) const
{    static set< pair<vrtx_t,int> > unprocessed, processed;
     unprocessed.clear( ), processed.clear( );
     unprocessed.insert( make_pair( v, 0 ) );
     while( !unprocessed.empty( ) )
     {    vrtx_t x = unprocessed.begin( )->first;
          int dx = unprocessed.begin( )->second;
          unprocessed.erase( unprocessed.begin( ) );
          processed.insert( make_pair( x, dx ) );
          for ( int j = 0; j < From(x).isize( ); j++ )
          {    vrtx_t y = From(x)[j];
	        int dy = dx + EdgeObjectByIndexFrom( x, j );
               if ( dy > max_dist ) continue;
               if ( Member( processed, make_pair( y, dy ) ) ) continue;
               unprocessed.insert( make_pair( y, dy ) );    }    }
     D.clear( );
     for ( set< pair<vrtx_t,int> >::iterator i = processed.begin( ); 
          i != processed.end( ); ++i )
     {    vrtx_t x = i->first;
          int d = i->second;
          if ( x == w ) D.push_back(d);    }
     UniqueSort(D);    }

template 
     void digraphE<int>::Distance( vrtx_t v, vrtx_t w, int max_dist, vec<int>& D ) const;

Bool digraph::AllPaths( vrtx_t v, vrtx_t w, vec< vec<vrtx_t> >& paths, int maxpaths,
     const Bool allow_self_loop, const int maxpushes ) const
{    paths.clear( );
     set< vec<vrtx_t> > pathss;
     int pushes = 0;
     if ( v < 0 && w < 0 )
     {    vec<vrtx_t> sinks;
          Sinks(sinks);
          static vec< vec<vrtx_t> > paths0;
          for ( int j = 0; j < sinks.isize( ); j++ )
          {    Bool OK = AllPaths( -1, sinks[j], paths0, maxpaths );
               if ( !OK ) return False;
               paths.append(paths0);
               if ( maxpaths > 0 && paths.isize( ) > maxpaths ) return False;    }
          return True;    }
     set< pair< vec<vrtx_t>, set<vrtx_t> > > partials;
     vec<vrtx_t> just;
     set<vrtx_t> justs;

     // Case 1.

     if ( v >= 0 && w >= 0 )
     {
          // First find all the vertices up2 which are upstream of w.

          set<vrtx_t> up1, up2;
          up1.insert(w);
          while( !up1.empty( ) )
          {    vrtx_t y = *up1.begin( );
               up1.erase( up1.begin( ) );
               up2.insert(y);
               for ( int i = 0; i < To(y).isize( ); i++ )
               {    vrtx_t x = To(y)[i];
                    if ( !Member( up2, x ) ) up1.insert(x);    }    }
          if ( !Member( up2, v ) ) return True;

          // Now build the answer.

          just.push_back(v), justs.insert(v);
          partials.insert( make_pair( just, justs ) );
          while( !partials.empty( ) )
          {    pair< vec<vrtx_t>, set<vrtx_t> > p = *partials.begin( );
               partials.erase( partials.begin( ) );
               vrtx_t x = p.first.back( );
               Bool to_process = True;
               if ( x == w ) 
               {    pathss.insert( p.first );
                    if ( maxpaths > 0 && (int) pathss.size( ) > maxpaths ) 
                         return False;
                    to_process = False;
                    if ( allow_self_loop && p.first.size( ) == 1 )
                         to_process = True;    }
               if (to_process)
               {    for ( int i = 0; i < From(x).isize( ); i++ )
                    {    vrtx_t y = From(x)[i];
                         if ( allow_self_loop && v == w && y == v )
                         {    vec<vrtx_t> pf = p.first;
                              pf.push_back(v);
                              pathss.insert(pf);
                              if ( maxpaths > 0 && (int) pathss.size( ) > maxpaths ) 
                                   return False;
                              continue;    }
                         if ( !Member( up2, y ) || Member( p.second, y ) ) continue;
                         pair< vec<vrtx_t>, set<vrtx_t> > q = p;
                         q.first.push_back(y);
                         q.second.insert(y);
                         partials.insert(q);    
                         if ( maxpushes >= 0 && ++pushes >= maxpushes ) return False;
                         if ( maxpaths > 0 && (int) partials.size( ) > maxpaths ) 
                              return False;    }    }    }    }

     // Case 2.

     if ( w < 0 )
     {    just.push_back(v), justs.insert(v);
          partials.insert( make_pair( just, justs ) );
          while( !partials.empty( ) )
          {    pair< vec<vrtx_t>, set<vrtx_t> > p = *partials.begin( );
               partials.erase( partials.begin( ) );
               vrtx_t x = p.first.back( );
               if ( From(x).empty( ) )
               {    pathss.insert( p.first );
                    if ( maxpaths > 0 && (int) pathss.size( ) > maxpaths ) 
                         return False;    }
               else
               {    for ( int i = 0; i < From(x).isize( ); i++ )
                    {    vrtx_t y = From(x)[i];
                         if ( Member( p.second, y ) ) continue;
                         pair< vec<vrtx_t>, set<vrtx_t> > q = p;
                         q.first.push_back(y);
                         q.second.insert(y);
                         partials.insert(q);
                         if ( maxpushes >= 0 && ++pushes >= maxpushes ) return False;
                         if ( maxpaths > 0 && (int) partials.size( ) > maxpaths ) 
                              return False;    }    }    }    }

     // Case 3.

     if ( v < 0 )
     {    just.push_back(w), justs.insert(w);
          partials.insert( make_pair( just, justs ) );
          while( !partials.empty( ) )
          {    pair< vec<vrtx_t>, set<vrtx_t> > p = *partials.begin( );
               partials.erase( partials.begin( ) );
               vrtx_t y = p.first.front( );
               if ( To(y).empty( ) )
               {    pathss.insert( p.first );
                    if ( maxpaths > 0 && (int) pathss.size( ) > maxpaths ) 
                         return False;    }
               else
               {    for ( int i = 0; i < To(y).isize( ); i++ )
                    {    vrtx_t x = To(y)[i];
                         if ( Member( p.second, x ) ) continue;
                         pair< vec<vrtx_t>, set<vrtx_t> > q = p;
                         q.first.push_front(x);
                         q.second.insert(x);
                         partials.insert(q);
                         if ( maxpushes >= 0 && ++pushes >= maxpushes ) return False;
                         if ( maxpaths > 0 && (int) partials.size( ) > maxpaths ) 
                              return False;    }    }    }    }

     paths.reserve( pathss.size( ) );
     for ( set< vec<vrtx_t> >::iterator i = pathss.begin( ); i != pathss.end( ); ++i )
          paths.push_back(*i);
     return True;    }


template<class E> void digraphE<E>::AllPathsFixedLength( 
     vrtx_t v, vrtx_t w, int L, vec< vec<vrtx_t> >& paths ) const
{    ForceAssertGe( L, 0 );
     paths.clear( );
     vec<vrtx_t> to_right;
     ToRight(to_right);
     vec< vec<vrtx_t> > partials;
     partials.push_back( vec<vrtx_t>( ) );
     while( partials.nonempty( ) )
     {    static vec<vrtx_t> p;
          p = partials.back( );
          partials.pop_back( );
          int l = 0;
          for ( int i = 0; i < p.isize( ); i++ )
               l += EdgeObject( p[i] );
          vrtx_t vn = ( p.empty( ) ? v : to_right[ p.back( ) ] );
          if ( l == L ) 
          {    if ( vn == w ) paths.push_back(p);    }
          else
          {    for ( int j = 0; j < From(vn).isize( ); j++ )
               {    int e = EdgeObjectIndexByIndexFrom( vn, j );
                    if ( l + EdgeObject(e) > L ) continue;
                    static vec<vrtx_t> p2;
                    p2 = p;
                    p2.push_back(e);
                    partials.push_back(p2);    }    }    }    }

template void digraphE<int>::AllPathsFixedLength( 
     vrtx_t v, vrtx_t w, int L, vec< vec<vrtx_t> >& paths ) const;

template<class E> Bool digraphE<E>::AllPathsLengthRange( vrtx_t v, vrtx_t w, int L1, 
     int L2, const vec<vrtx_t>& to_right, vec< vec<vrtx_t> >& paths, int max_paths,
     int max_loops ) const
{    ForceAssertGe( L1, 0 );
     paths.clear( );
     if ( L2 < L1 ) return True;
     vec< vec<vrtx_t> > partials;
     partials.push_back( vec<vrtx_t>( ) );
     int loopcount = 0;
     while( partials.nonempty( ) )
     {    if ( max_loops > 0 && loopcount++ > max_loops ) return False;
          static vec<vrtx_t> p;
          p = partials.back( );
          partials.pop_back( );
          int l = 0;
          for ( int i = 0; i < p.isize( ); i++ )
               l += EdgeObject( p[i] );
          vrtx_t vn = ( p.empty( ) ? v : to_right[ p.back( ) ] );
          if ( l >= L1 && l <= L2 && vn == w ) 
          {    paths.push_back(p);
               if ( max_paths > 0 && paths.isize( ) > max_paths ) return False;    }
          if ( l < L2 )
          {    for ( int j = 0; j < From(vn).isize( ); j++ )
               {    int e = EdgeObjectIndexByIndexFrom( vn, j );
                    if ( l + EdgeObject(e) > L2 ) continue;
                    static vec<vrtx_t> p2;
                    p2 = p;
                    p2.push_back(e);
                    partials.push_back(p2);    }    }    }
     return True;    }

template Bool digraphE<int>::AllPathsLengthRange( vrtx_t v, vrtx_t w, int L1, 
     int L2, const vec<vrtx_t>& to_right, vec< vec<vrtx_t> >& paths, int max_paths,
     int max_loops ) const;

template void BinaryWrite( int, const digraphE<int>& );
template void BinaryRead( int, digraphE<int>& );

void digraph::DeleteEdgesAtVertex( vrtx_t v )
{    for ( int i = 0; i < from_[v].isize( ); i++ )
     {    vrtx_t w = from_[v][i];
          if ( v == w ) continue;
          for ( int j = to_[w].isize( ) - 1; j >= 0; j-- )
               if ( to_[w][j] == v ) to_[w].erase( to_[w].begin( ) + j );    }
     for ( int i = 0; i < to_[v].isize( ); i++ )
     {    vrtx_t w = to_[v][i];
          if ( v == w ) continue;
          for ( int j = from_[w].isize( ) - 1; j >= 0; j-- )
               if ( from_[w][j] == v ) from_[w].erase( from_[w].begin( ) + j );    }
     from_[v].clear( ), to_[v].clear( );    }

