// Copyright (c) 2000-2003 Whitehead Institute for Biomedical Research
// 


#ifndef SORT_KMERS_H
#define SORT_KMERS_H

/**
   File: SortKmers.h

   Extracting and sorting of kmers.  See <SortKmers()>.

   @file
*/

#include <iosfwd>

#include "Basevector.h"
#include "Vec.h"
#include "KmerRecord.h"
#include "KmerShape.h"

/**
   Class: dummy
   
   Used to select the right version of <SortKmers()>: 1-pass, 10-pass or 100-pass.
 */
template<int Passes> class dummy { };

/**
   Macro: SortKmersNumPasses
   
   A slightly better name for the class (<dummy>) used to statically
   select the version of <SortKmers()> that is called.
 */
#define SortKmersNumPasses dummy

/**
   Type Concept: SortKmersOutputRecord

   A record representing one occurrence of a kmer in a read.  Classes
   that model this concept can be used as the _RECORD_ template
   argument to <SortKmers()>, which extracts all kmer occurrences from
   the reads and records each occurrence in a record of this class.
   To model this type concept, a type needs to define the members
   described below.  Not all these members are used by SortKmers(),
   but we still include them in this type concept as they are used
   by related code and it's useful to have a single type concept
   that captures all the related members.
   <kmer_record> and <kmer> model this type concept.

   Constant: BASES_SIZE

   The size of the kmer stored in this record.
   
   >static const int BASES_SIZE;
   
   Method: Set

   Records the occurrence of a kmer in a read.

   >  void Set( const kmer_t& b, int read_id, int read_pos );
   
   Parameters:

      b - the kmer (its sequence)
      read_id - the <read id> of the read in which this kmer occurs
      read_pos - the position in the read of this kmer's occurrence;
        it is 1-offset, and is positive if the <kmer's canonical form>
	occurs in the read, and negative if the reverse complement
	of the kmer's canonical form occurs in the read.

   Method: Bytes

   Return the kmer data as an array of bytes.

   > const unsigned char* Bytes( ) const;

   Method: Ints

   Return the kmer data as an array of unsigned ints.

   > const unsigned int* Ints( ) const;

   Method: GetBasevector

   Sets a <basevector> to the content of this kmer.
   
   > void GetBasevector( basevector& kmer ) const;
   
*/

// End: Section

/**
   Declare two versions of SortKmers() for the given number of passes:
   the version where all template arguments need to be specified,
   and the version where all but the first two are set to defaults.
*/
#define DECLARE_SORT_KMERS(numPasses)                                                               \
  template<class KSHAPE, class RECORD>                                                              \
    void SortKmers ( SortKmersNumPasses<numPasses>, const vecbasevector& reads,                     \
		    const vec<int>& read_ids, int pass, vec<RECORD>& R,                             \
                     unsigned int& S, bool use_stable_sort = false );                               \
                                                                                                    \
  template<int K, int I>                                                                            \
   void SortKmers ( SortKmersNumPasses<numPasses> d1, const vecbasevector& reads,                   \
		    const vec<int>& read_ids, int pass, vec< kmer_record<K,I> >& R,                 \
		    unsigned int& S, bool use_stable_sort = false)                                  \
{    SortKmers< KmerShapeDefaultClass(K), kmer_record<K,I> >                                             \
       ( d1, reads, read_ids, pass, R, S, use_stable_sort );    }   typedef int eatSemicolon ## numPasses

DECLARE_SORT_KMERS(1);
DECLARE_SORT_KMERS(10);
DECLARE_SORT_KMERS(100);

/**
   Macro for simplifying explicit template instantiation of SortKmers().
   See also: FOR_ALL_K(), FOR_ALL_GAPS(), FOR_ALL_K_GAP(), 
*/
#define INSTANTIATE_SORTKMERS(_KSHAPE, RECORD, _passes) \
  template void SortKmers<_KSHAPE,RECORD> ( \
     dummy<_passes>, const vecbasevector& reads, const vec<int>& read_ids, \
     int pass, vec<RECORD>& R, unsigned int& S, bool use_stable_sort )

#define INSTANTIATE_SORTKMERS_FOR_KSHAPE(KSHAPE, dummy) \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 1), 1); \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 2), 1); \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 1), 10); \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 2), 10); \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 1), 100); \
    INSTANTIATE_SORTKMERS(KSHAPE, KmerRecordType(KSHAPE, 2), 100); \
    INSTANTIATE_SORTKMERS(KSHAPE, kmer<KSHAPE::KSIZE>, 100)

#define TYPE_NAME_I_K(prefix1, prefix2, I, K) prefix1 ## _ ## prefix2 ## _ ## I ## _ ## K ## _t

/**
   Macro: INSTANTIATE_SORTKMERS_FOR_I_K

   Instantiate the SortKmers routine for the <default kmer form> and the specified
   kmer size.  Useful if a particular program needs to use kmer sizes outside the
   <supported kmers>.  See <FindMatches.cc> for an example.

   *NOTE*: before invoking this macro in your .cc file, make sure to include the following
   header files:
   
  >#include "SortKmersImpl.h"
  >#include "VecTemplate.h"
*/
#define INSTANTIATE_SORTKMERS_FOR_I_K(I, K, prefix)                            \
    typedef KmerShapeDefaultClass(K) TYPE_NAME_I_K(prefix, kmer_shape, I, K);  \
    typedef kmer_record<K,I> TYPE_NAME_I_K(prefix, kmer_record, I, K);         \
    INSTANTIATE_SORTKMERS(TYPE_NAME_I_K(prefix, kmer_shape, I, K),             \
			  TYPE_NAME_I_K(prefix, kmer_record, I, K), 1);        \
    INSTANTIATE_SORTKMERS(TYPE_NAME_I_K(prefix, kmer_shape, I, K),             \
			  TYPE_NAME_I_K(prefix, kmer_record, I, K), 10);       \
    INSTANTIATE_SORTKMERS(TYPE_NAME_I_K(prefix, kmer_shape, I, K),             \
			  TYPE_NAME_I_K(prefix, kmer_record, I, K), 100)

#endif
//#ifndef SORT_KMERS_H
