/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/// \file
/// This file defines the typedef "qualvector", which stores quality scores 
/// between 0 and 255 as a vector of unsigned chars, and the typedef 
/// "vecqualvector", which stores a vector of qualvectors.
/// \ingroup grp_quals

#ifndef QUALVECTOR
#define QUALVECTOR

#include "CoreTools.h"
#include "Feudal.h"

/// Logical type for quality scores
typedef unsigned char qual_t;

/// Vector of quality scores, for example representing the quality of each base in one read
typedef serfvec<qual_t> qualvector;
typedef qualvector qvec;

/// Vector of vectors of quality scores, for example representing the quality of each base in each
/// read in a set of reads.
typedef vecvec<qual_t> vecqualvector;
typedef vecqualvector vecqvec;

///Produces fasta format quals, mirrors basevector::Print()
void Print( ostream &out, const qualvector &q, const String &name, 
            const int scores_per_line = 25 );

/// CopyQuals: copy elements from one qualvector to another. If rev_from=True, 
/// then copy from reverse(from), starting at from_start on reverse(from).
/// This mirrors CopyBases in Basevector.h.

inline void CopyQuals( const qualvector& from, int from_start, qualvector& to, 
     int to_start, int count, Bool rev_from = False )
{    if ( !rev_from )
     {    for ( int i = 0; i < count; i++ )
               to[ to_start + i ] = from[ from_start + i ];    }
     else
     {    for ( int i = 0; i < count; i++ )
               to[ to_start + i ]
                    = from[ from.size( ) - (from_start + i) - 1 ];    }    }

// ReadFastaQuals: read quality scores from a fasta quality score file.  If 
// ids_to_read is supplied, it should be a sorted list of indices of the records
// to be read.

void ReadFastaQuals( const String& fn, vecqualvector& qual, 
     const vec<int>* ids_to_read = 0 );

#endif
