// Copyright (c) 2000-2003 Whitehead Institute for Biomedical Research
// 

#ifndef HISTOGRAM_H
#define HISTOGRAM_H

// This template defines a class that encapsulates the generation of histograms.
// The type T must have the following operations defined:
// operator< ( const T& )
// operator+= ( const T& )
// operator*= ( const T& )
// operator* ( const T&, const T& )
//
// Note that in this implementation, neither vecData nor vecBinLimits
// needs to be sorted.

#include "Vec.h"

#include <map>

template <typename T>
class histogram {

 public:
  // Convenience typedef.
  typedef map<T,longlong> bin_map;
  // the only kind of iterator we support is const iterator of the bin_map
  typedef typename bin_map::const_iterator iterator; 
  typedef typename bin_map::const_iterator const_iterator; 

  histogram( ) 
    : mUnderflowCount( 0 ),
      mShowUnderflow( false ),
      mIsEmpty( true )
  { }

  /// Reset the bins' counts to zero.
  void ResetCounts()
  {
    for ( typename bin_map::iterator binIter = mBins.begin();
          binIter != mBins.end(); ++binIter )
      binIter->second = 0;
    mUnderflowCount = 0;
  }

  /// Add a bin, reset counts if not empty.
  void AddBin( const T &binStart )
  {
    mBins.insert( make_pair( binStart, 0 ) );
    if ( ! mIsEmpty )
      ResetCounts();
  }

  /// Adds a set of bins starting at the given values.
  void AddBins( const vec<T> &vecBinStarts )
  {
    for ( typename vec<T>::const_iterator binStartIter = vecBinStarts.begin();
          binStartIter != vecBinStarts.end(); ++binStartIter )
      AddBin( *binStartIter );
  }
   
  /// This adds a set of bins whose starts follow a linear progression.

  void AddLinearProgressionOfBins( const T &lowestBinStart,
                                   const T &binSize,
                                   const int numBins )
  {
    T binStart = lowestBinStart;
    
    for ( int binNum = 0; binNum < numBins; ++binNum, binStart += binSize )
      AddBin( binStart );
  }

  /// This adds a set of bins whose starts follow a quasi-logarithmic
  /// progression.  No bin is added that is lower than the given lowest
  /// bin, and no bin is added that is higher than the given highest
  /// bin.  We set the current bin magnitude to the given minimum, then
  /// walk through the set of multiple progressions, adding a bin for
  /// each multiple equal to the product of the multiple and the
  /// current magnitude.  After each pass through the progression set,
  /// we multiple the current magnitude by the given bin magnitude
  /// jump.  
  ///
  /// For example, to generate the logarithmic progression of bins (1,
  /// 10, 100, 1000), we would call this method with the parameters
  /// lowestBin=1, highestBin=1000, minBinMagnitude=1,
  /// binMagnitudeJump=10, and vecBinMultipleProgression=(1).
  ///
  /// To generator the quasi-logarithmic progression (10, 20, 50, 100, 200,
  /// 500, 1000), we would call this method with the parameters
  /// lowestBin=1, highestBin=100, minBinMagnitude=10,
  /// binMagnitudeJump=10, and vecBinMultipleProgression=(1,2,5).

  void AddLogProgressionOfBins( const T &lowestBin,
                                const T &highestBin,
                                const T &minBinMagnitude,  // must be > 0
                                const T &binMagnitudeJump, // must be > 1
                                const vec<T> &vecBinMultipleProgression )
  {
    ForceAssertGt( minBinMagnitude, 0 );
    ForceAssertGt( binMagnitudeJump, 1 );

    T binMagnitudeLimit = max( -lowestBin, highestBin );

    for ( T binMagnitude = minBinMagnitude;
          binMagnitude <= binMagnitudeLimit; 
          binMagnitude *= binMagnitudeJump )
      for ( typename vec<T>::const_iterator multipleIter = vecBinMultipleProgression.begin(); 
            multipleIter != vecBinMultipleProgression.end(); ++multipleIter )
      {
        int binStart = *multipleIter * binMagnitude;
        if ( -binStart >= lowestBin )
          AddBin( -binStart );
        if ( binStart <= highestBin )
          AddBin( binStart );
      }
  }

  /// Renew this histogram from the data with default values that cover
  /// the complete range of the data.
  template <class T_iter>
  void SetAllFromData(T_iter begin, T_iter end, int nbins = 20) {
    T min = *min_element(begin, end);
    T max = *max_element(begin, end);
    T binSize = (max - min) / nbins;
    SetAllFromData(begin, end, min, binSize, nbins);
  }

  /// Renew this histogram from the data with bins set as in 
  /// AddLinearProgressionOfBins
  template <class T_iter>
  void SetAllFromData(T_iter begin, T_iter end, 
		      const T & min, const T & binSize, int nbins) {
    mBins.clear();
    mUnderflowCount = 0;
    mShowUnderflow = mIsEmpty = false;
    AddLinearProgressionOfBins(min, binSize, nbins);
    AddData(begin,end);
  }


  /// Get the number of bins.
  int size() const { return mBins.size(); }

  /// Get the total of all bins
  longlong CountData() const {
    longlong tot = mUnderflowCount;
    for ( typename bin_map::const_iterator binIter = mBins.begin();
          binIter != mBins.end(); ++binIter )
      tot += binIter->second;
    return tot;
  }

  /// Add a datum to the histogram, incrementing the appropriate bin
  void AddDatum( const T &datum ) { AddIdenticalData(datum,1); }

  /// Add N copies of the datum
  void AddIdenticalData( const T &datum, const longlong N )
  {
    mIsEmpty = false;

    // Find the bin after the one that should contain this datum.
    typename bin_map::iterator binIter = mBins.upper_bound( datum );

    // If we're pointing to the first bin, it's an underflow.
    if ( binIter == mBins.begin() )
      mUnderflowCount += N;

    // Otherwise, we move back one bin and increment its counter.
    else
    {
      --binIter;
      binIter->second += N;
    }
  }

  /// Increment the bin counts with the given data.
  template <class T_iter>
  void AddData( T_iter dataBegin, T_iter dataEnd )
  {
    for ( ; dataBegin != dataEnd; ++dataBegin )
      this->AddDatum( *dataBegin );
  }

  /// Get the number of elements which were less than the lowest bin limit.
  longlong GetUnderflowCount() const { return mUnderflowCount; }

  /// Should we print the underflow count?
  void ShowUnderflow() { mShowUnderflow = true; }
  /// Should we print the underflow count?
  void HideUnderflow() { mShowUnderflow = false; }

  /// Get constant iterators into the bin map.
  typename bin_map::const_iterator begin() const { return mBins.begin(); }
  typename bin_map::const_iterator end()   const { return mBins.end(); }


  /// Print the histogram.
  friend 
  ostream& operator<<( ostream &out, const histogram &h )
  {
    if ( h.mBins.empty() )
      out << "No bins in histogram." << "\n";
    else
    {
      if ( h.mShowUnderflow )
        out << "< " << h.mBins.begin()->first << "\t" << h.mUnderflowCount << "\n";
      for ( typename bin_map::const_iterator binIter = h.mBins.begin();
            binIter != h.mBins.end(); ++binIter )
        out << binIter->first << "\t" << binIter->second << "\n";
    }
    return out;
  }

  /// Print the histogram as a line.
  void PrintDataAsLine(ostream & out) {
    if ( mBins.empty() )
      out << "No bins in histogram.";
    else
    {
      if ( mShowUnderflow )
        out << mUnderflowCount << "\t";
      for ( typename bin_map::const_iterator binIter = mBins.begin();
            binIter != mBins.end(); ++binIter ) {
        out << binIter->second << "\t";
      }
    }
    out << "\n";
  }
    
  ///Print the bin starts as a line
  void PrintBinsAsLine(ostream & out) {
    if ( mBins.empty() )
      out << "No bins in histogram." << "\n";
    else
    {
      if ( mShowUnderflow )
        out << "under\t";
      for ( typename bin_map::const_iterator binIter = mBins.begin();
            binIter != mBins.end(); ++binIter ) {
        out << binIter->first << "\t";
      }
      out << "\n";
    }
  }
    
  ///Print the histogram as a line, in percentages of total counts.
  void PrintDataPercentAsLine(ostream & out) {
    double total = CountData();
    if ( mBins.empty() )
      out << "No bins in histogram." << "\n";
    else
    {
      if ( mShowUnderflow )
        out << mUnderflowCount << "\t";
      for ( typename bin_map::const_iterator binIter = mBins.begin();
            binIter != mBins.end(); ++binIter ) {
        out << binIter->second/total << "\t";
      }
      out << "\n";
    }
  }

 private:
  bin_map mBins;
  longlong mUnderflowCount;
  bool mShowUnderflow;
  bool mIsEmpty;
};

#endif
