/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

// This file defines class fast_ifstream and fast_pipe_ifstream, which has
// essentially identical member functions.

#ifndef FAST_IFSTREAM
#define FAST_IFSTREAM

#include "CoreTools.h"

const int opt_buf_size = 8192;

class fast_ifstream {

     public:

     fast_ifstream( ) : file_desc_(-1) { }
     fast_ifstream( const String& filename );
     void Open( const String& filename );

     ~fast_ifstream( );

     /// Rewinds the currently open file back to start position;
     /// has no effect if the stream is not associated with a file
     /// (i.e. default constructor was used and no Open() was issued yet)
     void Rewind() ;

     // get(c) reads a char and puts it in c

     void get( char& c, Bool advance = True );

     ///Read additional characters beyond the internal buffer.
     void ReadSlow(char * & start, char * & end);

     // peek(c) peeks at a char and puts it in c
     // If there is nothing left in the file, then fail is set.

     void peek( char& c );

     Bool fail( ) const
     {    return fail_;    }

     // Read a line from "in" and put it in "s".

     friend void getline( fast_ifstream& in, String& s );

     // Read a line form "in".  If it begins with "begin", put it in "s", return 
     // True.  Otherwise, return False.
     //
     // This is slightly faster than calling getline.

     friend Bool getline_if_match( fast_ifstream& in, String& s, 
          const String& begin );

     // Read until a string "tail" is encountered.  Set "s" to everything read,
     // including the tail.  If the tail is not found, return False;

     friend Bool get_to( fast_ifstream& in, String& s, const String& tail );

     friend void ReadParagraphs( fast_ifstream& in, vec<String>& paragraphs );

     private:

     String filename_;
     int file_desc_;
     char buf_[opt_buf_size + 1];
     int buf_ptr_, buf_top_;
     Bool fail_;

     void fill_buffer( );

};

class fast_pipe_ifstream {

     public:

     fast_pipe_ifstream( String command );

     ~fast_pipe_ifstream( );

     void get( char& c, Bool advance = True );

     void peek( char& c );

     Bool fail( ) const
     {    return fail_;    }

     friend void getline( fast_pipe_ifstream& in, String& s );

     friend Bool getline_if_match( fast_pipe_ifstream& in, String& s, 
          const String& begin );

     friend Bool get_to( fast_pipe_ifstream& in, String& s, const String& tail );

     friend void ReadParagraphs( fast_pipe_ifstream& in, vec<String>& paragraphs );

     private:

     String command_;
     FILE* file_;
     char buf_[opt_buf_size + 1];
     int buf_ptr_, buf_top_;
     Bool fail_;

     void fill_buffer( );

};

#endif
