/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2005) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Equiv.h"
#include "Vec.h"

bool equivalence_relation::equiv(int a, int b) const
{    if ( a == b ) return true;
     int c = a;
     while ( (c = (*this)[c]) != a )
          if ( c == b ) return true;
     return false;    }

void equivalence_relation::join(int a, int b)
{    if ( a == b ) return;
     // Determine if a is already equivalent to b.
     // Also, find out what comes before a.
     int c = (*this)[a], before_a = a;
     while( c != a )
     {    if ( c == b ) return;  // a ~ b already
          before_a = c;
          c = (*this)[c];    }
     // Make a equivalent to b.
     (*this)[before_a] = (*this)[b];
     (*this)[b] = a;    }

int equivalence_relation::size(int a) const
{    int s = 1, b = a;
     while( (b = (*this)[b]) != a ) ++s;
     return s;    }

int equivalence_relation::orbit_count( ) const
{    int count = 0;
     for ( unsigned int i = 0; i < vec<int>(*this).size( ); i++ )
     {    int a = (*this)[i];
          int b = a;
          while ( (b = (*this)[b]) != a )
               if ( b < a ) break;
          if ( a == b ) ++count;    }
     return count;    }

vec<int> equivalence_relation::OrbitReps( ) const
{    vec<int> R;
     for ( unsigned int i = 0; i < vec<int>(*this).size( ); i++ )
     {    int a = (*this)[i];
          int b = a;
          while ( (b = (*this)[b]) != a )
               if ( b < a ) break;
          if ( a == b ) R.push_back(a);    }
     return R;    }

vec<int> equivalence_relation::orbit(int f) const
{    vec<int> L;
     L.push_back(f);
     int n = f;
     while(1)
     {    n = (*this)[n];
          if ( n == f ) return L;
          L.push_back(n);    }    }

// ===============================================================================

equiv_rel::equiv_rel(int n)
{    x_.resize(n);
     y_.resize(n);
     for ( int i = 0; i < n; i++ )
          x_[i] = y_[i] = i;    }

void equiv_rel::Initialize(int n)
{    x_.resize(n);
     y_.resize(n);
     for ( int i = 0; i < n; i++ )
          x_[i] = y_[i] = i;    }

int equiv_rel::Size() const
{    return x_.size();    }

Bool equiv_rel::Equiv( int i, int j ) const
{    return y_[i] == y_[j];    }

void equiv_rel::Next( int& n ) const
{    n = x_[n];    }

Bool equiv_rel::Join(int a, int b)
{    if ( y_[a] == y_[b] ) return false; // already equivalent
     swap( x_[a], x_[b]); 
     for( int n = x_[a]; y_[n] != y_[a]; n = x_[n] )
          y_[n] = y_[a];
     return true;
}

void equiv_rel::Orbit( int a, vec<int>& o ) const
{    o.resize(0);
     o.push_back(a);
     int b = a;
     while(1)
     {    b = x_[b];
          if ( b == a ) break;
          o.push_back(b);    }    }

void equiv_rel::Orbit( int a, vec<TraceInt>& o ) const
{    o.resize(0);
     o.push_back(a);
     int b = a;
     while(1)
     {    b = x_[b];
          if ( b == a ) break;
          o.push_back(b);    }    }


int equiv_rel::OrbitSize( int a ) const
{    int ans=1;
     int b = a;
     while(1)
     {    b = x_[b];
          if ( b == a ) return ans;
          ans++;    }    }

int equiv_rel::OrbitCount( ) const
{    int count = 0;
     for ( int i = 0; i < (int) x_.size( ); i++ )
          if ( x_[i] == y_[i] ) ++count;
     return count;    }

void equiv_rel::OrbitReps( vec<int>& reps ) const
{    reps.clear( );
     for ( int i = 0; i < (int) x_.size( ); i++ )
          if ( x_[i] == y_[i] ) reps.push_back(i);    }

void equiv_rel::OrbitReps( vec<TraceInt>& reps ) const
{    reps.clear( );
     for ( int i = 0; i < (int) x_.size( ); i++ )
          if ( x_[i] == y_[i] ) reps.push_back(i);    }

void equiv_rel::OrbitRepsAlt( vec<int>& reps ) const
{    reps.clear( );
     for ( int i = 0; i < (int) x_.size( ); i++ )
          if ( i == y_[i] ) reps.push_back(i);    }

void equiv_rel::OrbitRepsAlt( vec<TraceInt>& reps ) const
{    reps.clear( );
     for ( int i = 0; i < (int) x_.size( ); i++ )
          if ( i == y_[i] ) reps.push_back(i);    }


void equiv_rel::Singletons( vec<int>& sings ) const {
  sings.clear();
  for (int i=0; i < x_.isize(); i++)
    if( x_[i]==i )
      sings.push_back(i);
}

bool equiv_rel::Singletons() const {
  for (int i=0; i < x_.isize(); i++)
    if( x_[i]==i )
      return true;
  return false;
}

