/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

/**
   Header: CommonSemanticTypes.h

   Defines some common <semantic types> used throughout the code.
   Putting these definitions into a separate header helps avoid creating some
   spurious dependencies among headers.
   
   @file
*/

#ifndef __INCLUDE_CommonSemanticTypes_h
#define __INCLUDE_CommonSemanticTypes_h

#include "SemanticTypes.h"

// Type: read_id_t
// Logical type for a read id -- the index of a read in a <reads_t> array of reads.
// A read can also be denoted by a reads_t::const_iterator.
SemanticTypeStd( int, read_id_t );

// Const: NULL_READ_ID
// A value guaranteed to be different from every valid read id.
const read_id_t NULL_READ_ID = -1;

// Type: genome_part_id_t
// Logical type for a genome part id -- the index of a <genome part> in a <genome_t> array of genome parts.
// A genome part can also be denoted by a genome_t::const_iterator.
SemanticTypeStd( int, genome_part_id_t );

// Type: nbases_t
// Represents the length of a DNA region in bases.
SemanticTypeStd( int, nbases_t );

// Type: nkmers_t
// Represents the length of a DNA region in kmers.  Of course, this
// presupposes there is one fixed kmer size in use throughout.
SemanticTypeStd( int, nkmers_t );

// Type: basevec_pos_t
// Position in a base vector.
SemanticType( nbases_t, basevec_pos_t );

// Type: read_pos_t
// Logical type for representing position in a read.  Must be signed -- this makes
// coding more convenient.
SemanticType( basevec_pos_t, read_pos_t );

// Type: genome_part_pos_t
// Logical type for representing position in a genome part.  Must be signed -- this makes
// coding more convenient.
SemanticType( basevec_pos_t, genome_part_pos_t );

// Logical type: orient_t
// Read orientation, relative to a given strand: forward or reverse complement.
// Use the named constants <ORIENT_FW> and <ORIENT_RC>, rather than the physical
// values False/True, to prevent any confusion.
SemanticTypeStd( Bool, orient_t );

// Constants: orient_t values
//
//    ORIENT_FW - the read or unipath occurs on the <genome part> at the specified
//        position as-is
//    ORIENT_RC - the reverse complement of the read or unipath occurs on the
//        genome part at the specified position.
const orient_t ORIENT_FW = False;
const orient_t ORIENT_RC = True;

#endif
// #ifndef __INCLUDE_CommonSemanticTypes_h



