/////////////////////////////////////////////////////////////////////////////
//                   SOFTWARE COPYRIGHT NOTICE AGREEMENT                   //
//       This software and its documentation are copyright (2007) by the   //
//   Broad Institute/Massachusetts Institute of Technology.  All rights    //
//   are reserved.  This software is supplied without any warranty or      //
//   guaranteed support whatsoever. Neither the Broad Institute nor MIT    //
//   can be responsible for its use, misuse, or functionality.             //
/////////////////////////////////////////////////////////////////////////////

#ifndef FORCE_DEBUG
     #define NDEBUG
#endif

#include "Bitvector.h"
#include "CoreTools.h"
#include "FeudalMimic.h"
#include "math/Functions.h"

bitvector::bitvector(const bitvector& b)
{    if ( b.data_ == 0 ) 
     {    data_ = 0;
          extra_space_and_self_owned_ = 1u << 31;    }
     else
     {    length_ = b.length_;
          extra_space_and_self_owned_ = (1 << 31) ^ b.ExtraSpace( );
          data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 31)/32];
          memcpy( data_, b.data_, sizeof(int) * (((int) length_+31)/32) );    }    }

// Note that operator= for class bitvector will try to avoid allocating
// memory.  So long as there is enough space in *this, new is not called.

bitvector& bitvector::operator=(const bitvector& b)
{    ForceAssert( b.data_ != 0 );
     if ( data_ == 0 )
     {    length_ = b.length_;
          extra_space_and_self_owned_ = (1 << 31) ^ b.ExtraSpace( );
          data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 31)/32];    }
     else if ( b.size( ) <= length_ + ExtraSpace( ) )
     {    extra_space_and_self_owned_ = 
               SelfOwnedBit( ) ^ ( length_ + ExtraSpace( ) - b.size( ) );
          length_ = b.size( );    }
     else
     {    if ( SelfOwned( ) ) delete [ ] data_;
          length_ = b.length_;
          extra_space_and_self_owned_ = (1 << 31) ^ b.ExtraSpace( );
          data_ = new unsigned int[(length_ + b.ExtraSpace( ) + 31)/32];    }
     memcpy( data_, b.data_, sizeof(int) * (((int) length_+31)/32) );
     return *this;    }

void bitvector::Set( unsigned int i, unsigned int j, unsigned char bit,
     Bool reverse )
{    AssertLe( i, j );
     AssertLe( j, length_ );
     AssertLt( bit, 2 );
     if (reverse)
     {    unsigned int inew = length_ - j, jnew = length_ - i;
          i = inew;
          j = jnew;    }
     unsigned int ieasy = ((i + 31) >> 5) << 5, jeasy = (j >> 5) << 5;
     if ( bit == 0 )
     {    if ( jeasy >= 32 + ieasy )
          {    for ( unsigned int k = ieasy; k < jeasy; k += 32 )
                    data_[k>>5] = 0;    
               for ( unsigned int k = i; k < ieasy; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = target & ~the_bit;    }
               for ( unsigned int k = jeasy; k < j; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = target & ~the_bit;    }    }
          else
          {    for ( unsigned int k = i; k < j; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = target & ~the_bit;    }    }    }
     else
     {    if ( jeasy >= 32 + ieasy )
          {    for ( unsigned int k = ieasy; k < jeasy; k += 32 )
                    data_[k>>5] = ~0u;
               for ( unsigned int k = i; k < ieasy; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = ( target & ~the_bit ) | the_bit;    }
               for ( unsigned int k = jeasy; k < j; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = ( target & ~the_bit ) | the_bit;    }    }
          else
          {    for ( unsigned int k = i; k < j; k++ )
               {    unsigned char& target = DataAsBytes( )[k>>3];
                    unsigned int the_bit = 1 << (k&7);
                    target = ( target & ~the_bit ) | the_bit;    }    }    }    }
     
#include "FeudalTemplate.h"

#ifdef __DECCXX_VER
#pragma define_template mastervec<bitvector, unsigned int>
#else
INSTANTIATE_MASTERVEC( bitvector, unsigned int )
#endif

float Coverage( const vecbitvector& v )
{    longlong covered = 0, total = 0;
     for ( int i = 0; i < v.size( ); i++ )
     {    for ( unsigned int j = 0; j < v[i].size( ); j++ )
               if ( v[i][j] ) ++covered;
          total += v[i].size( );    }
     ForceAssert( total != 0 );
     return float(covered) / float(total);    }

void bitvector::ReverseMe( )        
{    for ( unsigned int i = 0; i < size( )/2; i++ )
     {    unsigned char temp = (*this)[i];
          Set( i, (*this)[ size( ) - i - 1 ] );
          Set( size( ) - i - 1, temp );    }    }

void bitvector::Reverse( const bitvector& b ) 
{    Setsize( b.size( ) );
     for ( unsigned int i = 0; i < b.size( ); i++ )
          Set( i, b[ b.size( ) - i - 1 ] );    } 

bitvector::bitvector( unsigned int n, Bool value ) : length_(n)
{    int words_needed = (n + 31) / 32;
     int actual_extra = words_needed * 32 - n;
     extra_space_and_self_owned_ = (1 << 31) ^ actual_extra;
     data_ = new unsigned int[words_needed];
     if ( value == False )
     {    for ( int i = 0; i < words_needed; i++ )
               data_[i] = 0;    }
     else
     {    unsigned int zero = 0;
          for ( int i = 0; i < words_needed; i++ )
               data_[i] = ~zero;    }    }

void bitvector::Setsize( unsigned int n, unsigned int extra )
{    if ( data_ != 0 && length_ == n && ExtraSpace( ) == extra ) return;
     if ( data_ == 0 )
     {    data_ = new unsigned int[(n + extra + 31)/32];
          length_ = n;
          extra_space_and_self_owned_ = (1 << 31) ^ extra;    }
     else if ( length_ + ExtraSpace( ) >= n + extra )
     {    extra_space_and_self_owned_ =
               SelfOwnedBit( ) ^ (length_ + ExtraSpace( ) - n);
          length_ = n;    }
     else
     {    if ( SelfOwned( ) ) delete [ ] data_;
          data_ = new unsigned int[(n + extra + 31)/32];
          length_ = n;
	  extra_space_and_self_owned_ = (1 << 31) ^ extra;    
     }    
}

void bitvector::resize( unsigned int n )
{    bitvector b(n);
     unsigned int n0 = Min( n, size( ) );
     for ( unsigned int i = 0; i < n0; i++ )
          b.Set( i, (*this)[i] );
     *this = b;    }

void bitvector::resize( unsigned int n, unsigned char bit )
{    bitvector b(n);
     unsigned int n0 = Min( n, size( ) );
     for ( unsigned int i = 0; i < n0; i++ )
          b.Set( i, (*this)[i] );
     for ( unsigned int i = n0; i < n; i++ )
          b.Set( i, bit );
     *this = b;    }

template<> bool IsGoodFeudalFile<bitvector>( const String & filename,
    const bitvector * dummy, bool verbose, bool ok3) {
  if (!IsGoodFeudalFile(filename, verbose, ok3)) return false;

  mv_file_control_block control;
  int fd = Open( filename, O_RDONLY );
  read( fd, &control, sizeof(control) );
  close(fd);
  if (control.IsOldFormat()) return true;

  if ( control.vecSize() != 16 && control.vecSize() != 24) {
    if (verbose) {
      cerr << "IsGoodFeudalFile: Failed vecSize test" << endl;
      cerr << "Should be 16 (or 24 for old 64-bit files), but is "
	<< int(control.vecSize()) << endl;
    }
    return false;
  }
  return true;
}

void And(vecbitvector & result, const vecbitvector & first, 
	 const vecbitvector & second) {
  Assert(SameSizes(first, second));
  Mimic(first, result);
  for (int i=0; i != result.size(); ++i) 
    for (int j=0; j !=result[i].isize(); ++j)
      result[i].Set(j,first[i][j] & second[i][j]);
}
  
 
void Or(vecbitvector & result, const vecbitvector & first, 
	const vecbitvector & second){
  Assert(SameSizes(first, second));
  Mimic(first, result);
  for (int i=0; i != result.size(); ++i) 
    for (int j=0; j !=result[i].isize(); ++j)
      result[i].Set(j,first[i][j] | second[i][j]);
}
 
void bitvector::SetToSubOf(const bitvector& orig_bv, const unsigned int start_pos,
     const int len )
{    bitvector b(len);
     for ( int i = 0; i < len; i++ )
          b.Set( i, orig_bv[ start_pos + i ] );
     *this = b;    }
