#ifndef __BASEVECTORITERATOR_H
#define __BASEVECTORITERATOR_H

#include "Basevector.h"

class BasevectorIterator {
 public:
  
  /** \brief Constructor. Creates a new iterator associated with basevector \c b and
   * ready to read the vector's bases starting from position \c pos (default = 0)
   */
  explicit BasevectorIterator(const basevector &b, unsigned int position = 0) : byte(0) {
    Reset(b,position);
  }

  /** \brief Prepares the iterator to read base vector \c b starting from position \c pos.
   *
   *  After a call to this method, the iterator becomes attached to the basevector \c b and
   *  the following call to Next() will return the base at <tt>b[pos]</tt>.
   *  Note: this method is unchecked; if \c pos is out of the bounds of the basevector,
   *  the result of this call (or subsequent Next() calls) is undefined.
   */
  void Reset(const basevector &b, unsigned int pos = 0) {
    // ptr points to the byte, in which the first requested base is located
    // (there are 4 bases packed into one byte, hence >>2 ):
    ptr = b.DataAsBytes() + ( pos >> 2 );

    // we want to keep the following invariant for the next() method: 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset > 0 - otherwise load it from *ptr. We *need* to de-synchronize 
    //    loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the end of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!

    internal_offset = pos & 0x3; // the position in the byte to be read next

    if ( internal_offset ) {
      // extract the byte (4 bases) that contains the first base to be read
      // and advance ptr to point to the next byte:
      byte = *(ptr++);
      
      switch ( internal_offset ) {
	// invariant (see above): the byte must be already shifted by internal_offset-1 bases!
	// next() will always shift once before reading the next base
      case 3: byte >>= 4; break;
      case 2: byte >>= 2; break;
	// case 1: internal_offset-1 = 0 in this case, no shift needed
	// case 0: see (3) in the above comment - 
	//        if internal_offset=0, next() will reload the byte; we do not even
	//        get into the embedding if() when internal_offset=0
      }
    }
  }

  /** \brief Returns next base from the basevector this iterator is associated with.
   *
   *  Each subsequent call to Next() returns the next base from the basevector 
   * and "advances" the iterator to the next position. After the constructor or Reset()
   * method are executed, the iterator "points" immediately before the first requested base
   * (the one at position \c pos passed to those methods, default is 0). Hence the first
   * call to Next() returns the base <tt>b[pos]</tt> (@see Reset(), @see BasevectorIterator),
   * the next call returns <tt>b[pos+1]</tt> \e etc where \c b is the basevector this
   * iterator's instance  is associated with. 
   * 
   * NOTE: this method is unchecked; if an attempt is made to advance the iterator past
   * the end of the associated basevector, the result is undefined.
   */
  unsigned char Next() {
    // we want to keep the following invariant for the next() method: 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset > 0 - otherwise load it from *ptr only when next() is called. 
    //    We *need* to de-synchronize loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the end of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!


    if ( ! internal_offset ) { 
      // if we are about to read the very first base from new byte (internal_offset==0):
      // load that byte, advance ptr to the next byte to be read (if and when we reach it)
      //      cerr << "<-- advancing\n";
      byte = *(ptr++); // byte = * ( ptr+ ((~(internal_offset | internal_offset >> 1)) & 0x1));
    } else {
      // otherwise, the currently loaded byte was read (and shifted) up-to internal_offset-1
      byte >>= 2; // shift it to the current position we are going to read from
    }
    internal_offset++;      // increment internal offset
    internal_offset &= 0x3; // rollover; if we have already read all 4 bases from the current
                            // byte, offset (next position to read) is 0;
    // invariant is held!
    //    cerr << as_base(byte & 0x3);
    return byte & 0x03; // return the lowest two bits (the base!), byte was shifted already
  }

 private:
  /// points to the next byte in the byte array (immediately after thge one
  /// we are currently reading)
  unsigned char * ptr;
  
  // next bit pais (base) offset to read
  unsigned char internal_offset; 

  // byte (4 bases) we are currently reading. 
  unsigned char byte;
};



class BasevectorReverseIterator {
 public:
  
  /** \brief Constructor. Creates a new iterator associated with basevector \c b and
   * ready to read the vector's bases backwards starting from position \c pos 
   * counted from the end of the vector (default = 0)
   */
  explicit BasevectorReverseIterator(const basevector &b, unsigned int position = 0) : byte(0) {
    Reset(b,position);
  }

  /** \brief Prepares the iterator to read base vector \c b starting from position \c pos
   * counted from the vector's last base.
   *
   *  After a call to this method, the iterator becomes attached to the basevector \c b and
   *  the following call to Next() will return the base at <tt>b[b.size()-1-pos]</tt>.
   *  Note: this method is unchecked; if \c pos is out of the bounds of the basevector,
   *  the result of this call (or subsequent Next() calls) is undefined.
   */
  void Reset(const basevector &b, unsigned int pos = 0) {
    // ptr points to the byte, in which the first requested base is located
    // (there are 4 bases packed into one byte, hence >>2 ):
    ptr = b.DataAsBytes() + ( (b.size()-1-pos) >> 2 );

    // we want to keep the following invariant for the next() method (remember that for the 
    // reverse iterator "next" is always understood in backwards direction!): 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset != 3 - otherwise load it from *ptr only when next() is called. 
    //    We *need* to de-synchronize 
    //    loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the beginning of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!

    internal_offset = (b.size()-1-pos) & 0x3; // the position in the byte to be read next

    if ( internal_offset != 3) {
      // if at next call to next() we are not about to read the very last base
      // from the 4-base block (byte), it's safe to pre-load the byte.
      // extract the byte (4 bases) that contains the first base to be read
      // and advance ptr to point to the previous byte (we are reading backwards!);
      // we also invert the order of bases in the byte so that we could right-shift
      // later (towards the lowest bits); note that RCtable gives rc, so we apply
      // additional ~ to get just r (ideally another table is required to 
      // avoid additional ~ , that's a todo item...)

      byte = ~basevector::RCtable[*(ptr--)];
      
      switch ( internal_offset ) {
	// invariant (see above): the byte must be already shifted by internal_offset-1 bases!
	// next() will always shift once before reading the next base; remember: bases are 
	// already reverted, while internal_offset tells what base from the *original* byte we need:
      case 1: byte >>= 2; break;
      case 0: byte >>= 4; break;
	// case 2: needs to be shifted by 2 bits only once to be accessible - next() will do that
	// case 3: see (3) in the above comment - 
	//        if internal_offset=3, next() will reload the byte; we do not even
	//        get into the embedding if() above when internal_offset=3
      }
    }
  }

  /** \brief Returns next base from the basevector this iterator is associated with (this
   *  iterator walks *backwards*).
   *
   *  Each subsequent call to Next() returns the previous base from the basevector
   * (or next base from the reverted basevector) 
   * and "advances" the iterator backwards to the next position. After the constructor or Reset()
   * method are executed, the iterator "points" immediately before the first requested base
   * (the one at position \c pos passed to those methods, default is 0). Hence the first
   * call to Next() returns the base <tt>b[b.size()-1-pos]</tt> (@see Reset(), @see BasevectorIterator),
   * the next call returns <tt>b[b.size()-1-pos-1]</tt> \e etc where \c b is the basevector this
   * iterator's instance  is associated with. 
   * 
   * NOTE: this method is unchecked; if an attempt is made to advance the iterator past
   * the end of the associated basevector, the result is undefined.
   */
  unsigned char Next() {
    // we want to keep the following invariant for the next() method: 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset !=3  - otherwise load it from *ptr. We *need* to de-synchronize 
    //    loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the end of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!


    if ( internal_offset == 3 ) { 
      // if we are about to read the very first base from new byte (internal_offset==3):
      // load that byte, advance ptr to the next byte to be read (if and when we reach it)
      //      cerr << "<-- advancing\n";
      byte = ~basevector::RCtable[*(ptr--)]; 
    } else {
      // otherwise, the currently loaded byte was read (and shifted) up-to internal_offset-1
      byte >>= 2; // shift it to the current position we are going to read from
    }
    internal_offset--;      // decrement internal offset
    internal_offset &= 0x3; // rollover; if we have already read all 4 bases from the current
                            // byte, offset (next position to read) is 3 (reading backwards!);
    // invariant is held!
    //    cerr << as_base(byte & 0x3);
    return byte & 0x03; // return the lowest two bits (the base!) from the reverted current byte, 
                        // byte was shifted already
  }

 private:
  /// points to the next byte in the byte array (immediately after thge one
  /// we are currently reading)
  unsigned char * ptr;
  
  // next bit pais (base) offset to read
  unsigned char internal_offset; 

  // byte (4 bases) we are currently reading. 
  unsigned char byte;
};


class BasevectorReverseComplementIterator {
 public:
  
  /** \brief Constructor. Creates a new iterator associated with basevector \c b and
   * ready to read the vector's complementary bases backwards starting from position \c pos 
   * counted from the end of the vector (default = 0)
   */
  explicit BasevectorReverseComplementIterator(const basevector &b, unsigned int position = 0) : byte(0) {
    Reset(b,position);
  }

  /** \brief Prepares the iterator to read complement of base vector \c b starting from position \c pos
   * counted from the vector's last base.
   *
   *  After a call to this method, the iterator becomes attached to the basevector \c b and
   *  the following call to Next() will return the base complementary to the one at 
   *  <tt>b[b.size()-1-pos]</tt>.
   *  Note: this method is unchecked; if \c pos is out of the bounds of the basevector,
   *  the result of this call (or subsequent Next() calls) is undefined.
   */
  void Reset(const basevector &b, unsigned int pos = 0) {
    // ptr points to the byte, in which the first requested base is located
    // (there are 4 bases packed into one byte, hence >>2 ):
    ptr = b.DataAsBytes() + ( (b.size()-1-pos) >> 2 );

    // we want to keep the following invariant for the next() method (remember that for the 
    // reverse iterator "next" is always understood in backwards direction!): 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset != 3 - otherwise load it from *ptr only when next() is called. 
    //    We *need* to de-synchronize 
    //    loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the beginning of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!

    internal_offset = (b.size()-1-pos) & 0x3; // the position in the byte to be read next

    if ( internal_offset != 3) {
      // if at next call to next() we are not about to read the very last base
      // from the 4-base block (byte), it's safe to pre-load the byte.
      // extract the byte (4 bases) that contains the first base to be read
      // and advance ptr to point to the previous byte (we are reading backwards!);
      // we also rc the bases in the byte so that we could right-shift
      // later (towards the lowest bits); 

      byte = basevector::RCtable[*(ptr--)];
      
      switch ( internal_offset ) {
	// invariant (see above): the byte must be already shifted by internal_offset-1 bases!
	// next() will always shift once before reading the next base; remember: bases are 
	// already reverted, while internal_offset tells what base from the *original* byte we need:
      case 1: byte >>= 2; break;
      case 0: byte >>= 4; break;
	// case 2: needs to be shifted by 2 bits only once to be accessible - next() will do that
	// case 3: see (3) in the above comment - 
	//        if internal_offset=3, next() will reload the byte; we do not even
	//        get into the embedding if() above when internal_offset=3
      }
    }
  }

  /** \brief Returns next base from the basevector this iterator is associated with (this
   *  iterator walks *backwards*).
   *
   *  Each subsequent call to Next() returns the previous base from the basevector
   * (or next base from the reverted basevector) 
   * and "advances" the iterator backwards to the next position. After the constructor or Reset()
   * method are executed, the iterator "points" immediately before the first requested base
   * (the one at position \c pos passed to those methods, default is 0). Hence the first
   * call to Next() returns the base <tt>b[b.size()-1-pos]</tt> (@see Reset(), @see BasevectorIterator),
   * the next call returns <tt>b[b.size()-1-pos-1]</tt> \e etc where \c b is the basevector this
   * iterator's instance  is associated with. 
   * 
   * NOTE: this method is unchecked; if an attempt is made to advance the iterator past
   * the end of the associated basevector, the result is undefined.
   */
  unsigned char Next() {
    // we want to keep the following invariant for the next() method: 
    // 1) ptr points to the next byte to be loaded from the basevector (if and when necessary)
    // 2) internal_offset points ahead, to the next base (2-bit field) to be read from the
    //    current byte (see below);
    // 3) the byte from which the next base should be read is already loaded if
    //    internal_offset !=3  - otherwise load it from *ptr. We *need* to de-synchronize 
    //    loading the byte from advancing the
    //    offset counter, since otherwise next() will be loading bytes ahead of reading
    //    actual bases from them. That could cause segmentation fault at the end of
    //    basevector - next() might not intend to read the bases but it would attempt to 
    //    pre-load the out-of-the-boundary byte nevertheless!


    if ( internal_offset == 3 ) { 
      // if we are about to read the very first base from the new byte, we load it
      // only here, when it is explicitly requested with next() - hence all the fuss
      // in the Reset() method above. No preloading: if we tried to preload the next
      // byte after we read the very last base from the current byte, we would run out
      // of boundaries at the beginning of the basevector.
      byte = basevector::RCtable[*(ptr--)]; // rc the current 4 bases right away for reading convenience
    } else {
      // otherwise, the currently loaded byte was read (and shifted) up-to internal_offset-1
      byte >>= 2; // shift it to the current position we are going to read from
    }
    internal_offset--;      // decrement internal offset
    internal_offset &= 0x3; // rollover; if we have already read all 4 bases from the current
                            // byte, offset (next position to read) is 3 (reading backwards!);
    // invariant is held!
    //    cerr << as_base(byte & 0x3);
    return byte & 0x03; // return the lowest two bits (the base!) from the reverted current byte, 
                        // byte was shifted already
  }

 private:
  /// points to the next byte in the byte array (immediately after thge one
  /// we are currently reading)
  unsigned char * ptr;
  
  // next bit pais (base) offset to read
  unsigned char internal_offset; 

  // byte (4 bases) we are currently reading. 
  unsigned char byte;
};


#endif
