import jebl.math.Random;

import java.io.*;

/**
 * @author rambaut
 *         Date: Dec 13, 2005
 *         Time: 10:15:46 PM
 */
public class OverlapSimulator {

	private final boolean toStartCodon = true;
	private final int repeatCount;

	public OverlapSimulator(int repeatCount, double extensionProbability) {

		this.repeatCount = repeatCount;

		double[] codonFrequencies = new double[CODON_COUNTS.length];

		double sum = 0.0;
		for (int i = 0; i < CODON_COUNTS.length; i++) {
			sum += (double)CODON_COUNTS[i];
		}

		codonFrequencies[0] = ((double)CODON_COUNTS[0]) / sum;
		for (int i = 1; i < CODON_COUNTS.length; i++) {
			codonFrequencies[i] = codonFrequencies[i - 1] + (((double)CODON_COUNTS[i]) / sum);
		}

        double[] lengthFrequencies = new double[GENE_LENGTHS.length];

        sum = 0.0;
        for (int i = 0; i < GENE_LENGTHS.length; i++) {
            sum += (double)GENE_LENGTHS[i];
        }

        lengthFrequencies[0] = ((double)GENE_LENGTHS[0]) / sum;
        for (int i = 1; i < GENE_LENGTHS.length; i++) {
            lengthFrequencies[i] = lengthFrequencies[i - 1] + (((double)GENE_LENGTHS[i]) / sum);
        }

		int[] lengthsDownPlus1 = new int[repeatCount];
		simulateDownstreamExtension(lengthsDownPlus1, codonFrequencies, 1, extensionProbability);

		int[] lengthsDownMinus1 = new int[repeatCount];
		simulateDownstreamExtension(lengthsDownMinus1, codonFrequencies, -1, extensionProbability);

		int[] lengthsUpPlus1 = new int[repeatCount];
		simulateUpstreamExtension(lengthsUpPlus1, codonFrequencies, 1, toStartCodon, extensionProbability);

		int[] lengthsUpMinus1 = new int[repeatCount];
		simulateUpstreamExtension(lengthsUpMinus1, codonFrequencies, -1, toStartCodon, extensionProbability);

		int[] lengthsInternalPlus1 = new int[repeatCount];
		simulateMaxInternal(lengthFrequencies, lengthsInternalPlus1, codonFrequencies, 1, extensionProbability);

		int[] lengthsInternalMinus1 = new int[repeatCount];
		simulateMaxInternal(lengthFrequencies, lengthsInternalMinus1, codonFrequencies, 2, extensionProbability);

		System.out.println("Down+1\tDown-1\tUp+1\tUp-1\tInt+1\tInt-1");
		for (int i = 0; i < repeatCount; i++) {
			System.out.println(
					lengthsDownPlus1[i] + "\t" + lengthsDownMinus1[i] + "\t" +
							lengthsUpPlus1[i] + "\t" + lengthsUpMinus1[i] + "\t" +
							lengthsInternalPlus1[i] + "\t" + lengthsInternalMinus1[i]);
		}

		try {
			PrintWriter writer = new PrintWriter("term_sim1");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsDownPlus1[i]);
			}
			writer.close();

			writer = new PrintWriter("term_sim2");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsDownMinus1[i]);
			}
			writer.close();

			writer = new PrintWriter("term_sim_up1");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsUpPlus1[i]);
			}
			writer.close();

			writer = new PrintWriter("term_sim_up2");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsUpMinus1[i]);
			}
			writer.close();

			writer = new PrintWriter("int_sim1");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsInternalPlus1[i]);
			}
			writer.close();

			writer = new PrintWriter("int_sim2");
			for (int i = 0; i < repeatCount; i++) {
				writer.println(lengthsInternalMinus1[i]);
			}
			writer.close();

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	private void simulateDownstreamExtension(int[] lengths, double[] codonFrequencies, int frame, double extensionProbability) {
		int k = 0;
		while (k < repeatCount) {
			boolean done = false;
			String lastCodon = drawCodon(codonFrequencies);

			int length = 0;
			while (!done) {
				String codon = drawCodon(codonFrequencies);

				String overlappingCodon = null;

				if (frame == 1) {
					overlappingCodon = "" + lastCodon.charAt(1) + lastCodon.charAt(2) + codon.charAt(0);
				} else if (frame == -1) {
					overlappingCodon = "" + lastCodon.charAt(2) + codon.charAt(0) + codon.charAt(1);
				}


				if (overlappingCodon.equals("TAA") ||
						overlappingCodon.equals("TAG") ||
						overlappingCodon.equals("TGA") ) {
					double r = Random.nextDouble();
					if (r > extensionProbability) {
						done = true;
					}
				} else {
					length++;
					lastCodon = codon;
				}
			}

			lengths[k] = length;
			k++;
		}
	}

	private void simulateUpstreamExtension(int[] lengths, double[] codonFrequencies, int frame, boolean toStartCodon, double extensionProbability) {
		int k = 0;
		while (k < repeatCount) {
			boolean done = false;
			boolean startCodon = false;
			String lastCodon = drawCodon(codonFrequencies);

			int length = 0;
			while (!done) {
				String codon = drawCodon(codonFrequencies);

				String overlappingCodon = null;

				if (frame == 1) {
					overlappingCodon = "" + codon.charAt(1) + codon.charAt(2) + lastCodon.charAt(0);
				} else if (frame == -1) {
					overlappingCodon = "" + codon.charAt(2) + lastCodon.charAt(0) + lastCodon.charAt(1);
				}

				if (overlappingCodon.equals("TAA") ||
						overlappingCodon.equals("TAG") ||
						overlappingCodon.equals("TGA") ) {
					done = true;
				} else if (toStartCodon && overlappingCodon.equals("ATG")) {
					double r = Random.nextDouble();
					if (r > extensionProbability) {
						startCodon = true;
						done = true;
					}
				}

				if (!done) {
					length++;
					lastCodon = codon;
				}
			}

			if ((toStartCodon && startCodon) || !toStartCodon) {
				lengths[k] = length;
				k++;
			}
		}
	}

	private void simulateMaxInternal(double[] lengthFrequencies, int[] lengths, double[] codonFrequencies, int frame, double extensionProbability) {
        for (int k = 0; k < repeatCount; k++) {

            int maxCodonCount = drawGeneLength(lengthFrequencies, GENE_LENGTHS_STEP_SIZE);
            String lastCodon = drawCodon(codonFrequencies);

			int codonCount = 0;
			int maxLength = 0;
			while (codonCount < maxCodonCount) {

				int length = 0;
				boolean stop = false;
				while (!stop && codonCount < maxCodonCount) {
					String codon = drawCodon(codonFrequencies);

					String overlappingCodon = null;

					if (frame == 1) {
						overlappingCodon = "" + lastCodon.charAt(1) + lastCodon.charAt(2) + codon.charAt(0);
					} else if (frame == 2) {
						overlappingCodon = "" + lastCodon.charAt(2) + codon.charAt(0) + codon.charAt(1);
					} else if (frame == -1) {
						overlappingCodon = "" + lastCodon.charAt(1) + lastCodon.charAt(0) + codon.charAt(2);
					} else if (frame == -2) {
						overlappingCodon = "" + lastCodon.charAt(0) + codon.charAt(2) + codon.charAt(1);
					}

					if (overlappingCodon.equals("TAA") ||
							overlappingCodon.equals("TAG") ||
							overlappingCodon.equals("TGA") ) {
						double r = Random.nextDouble();
						if (r > extensionProbability) {
							stop = true;
						}
					} else {
						length++;
					}
					lastCodon = codon;
					codonCount++;
				}
				if (stop && length > maxLength) {
					maxLength = length;
				}
			}
			lengths[k] = maxLength;
		}
	}

	private String drawCodon(double[] frequencies) {
		double r = Random.nextDouble();

		for (int i = 0; i < frequencies.length; i++) {
			if (r < frequencies[i]) {
				return CODONS[i];
			}
		}

		throw new RuntimeException("Error drawing codon");
	}

    private int drawGeneLength(double[] frequencies, int stepSize) {
        double r = Random.nextDouble();

        int length = stepSize;
        for (int i = 0; i < frequencies.length; i++) {
            if (r < frequencies[i]) {
                return length;
            }
            length += stepSize;
        }

        throw new RuntimeException("Error drawing codon");
    }

	public static void main(String[] argv) {
		if (argv.length != 2) {
			System.out.println("Usage: overlap <rep_count> <extension_prob>");
			return;
		}
		int repeatCount = Integer.parseInt(argv[0]);
		double extensionProbability = Double.parseDouble(argv[1]);
		new OverlapSimulator(repeatCount, extensionProbability);
	}

	private final static int[] CODON_COUNTS = new int[] {
			42271, // AAA
			26262, // AAC
			38443, // AAG
			35400, // AAT
			29107, // ACA
			19266, // ACC
			11402, // ACG
			25978, // ACT
			23354, // AGA
			13636, // AGC
			15845, // AGG
			18507, // AGT
			24297, // ATA
			23958, // ATC
			35582, // ATG
			31838, // ATT
			26737, // CAA
			14203, // CAC
			21251, // CAG
			17637, // CAT
			22215, // CCA
			13045, // CCC
			8799,  // CCG
			19247, // CCT
			8524,  // CGA
			8756,  // CGC
			6031,  // CGG
			9913,  // CGT
			15848, // CTA
			18283, // CTC
			20101, // CTG
			21701, // CTT
			42394, // GAA
			29573, // GAC
			36664, // GAG
			43985, // GAT
			28581, // GCA
			21067, // GCC
			12355, // GCG
			30020, // GCT
			26042, // GGA
			15389, // GGC
			16297, // GGG
			23333, // GGT
			16510, // GTA
			19397, // GTC
			27656, // GTG
			29393, // GTT
			//602, // TAA
			21323, // TAC
			//444, // TAG
			26070, // TAT
			23497, // TCA
			14685, // TCC
			8637,  // TCG
			21902, // TCT
			//460, // TGA
			11155, // TGC
			18844, // TGG
			14494, // TGT
			22480, // TTA
			25700, // TTC
			27622, // TTG
			31316, // TTT
	};

	private final String[] CODONS = new String[] {
			"AAA",
			"AAC",
			"AAG",
			"AAT",
			"ACA",
			"ACC",
			"ACG",
			"ACT",
			"AGA",
			"AGC",
			"AGG",
			"AGT",
			"ATA",
			"ATC",
			"ATG",
			"ATT",
			"CAA",
			"CAC",
			"CAG",
			"CAT",
			"CCA",
			"CCC",
			"CCG",
			"CCT",
			"CGA",
			"CGC",
			"CGG",
			"CGT",
			"CTA",
			"CTC",
			"CTG",
			"CTT",
			"GAA",
			"GAC",
			"GAG",
			"GAT",
			"GCA",
			"GCC",
			"GCG",
			"GCT",
			"GGA",
			"GGC",
			"GGG",
			"GGT",
			"GTA",
			"GTC",
			"GTG",
			"GTT",
			//"TAA",
			"TAC",
			//"TAG",
			"TAT",
			"TCA",
			"TCC",
			"TCG",
			"TCT",
			//"TGA",
			"TGC",
			"TGG",
			"TGT",
			"TTA",
			"TTC",
			"TTG",
			"TTT",
	};

    private final static int GENE_LENGTHS_STEP_SIZE = 100;

    private final static int[] GENE_LENGTHS = new int[] {
			3,	//100
			111,	//200
			145,	//300
			190,	//400
			123,	//500
			175,	//600
			207,	//700
			166,	//800
			100,	//900
			101,	//1000
			91,	//1100
			75,	//1200
			57,	//1300
			62,	//1400
			47,	//1500
			79,	//1600
			90,	//1700
			50,	//1800
			45,	//1900
			54,	//2000
			45,	//2100
			46,	//2200
			39,	//2300
			41,	//2400
			29,	//2500
			28,	//2600
			35,	//2700
			13,	//2800
			20,	//2900
			30,	//3000
			15,	//3100
			23,	//3200
			12,	//3300
			28,	//3400
			23,	//3500
			16,	//3600
			15,	//3700
			20,	//3800
			20,	//3900
			14,	//4000
			14,	//4100
			7,	//4200
			9,	//4300
			15,	//4400
			2,	//4500
			2,	//4600
			10,	//4700
			6,	//4800
			14,	//4900
			9,	//5000
			9,	//5100
			6,	//5200
			7,	//5300
			11,	//5400
			10,	//5500
			13,	//5600
			18,	//5700
			11,	//5800
			7,	//5900
			14,	//6000
			4,	//6100
			5,	//6200
			6,	//6300
			11,	//6400
			18,	//6500
			20,	//6600
			26,	//6700
			21,	//6800
			11,	//6900
			8,	//7000
			7,	//7100
			3,	//7200
			6,	//7300
			5,	//7400
			9,	//7500
			6,	//7600
			0,	//7700
			4,	//7800
			2,	//7900
			0,	//8000
			0,	//8100
			0,	//8200
			0,	//8300
			1,	//8400
			1,	//8500
			4,	//8600
			8,	//8700
			2,	//8800
			0,	//8900
			3,	//9000
			5,	//9100
			16,	//9200
			15,	//9300
			4,	//9400
			4,	//9500
			2,	//9600
			3,	//9700
			1,	//9800
			2,	//9900
			0,	//10000
			5,	//10100
			9,	//10200
			13,	//10300
			4,	//10400
			2,	//10500
			0,	//10600
			0,	//10700
			2,	//10800
			0,	//10900
			0,	//11000
			0,	//11100
			0,	//11200
			0,	//11300
			0,	//11400
			0,	//11500
			0,	//11600
			4,	//11700
			0,	//11800
			4,	//11900
			2,	//12000
			1,	//12100
			1,	//12200
			1,	//12300
			0,	//12400
			0,	//12500
			0,	//12600
			0,	//12700
			0,	//12800
			0,	//12900
			0,	//13000
			0,	//13100
			2,	//13200
			0,	//13300
			0,	//13400
			2,	//13500
			0,	//13600
			0,	//13700
			0,	//13800
			1,	//13900
			0,	//14000
			0,	//14100
			0,	//14200
			0,	//14300
			0,	//14400
			0,	//14500
			0,	//14600
			0,	//14700
			0,	//14800
			0,	//14900
			0,	//15000
			0,	//15100
			0,	//15200
			0,	//15300
			0,	//15400
			0,	//15500
			0,	//15600
			0,	//15700
			0,	//15800
			0,	//15900
			0,	//16000
			0,	//16100
			0,	//16200
			0,	//16300
			0,	//16400
			0,	//16500
			0,	//16600
			0,	//16700
			0,	//16800
			0,	//16900
			0,	//17000
			0,	//17100
			0,	//17200
			0,	//17300
			0,	//17400
			0,	//17500
			0,	//17600
			0,	//17700
			0,	//17800
			0,	//17900
			0,	//18000
			0,	//18100
			0,	//18200
			0,	//18300
			0,	//18400
			0,	//18500
			0,	//18600
			0,	//18700
			0,	//18800
			0,	//18900
			0,	//19000
			0,	//19100
			0,	//19200
			0,	//19300
			0,	//19400
			0,	//19500
			0,	//19600
			0,	//19700
			0,	//19800
			2,	//19900
			0,	//20000
			1,	//20100
			1,	//20200
			1,	//20300
			1,	//20400
			0,	//20500
			0,	//20600
			0,	//20700
			0,	//20800
			0,	//20900
			0,	//21000
			0,	//21100
			0,	//21200
			3,	//21300
			0,	//21400
			0,	//21500
			3,	//21600
			0,	//21700
			0,	//21800
			0	//21900
	};
}
